/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Setup.cc
 *
 * Simulation utilities.
 */

#include <stdlib.h>		// exit
#include <iostream.h>
#include "glo.h"		// HASHSIZE, MAXBUF
#include "intrface.h"		// p_* types
#include "Error.h"		// Sim_errno, etc...
#include "Event.h"
#include "Symtab.h"
#include "TWheel.h"
#include "Base.h"
#include "PExpr.h"
#include "PTypes.h"
#include "Systask.h"

EvntQueue<Stmts *> eventqueue;		// Event queue for simulation
TimeWheel<Stmts *> timewheel;		// Time wheel for simulation

void
p_perror(char *type, char *mesg, char *word, char *fn, int ln)
	{
	String s = String(mesg);
	s = s + String(": ");
	s = s + String(word);
	if (strcmp(type, "symbol") == 0)
		Sim_errno = SE_PSYMBOL;
	else if (strcmp(type, "keyword") == 0)
		Sim_errno = SE_PKEYWORD;
	else
		Sim_errno = SE_PEOF;
	Sim_filename = String(fn);
	Sim_lineno = ln;
	Sim_perror(SEF_EXIT | SEF_COMP, "%s", (char *)s);
	}

void
Sim_init(char *pn)
	{
	String s(pn);
	// Setup variables for error reporting.
	Sim_programname = s;
	Sim_lineno = -1;
	Sim_errno = SE_NONE;

	// Symbol table for simulation
	symboltable.Init(HASHSIZE);

	// Our global system tasks and functions.
	int glo = 0;
	Stack<int> scope;
	Push(scope, glo);
	SysTaskWrite stw;
	stw.Sim_setup(scope);
	SysTaskFinish stf;
	stf.Sim_setup(scope);
	SysFunctionTime sft;
	sft.Sim_setup(scope);
	}

void
Sim_debug(char *)
	{ cout << symboltable << endl; }

void
parsed_module(p_Module *m)
	{
	// After we parsed a complete module, we need to store it into
	// the symbol table for later use.  Currently, we limit the
	// name of the top-level module to "main".  This is because we
	// do not know how to find the top-level module in the symbol
	// table.
	Module *mod = (Module *)m->data;
	m->data = NULL;
	RangeId &id = Id(*mod);
	String name = String(id);
	List<Port *> *plst = new List<Port *>(PortLst(*mod));
	List<ModuleItem *> *itm = new List<ModuleItem *>(Items(*mod));

	// Now enter the module into the symbol table.  The scope should
	// always be global.  There are no nested module definitions.
	STnode *newmod = new STnode(0, name);
	newmod->filename = mod->filename;
	newmod->lineno = mod->lineno;
	newmod->SetType(NT_MOD);
	if (Size(*plst) > 0)
		newmod->SetPortLst(plst);
	else
		delete plst;
	if (Size(*itm) > 0)
		newmod->SetModuleItemLst(itm);
	else
		delete itm;
	// Enter name into symbol table.
	NewSymbol(newmod);
	delete mod;
	free(m);
	}

// Function to call a member function in <eventqueue>.  I don't know
// if we should use a member function pointer.  Member function
// pointers require the class to be known, this method is more
// generic.
void
stmtstrigger(Stmts *st)
	{ st->Sim_trigger(st); }
void
eventtrigger(void)
	{ eventqueue.Sim_trigger(stmtstrigger); }

void
Sim_start(void)
	{
	// Start work.  First we need to setup for simulation.  This means
	// grabbing the top-level module from the symbol table and call
	// Sim_setup for this module.
	HashValue idx = hash(String("main"), 0);
	STnode *toplevel = symboltable[idx];
	if (toplevel == NULL)
		{
		Sim_errno = SE_NTOPLVL;
		Sim_filename = String("None");
		Sim_lineno = -1;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "main");
		}
	Stack<int> scope;
	int thisscope = symboltable.Scope()++;
	Push(scope, thisscope);
	toplevel->Sim_setup(scope, String("main"));

#if defined(DEBUG_ST)
	cout << "DEBUG_ST: Symbol Table:" << endl << symboltable
		<< endl << endl;
#endif
	cout << "Begin simulation:" << endl << endl;

	timewheel.Sim_trigger(eventtrigger);

	cout << endl << "No more events, ending simulation." << endl << endl;
	}

void
ModuleInstan::Sim_setup(Stack<int> scope, STnode *)
	{
	modname.Sim_setup(scope);
	STnode *mod = symboltable[modname.idx];
	ModuleInstance *mi;
	for (int i = 0; i < Size(instances); i++)
		{
		mi = instances[i];
		mi->Sim_setup(scope, mod);
		}
	}

void
port_setup(int localscope, String &instname, String &filename, int lineno,
		List<Port *> *plst,
		List<PortConnection *> *pclst)
	{
	HashValue newidx, *idx, *tmpidx;
	STnode *newport, *orig;
	Port *port;
	PortConnection *pc;
	unsigned long ms, ls;
	for (int i = 0; i < Size(*plst); i++)
		{
		// Create a new symbol node for this port.
		port = (*plst)[i];
		newport = new STnode(localscope, String(*port));
		newport->filename = filename;
		newport->lineno = lineno;
		newport->SetType(NT_WIRE);
		newidx = NewSymbol(newport);

		// Attach incoming port connection.
		pc = (*pclst)[i];
		idx = pc->Index();
		if (idx == NULL)
			continue;
		tmpidx = new HashValue(*idx);
		newport->SetPortIndex(tmpidx);
		if (pc->GetRange(ms, ls))
			newport->SetPortRange(ms, ls);

		// Update incoming symbol so that changes
		// get propagated to us.
		orig = symboltable[*idx];
		orig->AddPropagate(newidx);
		}
	}

void
ModuleInstance::Sim_setup(Stack<int> &scope, STnode *mod)
	{
	int localscope = Pop(scope);
	Push(scope, localscope);
	STnode *newinstan = new STnode(localscope, String(name));
	newinstan->filename = filename;
	newinstan->lineno = lineno;
	newinstan->SetType(NT_INSTAN);
	NewSymbol(newinstan);
	// Get the index of the port connections.
	PortConnection *pc;
	for (int i = 0; i < Size(connections); i++)
		{
		pc = connections[i];
		pc->Sim_setup(scope);
		}

	// Setup the rest of the module items.
	mod->Sim_setup(scope, String(name), filename, lineno,
		&connections, port_setup);
	}

void
AlwaysStmt::Sim_setup(Stack<int> scope, STnode *)
	{
	// Make a copy because we only need the statement, everything
	// else will be deleted.  Same goes for other module items.
	Stmts *tmp = stmt->copy();
	// This statement is in an always statement.
	// Set flag before doing anything...
	tmp->always = TRUE;
	tmp->Sim_setup(scope);

	// Call Sim_setup of their types and have it do the work.
	if (tmp->dec != NULL)
		tmp->dec->Sim_setup(scope, tmp);
	}

void
InitialStmt::Sim_setup(Stack<int> scope, STnode *)
	{
	Stmts *tmp = stmt->copy();
	// Setup each statement then append to time wheel.
	tmp->Sim_setup(scope);

#if defined(DEBUG_INIT)
	cout << "Appending initial statement..." << endl;
#endif

	TWnode<Stmts *> newnode(tmp, 0);
	timewheel += newnode;
	}

void
ftsetup(Stack<int> scope, STnode *newft, Stmts *st, List<TFDecl *> &decl,
		String &filename, int lineno)
	{
	// Increment the scope for local data.
	int localscope = symboltable.Scope()++;
	Push(scope, localscope);

	// Must setup declarations first, because the statement might
	// need variables declared locally.
	for (int i = 0; i < Size(decl); i++)
		decl[i]->Sim_setup(scope, newft);

	// The name of the function is the return value, so must setup
	// statement after entering the function name into the symbol
	// table.
	st->Sim_setup(scope, FALSE, NS_DELAY);

	// Check variables are complete.  <decl> above should have set
	// this up if there are IO variables.
	List<HashValue> *locvars = newft->IOVar();
	if (locvars == NULL)
		return;

	Reset(*locvars);
	HashValue idx;
	STnode *node;
	while (Data(*locvars, idx))
		{
		node = symboltable[idx];
		if (!node->Complete())
			{
			Sim_seterror(SE_COMPLETE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP,
				"%s", (char *)String(*node));
			}
		}
	}

void
Function::Sim_setup(Stack<int> scope, STnode *)
	{
	Number lnum, rnum;
	GetRangeData(retsize, lnum, rnum);
	int ms = lnum;
	int ls = rnum;
	Stmts *st = stmt->copy();
	int thisscope = Pop(scope);
	Push(scope, thisscope);
	STnode *newfunc = new STnode(thisscope, String(name));
	newfunc->filename = filename;
	newfunc->lineno = lineno;
	newfunc->SetType(NT_FUNC);
	newfunc->SetStorage(ms, ls);
	newfunc->SetStmts(st);
	NewSymbol(newfunc);
	ftsetup(scope, newfunc, st, decl, filename, lineno);
	}

void
Task::Sim_setup(Stack<int> scope, STnode *)
	{
	Stmts *st = stmt->copy();
	int thisscope = Pop(scope);
	Push(scope, thisscope);
	STnode *newtask = new STnode(thisscope, String(name));
	newtask->filename = filename;
	newtask->lineno = lineno;
	newtask->SetType(NT_TASK);
	newtask->SetStmts(st);
	NewSymbol(newtask);
	ftsetup(scope, newtask, st, decl, filename, lineno);
	}

void
check_monitor(List< Events<Stmts *> *> *evntlst,
		Number &before, Number &after)
	{
	List< Events<Stmts *> *> &lst = *evntlst;
	Events<Stmts *> *ev;
	for (int i = 0; i < Size(lst); i++)
		{
		ev = lst[i++];
		if (ev->trigger != TRIG_CHNG)
			{
			if (before[ev->bit] == LO && after[ev->bit] == HI)
				{
				ev->QueueEvnt(eventqueue,
					TrigType(TRIG_CHNG | TRIG_POS));
				}
			else if (before[ev->bit] == HI && after[ev->bit] == LO)
				{
				ev->QueueEvnt(eventqueue,
					TrigType(TRIG_CHNG | TRIG_NEG));
				}
				// Otherwise, this variable didn't change...
			}
		else if (before != after)	// Did we change?
			ev->QueueEvnt(eventqueue, TRIG_CHNG);
		}
	}

void
net_decl(Stack<int> &scope, String &filename, int lineno,
		Range &range, List<RangeId *> &ids, NodeType type)
	{
	Number lnum, rnum;
	GetRangeData(range, lnum, rnum);
	int ms = lnum;
	int ls = rnum;

	// Enter each id into the symbol table.
	int thisscope = Pop(scope);
	Push(scope, thisscope);
	// Search only local variables.
	Stack<int> localscope;
	Push(localscope, thisscope);

	// Add or update
	RangeId *id;
	STnode *st;
	HashValue idx;
	Reset(ids);
	for (int i = 0; i < Size(ids); i++)
		{
		Data(ids, id);
		idx = symboltable.Lookup(String(*id), localscope);
		st = symboltable[idx];
		if (st == NULL)
			{
			st = new STnode(thisscope, String(*id));
			st->filename = filename;
			st->lineno = lineno;
			NewSymbol(st);
			}
		st->SetType(type);
		st->SetStorage(ms, ls);
		// Both wires and registers can be monitored.
		st->SetOnAssign(check_monitor);
		}
	}

void
WireDecl::Sim_setup(Stack<int> scope, STnode *)
	{ net_decl(scope, filename, lineno, range, ids, NT_WIRE); }

void
RegDecl::Sim_setup(Stack<int> scope, STnode *)
	{ net_decl(scope, filename, lineno, range, ids, NT_REG); }

void
IODecl_setup(Stack<int> scope, STnode *node, IODir dir,
		Range &range, List<RangeId *> &ids,
		String &filename, int lineno)
	{
	// First, get the range.
	Number lnum, rnum;
	GetRangeData(range, lnum, rnum);
	int ms = lnum;
	int ls = rnum;

	// Save a list of IO variables to check.
	int thisscope = Pop(scope);
	Push(scope, thisscope);
	List<HashValue> *locvars = node->IOVar();
	if (locvars == NULL)
		{
		locvars = new List<HashValue>;
		node->SetIOVar(locvars);
		}

	// Temporary scope for symbol table lookup.
	Stack<int> localscope;
	Push(localscope, thisscope);

	RangeId *id;
	STnode *st;
	HashValue idx;
	Reset(ids);
	for (int i = 0; i < Size(ids); i++)
		{
		// Lookup each id.
		Data(ids, id);
		idx = symboltable.Lookup(String(*id), localscope);
		st = symboltable[idx];
		if (st == NULL)
			{
			st = new STnode(thisscope, String(*id));
			st->filename = filename;
			st->lineno = lineno;
			// IO_IN and IO_INOUT are wires.
			if (dir != IO_OUT)
				st->SetType(NT_WIRE);
			idx = NewSymbol(st);
			}
		st->SetDir(dir);
		st->SetOnAssign(check_monitor);
		// Output declarations also need a register declaration.
		if (dir != IO_OUT)
			st->SetStorage(ms, ls);
		*locvars += idx;
		}
	}

void
InputDecl::Sim_setup(Stack<int> scope, STnode *node)
	{ IODecl_setup(scope, node, IO_IN, range, ids, filename, lineno); }

void
OutputDecl::Sim_setup(Stack<int> scope, STnode *node)
	{ IODecl_setup(scope, node, IO_OUT, range, ids, filename, lineno); }

void
InoutDecl::Sim_setup(Stack<int> scope, STnode *node)
	{ IODecl_setup(scope, node, IO_INOUT, range, ids, filename, lineno); }

void
check_notsupported(int ns, DelayEvntCtl *dec, String &filename, int lineno)
	{
	if ((ns & NS_DELAY) != 0 && dec != NULL)
		{
		Sim_seterror(SE_SUPPORT, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "#num");
		}
	}
void
AssignStmt::Sim_setup(Stack<int> &scope, Bool, int ns)
	{
	// Can we allow delays in this statement?
	check_notsupported(ns, dec, filename, lineno);
	lval.Sim_setup(scope);
	rval->Sim_setup(scope);
	}

void
SeqBlkStmt::Sim_setup(Stack<int> &scope, Bool lp, int ns)
	{
	check_notsupported(ns, dec, filename, lineno);
	if (lp)
		loop = TRUE;			// We are part of a loop.
	for (int i = 0; i < Size(stmts); i++)
		stmts[i]->Sim_setup(scope, lp, ns);
	}

void
IfStmt::Sim_setup(Stack<int> &scope, Bool lp, int ns)
	{
	check_notsupported(ns, dec, filename, lineno);
	expr->Sim_setup(scope);
	stmt->Sim_setup(scope, lp, ns);
	}

void
IfElseStmt::Sim_setup(Stack<int> &scope, Bool lp, int ns)
	{
	check_notsupported(ns, dec, filename, lineno);
	expr->Sim_setup(scope);
	stmt1->Sim_setup(scope, lp, ns);
	stmt2->Sim_setup(scope, lp, ns);
	}

void
ForStmt::Sim_setup(Stack<int> &scope, Bool, int ns)
	{
	check_notsupported(ns, dec, filename, lineno);
	assign1.Sim_setup(scope);
	assign2.Sim_setup(scope);
	expr->Sim_setup(scope);
	// Whether <lp> was true or not, we are now true...
	stmt->Sim_setup(scope, TRUE, NS_DELAY);
  	}

void
CaseItem::Sim_setup(Stack<int> &scope, Bool lp)
	{
	for (int i = 0; i < Size(expr); i++)
		expr[i]->Sim_setup(scope);
	stmt->Sim_setup(scope, lp, NS_DELAY);
	}

void
CaseStmt::Sim_setup(Stack<int> &scope, Bool lp, int ns)
	{
	check_notsupported(ns, dec, filename, lineno);
	CaseItem *tmp;
	expr->Sim_setup(scope);
	int defaultidx = -1;
	for (int i = 0; i < Size(ci); i++)
		{
		tmp = ci[i];
		tmp->Sim_setup(scope, lp);
		if (IsDefault(*tmp))
			{
			if (def != NULL)
				{
				// Print out error mesg and quit.
				Sim_seterror(SE_MULTDEF, filename, lineno);
				Sim_perror(SEF_EXIT | SEF_SETUP,
					"%s", "default:");
				}
			def = tmp;
			defaultidx = i;
			}
		}
	// We must delete the default case from the list.  Note, this
	// doesn't delete the default CaseItem, just the node.  We could
	// not delete it before because we can not modify the list while
	// in a for loop.
	if (defaultidx >= 0)
		Delete(ci, defaultidx);
	}

void
DelayNum::Sim_setup(Stack<int> &, Stmts *st)
	{
	TWnode<Stmts *> newnode(st, amt);
	timewheel += newnode;
	}

void
DelayRangeId::Sim_setup(Stack<int> &scope, Stmts *st)
	{
	id.Sim_setup(scope);
	STnode *node = symboltable[id.idx];
	Number amt = node->Sim_evaluate();
	TWnode<Stmts *> newnode(st, amt);
	timewheel += newnode;
	}

void
EvntExpression::Sim_setup(Stack<int> &scope, Events<Stmts *> *es)
	{
	// Get the symbol table index for our variable.
	expr->Sim_setup(scope);
	HashValue *idx = expr->Index();
	if (idx == NULL)
		{
		Sim_seterror(SE_TYPE, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "EvntExpression");
		}

	// The symbol table contains a list of statements to be executed
	// when our variable is changed.  Add our statement <st> to
	// this list.
	List< Events<Stmts *> *> *lst;
	STnode *tmp = symboltable[*idx];
	if (tmp->Type() != NT_WIRE && tmp->Type() != NT_REG)
		{
		Sim_seterror(SE_TYPE, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "EvntExpression");
		}
	else
		{
		lst = tmp->EventLst();
		if (lst == NULL)
			{
			lst = new List< Events<Stmts *> *>;
			tmp->SetEventLst(lst);
			}
		}

	// Include our trigger type in this event.
	es->trigger = TrigType(es->trigger | edge);

	/*
	 * If the trigger type is edge type, the evaluated data must be a
	 * single bit.  Thus we need to test it now.
	 */
	if (es->trigger != TRIG_CHNG)
		{
		Number lnum, rnum;
		// <expr> should be a range id if the above test passed.
		RangeId &id = *(RangeId *)expr;
		GetRangeData(Range(id), lnum, rnum);
		// If we do not specify a size, Sim_evaluate() will return
		// the actual number, so <tmpnum> will have the correct
		// data.  Even though <tmpnum> is not a reference.
		Number tmpnum(tmp->Sim_evaluate());
		// Convert to unsigned long for easier comparison.
		unsigned long sb = lnum;
		if ((lnum - rnum) != 0 || sb < StartBit(tmpnum))
			{
			char buf[MAXBUF];
			sprintf(buf, "[%ld:%ld]",
				(unsigned long)lnum, (unsigned long)rnum);
			Sim_seterror(SE_RANGE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP, "%s", buf);
			}
		es->bit = lnum;		// Save the bit being monitored.
		}
	*lst += es;
	}

void
OredEvntExpression::Sim_setup(Stack<int> &scope, Stmts *st)
	{
	// Only one event can exist for each "always" statement.
	// Otherwise, we will trigger two events when we should
	// only trigger one.  Add this event to the list for all
	// events in this expression.
	Events<Stmts *> *es = new Events<Stmts *>(st);
	for (int i = 0; i < Size(vars); i++)
		vars[i]->Sim_setup(scope, es);
	}

void
Lvalue::Sim_setup(Stack<int> &scope)
	{
	id.Sim_setup(scope);

	// Setup for RangeId should have tested for valid index.  So
	// no need check.
	STnode *node = symboltable[id.idx];

	// Lvalues are wires, registers, and function names.
	// Test for the rest, since there are less of them.
	if (node->Type() == NT_TASK || node->Type() == NT_MOD)
		{
		char buf[MAXBUF];
		sprintf(buf, "%s(%s)", (char *)String(id),
			(char *)node->TypeString());
		Sim_seterror(SE_NLVAL, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", buf);
		}
	if (!node->Complete())
		{
		Sim_seterror(SE_COMPLETE, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(id));
		}
	}

void
RangeId::Sim_setup(Stack<int> &scope)
	{
	idx = symboltable.Lookup(name, scope);
	if (idx.scope < 0)
		{
		Errno ste = SE_NONE;
		switch (idx.scope)
			{
			case STE_INIT: ste = SE_STINIT; break;
			case STE_FULL: ste = SE_STFULL; break;
			case STE_DUP: ste = SE_STDUP; break;
			case STE_MISS: ste = SE_STDEF; break;
			default: break;
			}
		Sim_seterror(ste, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
		}

	// Check for valid range...
	if (flag)
		{
		STnode *node = symboltable[idx];
		Number *n = node->Storage();
		if (n == NULL)
			{
			Sim_seterror(SE_COMPLETE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
			}
		Number reg(*n);
		Number lnum, rnum;
		GetRangeData(range, lnum, rnum);
		unsigned long ms = lnum;
		unsigned long ls = rnum;
		if (Size(reg(ms,ls)) == 0)
			{
			char buf[MAXBUF];
			sprintf(buf, "[%ld:%ld]", ms, ls);
			Sim_seterror(SE_IDXORDER, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP, "%s", buf);
			}
		}
	}

void
NotOp::Sim_setup(Stack<int> &scope)
	{ exp->Sim_setup(scope); }

void
InvertOp::Sim_setup(Stack<int> &scope)
	{ exp->Sim_setup(scope); }

void
AddOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
SubOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
EqEqComp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
NotEqComp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
OrOrOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
AndAndOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LogOrOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LogAndOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LogLShiftOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LogRShiftOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
GrtThanOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
GrtEqOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LesThanOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}

void
LesEqOp::Sim_setup(Stack<int> &scope)
	{
	exp1->Sim_setup(scope);
	exp2->Sim_setup(scope);
	}
