/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Symtab.cc
 *
 * Symbol table utilities.
 */

#include <iostream.h>
#include "Bool.h"
#include "String.h"
#include "List.h"
#include "Event.h"
#include "Error.h"
#include "Base.h"
#include "Symtab.h"

/* Global variables. */
SymTab symboltable;

/* Static table size. */
unsigned long TABSIZE = 0;

String NodeTypeString[] =
	{
	String("None"),
	String("Wire"),
	String("Register"),
	String("Function"),
	String("Task"),
	String("Module")
	};

/*
 * Utility function to add new symbols into the symbol table.
 */
HashValue
NewSymbol(STnode *n)
	{
	HashValue hv;
	switch (symboltable.Enter(n, hv))
		{
		case STE_DUP:
			{
			Sim_seterror(SE_STDUP, n->filename, n->lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP,
				"%s", (char *)String(*n));
			}
		case STE_FULL:
			{
			Sim_seterror(SE_STFULL, n->filename, n->lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP,
				"%s", (char *)String(*n));
			}
		default:
			break;
		}
	return hv;
	}

/*
 * Symbol table node.
 */

void
STnode::init(void)
	{
	storage = NULL;
	portidx = NULL;
	propagate = NULL;
	monitors = NULL;
	portlst = NULL;
	moditems = NULL;
	iovars = NULL;
	expr = NULL;
	stmt = NULL;
	on_assignment = NULL;
	}

STnode::STnode()
	: complete(FALSE), ms(0UL), ls(0UL),
		scope(STE_MISS), dir(IO_NONE), type(NT_NONE),
		port_ms(0UL), port_ls(0UL), port_range(FALSE),
		filename("unknown"), lineno(-1)
	{ init(); }

STnode::STnode(int sc, String &s, IODir d, NodeType t)
	: complete(FALSE), ms(0UL), ls(0UL),
		scope(sc), dir(d), type(t), name(s),
		port_ms(0UL), port_ls(0UL), port_range(FALSE),
		filename("unknown"), lineno(-1)
	{ init(); }

STnode::STnode(STnode &n)
	: complete(n.complete), ms(n.ms), ls(n.ls),
		scope(n.scope), dir(n.dir), type(n.type), name(n.name),
		port_ms(n.port_ms), port_ls(n.port_ls),
		port_range(n.port_range),
		filename("unknown"), lineno(-1)
	{ init(); }

void
STnode::assign(Number &num, unsigned long m, unsigned long l, Bool flag)
	{
	Number before(*storage);
	if (flag)
		(*storage)(m, l) = BitVector(num);
	else
		*storage = num;

	// Handle port type.
	if (portidx != NULL && (dir == IO_OUT || dir == IO_INOUT))
		{
		// We only want to update the port if it's output or
		// inout direction.  We never update the port if it's
		// input or no direction.
		STnode *node = symboltable[*portidx];
		if (port_range)
			node->Assignment(*storage, port_ms, port_ls);
		else
			node->Assignment(*storage);
		// Even if we are a port, we must still handle the local
		// variables.  So continue;
		}
	if (propagate != NULL)
		{
		HashValue idx;
		STnode *node;
		for (int i = 0; i < Size(*propagate); i++)
			{
			idx = (*propagate)[i];
			node = symboltable[idx];
			// Only propagate to input/inout ports.
			if (node->Dir() != IO_IN && node->Dir() != IO_INOUT)
				continue;
			node->Assignment(*storage);
			}
		}

	// Now check to see whether this variable was monitored.
	// If so, add the statements in the list to the event queue.
	if (on_assignment != NULL && monitors != NULL)
		{
		Number after;
		after = *storage;
		on_assignment(monitors, before, after);
		}
	}

void
STnode::Assignment(Number &num)
	{ assign(num); }

void
STnode::Assignment(Number &num, unsigned long m, unsigned long l)
	{ assign(num, m, l, TRUE); }

ostream &
STnode::display(ostream &s)
	{
	// Move type down into lower bit field.
	String tmp = NodeTypeString[type];
	s << tmp << '(' << name << ',' << scope << ')';
	return s;
	}

String
STnode::TypeString(void)
	{ return NodeTypeString[type]; }

void
STnode::SetDir(IODir iod)
	{
	if (dir != IO_NONE)
		{
		Sim_seterror(SE_REPEAT, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
		}
	switch(iod)
		{
		case IO_IN:
			{
			dir = IO_IN;
			break;
			}
		case IO_OUT:
			{
			if (type != NT_REG && type != NT_WIRE &&
					type != NT_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("output: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			dir = IO_OUT;
			break;
			}
		case IO_INOUT:
			{
			if (type != NT_WIRE && type != NT_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("inout: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			dir = IO_INOUT;
			break;
			}
		default:
			break;
		}
	}

void
STnode::SetType(NodeType nt)
	{
	if (type != NT_NONE && type != NT_WIRE)
		{
		Sim_seterror(SE_REPEAT, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
		}
	switch(nt)
		{
		case NT_WIRE:
			{
			if (dir == IO_OUT)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("wire: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			type = NT_WIRE;
			break;
			}
		case NT_REG:
			{
			if (dir != IO_OUT && dir != IO_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("register: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			type = NT_REG;
			break;
			}
		case NT_FUNC:
			{
			if (dir != IO_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("function: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			type = NT_FUNC;
			break;
			}
		case NT_TASK:
			{
			if (dir != IO_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("task: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			type = NT_TASK;
			break;
			}
		case NT_MOD:
			{
			if (dir != IO_NONE)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				String tmp("module: ");
				tmp = tmp + name;
				Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
					(char *)tmp);
				}
			type = NT_MOD;
			complete = TRUE;
			break;
			}
		case NT_INSTAN:
			{
			type = NT_INSTAN;
			complete = TRUE;
			break;
			}
		default:
			break;
		}
	}

void
STnode::SetStorage(unsigned long m, unsigned long l)
	{
	if (type == NT_TASK || type == NT_INSTAN || type == NT_MOD)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetStorage: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	// It is ok to call SetStorage more than once.  But do not
	// allocate storage more than once.
	if (storage == NULL)
		storage = new Number(m, l);
	else if (ms != m || ls != l)
		{
		Sim_seterror(SE_RANGE, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
		}

	// Functions need an expression or a statement.
	if (type == NT_FUNC && (stmt == NULL && expr == NULL))
		return;
	complete = TRUE;
	}

void
STnode::SetPortIndex(HashValue *idx)
	{
	if (portidx != NULL)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetPortIndex: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	portidx = idx;
	}

void
STnode::SetPortRange(unsigned long m, unsigned long l)
	{
	if (portidx == NULL)
		{
		Sim_seterror(SE_TYPE, filename, lineno);
		String tmp("SetPortRange: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	port_ms = m;
	port_ls = l;
	port_range = TRUE;
	}

void
STnode::AddPropagate(HashValue &idx)
	{
	if (propagate == NULL)
		propagate = new List<HashValue>;
	(*propagate) += idx;
	}

void
STnode::SetEventLst(List< Events<Stmts *> *> *lst)
	{
	if (type != NT_REG && type != NT_WIRE)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetEventLst: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	monitors = lst;
	}

void
STnode::SetIOVar(List<HashValue> *iov)
	{
	if (type != NT_FUNC && type != NT_TASK && type != NT_MOD)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetIOVar: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	iovars = iov;
	}

void
STnode::SetPortLst(List<Port *> *plst)
	{
	if (type != NT_MOD)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetPortLst: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	portlst = plst;
	}

void
STnode::SetModuleItemLst(List<ModuleItem *> *mitm)
	{
	if (type != NT_MOD)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetModuleItemLst: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	moditems = mitm;
	}

void
STnode::SetExpression(Expression *exp)
	{
	if (type != NT_FUNC)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetExpression: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	expr = exp;
	if (storage != NULL)
		complete = TRUE;
	}

void
STnode::SetStmts(Stmts *st)
	{
	if (type != NT_FUNC && type != NT_TASK)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetStmts: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	stmt = st;
	if (type == NT_FUNC && storage == NULL)
		return;
	complete = TRUE;
	}

void
STnode::SetOnAssign(void (*handler)(List< Events<Stmts *> *> *,
		Number &, Number &))
	{
	if (type == NT_FUNC || type == NT_MOD || type == NT_INSTAN)
		{
		Sim_seterror(SE_INTERNAL, filename, lineno);
		String tmp("SetOnAssign: ");
		tmp = tmp + name;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)tmp);
		}
	on_assignment = handler;
	}

void
STnode::entry_iovars(List<Expression *> *arg)
	{
	if (arg == NULL)
		{
		Sim_seterror(SE_NARGLST, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SIM, "%s", (char *)name);
		}
	Reset(*arg);
	Reset(*iovars);
	HashValue idx;
	Expression *exp;
	STnode *node;
	// Go through the list and initialize all input variables.
	for (int i = 0; i < Size(*iovars); i++)
		{
		if (!Data(*arg, exp) || !Data(*iovars, idx))
			break;
		// Can not be null, since we created this node.
		node = symboltable[idx];
		// We do not use assignment here because this is not
		// an assignment.  We're just updating the local variables.
		*node->storage = exp->Sim_evaluate();
		}
	}

void
STnode::exit_iovars(List<Expression *> *arg)
	{
	if (arg == NULL)
		{
		Sim_seterror(SE_NARGLST, filename, lineno);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)name);
		}
	Reset(*arg);
	Reset(*iovars);
	HashValue idx;
	Expression *exp;
	STnode *them, *us;
	// Go through the list and set all output variables.
	for (int i = 0; i < Size(*iovars); i++)
		{
		if (!Data(*arg, exp) || !Data(*iovars, idx))
			break;
		// Can not be null, since we created this node.
		us = symboltable[idx];
		if (us->Dir() == IO_OUT || us->Dir() == IO_INOUT)
			{
			HashValue *index = exp->Index();
			if (index == NULL)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				Sim_perror(SEF_EXIT | SEF_SIM,
					"%s", "unknown expression");
				}
			// Can not be NULL.
			them = symboltable[*index];
			if (them->Type() != NT_REG)
				{
				Sim_seterror(SE_TYPE, filename, lineno);
				Sim_perror(SEF_EXIT | SEF_SIM,
					"%s", (char *)String(*them));
				}
			// <ms>/<ls> are not used here because it's
			// upto <them> to select which bits to keep.
			// We just send them everything we have.
			them->Assignment(*us->storage);
			}
		}
	}

/*
 * Function trigger and evaluate.
 */
void
STnode::Sim_trigger(String &fmt, List<Expression *> *arg)
	{
	int base, size;
	get_base_size(fmt, base, size);
	switch(type)
		{
		case NT_FUNC:
			{
			// Initialize local variables.
			if (iovars != NULL)
				entry_iovars(arg);
			if (stmt)
				stmt->Sim_trigger(stmt);
			if (expr)
				*storage = expr->Sim_evaluate();
			// Set output variables.
			if (iovars != NULL)
				exit_iovars(arg);
			}
		case NT_WIRE:
		case NT_REG:
			{
			// Display result.
			String str(Convert(*storage, base, size));
			cout << str;
			break;
			}
		default:
			{
			Sim_seterror(SE_TYPE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SIM, "%s", (char *)name);
			}
		}
	}

void
STnode::Sim_setup(Stack<int> scope, String &instname, String &fn, int ln,
		List<PortConnection *> *pclst,
		void (*port_setup)(int, String &, String &, int,
		List<Port *> *, List<PortConnection *> *)
		)
	{
	// Process each module item.  The scope of 'initial' and 'always'
	// procedural block is the next level, but functions and tasks
	// must increment symboltable scope again...
	if (type != NT_MOD)
		return;
	int localscope = symboltable.Scope()++;
	Push(scope, localscope);

	if (portlst != NULL && port_setup != NULL)
		{
		if (pclst == NULL)
			{
			Sim_seterror(SE_NARGLST, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
				(char *)instname);
			}
		else if (Size(*pclst) != Size(*portlst))
			{
			Sim_seterror(SE_NARGLST, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP, "%s",
				(char *)instname);
			}
		port_setup(localscope, instname, fn, ln, portlst, pclst);
		}
	else if (pclst != NULL)
		{
		Sim_seterror(SE_NARGLST, fn, ln);
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "port connection");
		}

	// Setup the module items.
	if (moditems == NULL)
		return;
	ModuleItem *itm;
	for (int i = 0; i < Size(*moditems); i++)
		{
		itm = (*moditems)[i];
		itm->Sim_setup(scope, this);
		}

	// Check the port list...
	if (iovars == NULL)
		return;

	Reset(*iovars);
	HashValue idx;
	STnode *node;
	while (Data(*iovars, idx))
		{
		node = symboltable[idx];
		if (!node->Complete())
			{
			Sim_seterror(SE_COMPLETE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SETUP,
				"%s", (char *)String(*node));
			}
		}
	}

Number
STnode::Sim_evaluate(List<Expression *> *arg)
	{
	switch(type)
		{
		case NT_FUNC:
			{
			// Initialize local variables.
			if (iovars != NULL)
				entry_iovars(arg);
			if (stmt)
				stmt->Sim_trigger(stmt);
			if (expr)
				*storage = expr->Sim_evaluate();
			// Set output variables.
			if (iovars != NULL)
				exit_iovars(arg);
			}
		case NT_WIRE:
		case NT_REG:
			{ return *storage; }
		default:
			{
			Sim_seterror(SE_TYPE, filename, lineno);
			Sim_perror(SEF_EXIT | SEF_SIM, "%s", (char *)name);
			}
		}
	// Should never get here.  But need to return something.
	String tmp("x");
	return Number(tmp);
	}

/*
 * Task trigger.
 */
int
STnode::Sim_trigger(Stmts *orig, List<Expression *> *arg)
	{
	// We need to change this so it behaves like functions.
	// Right now, we only support system tasks, so this will
	// work.
	// <orig> is not the original sequential block.  It is the
	// actual task itself.  We need the argument list.
	if (iovars != NULL)
		entry_iovars(arg);
	stmt->Sim_trigger(orig);
	if (iovars != NULL)
		exit_iovars(arg);
	return TRUE;
	}

/*
 * Symbol table.
 */
SymTab::SymTab(unsigned long siz)
	: size(siz), amt(0), scope(0)
	{
	if (siz > 0)
		{
		TABSIZE = size;
		table = new List<STnode *>[size];
		}
	else
		{
		TABSIZE = 0;
		table = NULL;
		}
	}

int
SymTab::Init(unsigned long siz)
	{
	if (size == 0)
		{
		size = siz;
		TABSIZE = siz;
		table = new List<STnode *>[size];
		return 1;
		}
	return 0;
	}

SymTab::~SymTab(void)
	{
	for (int i = 0; i < amt; i++)
		Delete(table[i]);
	delete [] table;
	}

STnode *
SymTab::operator[](HashValue &idx)
	{
	if (idx.scope < 0 || size == 0)
		return NULL;
	// The list should not grow, so we can use the reference to it.
	List<STnode *> &lst = table[idx.value];
	STnode *node;
	for (int i = 0; i < Size(lst); i++)
		{
		// Each bucket is a list of symbols with different scope.
		// So traverse the list and find the scope we're looking
		// for.  Duplicates are checked when the symbol was first
		// entered into the table.
		node = lst[i];
		if (node->scope == idx.scope)
			return node;
		}
	return NULL;
	}

SymTabError
SymTab::Enter(STnode *node, HashValue &hv)
	{
	// Do we still have room?  If not, return error.
	if (size == 0)
		return STE_INIT;
	else if (amt >= size)
		return STE_FULL;

	HashValue idx = hash(node->name, node->scope);
	STnode *tmp;
	for (int i = 0; i < size; i++)
		{
		tmp = (*this)[idx];
		if (tmp == NULL)
			{
			table[idx.value] += node;
			amt++;
			break;
			}

		// Collision occured!
		if (tmp->name == node->name)
			// When the scope, hash value, and the symbol
			// are the same, we have a duplicate!
			return STE_DUP;

		// Hmm, not a duplicate.  Should we worry?  Neah!
		idx = rehash(idx, i);
		}
	hv = idx;
	return STE_NONE;
	}

HashValue
SymTab::Lookup(String &name, Stack<int> scope)
	{
	// Are there any elements yet?
	// If not, return invalid element.
	if (size == 0)
		return HashValue(0, STE_INIT);
	else if (amt < 1)
		return HashValue(0, STE_MISS);

	int sc = -1;
	HashValue idx, idxorig = hash(name, sc);
	STnode *node;
	// Sooner or later, the scope will reach zero (global).
	// So stop there.
	for (sc = Pop(scope); sc > 0; sc = Pop(scope))
		{
		// Try each scope in turn.
		idx.value = idxorig.value;
		idx.scope = sc;
		for (int i = 0; i < size; i++)
			{
			node = (*this)[idx];
			// Is this node taken at all?
			if (node == NULL)
				break;
			else if (node->name == name)
				return idx;
			// Collision, but not what we're looking for.
			idx = rehash(idx, i);
			}
		}
	// We skipped globals, so try it now.
	idx.value = idxorig.value;
	idx.scope = sc;
	for (int i = 0; i < size; i++)
		{
		node = (*this)[idx];
		if (node == NULL)
			break;
		else if (node->name == name)
			return idx;
		idx = rehash(idx, i);
		}

	// Well, I guess it's not in the symbol table.
	return HashValue(0, STE_MISS);	// Return invalid element.
	}

ostream &
operator<<(ostream &s, SymTab &st)
	{
	unsigned int i;
	s << "size/amt:" << st.size << '/' << st.amt << endl;
	for (i = 0; i < st.size; i++)
		{
		if (Size(st.table[i]) > 0)
			s << i << ": " << st.table[i] << endl;
		}
	return s;
	}

/*
 * Tools.
 *
 * See "Compiler Design in C" by Allen Holub, (C) 1990 Prentice Hall
 */

#define VALSIZE			16
#define TWELVE_PERCENT		( (int)(VALSIZE * .125) )
#define SEVENTY_FIVE_PERCENT	( (int)(VALSIZE * .75) )
#define HIGH_BITS		( ~(unsigned)(~0 >> TWELVE_PERCENT) )

HashValue
hash(String &s, int sc)
	{
	if (TABSIZE == 0)
		return HashValue(0, STE_INIT);

	HashValue hv(0, sc);
	int tmp;
	for(int i = 0; i < Length(s); i++)
		{
		hv.value = (hv.value << TWELVE_PERCENT) + s[i];
		if ( (tmp = hv.value & HIGH_BITS) )
			{
			hv.value = (hv.value ^ (tmp >> 
				SEVENTY_FIVE_PERCENT)) &
				HIGH_BITS;
			}
		}
	hv.value = hv.value % TABSIZE;		// Reduce to 17 bits.
	return hv;
	}

HashValue
rehash(HashValue &hv, int i)
	{
	if (TABSIZE == 0)
		return HashValue(0, STE_INIT);

	hv.value = hv.value * i;
	hv.value = hv.value % TABSIZE;
	return hv;
	}
