/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Symtab.h
 *
 * Symbol Table.  This symbol table is an array of buckets.  Each bucket
 * contains a list of nodes.  The list grows with the symbols of different
 * scope.  Each scope has it's own node in the list, but only one node.
 * If the symbol resulted in the same hash value, and has the same scope,
 * it is either a duplicate or has collided.  If collision has occured, we
 * rehash and repeat.
 *
 * Each node in the symbol table contains pointers to all necessary data
 * types for all the different types of nodes.  The different type of nodes
 * are enumerated in NodeType.
 */

#ifndef SYMTAB_H
#define SYMTAB_H

#include <iostream.h>
#include "Bool.h"
#include "String.h"
#include "List.h"
#include "Event.h"
#include "Base.h"

class Port;
class PortConnection;

/*
 * Errors generated by symbol table.
 */
enum SymTabError
	{
	STE_INIT	= -4,
	STE_FULL	= -3,
	STE_DUP		= -2,
	STE_MISS	= -1,
	STE_NONE	= 0
	};


/* Direction of NET types. */
enum IODir
	{
	IO_NONE,
	IO_IN,
	IO_OUT,
	IO_INOUT
	};
/* Node types. */
enum NodeType
	{
	NT_NONE,
	NT_WIRE,
	NT_REG,
	NT_FUNC,
	NT_TASK,
	NT_MOD,
	NT_INSTAN
	};

/*
 * Hash number.
 */
class HashValue
	{
public:
	unsigned long value;
	int scope;
	HashValue(unsigned long idx = 0UL, int sc = -1)
		: value(idx), scope(sc) {}
	HashValue(HashValue &hv)
		: value(hv.value), scope(hv.scope) {}
	HashValue &operator=(HashValue &hv)
		{ value = hv.value; scope = hv.scope; return *this; }
	friend HashValue hash(String &, int);
	friend HashValue rehash(HashValue &, int);
	friend ostream &operator<<(ostream &s, HashValue &hv)
		{ s << '(' << hv.value << ',' << hv.scope << ')'; }
	};

/*
 * Symbol table node.
 *
 * Verilog has the information about a symbol spread over a few
 * declarations.  Thus we need to set these information one at a time.
 * This prevents us from using polymorphism, or at least I don't know
 * how to use it in this situation.
 */
class STnode
	{
	friend class SymTab;

	// Is this node a complete description of a symbol?
	Bool complete;
	unsigned long ms, ls;	// Check for valid ranges.

	int scope;		// Must keep track of scope.
	IODir dir;		// Direction.
	NodeType type;		// Type.
	String name;		// Name of symbol.

	// Storage for anything that needs it.
	Number *storage;

	// For module instantiation, ports do not behave like arguments
	// to functions and tasks.  This is because arguments to functions
	// and tasks are updated on entry and exit.  But ports must be
	// updated immediately.  Thus, we need the actual index of the
	// register to update.  This is also a flag to indicate whether
	// we are a port or not.
	HashValue *portidx;
	unsigned long port_ms, port_ls;
	Bool port_range;	// Does port connection have range?

	// For module instantiation.
	List<HashValue> *propagate;

	// List of monitor statements for reg and wires.
	List< Events<Stmts *> *> *monitors;

	// List of ports for modules.
	List<Port *> *portlst;

	// List of module items for modules.
	List<ModuleItem *> *moditems;

	// List of indices for arguments in function calls.
	// This contains inputs/outputs/inouts in the order they
	// are declared.
	List<HashValue> *iovars;

	// Expressions and statements for functions and tasks.
	Expression *expr;
	Stmts *stmt;

	// When we do assignment, we usually like to check the <monitors>
	// list to see if we need to add events to the event queue.
	// We use a function pointer so that we don't have to include
	// the event queue in Symtab.cc.
	void (*on_assignment)(List< Events<Stmts *> *> *, Number &, Number &);

	// Hidden member functions.
	void init(void);
	void entry_iovars(List<Expression *> *arg);
	void exit_iovars(List<Expression *> *arg);
	void assign(Number &, unsigned long = 0, unsigned long = 0,
		Bool = FALSE);
public:
	String filename;
	int lineno;
	STnode();
	STnode(int sc, String &s, IODir d = IO_NONE, NodeType t = NT_NONE);
	STnode(STnode &n);
	void Assignment(Number &);
	void Assignment(Number &, unsigned long, unsigned long);
	operator String(void)
		{ return name; }
	String TypeString(void);
	// Must set type and direction before anything else.  Otherwise,
	// we can not check for valid data.
	Bool Complete(void)
		{ return complete; }
	IODir Dir(void)
		{ return dir; }
	NodeType Type(void)
		{ return type; }
	Number *Storage(void)
		{ return storage; }
	HashValue *PortIndex(void)
		{ return portidx; }
	List< Events<Stmts *> *> *EventLst(void)
		{ return monitors; }
	List<HashValue> *IOVar(void)
		{ return iovars; }
	List<Port *> *PortLst(void)
		{ return portlst; }
	List<ModuleItem *> *ModuleItemLst(void)
		{ return moditems; }
	void SetDir(IODir);
	void SetType(NodeType);
	void SetStorage(unsigned long, unsigned long);
	void SetPortIndex(HashValue *);
	void SetPortRange(unsigned long, unsigned long);
	void AddPropagate(HashValue &);
	void SetEventLst(List< Events<Stmts *> *> *);
	void SetIOVar(List<HashValue> *);
	void SetPortLst(List<Port *> *);
	void SetModuleItemLst(List<ModuleItem *> *);
	void SetExpression(Expression *);
	void SetStmts(Stmts *);
	void SetOnAssign(void (*)(List< Events<Stmts *> *> *,
		Number &, Number &));
	ostream &display(ostream &s);
	// Sim_evaluate for STnode must take the argument list for
	// functions.  But notice, only functions will past in arguments
	// since other expressions do not have argument lists.  Thus,
	// we do not need to modify the Expression class, just used a
	// default argument in the following virtual definition.
	Number Sim_evaluate(List<Expression *> * = NULL);
	void Sim_trigger(String &, List<Expression *> * = NULL);
	int Sim_trigger(Stmts *, List<Expression *> *);
	void Sim_setup(Stack<int>, String &,
		String & = String("unknown"), int = -1,
		List<PortConnection *> * = NULL,
		void (*)(int, String &, String &, int,
		List<Port *> *, List<PortConnection *> *) = NULL);
	friend ostream &operator<<(ostream &s, STnode &stn)
		{ stn.display(s); return s; }
	friend ostream &operator<<(ostream &s, STnode *stn)
		{ stn->display(s); return s; }
	};

/*
 * Symbol Table
 *
 * This class handles the symbol allocation and collisions.
 */
class SymTab
	{
	List<STnode *> *table;
	unsigned long size;
	unsigned long amt;
	int scope;
public:
	SymTab(unsigned long = 0);
	Init(unsigned long sz);
	~SymTab(void);
	STnode *operator[](HashValue &);
	// Increment scope.
	int &Scope(void)
		{ return scope; }
	// Enter node into symbol table.
	SymTabError Enter(STnode *, HashValue &);
	// Lookup string in symbol table.
	HashValue Lookup(String &, Stack<int>);
	friend ostream &operator<<(ostream &, SymTab &);
	};
	
extern HashValue NewSymbol(STnode *);
extern SymTab symboltable;

#endif // SYMTAB_H
