/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Systask.cc
 *
 * Utilities for all supported system tasks and functions.
 */

#include <string.h>		// strlen
#include <stdlib.h>		// exit
#include <iostream.h>
#include "glo.h"		// MAXBUF
#include "Bool.h"
#include "String.h"		// FindIndex
#include "List.h"
#include "Error.h"
#include "Symtab.h"
#include "TWheel.h"
#include "Base.h"
#include "Systask.h"

extern TimeWheel<Stmts *> timewheel;

/*
 * Available system tasks and functions...
 */

int
print_escapecode(QString &str, int idx)
	{
	char buf[MAXBUF];
	int i = 0;
	switch(str[idx])
		{
		case 'n': cout << endl; break;
		case 't': cout << '\t'; break;
		case '\\': cout << '\\'; break;
		case '\"': cout << '\"'; break;
		case '0':
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
		case '7':
			{
			buf[i++] = str[idx++];
			while (str[idx] >= '0' && str[idx] <= '7')
				{
				buf[i++] = str[idx++];
				// Only 1-3 octal digits.
				if (i > 2)
					break;
				}
			buf[i] = '\0';
			// Now, convert the octal number to the equivalent
			// integer.  Then print out the ASCII value.
			int tmp = strtol(buf, NULL, 8);
			sprintf(buf, "%c", tmp);
			cout << buf;
			// <idx> is one further than the character we used.
			// So move it back...
			idx--;
			break;
			}
		default: cout << str[idx]; break;
		}
	return idx;		// <idx> is last character used in <str>.
	}

int
find_format(int i, QString &fmt, String &str)
	{
	// Extract the format string from this format.
	// Return next unused position in <fmt> or -1 if error.
	int k = 0;
	char s[MAXBUF] = { 0 };
#ifdef __linux
	for (int j = i; fmt[j] != (char)NULL; j++)
#else
	for (int j = i; fmt[j] != NULL; j++)
#endif
		{
		if (fmt[j] == '\"')
			break;
		else if (fmt[j] >= '0' && fmt[j] <= '9')
			s[k++] = fmt[j];
		else if (fmt[j] == 'd' || fmt[j] == 'h'
				|| fmt[j] == 'b' || fmt[j] == 'o'
				|| fmt[j] == 's')
			{
			// Found the end.  These are our supported types.
			s[k++] = fmt[j];
			s[k++] = '\0';
			str = String(s);
			return j;
			}
		else if (fmt[j] == ' ')
			{
			// Save the string for error reporting.
			str = String(s);
			Sim_errno = SE_FMTSPEC;
			return -1;
			}
		else
			{
			char tmp[2];
			tmp[0] = fmt[j];
			tmp[1] = '\0';
			str = String(tmp);
			Sim_errno = SE_FMTSYM;
			return -1;
			}
		}
	str = String(s);
	Sim_errno = SE_FMTNOTERM;
	return -1;
	}

/*
 * Tasks and functions.
 */

void
SysTaskWrite::Sim_setup(Stack<int> &scope, Bool, int)
	{
	filename = String("internal");
	lineno = 0;
	Stmts *st = new SysTaskWrite(*this);
	int localscope = Pop(scope);
	Push(scope, localscope);
	STnode *node = new STnode(localscope, name);
	node->filename = filename;
	node->lineno = lineno;
	node->SetType(NT_TASK);
	node->SetStmts(st);
	NewSymbol(node);
	}

int
SysTaskWrite::Sim_trigger(Stmts *st)
	{
	int i, j = 0;
	// Get the parameter list.  If none, there's nothing to do.
	TaskStmt *sts = (TaskStmt *)st;
	Expression *expr = sts->args[j++];
	if (expr == NULL)
		return TRUE;

	QString &fmt = *(QString *)expr;
	for (i = 1; i < Length(fmt); i++)
		{
		if (fmt[i] == '\"')
			break;
		switch(fmt[i])
			{
			case '%':
				{
				String s;
				i++;	// Move to next character.
				i = find_format(i, fmt, s);
				if (i < 0)
					{
					Sim_perror(SEF_EXIT | SEF_SIM,
						"%s", (char *)s);
					}
				expr = sts->args[j++];
				if (expr != NULL)
					expr->Sim_trigger(s);
				break;
				}
			case '\\':
				{
				// Escape character found.
				i = print_escapecode(fmt, i+1);
				break;
				}
			default:
				{
				cout << fmt[i];
				break;
				}
			}
		}
	return TRUE;
	}

void
SysTaskFinish::Sim_setup(Stack<int> &scope, Bool, int)
	{
	filename = String("internal");
	lineno = 0;
	Stmts *st = new SysTaskFinish(*this);
	int localscope = Pop(scope);
	Push(scope, localscope);
	STnode *node = new STnode(localscope, name);
	node->filename = filename;
	node->lineno = lineno;
	node->SetType(NT_TASK);
	node->SetStmts(st);
	NewSymbol(node);
	}

int
SysTaskFinish::Sim_trigger(Stmts *)
	{
	cout << endl << "Ending simulation at time unit "
		<< CurrentTime(timewheel)
		<< " due to $finish called."
		<< endl << endl;
	Sim_errno = SE_NONE;
	exit(Sim_errno);
	}

void
SysFunctionTime::Sim_setup(Stack<int> &scope)
	{
	filename = String("internal");
	lineno = 0;
	Expression *exp = new SysFunctionTime(*this);
	int localscope = Pop(scope);
	Push(scope, localscope);
	// We only support time of upto 32 bits.
	STnode *node = new STnode(localscope, name);
	node->filename = filename;
	node->lineno = lineno;
	node->SetType(NT_FUNC);
	node->SetStorage(31, 0);
	node->SetExpression(exp);
	NewSymbol(node);
	}

void
SysFunctionTime::Sim_trigger(String &fmt)
	{
	/* Get format and put current time into format. */
	int base, size;
	get_base_size(fmt, base, size);
	Number num = CurrentTime(timewheel);
	String str = Convert(num, base, size);
	cout << str;
	}

Number
SysFunctionTime::Sim_evaluate(void)
	{ return CurrentTime(timewheel); }

/*
 * Function calls and task enable.
 */

ostream &
FunctionCall::display(ostream &s)
	{ s << name; return s; }

Expression *
FunctionCall::copy(void)
	{ return new FunctionCall(*this); }

void
FunctionCall::Sim_setup(Stack<int> &scope)
	{
	idx = symboltable.Lookup(String(name), scope);
	if (idx.scope == STE_MISS)
		{
		Sim_errno = SE_STDEF;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	for (int i = 0; i < Size(args); i++)
		args[i]->Sim_setup(scope);

	// Check that we have the correct number of arguments.
	STnode *node = symboltable[idx];
	if (node->Type() != NT_FUNC)
		{
		Sim_errno = SE_NFUNCTION;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	List<HashValue> *lst = node->IOVar();
	int num = 0;
	if (lst != NULL)
		num = Size(*lst);
	if (num != Size(args))
		{
		Sim_errno = SE_NARGLST;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	}

void
FunctionCall::Sim_trigger(String &fmt)
	{
	// <node> can not be NULL.  Setup should have checked.
	STnode *node = symboltable[idx];

	// Set the filename and line number in case the function is a
	// system function which does error checking.  Right now, there
	// isn't any.  But in the future, we won't have to worry about
	// this.  Of course, the trigger function could always reset
	// the filename/lineno to something else.
	Sim_filename = filename;
	Sim_lineno = lineno;
	node->Sim_trigger(fmt, &args);
	}

Number
FunctionCall::Sim_evaluate(void)
	{
	STnode *node = symboltable[idx];
	// Must past in argument list to user defined function.  It
	// is up to Sim_evaluate to check for arguments.
	Sim_filename = filename;
	Sim_lineno = lineno;
	return node->Sim_evaluate(&args);
	}

ostream &
TaskStmt::display(ostream &s)
	{
	if (dec)
		s << *dec << ' ';
	s << name << '(' << args << ')' << endl;
	return s;
	}

Stmts *
TaskStmt::copy(void)
	{ return new TaskStmt(*this); }

void
TaskStmt::Sim_setup(Stack<int> &scope, Bool, int ns)
	{
	// Allow delays?
	if ((ns & NS_DELAY) != 0 && dec != NULL)
		{
		Sim_errno = SE_SUPPORT;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", "#num");
		}
	idx = symboltable.Lookup(String(name), scope);
	if (idx.scope == STE_MISS)
		{
		Sim_errno = SE_STDEF;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	for (int i = 0; i < Size(args); i++)
		args[i]->Sim_setup(scope);

	// Check that we have the correct number of arguments.
	STnode *node = symboltable[idx];
	if (node->Type() != NT_TASK)
		{
		Sim_errno = SE_NTASK;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	List<HashValue> *lst = node->IOVar();
	int num = 0;
	if (lst != NULL)
		num = Size(*lst);
	// If task doesn't take arguments, but they are passed in,
	// what is the harm?  Also, system tasks ($write) takes infinite
	// arguments.
	if (num > 0 && num != Size(args))
		{
		Sim_errno = SE_NARGLST;
		Sim_filename = filename;
		Sim_lineno = lineno;
		Sim_perror(SEF_EXIT | SEF_SETUP, "%s", (char *)String(name));
		}
	}

int
TaskStmt::Sim_trigger(Stmts *orig)
	{
	// See Trigger.cc for an explanation on how this function works.
	if (always == TRUE || orig->always == TRUE)
		{
		// In an "always" statement, we always trigger the
		// system task.  But instead of passing in the original
		// statement, we pass in <this> (which has the args to the
		// system task.
		STnode *node = symboltable[idx];
		Sim_filename = filename;
		Sim_lineno = lineno;
		node->Sim_trigger(this, &args);

		if (dec != NULL)
			dec->Sim_trigger(this);
		return TRUE;
		}

	if (dec != NULL)
		{
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	STnode *node = symboltable[idx];
	Sim_filename = filename;
	Sim_lineno = lineno;
	node->Sim_trigger(this, &args);
	return TRUE;
	}
