/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * Trigger.cc
 *
 * Time Wheel simulation methods.
 */

#include <stdio.h>
#include <stdlib.h>
#include "List.h"
#include "Error.h"
#include "Event.h"
#include "Symtab.h"
#include "TWheel.h"
#include "Base.h"
#include "Systask.h"
#include "PExpr.h"
#include "PTypes.h"

extern EvntQueue<Stmts *> eventqueue;
extern TimeWheel<Stmts *> timewheel;

int
AssignStmt::Sim_trigger(Stmts *orig)
	{
	// Does this statement execute now or in the future?
	// It depends on whether we are in an 'always' statement or not.
	if (always == TRUE || orig->always == TRUE)
		{
		// Trigger lvalue by passing in the expression.
		lval.Sim_trigger(*rval);

		// If there's also a delay, then we reschedule "this"
		// statement.  This delay is usually due to a single
		// statement in an 'always' procedural block.  i.e.
		//
		//	always #5 statement_or_null;
		//
		if (dec != NULL)
			dec->Sim_trigger(this);

		// We do not support delays inside a sequential block with
		// an 'always' statement.  If someone tries it, we have an
		// undefined behavior.
		return TRUE;		// Always return true.
		}

	// The way we handle delays inside the 'initial' procedural block
	// is to append the entire procedural block to the future instead
	// of just this statement (like before).  Thus, we do not handle
	// future statements until we arrive at the future.
	if (dec != NULL)
		{
		// If there's a delay, we append to the future.  Remove this
		// delay so the next time we come here, we trigger this
		// statment.
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	// We do not know at this point whether we were put into the
	// future or this is the first call.  But we don't care either
	// way.
	lval.Sim_trigger(*rval);
	return TRUE;
	}

int
SeqBlkStmt::Sim_trigger(Stmts *orig)
	{
	// Sequential block statements are handled differently because of
	// the multiple statements it contains.
	if (always == TRUE || orig->always == TRUE)
		{
		// Go through each statement in the block and trigger them.
		// Since we do not support delays in sequential blocks with
		// 'always', this should work.
		Reset(stmts);
		for (int i = 0; i < Size(stmts); i++)
			stmts[i]->Sim_trigger(orig);
		if (dec != NULL)
			dec->Sim_trigger(this);
		return TRUE;
		}

	// For 'initial' procedural blocks.  We must handle one statement
	// at a time.  This means we do not go through the entire list
	// and append each statement to the future.  Instead, we get one
	// statement and try and execute it.  If there's a delay, we append
	// this entire statement into the future.

	// Note:  We must push the list of statements into the stack if it's
	// not already on the stack.
	SeqBlkStmt *sqblk = (SeqBlkStmt *)orig;
	if (pushed == FALSE)
		{
#if defined(DEBUG_NES)
cout << "Adding list: " << &(this->stmts) << endl;
#endif
		Reset(stmts);
		Push(sqblk->stk, this);
		pushed = TRUE;
		}

	// Let's begin.
	Stmts *tmp;
	SeqBlkStmt *nesting;
	while (TRUE)
		{
		// First, get next statement.
		while (sqblk->curstmt == NULL)
			{
			if (Size(sqblk->stk) < 1)
				// We are at the end of the original statement.
				return TRUE;
			nesting = Pop(sqblk->stk);
			if (Data(nesting->stmts, sqblk->curstmt))
				{
#if defined(DEBUG_NES)
cout << "Using list: " << &(nesting->stmts) << endl;
#endif
				Push(sqblk->stk, nesting);
				break;
				}
			else
				{
				// No more statements in this nested level.
				// Reset <pushed>, because we might be
				// triggered again.
#if defined(DEBUG_NES)
cout << "Ended one nesting level..." << endl;
#endif
				nesting->pushed = FALSE;
				if (nesting->loop == TRUE)
					return TRUE;
				}
			}

		// Now trigger this statement.  Let the statement handle
		// the delay.
		tmp = sqblk->curstmt;
		Push(sqblk->savestmts, sqblk->curstmt);
#if defined(DEBUG_NES)
cout << "Savestmts size (" << Size(sqblk->savestmts) << ") ";
cout << "Triggering: " << *sqblk->curstmt << endl;
#endif
		sqblk->curstmt = NULL;
		if (tmp->Sim_trigger(orig) == FALSE)
			{
			// Due to the recursiveness of nested blocks.  What we
			// pop might not be what we just pushed into the stack.
			// It depends on whether the statement we've just triggered
			// was a sequential block or not.
			if (sqblk->curstmt == NULL)
				// Non-sequential block, restore statement.
				sqblk->curstmt = Pop(sqblk->savestmts);
#if defined(DEBUG_NES)
cout << "Delayed * level: " << *sqblk->curstmt << endl;
#endif
			return FALSE;
			}
#if defined(DEBUG_NES)
cout << "Releasing saved statement: " <<
#endif
		Pop(sqblk->savestmts);		// Dump saved statement.
#if defined(DEBUG_NES)
cout << endl;
#endif
		sqblk->curstmt = NULL;		// Setup for next statement.
		}
	}

int
IfStmt::Sim_trigger(Stmts *orig)
	{
	// All statements have the same procedure as above.  The only change
	// is the actual trigger of the statement.
	int ret = TRUE;
	if (always == TRUE || orig->always == TRUE)
		{
		Number num = expr->Sim_evaluate();
		if (num != FALSE)
			ret = stmt->Sim_trigger(orig);
		if (dec != NULL)
			dec->Sim_trigger(this);
		return ret;
		}

	if (dec != NULL)
		{
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	Number num = expr->Sim_evaluate();
	if (num != FALSE)
		ret = stmt->Sim_trigger(orig);
	return ret;
	}

int
IfElseStmt::Sim_trigger(Stmts *orig)
	{
	int ret = TRUE;
	if (always == TRUE || orig->always == TRUE)
		{
		Number num = expr->Sim_evaluate();
		if (num != FALSE)
			ret = stmt1->Sim_trigger(orig);
		else
			ret = stmt2->Sim_trigger(orig);
		if (dec != NULL)
			dec->Sim_trigger(this);
		return ret;
		}

	if (dec != NULL)
		{
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	Number num = expr->Sim_evaluate();
	if (num != FALSE)
		ret = stmt1->Sim_trigger(orig);
	else
		ret = stmt2->Sim_trigger(orig);
	return ret;
	}

int
ForStmt::Sim_trigger(Stmts *orig)
	{
	int ret = TRUE;
	if (always == TRUE || orig->always == TRUE)
		{
		assign1.Sim_trigger(orig);	// There should be no delay.
		while (expr->Sim_evaluate() != FALSE)
			{
			ret = stmt->Sim_trigger(orig);
			assign2.Sim_trigger(orig);	// No delay.
			}
		if (dec != NULL)
			dec->Sim_trigger(this);
		return ret;
		}

	if (dec != NULL)
		{
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	assign1.Sim_trigger(orig);
	while (expr->Sim_evaluate() != FALSE)
		{
		if ((ret = stmt->Sim_trigger(orig)) == FALSE)
			{
			// There was a delay inside a 'for' loop?
			TWnode<Stmts *> nnode(orig, CurrentTime(timewheel));
			timewheel += nnode;
			return FALSE;
			}
		assign2.Sim_trigger(orig);
		}
	return ret;
	}

int
CaseItem::Sim_trigger(Number &num, Bool &status, Stmts *orig)
	{
	int ret = TRUE;
	if (def == TRUE)
		ret = stmt->Sim_trigger(orig);
	else
		{
		for (int i=0; i < Size(expr); i++)
			{
			// The result of evaluate is of different size,
			// thus we cannot use a temporary at a fixed size.
			if (expr[i]->Sim_evaluate() == num)
				{
				ret = stmt->Sim_trigger(orig);
				status = TRUE;
				break;
				}
			}
		}
	return ret;
	}

int
CaseStmt::Sim_trigger(Stmts *orig)
	{
	int ret = TRUE;
	if (always == TRUE || orig->always == TRUE)
		{
		Number num = expr->Sim_evaluate();
		Bool status = FALSE;
		for (int i=0; i < Size(ci); i++)
			{
			ret  = ci[i]->Sim_trigger(num, status, orig);
			if (status == TRUE)
				break;
			}
		if (def != NULL && status == FALSE)
			ret = def->Sim_trigger(num, status, orig);

		if (dec != NULL)
			dec->Sim_trigger(this);
		return ret;
		}

	if (dec != NULL)
		{
		dec->Sim_trigger(orig);
		delete dec;
		dec = NULL;
		return FALSE;
		}
	Number num = expr->Sim_evaluate();
	Bool status = FALSE;
	for (int i=0; i < Size(ci); i++)
		{
		ret = ci[i]->Sim_trigger(num, status, orig);
		if (status == TRUE)
			break;
		}
	if (def != NULL && status == FALSE)
		ret = def->Sim_trigger(num, status, orig);
	return ret;
	}

void
DelayNum::Sim_trigger(Stmts *st)
	{
	// Find the current time and add the delay.  Append to this time
	// unit.
	unsigned long curtime = CurrentTime(timewheel);
	curtime += amt;
	TWnode<Stmts *> newnode(st, curtime);
	timewheel += newnode;
	}

void
DelayRangeId::Sim_trigger(Stmts *st)
	{
	// Get the number from the id.
	STnode *node = symboltable[id.idx];
	Number amt = node->Sim_evaluate();
	unsigned long curtime = CurrentTime(timewheel);
	curtime += amt;
	TWnode<Stmts *> newnode(st, curtime);
	timewheel += newnode;
	}

void
OredEvntExpression::Sim_trigger(Stmts *)
	{}

void
Lvalue::Sim_trigger(Expression &expr)
	{
	// Get the storage object from the symbol table.
	STnode *node = symboltable[id.idx];

	// Evaluate the expr and assign to num.
	Number res = expr.Sim_evaluate();
	if (HaveRange(id))
		{
		Number lnum, rnum;
		GetRangeData(Range(id), lnum, rnum);
		unsigned long ms = lnum;
		unsigned long ls = rnum;
		node->Assignment(res, ms, ls);
		}
	else
		node->Assignment(res);
	}

void
RangeId::Sim_trigger(String &fmt)
	{
	// Grab the data from the symbo table.
	STnode *node = symboltable[idx];
	// Get the data only.
	Number reg(node->Sim_evaluate());
	// Print out the content.
	if (flag)
		{
		Number lnum, rnum;
		GetRangeData(range, lnum, rnum);
		unsigned long ms = lnum;
		unsigned long ls = rnum;
		Number tmp = BitVector(reg(ms,ls));
		tmp.Sim_trigger(fmt);
		}
	else
		reg.Sim_trigger(fmt);
	}

void
ConstExpression::Sim_trigger(String &fmt)
	{ amt.Sim_trigger(fmt); }

void
NotOp::Sim_trigger(String &)
	{}

void
InvertOp::Sim_trigger(String &)
	{}

void
AddOp::Sim_trigger(String &)
	{}

void
SubOp::Sim_trigger(String &)
	{}

void
EqEqComp::Sim_trigger(String &)
	{}

void
NotEqComp::Sim_trigger(String &)
	{}

void
OrOrOp::Sim_trigger(String &)
	{}

void
AndAndOp::Sim_trigger(String &)
	{}

void
LogOrOp::Sim_trigger(String &)
	{}

void
LogAndOp::Sim_trigger(String &)
	{}

void
LogLShiftOp::Sim_trigger(String &)
	{}

void
LogRShiftOp::Sim_trigger(String &)
	{}

void
GrtThanOp::Sim_trigger(String &)
	{}

void
GrtEqOp::Sim_trigger(String &)
	{}

void
LesThanOp::Sim_trigger(String &)
	{}

void
LesEqOp::Sim_trigger(String &)
	{}
