/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * intrface.c
 *
 * See intrface.h for a description.
 */

#include <stdio.h>			/* vsprintf */
#include <string.h>
#include <stdlib.h>			/* exit */
#include <ctype.h>
#include <stdarg.h>			/* varargs */
#include "glo.h"
#include "intrface.h"

char **filelist;			/* List of files to parse. */
int maxfiles, nfiles;		/* Number of files left to parse. */

void
sys_error(char *fmt, ...)
	{
	va_list ap;

	va_start(ap, fmt);
	vfprintf(stdout, fmt, ap);
	va_end(ap);
	}

char *
my_strdup(const char *str)
	{
	char *p;
	int len;
	len = strlen(str);
	p = (char *)malloc(len + 1);
	strcpy(p, str);
	return p;
	}

int
p_init(int amt, char **lst)
	{
	/*
	 * Initialize parser variables.  these are global because all
	 * other files are in C++.
	 */

	filelist = lst;
	maxfiles = nfiles = amt;

	/* Setup yacc input file. */
	if (nfiles <= 0)
		return 0;

	yyin = fopen(filelist[maxfiles - nfiles], "r");
	if (!yyin)
		{
		sys_error("%s -- unable to open file.\n",
			filelist[maxfiles - nfiles]);
		return 0;
		}
	printf("Compiling file '%s'...\n", filelist[maxfiles - nfiles]);
	cur_filename = filelist[maxfiles - nfiles];
	nfiles--;
	return 1;
	}

int
p_start(void)
	{ return yyparse(); }

int
yywrap(void)
	{
	/*
	 * yywarp used by yacc after EOF is returned from yyin.
	 * Let's try our hand at multiple input files.
	 */
	char *curfile = filelist[maxfiles - nfiles];

#if defined(DEBUG_PAR)
	printf("\nFinished parsing a file...\n");
#endif

	if (nfiles > 0)
		{
		/*
		 * Close old file pointer and open new file, if any.
		 * Since <nfiles> starts at non-zero, and descreases
		 * to zero, we are actually going backwards.
		 */
		fclose(yyin);
		yyin = fopen(curfile, "r");
		if (!yyin)
			{
			sys_error("%s -- unable to open file.\n",
				curfile);
			exit(1);
			}
		printf("Compiling file '%s'...\n", curfile);
		cur_filename = curfile;
		nfiles--;
		return 0; /* More to do. */
		}

	return 1; /* Nothing left. */
	}

/*
 * C interface routines to create C++ objects for the parser.
 */

p_Number *
p_create_num(char *s)
	{
	int len, type;
	char *siz, *num, *tmp, base;
	p_Number *ret;

#if defined(DEBUG_PAR)
	printf("Creating number: %s\n", s);
#endif
	
	if (strchr(s, '\'') == NULL)
		{
		len = 0;
		base = 'd';
		num = s;
		}
	else
		{
		tmp = my_strdup(s);
		siz = strtok(tmp, "\'");
		len = atoi(siz);
		base = *(s + strlen(siz) + 1);
		num = s + strlen(siz) + 2;
		free(tmp);
		}
	switch(base)
		{
		case 'b':
		case 'B':	type = 2; break;
		case 'o':
		case 'O': type = 8; break;
		case 'd':
		case 'D': type = 10; break;
		case 'h':
		case 'H': type = 16; break;
		}
	/* Allocate structure for storage. */
	ret = (p_Number *)malloc(sizeof(p_Number));
	ret->data = P_CreateNumber(num, type, len);
	free(s);
	return ret;
	}

p_ConstExpression *
p_create_constexpr(p_Expression *expr)
	{
	p_ConstExpression *ret;
#if defined(DEBUG_PAR)
	printf("Creating constant expression\n");
#endif
	ret = (p_ConstExpression *)malloc(sizeof(p_ConstExpression));
	ret->data = P_CreateConstExpr(expr);
	free(expr);
	return ret;
	}

p_Expression *
p_create_expr(int type, void *p1, void *p2)
	{
	p_Expression *ret;
#if defined(DEBUG_PAR)
	printf("Creating expression\n");
#endif
	ret = (p_Expression *)malloc(sizeof(p_Expression));
	ret->data = P_CreateExpr(type, p1, p2);
	free(p1);
	if (p2)
		free(p2);
	return ret;
	}

p_ExpressionLst *
p_create_exprlst(p_ExpressionLst *lst, p_Expression *p)
	{
	p_ExpressionLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating expression list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_ExpressionLst *)malloc(sizeof(p_ExpressionLst));
	ret->data = P_CreateExprLst(lst, p);
	free(p);
	return ret;
	}

p_Range *
p_create_range(p_ConstExpression *l, p_ConstExpression *r)
	{
	p_Range *ret;
#if defined(DEBUG_PAR)
	printf("Creating range\n");
#endif
	ret = (p_Range *)malloc(sizeof(p_Range));
	ret->data = P_CreateRange(l, r);
	free(l);
	free(r);
	return ret;
	}

p_RangeId *
p_create_rangeid(char *str, p_Range *r)
	{
	p_RangeId *ret;
#if defined(DEBUG_PAR)
	printf("Creating range id\n");
#endif
	ret = (p_RangeId *)malloc(sizeof(p_RangeId));
	ret->data = P_CreateRangeId(str, r);
	free(r);
	free(str);
	return ret;
	}

p_RangeIdLst *
p_create_rangeidlst(p_RangeIdLst *lst, p_RangeId *id)
	{
	p_RangeIdLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating range id list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_RangeIdLst *)malloc(sizeof(p_RangeIdLst));
	ret->data = P_CreateRangeIdLst(lst, id);
	free(id);
	return ret;
	}

p_FunctionCall *
p_create_functioncall(p_RangeId *ri, p_ExpressionLst *lst)
	{
	p_FunctionCall *ret;
#if defined(DEBUG_PAR)
	printf("Creating a function call\n");
#endif
	ret = (p_FunctionCall *)malloc(sizeof(p_FunctionCall));
	ret->data = P_CreateFunctionCall(ri, lst);
	free(ri);
	return ret;
	}

p_Lvalue *
p_create_lval(p_RangeId *ri)
	{
	p_Lvalue *ret;
#if defined(DEBUG_PAR)
	printf("Creating lvalue\n");
#endif
	ret = (p_Lvalue *)malloc(sizeof(p_Lvalue));
	ret->data = P_CreateLvalue(ri);
	free(ri);
	return ret;
	}

p_DelayEvntCtl *
p_create_delayevntctl(int type, void *p)
	{
	p_DelayEvntCtl *ret;
#if defined(DEBUG_PAR)
	printf("Creating delay or event control\n");
#endif
	ret = (p_DelayEvntCtl *)malloc(sizeof(p_DelayEvntCtl));
	ret->data = P_CreateDelayEvntCtl(type, p);
	free(p);
	return ret;
	}

p_CaseItem *
p_create_caseitem(int type, p_ExpressionLst *p, p_Stmts *s)
	{
	p_CaseItem *ret;
#if defined(DEBUG_PAR)
	printf("Creating case item\n");
#endif
	ret	= (p_CaseItem *)malloc(sizeof(p_CaseItem)); 
	ret->data = P_CreateCaseItem(type, p, s);
	if (p != NULL)
		free(p);
	free(s);
	return ret;
	}

p_CaseItemLst *
p_create_caseitemlst(p_CaseItemLst *cl, p_CaseItem *c)
	{
	p_CaseItemLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating case item list\n");
#endif
	ret	= (p_CaseItemLst *)malloc(sizeof(p_CaseItemLst)); 
	ret->data = P_CreateCaseItemLst(cl, c);
	free(c);
	free(cl);
	return ret;
	}

p_EvntExpression *
p_create_evntexpression(int type, p_Expression *p)
	{
	p_EvntExpression *ret;
#if defined(DEBUG_PAR)
	printf("Creating event expression\n");
#endif
	ret = (p_EvntExpression *)malloc(sizeof(p_EvntExpression));
	ret->data = P_CreateEvntExpression(type, p);
	free(p);
	return ret;
	}

p_EvntExpressionLst *
p_create_evntexpressionlst(p_EvntExpressionLst *lst,
		p_EvntExpression *p)
	{
	p_EvntExpressionLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating event expression list\n");
#endif
	ret = (p_EvntExpressionLst *)malloc(sizeof(p_EvntExpressionLst));
	ret->data = P_CreateEvntExpressionLst(lst, p);
	free(p);
	return ret;
	}

p_TaskStmt *
p_create_taskstmt(p_RangeId *r, p_ExpressionLst *lst)
	{
	p_TaskStmt *ret;
#if defined(DEBUG_PAR)
	printf("Creating system task\n");
#endif
	ret = (p_TaskStmt *)malloc(sizeof(p_TaskStmt));
	ret->data = P_CreateTaskStmt(r, lst);
	free(r);
	free(lst);
	return ret;
	}

p_AssignStmt *
p_create_assign(p_Lvalue *lval, p_Expression *expr)
	{
	p_AssignStmt *ret;
#if defined(DEBUG_PAR)
	printf("Creating assignment statement\n");
#endif
	ret = (p_AssignStmt *)malloc(sizeof(p_AssignStmt));
	ret->data = P_CreateAssignStmt(lval, expr);
	free(lval);
	free(expr);
	return ret;
	}

p_SeqBlkStmt *
p_create_seqblk(p_RangeId *id, p_StmtsLst *stmts)
	{
	p_SeqBlkStmt *ret;
#if defined(DEBUG_PAR)
	printf("Creating sequential block\n");
#endif
	ret = (p_SeqBlkStmt *)malloc(sizeof(p_SeqBlkStmt));
	ret->data = P_CreateSeqBlkStmt(id, stmts); 
	if (id)
		free(id);
	free(stmts);
	return ret;
	}

p_Stmts *
p_create_stmts(int type, void *p1, void *p2, void *p3, void *p4)
	{
	p_Stmts *ret;
#if defined(DEBUG_PAR)
	printf("Creating a statement\n");
#endif
	ret = (p_Stmts *)malloc(sizeof(p_Stmts));
	ret->data = P_CreateStmts(type, p1, p2, p3, p4);
	free(p1);
	if (p2 != NULL)
		free(p2);
	if (p3 != NULL)
		free(p3);
	if (p4 != NULL)
		free(p4);
	return ret;
	}

p_StmtsLst *
p_create_stmtslst(p_StmtsLst *lst, int type, p_Stmts *p)
	{
	p_StmtsLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a statement list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_StmtsLst *)malloc(sizeof(p_StmtsLst));
	ret->data = P_CreateStmtsLst(lst, type, p);
	free(p);
	return ret;
	}

p_WireDecl *
p_create_wiredecl(p_Range *range, p_RangeIdLst *ids)
	{
	p_WireDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a wire declaration\n");
#endif
	ret = (p_WireDecl *)malloc(sizeof(p_WireDecl));
	ret->data = P_CreateWireDecl(range, ids);
	if (range)
		free(range);
	free(ids);
	return ret;
	}

p_RegDecl *
p_create_regdecl(p_Range *range, p_RangeIdLst *ids)
	{
	p_RegDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a register declaration\n");
#endif
	ret = (p_RegDecl *)malloc(sizeof(p_RegDecl));
	ret->data = P_CreateRegDecl(range, ids);
	if (range)
		free(range);
	free(ids);
	return ret;
	}

p_InputDecl *
p_create_inputdecl(p_Range *range, p_RangeIdLst *ids)
	{
	p_InputDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a input declaration\n");
#endif
	ret = (p_InputDecl *)malloc(sizeof(p_InputDecl));
	ret->data = P_CreateInputDecl(range, ids);
	if (range)
		free(range);
	free(ids);
	return ret;
	}

p_OutputDecl *
p_create_outputdecl(p_Range *range, p_RangeIdLst *ids)
	{
	p_OutputDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a output declaration\n");
#endif
	ret = (p_OutputDecl *)malloc(sizeof(p_OutputDecl));
	ret->data = P_CreateOutputDecl(range, ids);
	if (range)
		free(range);
	free(ids);
	return ret;
	}

p_InoutDecl *
p_create_inoutdecl(p_Range *range, p_RangeIdLst *ids)
	{
	p_InoutDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a inout declaration\n");
#endif
	ret = (p_InoutDecl *)malloc(sizeof(p_InoutDecl));
	ret->data = P_CreateInoutDecl(range, ids);
	if (range)
		free(range);
	free(ids);
	return ret;
	}

p_TFDecl *
p_create_tfdecl(int type, void *p)
	{
	p_TFDecl *ret;
#if defined(DEBUG_PAR)
	printf("Creating a tf declaration\n");
#endif
	ret = (p_TFDecl *)malloc(sizeof(p_TFDecl));
	ret->data = P_CreateTFDecl(type, p);
	free(p);
	return ret;
	}

p_TFDeclLst *
p_create_tfdecllst(p_TFDeclLst *lst, int type, p_TFDecl *p)
	{
	p_TFDeclLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a tf declaration list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_TFDeclLst *)malloc(sizeof(p_TFDeclLst));
	ret->data = P_CreateTFDeclLst(lst, type, p);
	free(p);
	return ret;
	}

p_Function *
p_create_function(p_Range *r, p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st)
	{
	p_Function *ret;
#if defined(DEBUG_PAR)
	printf("Creating a function\n");
#endif
	ret = (p_Function *)malloc(sizeof(p_Function));
	ret->data = P_CreateFunction(r, n, decl, st);
	if (r != NULL)
		free(r);
	free(n);
	free(decl);
	free(st);
	return ret;
	}

p_Task *
p_create_task(p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st)
	{
	p_Task *ret;
#if defined(DEBUG_PAR)
	printf("Creating a task\n");
#endif
	ret = (p_Task *)malloc(sizeof(p_Task));
	ret->data = P_CreateTask(n, decl, st);
	free(n);
	free(decl);
	free(st);
	return ret;
	}

p_InitialStmt *
p_create_initialstmt(p_Stmts *st)
	{
	p_InitialStmt *ret;
#if defined(DEBUG_PAR)
	printf("Creating an initial statement\n");
#endif
	ret = (p_InitialStmt *)malloc(sizeof(p_InitialStmt));
	ret->data = P_CreateInitialStmt(st);
	free(st);
	return ret;
	}

p_AlwaysStmt *
p_create_alwaysstmt(p_Stmts *st)
	{
	p_AlwaysStmt *ret;
#if defined(DEBUG_PAR)
	printf("Creating an always statement\n");
#endif
	ret = (p_AlwaysStmt *)malloc(sizeof(p_AlwaysStmt));
	ret->data = P_CreateAlwaysStmt(st);
	free(st);
	return ret;
	}

p_PortConn *
p_create_portconn(p_Expression *p)
	{
	p_PortConn *ret;
#if defined(DEBUG_PAR)
	printf("Creating port connection\n");
#endif
	ret = (p_PortConn *)malloc(sizeof(p_PortConn));
	ret->data = P_CreatePortConn(p);
	return ret;
	}

p_PortConnLst *
p_create_portconnlst(p_PortConnLst *lst, p_PortConn *p)
	{
	p_PortConnLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a port connection list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_PortConnLst *)malloc(sizeof(p_PortConnLst));
	ret->data = P_CreatePortConnLst(lst, p);
	free(p);
	return ret;
	}

p_ModInstance *
p_create_modinstance(p_RangeId *id, p_PortConnLst *lst)
	{
	p_ModInstance *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module instance\n");
#endif
	ret = (p_ModInstance *)malloc(sizeof(p_ModInstance));
	ret->data = P_CreateModInstance(id, lst);
	free(id);
	if (lst != NULL)
		free(lst);
	return ret;
	}

p_ModInstanceLst *
p_create_modinstancelst(p_ModInstanceLst *lst, p_ModInstance *mi)
	{
	p_ModInstanceLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module instance list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_ModInstanceLst *)malloc(sizeof(p_ModInstanceLst));
	ret->data = P_CreateModInstanceLst(lst, mi);
	free(mi);
	return ret;
	}

p_ModInstan *
p_create_modinstan(p_RangeId *id, p_ModInstanceLst *lst)
	{
	p_ModInstan *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module instantiation\n");
#endif
	ret = (p_ModInstan *)malloc(sizeof(p_ModInstan));
	ret->data = P_CreateModInstan(id, lst);
	free(id);
	free(lst);
	return ret;
	}

p_ModuleItem *
p_create_moditem(int type, void *p)
	{
	p_ModuleItem *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module item\n");
#endif
	ret = (p_ModuleItem *)malloc(sizeof(p_ModuleItem));
	ret->data = P_CreateModItem(type, p);
	free(p);
	return ret;
	}

p_ModuleItemLst *
p_create_moditemlst(p_ModuleItemLst *lst, int type, p_ModuleItem *p)
	{
	p_ModuleItemLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module item list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_ModuleItemLst *)malloc(sizeof(p_ModuleItemLst));
	ret->data = P_CreateModItemLst(lst, type, p);
	free(p);
	return ret;
	}

p_Port *
p_create_port(p_RangeId *id)
	{
	p_Port *ret;
#if defined(DEBUG_PAR)
	printf("Creating a port\n");
#endif
	ret = (p_Port *)malloc(sizeof(p_Port));
	ret->data = P_CreatePort(id);
	free(id);
	return ret;
	}

p_PortLst *
p_create_portlst(p_PortLst *lst, p_Port *p)
	{
	p_PortLst *ret;
#if defined(DEBUG_PAR)
	printf("Creating a port list\n");
#endif
	if (lst != NULL)
		ret = lst;
	else
		ret = (p_PortLst *)malloc(sizeof(p_PortLst));
	ret->data = P_CreatePortLst(lst, p);
	free(p);
	return ret;
	}

p_Module *
p_create_module(p_RangeId *id, p_ModuleItemLst *lst, p_PortLst *plst)
	{
	p_Module *ret;
#if defined(DEBUG_PAR)
	printf("Creating a module\n");
#endif
	ret = (p_Module *)malloc(sizeof(p_Module));
	ret->data = P_CreateModule(id, lst, plst);
	if (id)
		free(id);
	if (plst)
		free(plst);
	free(lst);
	return ret;
	}
