/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * intrface.h
 *
 * intrface.h and intrface.c contain structures and functions for the
 * yacc parser.  Functions in intrface.c are frontend functions for
 * the C++ functions in PTypes.cc.
 */

#ifndef INTRFACE_H
#define INTRFACE_H

#include <stdio.h>

#ifdef __cplusplus
extern "C" {
#endif

typedef struct { void *data; } Ptr;	/* Generic Pointer type */
typedef Ptr p_Number;
typedef Ptr p_ConstExpression;
typedef Ptr p_Expression;
typedef Ptr p_ExpressionLst;
typedef Ptr p_Range;
typedef Ptr p_RangeId;
typedef Ptr p_RangeIdLst;
typedef Ptr p_FunctionCall;
typedef Ptr p_Lvalue;
typedef Ptr p_CaseItem;
typedef Ptr p_CaseItemLst;
typedef Ptr p_DelayEvntCtl;
typedef Ptr p_EvntExpression;
typedef Ptr p_EvntExpressionLst;
typedef Ptr p_TaskStmt;
typedef Ptr p_AssignStmt;
typedef Ptr p_SeqBlkStmt;
typedef Ptr p_Stmts;
typedef Ptr p_StmtsLst;
typedef Ptr p_WireDecl;
typedef Ptr p_RegDecl;
typedef Ptr p_InputDecl;
typedef Ptr p_OutputDecl;
typedef Ptr p_InoutDecl;
typedef Ptr p_TFDecl;
typedef Ptr p_TFDeclLst;
typedef Ptr p_Function;
typedef Ptr p_Task;
typedef Ptr p_InitialStmt;
typedef Ptr p_AlwaysStmt;
typedef Ptr p_PortConn;
typedef Ptr p_PortConnLst;
typedef Ptr p_ModInstance;
typedef Ptr p_ModInstanceLst;
typedef Ptr p_ModInstan;
typedef Ptr p_ModuleItem;
typedef Ptr p_ModuleItemLst;
typedef Ptr p_Port;
typedef Ptr p_PortLst;
typedef Ptr p_Module;

enum EXPR
	{			/* Expression */
	EXPR_STR,		/*	String constant */
	EXPR_NUM,		/*	Numeric constant */
	EXPR_ID,		/*	Identifier */
	EXPR_FUNC,		/*	System function */
	EXPR_NOTOP,		/*	Not (!) operator */
	EXPR_INVOP,		/*	Invert (~) operator */
	EXPR_ADD,		/*	Add (+) operator */
	EXPR_SUB,		/*	Subtract (-) operator */
	EXPR_EQEQ,		/*	Equality comparator (==) */
	EXPR_NOTEQ,		/*	Not equal comparator (!=) */
	EXPR_OROR,		/*	Union of expresions (||) */
	EXPR_ANDAND,		/*	Intersection of expressions (&&) */
	EXPR_LOGOR,		/*	Logical bitwise or (|) */
	EXPR_LOGAND,		/*	Logical bitwise and (&) */
	EXPR_LOGLSHIFT,		/*	Logical left shift */
	EXPR_LOGRSHIFT,		/*	Logical right shift */
	EXPR_GREATEREQ,		/*	Greater than or equal */
	EXPR_GRTTHAN,		/*	Greater than */
	EXPR_LESSEQ,		/*	Less than or equal */
	EXPR_LESTHAN,		/*	Less than */
	EXPR_EXP		/*	Another expression */
	};
enum LVAL
	{			/* Lvalue */
	LVAL_RNGE,		/*	Range identifier */
	LVAL_ID			/*	Identifier */
	};
enum DEC
	{			/* Delay/Event control */
	DEC_DLAYNUM,		/*	Delay by constant number */
	DEC_DLAYID,		/*	Delay by variable name */
	DEC_EVNTID,		/*	Event by variable name */
	DEC_EVNTEXPR		/*	Event by expression */
	};
enum EVNT
	{			/* Event trigger type */
	EVNT_POS,		/*	Posedge trigger */
	EVNT_NEG,		/*	Negedge trigger */
	EVNT_CHNG,		/*	Any state change */
	EVNT_NONE		/*	Invalid */
	};
enum CASE
	{			/* Case */
	CASEITEM,		/*	Individual case item */
	DEFAULTITEM		/*	Default case */
	};
enum STMT
	{			/* Statements */
	STMT_TASK,		/*	Task/system task enable */
	STMT_ASSIGN,		/*	Assignment */
	STMT_SEQBLK,		/*	Sequential block */
	STMT_STMT,		/*	Another statement */
	STMT_IF,		/*	If statement */
	STMT_IF_ELSE,		/*	If else statement */
	STMT_FOR,		/*	For loop */
	STMT_CASE,		/*	Case statement */
	STMT_NULL		/*	Null statement */
	};
enum PS
	{			/* Procedural statements */
	PS_ALWAYS,		/*	Always */
	PS_INITIAL		/*	Initial */
	};
enum DECL
	{			/* Declarations */
	DECL_REG,		/*	Register */
	DECL_INPUT,		/*	Input */
	DECL_OUTPUT,		/*	Output */
	DECL_INOUT,		/*	Inout */
	DECL_DECL,		/*	Another declaration */
	DECL_NULL		/*	Null declaration */
	};
enum MODITM
	{			/* Module item */
	ITEM_WIREDECL,		/*	Wire */
	ITEM_REGDECL,		/*	Register */
	ITEM_INPUTDECL,		/*	Input */
	ITEM_OUTPUTDECL,	/*	Output */
	ITEM_INOUTDECL,		/*	Inout */
	ITEM_FUNC,		/*	Function */
	ITEM_TASK,		/*	Task */
	ITEM_INITIAL,		/*	Initial Statement */
	ITEM_ALWAYS,		/*	Always Statement */
	ITEM_INSTAN,		/*	Module instantiation */
	ITEM_ITEM,		/*	Another item */
	ITEM_NULL		/*	Null item */
	};

/* Misc functions */
extern int p_init(int amt, char **lst);
extern int p_start(void);

/* Number */
extern p_Number *p_create_num(char *s);
extern void *P_CreateNumber(char *str, int b, int l);

/* Constant Expression */
extern p_ConstExpression *p_create_constexpr(p_Expression *expr);
extern void *P_CreateConstExpr(p_Expression *expr);

/* Expression */
extern p_Expression *p_create_expr(int type, void *p1, void *p2);
extern p_ExpressionLst *p_create_exprlst(p_ExpressionLst *lst,
		p_Expression *p);
extern void *P_CreateExpr(int type, void *p1, void *p2);
extern void *P_CreateExprLst(p_ExpressionLst *lst, p_Expression *p);

/* Range */
extern p_Range *p_create_range(p_Expression *l, p_Expression *r);
extern void *P_CreateRange(p_Expression *l, p_Expression *r);

/* RangeId */
extern p_RangeId *p_create_rangeid(char *str, p_Range *r);
extern p_RangeIdLst *p_create_rangeidlst(p_RangeIdLst *lst, p_RangeId *id);
extern void *P_CreateRangeId(char *str, p_Range *r);
extern void *P_CreateRangeIdLst(p_RangeIdLst *lst, p_RangeId *id);

/* Function call */
extern p_FunctionCall *p_create_functioncall(p_RangeId *ri,
	p_ExpressionLst *lst);
extern void *P_CreateFunctionCall(p_RangeId *ri, p_ExpressionLst *lst);

/* Lvalue */
extern p_Lvalue *p_create_lval(p_RangeId *ri);
extern void *P_CreateLvalue(p_RangeId *ri);

/* Delay or Event control */
extern p_DelayEvntCtl *p_create_delayevntctl(int type, void *p);
extern void *P_CreateDelayEvntCtl(int type, void *p);
extern p_EvntExpression *p_create_evntexpression(int type, p_Expression *p);
extern void *P_CreateEvntExpression(int type, p_Expression *p);
extern p_EvntExpressionLst *
	p_create_evntexpressionlst(p_EvntExpressionLst *lst,
		p_EvntExpression *p);
extern void *P_CreateEvntExpressionLst(p_EvntExpressionLst *lst,
	p_EvntExpression *p);

/* Case item */
extern p_CaseItem *p_create_caseitem(int type, p_ExpressionLst *p, p_Stmts *s);
extern p_CaseItemLst *p_create_caseitemlst(p_CaseItemLst *cl, p_CaseItem *c);
extern void *P_CreateCaseItem(int type, p_ExpressionLst *p, p_Stmts *s);
extern void *P_CreateCaseItemLst(p_CaseItemLst *cl, p_CaseItem *c );

/* Task enable statement */
extern p_TaskStmt *p_create_taskstmt(p_RangeId *r, p_ExpressionLst *lst);
extern void *P_CreateTaskStmt(p_RangeId *r, p_ExpressionLst *p);

/* Assignment statement */
extern p_AssignStmt *p_create_assign(p_Lvalue *lval, p_Expression *expr);
extern void *P_CreateAssignStmt(p_Lvalue *lval, p_Expression *expr);

/* Sequential (begin ... end) statement */
extern p_SeqBlkStmt *p_create_seqblk(p_RangeId *name, p_StmtsLst *stmts);
extern void *P_CreateSeqBlkStmt(p_RangeId *id, p_StmtsLst *p);

/* Statements */
extern p_Stmts *p_create_stmts(int type, void *p1, void *p2, void *p3,
		void *p4);
extern p_StmtsLst *p_create_stmtslst(p_StmtsLst *lst, int type, p_Stmts *p);
extern void *P_CreateStmts(int type, void *p1, void *p2, void *p3, void *p4);
extern void *P_CreateStmtsLst(p_StmtsLst *lst, int type, p_Stmts *p);

/* Declarations */
extern p_RegDecl *p_create_regdecl(p_Range *range, p_RangeIdLst *ids);
extern void *P_CreateRegDecl(p_Range *range, p_RangeIdLst *ids);
extern p_WireDecl *p_create_wiredecl(p_Range *range, p_RangeIdLst *ids);
extern void *P_CreateWireDecl(p_Range *range, p_RangeIdLst *ids);

/* IO Declarations */
extern p_InputDecl *p_create_inputdecl(p_Range *range, p_RangeIdLst *id);
extern void *P_CreateInputDecl(p_Range *range, p_RangeIdLst *lst);
extern p_OutputDecl *p_create_outputdecl(p_Range *range, p_RangeIdLst *id);
extern void *P_CreateOutputDecl(p_Range *range, p_RangeIdLst *lst);
extern p_InoutDecl *p_create_inoutdecl(p_Range *range, p_RangeIdLst *id);
extern void *P_CreateInoutDecl(p_Range *range, p_RangeIdLst *lst);

/* TF Declarations */
extern p_TFDecl *p_create_tfdecl(int type, void *p);
extern p_TFDeclLst *p_create_tfdecllst(p_TFDeclLst *lst, int type,
	p_TFDecl *p);
extern void *P_CreateTFDecl(int type, void *p);
extern void *P_CreateTFDeclLst(p_TFDeclLst *lst, int type,
	p_TFDecl *p);

/* Function */
extern p_Function *p_create_function(p_Range *r, p_RangeId *n,
	p_TFDeclLst *decl, p_Stmts *st);
extern void *P_CreateFunction(p_Range *r, p_RangeId *n,
	p_TFDeclLst *decl, p_Stmts *st);

/* Task */
extern p_Task *p_create_task(p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st);
extern void *P_CreateTask(p_RangeId *n, p_TFDeclLst *decl, p_Stmts *st);

/* Procedural Statements */
extern p_InitialStmt *p_create_initialstmt(p_Stmts *st);
extern void *P_CreateInitialStmt(p_Stmts *st);
extern p_AlwaysStmt *p_create_alwaysstmt(p_Stmts *st);
extern void *P_CreateAlwaysStmt(p_Stmts *st);

/* Module instantiation */
extern p_PortConn *p_create_portconn(p_Expression *p);
extern void *P_CreatePortConn(p_Expression *p);
extern p_PortConnLst *p_create_portconnlst(p_PortConnLst *lst, p_PortConn *p);
extern void *P_CreatePortConnLst(p_PortConnLst *lst, p_PortConn *p);
extern p_ModInstance *p_create_modinstance(p_RangeId *id, p_PortConnLst *lst);
extern void *P_CreateModInstance(p_RangeId *id, p_PortConnLst *lst);
extern p_ModInstanceLst *p_create_modinstancelst(p_ModInstanceLst *lst,
	p_ModInstance *p);
extern void *P_CreateModInstanceLst(p_ModInstanceLst *lst, p_ModInstance *p);
extern p_ModInstan *p_create_modinstan(p_RangeId *id, p_ModInstanceLst *lst);
extern void *P_CreateModInstan(p_RangeId *id, p_ModInstanceLst *lst);

/* Module Items */
extern p_ModuleItem *p_create_moditem(int type, void *p);
extern p_ModuleItemLst *p_create_moditemlst(p_ModuleItemLst *lst,
	int type, p_ModuleItem *p);
extern void *P_CreateModItem(int type, void *p);
extern void *P_CreateModItemLst(p_ModuleItemLst *lst, int type,
	p_ModuleItem *p);

/* Ports */
extern p_Port *p_create_port(p_RangeId *id);
extern void *P_CreatePort(p_RangeId *id);
extern p_PortLst *p_create_portlst(p_PortLst *lst, p_Port *id);
extern void *P_CreatePortLst(p_PortLst *lst, p_Port *id);

/* Module */
extern p_Module *p_create_module(p_RangeId *id, p_ModuleItemLst *lst,
	p_PortLst *plst);
extern void *P_CreateModule(p_RangeId *id, p_ModuleItemLst *lst,
	p_PortLst *plst);

#ifdef __cplusplus
}
#endif

#endif /* INTRFACE_H */
