%{
/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * vbs.l
 *
 * Verilog Behavioral Simulator lexical scanner.
 */

#include <stdlib.h>
#include <string.h>
#include <unistd.h>		/* isatty(flex) */
#include <stdio.h>		/* fileno(flex) */
#include "glo.h"
#include "vbs_yacc.h"

extern void copy_word(char *str);
extern void save_word(char *str);
int cur_lineno = 1;
char *cur_filename = NULL;
char *savedword = NULL;

%}

%s COMMENT

NL		[\n]
WhiteSpace	[\t\r\b{NL} ]
Digit		[0-9]
DigitU		[0-9_]
Letter		[a-zA-Z]
LetterU		[a-zA-Z_]
WordNum		[0-9a-zA-Z]
WordNumU	[0-9a-zA-Z_]
WhiteSpaces	{WhiteSpace}+
Number		{Digit}{DigitU}*
Word		{LetterU}{WordNumU}*
Binary		({Number})?'[bB][01xXzZ?][01xXzZ?_]*
Octal		({Number})?'[oO][0-7xXzZ?][0-7xXzZ?_]*
Decimal		({Number})?'[dD]{Number}
Hexdecimal	({Number})?'[hH][0-9a-fA-FxXzZ?][0-9a-fA-FxXzZ?_]*
QoutedString	\"[^"\n]*["{NL}]
Comment		"//"[^\n]*{NL}

%%

<INITIAL>{NL}		{ cur_lineno++; continue; }
<INITIAL>{WhiteSpaces}	{ if (yytext[0] == '\n') cur_lineno++; continue; }
<INITIAL>{Comment}	{ cur_lineno++; continue; }
<INITIAL>"/*"		{ BEGIN COMMENT; continue; }
<COMMENT>{NL}		{ cur_lineno++; continue; }
<COMMENT>.		{ continue; }
<COMMENT>"*/"		{ BEGIN INITIAL; continue; }

<INITIAL>">="		{ save_word(">="); return YYGREATEREQ; }
<INITIAL>"<="		{ save_word("<="); return YYLESSEQ; }
<INITIAL>"=="		{ save_word("=="); return YYEQUALEQUAL; }
<INITIAL>"!="		{ save_word("!="); return YYNOTEQUAL; }
<INITIAL>"||"		{ save_word("||"); return YYOROR; }
<INITIAL>"&&"		{ save_word("&&"); return YYANDAND; }
<INITIAL>"<<"		{ save_word("<<"); return YYLOGLSHIFT; }
<INITIAL>">>"		{ save_word(">>"); return YYLOGRSHIFT; }
<INITIAL>always		{ save_word("always"); return YYALWAYS; }
<INITIAL>begin		{ save_word("begin"); return YYBEGIN; }
<INITIAL>case		{ save_word("case"); return YYCASE; }
<INITIAL>default	{ save_word("default"); return YYDEFAULT; }
<INITIAL>else		{ save_word("else"); return YYELSE; }
<INITIAL>end		{ save_word("end"); return YYEND; }
<INITIAL>endcase	{ save_word("endcase"); return YYENDCASE; }
<INITIAL>endfunction	{ save_word("endfunction"); return YYENDFUNCTION; }
<INITIAL>endmodule	{ save_word("endmodule"); return YYENDMODULE; }
<INITIAL>endtask	{ save_word("endtask"); return YYENDTASK; }
<INITIAL>for		{ save_word("for"); return YYFOR; }
<INITIAL>function	{ save_word("function"); return YYFUNCTION; }
<INITIAL>if		{ save_word("if"); return YYIF; }
<INITIAL>initial	{ save_word("initial"); return YYINITIAL; }
<INITIAL>inout		{ save_word("inout"); return YYINOUT; }
<INITIAL>input		{ save_word("input"); return YYINPUT; }
<INITIAL>module		{ save_word("module"); return YYMODULE; }
<INITIAL>negedge	{ save_word("negedge"); return YYNEGEDGE; }
<INITIAL>or		{ save_word("or"); return YYOR; }
<INITIAL>output		{ save_word("output"); return YYOUTPUT; }
<INITIAL>posedge	{ save_word("posedge"); return YYPOSEDGE; }
<INITIAL>reg		{ save_word("reg"); return YYREG; }
<INITIAL>task		{ save_word("task"); return YYTASK; }
<INITIAL>wire		{ save_word("wire"); return YYWIRE; }
<INITIAL>\${Word}	{ copy_word(yytext); return YYSYSIDENT; }
<INITIAL>{Binary}	{ copy_word(yytext); return YYNUMBER; }
<INITIAL>{Octal}	{ copy_word(yytext); return YYNUMBER; }
<INITIAL>{Decimal}	{ copy_word(yytext); return YYNUMBER; }
<INITIAL>{Hexdecimal}	{ copy_word(yytext); return YYNUMBER; }
<INITIAL>{Number}	{ copy_word(yytext); return YYNUMBER; }
<INITIAL>{QoutedString}	{ copy_word(yytext); return YYQSTRING; }
<INITIAL>{Word}		{ copy_word(yytext); return YYWORD; }
<INITIAL>.		{ return(yytext[0]); }

%%

void
copy_word(char *str)
	{
	yylval.word = my_strdup(str);
	if (savedword != NULL)
		{
		free(savedword);
		savedword = NULL;
		}
	}

void
save_word(char *str)
	{
	if (savedword != NULL)
		free(savedword);
	savedword = my_strdup(str);
	}
