%{
/*
 * Verilog Behavioral Simulator
 * Copyright (C) 1995 Lay Hoon Tho, Jimen Ching
 *
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Special Contributions:
 *
 * The authors of this software would like to thank the University of 
 * Hawaii, College of Engineering for the use of their computer 
 * facilities in the course of the development of this software.  Special 
 * thanks to Dr. Alex Quilici, who is the advisor for this project, and Dr. 
 * Michael Smith, who taught us how to use Verilog.  We would also like
 * to thank the College of Engineering for the knowledge we have gained
 * through their engineering curriculum.
 *
 * Authors:
 *	Lay Hoon Tho
 *		8, Jalan Setia 6 Chamek,
 *		Kluang, 86000 Johore,
 *		Malaysia
 *
 *	Jimen Ching
 *		2108 Citron St. Apt. #2
 *		Honolulu, HI 96826
 *		USA
 *		(jching@aloha.com)
 */
/*
 * vbs.y
 *
 * Verilog Behavioral Simulator yacc grammar parser.
 */

#include <stdio.h>
#include <stdlib.h>
#include "glo.h"

%}

%union
	{
	char			*word;
	p_Number		*num;
	p_Expression		*expr;
	p_ConstExpression	*cexpr;
	p_ExpressionLst		*exprlst;
	p_Range			*range;
	p_RangeId		*rangeid;
	p_RangeIdLst		*rangeidlst;
	p_FunctionCall		*funccall;
	p_Lvalue		*lval;
	p_DelayEvntCtl		*dec;
	p_EvntExpression	*evntexpr;
	p_EvntExpressionLst	*evntexprlst;
	p_CaseItem		*caseitem;
	p_CaseItemLst		*caseitemlst;
	p_TaskStmt		*taskstmt;
	p_AssignStmt		*assign;
	p_SeqBlkStmt		*seqblk;
	p_Stmts			*stmts;
	p_StmtsLst		*stmtslst;
	p_TFDecl		*tfdecl;
	p_TFDeclLst		*tfdecllst;
	p_WireDecl		*wiredecl;
	p_RegDecl		*regdecl;
	p_InputDecl		*inputdecl;
	p_OutputDecl		*outputdecl;
	p_InoutDecl		*inoutdecl;
	p_Task			*task;
	p_Function		*func;
	p_InitialStmt		*initial;
	p_AlwaysStmt		*always;
	p_PortConn		*portconn;
	p_PortConnLst		*portconnlst;
	p_ModInstance		*modinstance;
	p_ModInstanceLst	*modinstancelst;
	p_ModInstan		*modinstan;
	p_ModuleItem		*moditem;
	p_ModuleItemLst		*moditemlst;
	p_Port			*port;
	p_PortLst		*portlst;
	p_Module		*mod;
	};

%token YYALWAYS			/* always */
%token YYANDAND			/* && */
%token YYBEGIN			/* begin */
%token YYCASE			/* case */
%token YYDEFAULT		/* default */
%token YYELSE			/* else */
%token YYEND			/* end */
%token YYENDCASE		/* endcase */
%token YYENDFUNCTION		/* endfunction */
%token YYENDMODULE		/* endmodule */
%token YYENDTASK		/* endtask */
%token YYEQUALEQUAL		/* == */
%token YYFOR			/* for */
%token YYFUNCTION		/* function */
%token YYGREATEREQ		/* >= */
%token YYIF			/* if */
%token YYINITIAL		/* initial */
%token YYINOUT			/* inout */
%token YYINPUT			/* input */
%token YYLESSEQ			/* <= */
%token YYLOGLSHIFT		/* << */
%token YYLOGRSHIFT		/* >> */
%token YYMODULE			/* module */
%token YYNEGEDGE		/* negedge */
%token YYNOTEQUAL		/* != */
%token YYOR			/* or */
%token YYOROR			/* || */
%token YYOUTPUT			/* output */
%token YYPOSEDGE		/* posedge */
%token YYREG			/* reg */
%token YYTASK			/* task */
%token YYWIRE			/* wire */
%token <word> YYSYSIDENT	/* $... */
%token <word> YYQSTRING		/* qouted string i.e. "abc" */
%token <word> YYNUMBER		/* number */
%token <word> YYWORD		/* symbol name */

%left YYOR
%left YYOROR
%left YYANDAND
%left '|'
%left '&'
%left YYEQUALEQUAL YYNOTEQUAL
%left YYGREATEREQ YYLESSEQ '>' '<'
%left YYLOGLSHIFT YYLOGRSHIFT
%left '+' '-'
%right '~' '!'
%nonassoc LT_YYELSE
%nonassoc YYELSE

%type <num> number
%type <rangeid> name_of_system_function
%type <funccall> function_call
%type <expr> primary
%type <expr> expression
%type <cexpr> const_expression
%type <exprlst> expression_plus
%type <rangeid> identifier
%type <lval> lvalue
%type <seqblk> seq_block
%type <rangeid> system_identifier
%type <rangeid> name_of_system_task
%type <taskstmt> system_task_enable
%type <taskstmt> task_enable
%type <dec> delay_or_event_cntl
%type <dec> delay_control
%type <dec> event_control
%type <caseitem> case_item
%type <caseitemlst> case_item_plus
%type <evntexpr> event_expression
%type <evntexprlst> ored_event_expression
%type <assign> blocking_assign
%type <stmts> statement
%type <stmts> statement_or_null
%type <stmtslst> statement_star
%type <initial> initial_statement
%type <always> always_statement
%type <rangeid> name_of_register
%type <rangeidlst> list_of_registers
%type <range> range
%type <range> range_question
%type <wiredecl> wire_declaration
%type <regdecl> reg_declaration
%type <inputdecl> input_declaration
%type <outputdecl> output_declaration
%type <inoutdecl> inout_declaration
%type <tfdecl> tf_declaration
%type <tfdecllst> tf_declaration_star
%type <tfdecllst> tf_declaration_plus
%type <task> task
%type <rangeid> name_of_task
%type <func> function
%type <rangeid> name_of_function
%type <portconn> module_port_connection
%type <portconnlst> module_connections_plus
%type <rangeid> name_of_instance
%type <modinstance> module_instance
%type <modinstancelst> module_instance_plus
%type <modinstan> module_instantiation
%type <moditem> module_item
%type <moditemlst> module_item_star
%type <rangeid> name_of_module
%type <port> port
%type <portlst> list_of_ports_question
%type <portlst> port_plus
%type <mod> module

%%

source_text:
		description_star
	;

description_star:
		/* empty */
	|	description_star description
	;

description:
		module
			{
			/* Save the module so we can let yywrap handle it. */
			parsed_module($1);
			}
	;

module:
		YYMODULE name_of_module list_of_ports_question ';'
				module_item_star
		YYENDMODULE
			{
			$$ = p_create_module($2, $5, $3);
			}
	;

name_of_module:
		YYWORD
			{
			$$ = p_create_rangeid($1, NULL);
			}
	;

list_of_ports_question:
		/* empty */
			{
			$$ = NULL;
			}
	|	'(' port_plus ')'
			{
			$$ = $2;
			}
	;

port_plus:
		port
			{
			$$ = p_create_portlst(NULL, $1);
			}
	|	port_plus ',' port
			{
			$$ = p_create_portlst($1, $3);
			}
	;

port:
		identifier
			{
			$$ = p_create_port($1);
			}
	;

module_item_star:
		/* empty */
			{
			$$ = p_create_moditemlst(NULL, ITEM_NULL, NULL);
			}
	|	module_item_star module_item
			{
			$$ = p_create_moditemlst($1, ITEM_ITEM, $2);
			}
	;

module_item:
		wire_declaration
			{
			$$ = p_create_moditem(ITEM_WIREDECL, $1);
			}
	|	reg_declaration
			{
			$$ = p_create_moditem(ITEM_REGDECL, $1);
			}
	|	input_declaration
			{
			$$ = p_create_moditem(ITEM_INPUTDECL, $1);
			}
	|	output_declaration
			{
			$$ = p_create_moditem(ITEM_OUTPUTDECL, $1);
			}
	|	inout_declaration
			{
			$$ = p_create_moditem(ITEM_INOUTDECL, $1);
			}
	|	task
			{
			$$ = p_create_moditem(ITEM_TASK, $1);
			}
	|	function
			{
			$$ = p_create_moditem(ITEM_FUNC, $1);
			}
	|	initial_statement
			{
			$$ = p_create_moditem(ITEM_INITIAL, $1);
			}
	|	always_statement
			{
			$$ = p_create_moditem(ITEM_ALWAYS, $1);
			}
	|	module_instantiation
			{
			$$ = p_create_moditem(ITEM_INSTAN, $1);
			}
	;

wire_declaration:
		YYWIRE range_question list_of_registers ';'
			{
			$$ = p_create_wiredecl($2, $3);
			}
	;

reg_declaration:
		YYREG range_question list_of_registers ';'
			{
			$$ = p_create_regdecl($2, $3);
			}
	;

input_declaration:
		YYINPUT range_question list_of_registers ';'
			{
			$$ = p_create_inputdecl($2, $3);
			}
	;

output_declaration:
		YYOUTPUT range_question list_of_registers ';'
			{
			$$ = p_create_outputdecl($2, $3);
			}
	;

inout_declaration:
		YYINOUT range_question list_of_registers ';'
			{
			$$ = p_create_inoutdecl($2, $3);
			}
	;

range_question:
		/* empty */
			{
			$$ = NULL;
			}
	|	range
			{
			$$ = $1;
			}
	;

range:
		'[' const_expression ':' const_expression ']'
			{
			$$ = p_create_range($2, $4);
			}
	;

list_of_registers:
		name_of_register
			{
			$$ = p_create_rangeidlst(NULL, $1);
			}
	|	list_of_registers ',' name_of_register
			{
			$$ = p_create_rangeidlst($1, $3);
			}
	;

name_of_register:
		YYWORD
			{
			$$ = p_create_rangeid($1, NULL);
			}
	;

task:
		YYTASK name_of_task ';'
				tf_declaration_star
				statement_or_null
		YYENDTASK
			{
			$$ = p_create_task($2, $4, $5);
			}

name_of_task:
		YYWORD
			{
			$$ = p_create_rangeid($1, NULL);
			}
	;

function:
		YYFUNCTION range_question name_of_function ';'
				tf_declaration_plus
				statement
		YYENDFUNCTION
			{
			$$ = p_create_function($2, $3, $5, $6);
			}
	;

name_of_function:
		YYWORD
			{
			$$ = p_create_rangeid($1, NULL);
			}
	;

tf_declaration_plus:
		tf_declaration
			{
			$$ = p_create_tfdecllst(NULL, DECL_DECL, $1);
			}
	|	tf_declaration_plus tf_declaration
			{
			$$ = p_create_tfdecllst($1, DECL_DECL, $2);
			}
	;

tf_declaration_star:
		/* empty */
			{
			$$ = p_create_tfdecllst(NULL, DECL_NULL, NULL);
			}
	|	tf_declaration_star tf_declaration
			{
			$$ = p_create_tfdecllst($1, DECL_DECL, $2);
			}
	;

tf_declaration:
		reg_declaration
			{
			$$ = p_create_tfdecl(DECL_REG, $1);
			}
	|	input_declaration
			{
			$$ = p_create_tfdecl(DECL_INPUT, $1);
			}
	|	output_declaration
			{
			$$ = p_create_tfdecl(DECL_OUTPUT, $1);
			}
	|	inout_declaration
			{
			$$ = p_create_tfdecl(DECL_INOUT, $1);
			}
	;

initial_statement:
		YYINITIAL statement
			{
			$$ = p_create_initialstmt($2);
			}
	;

always_statement:
		YYALWAYS statement
			{
			$$ = p_create_alwaysstmt($2);
			}
	;

module_instantiation:
		name_of_module module_instance_plus ';'
			{
			$$ = p_create_modinstan($1, $2);
			}
	;

module_instance_plus:
		module_instance
			{
			$$ = p_create_modinstancelst(NULL, $1);
			}
	|	module_instance_plus ',' module_instance
			{
			$$ = p_create_modinstancelst($1, $3);
			}
	;

module_instance:
		name_of_instance '(' module_connections_plus ')'
			{
			/*
			 * For some reason, if I make module_connections
			 * optional, I get a reduce/reduce warning from
			 * yacc.  I do not know how to fix this problem
			 * so optional module port connections are not
			 * supported.
			 */
			$$ = p_create_modinstance($1, $3);
			}
	;

name_of_instance:
		identifier
			{
			$$ = $1;
			}
	;

module_connections_plus:
		module_port_connection
			{
			$$ = p_create_portconnlst(NULL, $1);
			}
	|	module_connections_plus ',' module_port_connection
			{
			$$ = p_create_portconnlst($1, $3);
			}
	;

module_port_connection:
		/* empty */
			{
			$$ = p_create_portconn(NULL);
			}
	|	expression
			{
			$$ = p_create_portconn($1);
			}
	;

statement_or_null:
		statement
			{
			$$ = p_create_stmts(STMT_STMT, $1, NULL, NULL, NULL);
			}
	|	';'
			{
			$$ = p_create_stmts(STMT_NULL, NULL, NULL, NULL, NULL);
			}
	;

statement_star:
		/* empty */
			{
			$$ = p_create_stmtslst(NULL, STMT_NULL, NULL);
			}
	|	statement_star statement
			{
			$$ = p_create_stmtslst($1, STMT_STMT, $2);
			}
	;

statement:
		blocking_assign ';'
			{
			$$ = p_create_stmts(STMT_ASSIGN, $1, NULL, NULL, NULL);
			}
	|	delay_or_event_cntl statement_or_null
			{
			$$ = p_create_stmts(STMT_STMT, $2, $1, NULL, NULL);
			}
	|	task_enable
			{
			$$ = p_create_stmts(STMT_TASK, $1, NULL, NULL, NULL);
			}
	|	system_task_enable
			{
			$$ = p_create_stmts(STMT_TASK, $1, NULL, NULL, NULL);
			}
	|	seq_block
			{
			$$ = p_create_stmts(STMT_SEQBLK, $1, NULL, NULL, NULL);
			}
	|	YYIF '(' expression ')' statement_or_null	%prec LT_YYELSE
			{
			$$ = p_create_stmts(STMT_IF, $3, $5, NULL, NULL);
			}
	|	YYIF '(' expression ')' statement_or_null YYELSE statement_or_null
			{
			$$ = p_create_stmts(STMT_IF_ELSE, $3, $5, $7, NULL);
			}
	|	YYFOR '(' blocking_assign ';' expression ';' blocking_assign ')'
			statement
			{
			$$ = p_create_stmts(STMT_FOR, $3, $5, $7, $9);
			}
	|	YYCASE '(' expression ')' case_item_plus YYENDCASE
			{
			$$ = p_create_stmts(STMT_CASE, $3, $5, NULL, NULL);
			}
	;

blocking_assign:
		lvalue '=' expression
			{
			$$ = p_create_assign($1, $3);
			}
	;

delay_or_event_cntl:
		delay_control
	|	event_control
	;

delay_control:
		'#' number
			{
			$$ = p_create_delayevntctl(DEC_DLAYNUM, $2);
			}
	|	'#' identifier
			{
			$$ = p_create_delayevntctl(DEC_DLAYID, $2);
			}
	;

event_control:
		'@' identifier
			{
			$$ = p_create_delayevntctl(DEC_EVNTID, $2);
			}
	|	'@' '(' ored_event_expression ')'
			{
			$$ = p_create_delayevntctl(DEC_EVNTEXPR, $3);
			}
	;

event_expression:
		expression
			{
			$$ = p_create_evntexpression(EVNT_CHNG, $1);
			}
	|	YYPOSEDGE expression
			{
			$$ = p_create_evntexpression(EVNT_POS, $2);
			}
	|	YYNEGEDGE expression
			{
			$$ = p_create_evntexpression(EVNT_NEG, $2);
			}
	;

case_item_plus:
		case_item
			{
			$$ = p_create_caseitemlst(NULL, $1);
			}
	|	case_item_plus case_item
			{
			$$ = p_create_caseitemlst($1, $2);
			}

case_item:
		expression_plus ':' statement_or_null
			{
			$$ = p_create_caseitem(CASEITEM, $1, $3);
			}
	|	YYDEFAULT ':' statement_or_null
			{
			$$ = p_create_caseitem(DEFAULTITEM, NULL, $3);
			}
	|	YYDEFAULT statement_or_null
			{
			$$ = p_create_caseitem(DEFAULTITEM, NULL, $2);
			}

ored_event_expression:
		event_expression
			{
			$$ = p_create_evntexpressionlst(NULL, $1);
			}
	|	ored_event_expression YYOR event_expression
			{
			$$ = p_create_evntexpressionlst($1, $3);
			}
	;

system_task_enable:
		name_of_system_task ';'
			{
			$$ = p_create_taskstmt($1, NULL);
			}
	|	name_of_system_task '(' expression_plus ')' ';'
			{
			$$ = p_create_taskstmt($1, $3);
			}
	;

name_of_system_task:
		system_identifier
	;

system_identifier:
		YYSYSIDENT
			{
			$$ = p_create_rangeid($1, NULL);
			}
	;

task_enable:
		name_of_task ';'
			{
			$$ = p_create_taskstmt($1, NULL);
			}
	|	name_of_task '(' expression_plus ')' ';'
			{
			$$ = p_create_taskstmt($1, $3);
			}
	;

seq_block:
		YYBEGIN statement_star YYEND
			{
			$$ = p_create_seqblk(NULL, $2);
			}
	;

lvalue:
		identifier
			{
			$$ = p_create_lval($1);
			}
	;

identifier:
		YYWORD
			{
			$$ = p_create_rangeid($1, NULL);
			}
	|	YYWORD range
			{
			$$ = p_create_rangeid($1, $2);
			}
	;

expression_plus:
		expression
			{
			$$ = p_create_exprlst(NULL, $1);
			}
	|	expression_plus ',' expression
			{
			$$ = p_create_exprlst($1, $3);
			}
	;

const_expression:
		expression
			{
			$$ = p_create_constexpr($1);
			}
	;

expression:
		primary
			{
			$$ = p_create_expr(EXPR_EXP, $1, NULL);
			}
	|	YYQSTRING
			{
			$$ = p_create_expr(EXPR_STR, $1, NULL);
			}
	|	'!' primary
			{
			$$ = p_create_expr(EXPR_NOTOP, $2, NULL);
			}
	|	'~' primary
			{
			$$ = p_create_expr(EXPR_INVOP, $2, NULL);
			}
	|	expression '+' expression
			{
			$$ = p_create_expr(EXPR_ADD, $1, $3);
			}
	|	expression '-' expression
			{
			$$ = p_create_expr(EXPR_SUB, $1, $3);
			}
	|	expression YYEQUALEQUAL expression
			{
			$$ = p_create_expr(EXPR_EQEQ, $1, $3);
			}
	|	expression YYNOTEQUAL expression
			{
			$$ = p_create_expr(EXPR_NOTEQ, $1, $3);
			}
	|	expression YYOROR expression
			{
			$$ = p_create_expr(EXPR_OROR, $1, $3);
			}
	|	expression YYANDAND expression
			{
			$$ = p_create_expr(EXPR_ANDAND, $1, $3);
			}
	|	expression '|' expression
			{
			$$ = p_create_expr(EXPR_LOGOR, $1, $3);
			}
	|	expression '&' expression
			{
			$$ = p_create_expr(EXPR_LOGAND, $1, $3);
			}
	|	expression YYLOGLSHIFT expression
			{
			$$ = p_create_expr(EXPR_LOGLSHIFT, $1, $3);
			}
	|	expression YYLOGRSHIFT expression
			{
			$$ = p_create_expr(EXPR_LOGRSHIFT, $1, $3);
			}
	|	expression YYGREATEREQ expression
			{
			$$ = p_create_expr(EXPR_GREATEREQ, $1, $3);
			}
	|	expression '>' expression
			{
			$$ = p_create_expr(EXPR_GRTTHAN, $1, $3);
			}
	|	expression YYLESSEQ expression
			{
			$$ = p_create_expr(EXPR_LESSEQ, $1, $3);
			}
	|	expression '<' expression
			{
			$$ = p_create_expr(EXPR_LESTHAN, $1, $3);
			}
	|	'(' expression ')'
			{
			$$ = p_create_expr(EXPR_EXP, $2, NULL);
			}
	;
	
primary:
		number
			{
			$$ = p_create_expr(EXPR_NUM, $1, NULL);
			}
	|	identifier
			{
			$$ = p_create_expr(EXPR_ID, $1, NULL);
			}
	|	function_call
			{
			$$ = p_create_expr(EXPR_FUNC, $1, NULL);
			}
	;


function_call:
		name_of_system_function
			{
			$$ = p_create_functioncall($1, NULL);
			}
	|	name_of_function '(' expression_plus ')'
			{
			$$ = p_create_functioncall($1, $3);
			}
	;

name_of_system_function:
		system_identifier
	;

number:
		YYNUMBER
			{
			$$ = p_create_num($1);
			}
	;

%%

void
yyerror(char *str)
	{
	if (savedword != NULL)
		p_perror("keyword", str, savedword, cur_filename, cur_lineno);
	else if (yylval.word != NULL && strlen(yylval.word) > 0)
		p_perror("symbol", str, yylval.word, cur_filename, cur_lineno);
	else
		p_perror("eof", str, "EOF", cur_filename, cur_lineno);
	}
