/*+-------------------------------------------------------------------------
	pcmd.c - ecu miscellaneous procedure commands
	wht@n4hgf.Mt-Park.GA.US

  Defined functions:
	get_big_endian_16(ptr)
	get_big_endian_32(ptr)
	pcmd_autorz(param)
	pcmd_baud(param)
	pcmd_cd(param)
	pcmd_clrx(param)
	pcmd_dcdwatch(param)
	pcmd_dial(param)
	pcmd_duplex(param)
	pcmd_echo(param)
	pcmd_exec(param)
	pcmd_exit(param)
	pcmd_flush(param)
	pcmd_getf(param)
	pcmd_hangup(param)
	pcmd_hexdump(param)
	pcmd_lbreak(param)
	pcmd_lgets(param)
	pcmd_logevent(param)
	pcmd_lookfor(param)
	pcmd_nap(param)
	pcmd_nice(param)
	pcmd_parity(param)
	pcmd_popd(param)
	pcmd_prompt(param)
	pcmd_ptrace(param)
	pcmd_pushd(param)
	pcmd_putf(param)
	pcmd_rname(param)
	pcmd_rtscts(param)
	pcmd_send(param)
	pcmd_set(param)
	pcmd_setline(param)
	pcmd_system(param)
	pcmd_xon(param)

--------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:09-10-1992-14:00-wht@n4hgf-ECU release 3.20 */
/*:09-06-1992-13:44-wht@n4hgf-rtscts would not accept a numeric argument */
/*:08-22-1992-15:39-wht@n4hgf-ECU release 3.20 BETA */
/*:01-12-1992-20:54-wht@n4hgf-add autorz command */
/*:12-12-1991-05:27-wht@n4hgf-proctrace of intvar shows char value if 0-255 */
/*:11-11-1991-14:38-wht@n4hgf-add pcmd_dcdwatch code */
/*:10-09-1991-21:54-wht@n4hgf-add -p and -v switch to send */
/*:10-09-1991-20:32-wht@n4hgf-proctrace code for send */
/*:09-01-1991-19:10-wht@n4hgf2-baud cmd can set rate even if no line open */
/*:09-01-1991-18:10-wht@n4hgf2-add setline */
/*:08-25-1991-14:39-wht@n4hgf-SVR4 port thanks to aega84!lh */
/*:08-06-1991-21:18-wht@n4hgf-nap -m test wrong sense ... old bug! */
/*:08-05-1991-16:22-wht@n4hgf-add nap -1 return and proctrace */
/*:07-25-1991-12:58-wht@n4hgf-ECU release 3.10 */
/*:07-17-1991-07:04-wht@n4hgf-avoid SCO UNIX nap bug */
/*:06-05-1991-22:50-wht@n4hgf-fix parity cmd not taking alpha str */
/*:05-21-1991-18:52-wht@n4hgf-add pcmd_pushd and pcmd_popd */
/*:03-16-1991-15:12-wht@n4hgf-add pcmd_nice */
/*:01-09-1991-22:31-wht@n4hgf-ISC port */
/*:12-26-1990-02:34-wht@n4hgf-add cmd_rtscts */
/*:12-03-1990-04:59-wht@n4hgf-beef up pcmd_exit */
/*:09-19-1990-19:36-wht@n4hgf-ecu_log_event now gets pid for log from caller */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#include "ecu.h"
#include "ecuerror.h"
#include "termecu.h"
#include "ecukey.h"
#include "esd.h"
#include "var.h"
#include "proc.h"

#define NAMED_VARIABLE_FLAG 0x1000L

#if defined(SVR4)
# include <sys/termiox.h>
extern int hx_flag;
#endif

extern int rc_ep_has_run;
extern ulong colors_current;
extern char errmsg[];
extern char curr_dir[CURR_DIRSIZ];		/* current working directory */

/*+-------------------------------------------------------------------------
	pcmd_autorz(param)
--------------------------------------------------------------------------*/
int
pcmd_autorz(param)
ESD *param;
{
char s8[8];

	if(get_alpha_zstr(param,s8,sizeof(s8)))
		return(eSyntaxError);
	if(!strcmp(s8,"on"))
		shm->autorz = 1;
	else if(!strcmp(s8,"off"))
		shm->autorz = 0;
	else
		return(eSyntaxError);
	shm->autorz_pos = 0;
	return(0);
}	/* end of pcmd_autorz */

/*+-------------------------------------------------------------------------
	pcmd_baud(param) - set line or default baud rate

The command sets shm->Lbaud whether or not a line is open.
If a line is open, the baud rate is actually set.
--------------------------------------------------------------------------*/
int
pcmd_baud(param)
ESD *param;
{
long new_baud;
int erc;


	if(erc = gint(param,&new_baud))
		return(erc);
	if(!valid_baud_rate((uint)new_baud))
	{
		pprintf("invalid baud rate: %lu\n",new_baud);
		return(eFATAL_ALREADY);
	}
	shm->Lbaud = (uint)new_baud;
	if(shm->Liofd >= 0)
		lset_baud_rate(1);
	if(proctrace)
	{
		pprintf("baud rate set to %u\n",shm->Lbaud);
	}
	return(0);

}	/* end of pcmd_baud */

/*+-------------------------------------------------------------------------
	pcmd_cd(param)
--------------------------------------------------------------------------*/
int
pcmd_cd(param)
ESD *param;
{
int erc;
ESD *tesd = esdalloc(256);

	if(!tesd)
		return(eNoMemory);
	if(erc = gstr(param,tesd,0))
		goto RETURN;
	if(expand_dirname(tesd->pb,tesd->maxcb))
	{
		pprintf("%s\n",errmsg);
		param->index = param->old_index;
		erc = eFATAL_ALREADY;
		goto RETURN;
	}
	if(chdir(tesd->pb) < 0)		/* now change to the new directory */
	{
		pperror(tesd->pb);		/* print error if we get one */
		pputs("\n");
		erc = eFATAL_ALREADY;
		goto RETURN;
	}
	get_curr_dir(curr_dir,256);

RETURN:
	esdfree(tesd);
	return(erc);
}	/* end of pcmd_cd */

/*+-------------------------------------------------------------------------
	pcmd_pushd(param)
--------------------------------------------------------------------------*/
int
pcmd_pushd(param)
ESD *param;
{
int erc = 0;
int arg_present;
ESD *tesd = (ESD *)0;

	if(arg_present = !!end_of_cmd(param))
	{
		if(!(tesd = esdalloc(256)))
			return(eNoMemory);
		if(erc = gstr(param,tesd,0))
			goto RETURN;
	}

	if(!push_directory((arg_present) ? tesd->pb : "",arg_present,1))
	{
		param->index = param->old_index;
		erc = eFATAL_ALREADY;
	}

RETURN:
	if(tesd)
		esdfree(tesd);
	return(erc);

}	/* end of pcmd_pushd */

/*+-------------------------------------------------------------------------
	pcmd_popd(param)
--------------------------------------------------------------------------*/
int
pcmd_popd(param)
ESD *param;
{
int erc = 0;
int arg_present;
char allstr[8];

	allstr[0] = 0;
	if(arg_present = !!end_of_cmd(param))
	{
		if(get_alpha_zstr(param,allstr,sizeof(allstr)))
		{
			param->index = param->old_index;
			return(eSyntaxError);
		} 
	}

	if(!pop_directory(allstr,arg_present,1))
	{
		param->index = param->old_index;
		erc = eFATAL_ALREADY;
	}

	return(erc);

}	/* end of pcmd_popd */

/*+-------------------------------------------------------------------------
	pcmd_clrx(param)
--------------------------------------------------------------------------*/
/*ARGSUSED*/
int
pcmd_clrx(param)
ESD *param;
{
	if(shm->Liofd < 0)
		return(eNoLineAttached);

	lclear_xmtr_xoff();
	if(proctrace)
		pputs("transmitter XOFF cleared\n");
	return(0);
}	/* end of pcmd_clrx */

/*+-------------------------------------------------------------------------
	pcmd_dcdwatch(param)
--------------------------------------------------------------------------*/
int
pcmd_dcdwatch(param)
ESD *param;
{
int erc;
char s16[16];
char *cptr;

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	if(erc = get_alpha_zstr(param,s16,sizeof(s16)))
		return(erc);

	erc = (ldcdwatch_str(s16)) ? eSyntaxError : 0;
	if(!erc && proctrace)
	{
		pputs("DCD watch set to ");
		cptr = "???";
		switch(shm->Ldcdwatch)
		{
			case DCDW_OFF:			cptr = "off"; break;
			case DCDW_ON:			cptr = "on"; break;
			case DCDW_TERMINATE:	cptr = "TERMINATE"; break;
		}
		pprintf("%s\n",cptr);
	}
	return(0);

}	/* end of pcmd_dcdwatch */

/*+-------------------------------------------------------------------------
	pcmd_dial(param) - connect to a remote DTE or to local DCE

  sets I0 to 0==connect,
             1==failed to connect,
             2==interrupted,
             3==modem error
  sets S0 to modem result code
--------------------------------------------------------------------------*/
int
pcmd_dial(param)
ESD *param;
{
int erc;
ESD *tesd = (ESD *)0;

	if(shm->Lconnected)
	{
		pprintf("Already connected (to %s)\n",shm->Llogical);
		return(eFATAL_ALREADY);
	}

	if(!(tesd = esdalloc(64)))
		return(eNoMemory);

	if(erc = gstr(param,tesd,0))
	{
		esdfree(tesd);
		return(erc);
	}

	if((erc = call_logical_telno(tesd->pb)) && (erc == eConnectFailed))
		erc = 0;

	if(!erc && (shm->Liofd < 0))
		erc = eNoLineAttached;

	esdfree(tesd);

	return(erc);
}	/* end of pcmd_dial */

/*+-------------------------------------------------------------------------
	pcmd_duplex(param)

duplex [f | h]
duplex ['f' | 'h']
duplex <int>  0 == half, non-0 == full
--------------------------------------------------------------------------*/
int
pcmd_duplex(param)
ESD *param;
{
int erc;
int new_duplex;
ESD *tesd;

	if(erc = skip_cmd_break(param))
		return(erc);
	if(!(tesd = esdalloc(64)))
		return(eNoMemory);
	erc = gstr(param,tesd,0);
	new_duplex = to_lower((erc) ? param->pb[param->index] : *tesd->pb);
	esdfree(tesd);
	erc = 0;

	switch(new_duplex)
	{
		case 'f':
			shm->Lfull_duplex = 1;
			break;
		case 'h':
			shm->Lfull_duplex = 0;
			break;
		default:
			erc = eBadParameter;
	}
	if(proctrace && !erc)
		pprintf("duplex set to %s\n",(shm->Lfull_duplex) ? "full" : "half");
	return(erc);

}	/* end of pcmd_duplex */

/*+-------------------------------------------------------------------------
	pcmd_echo(param)
echo [-n] <str>
--------------------------------------------------------------------------*/
int
pcmd_echo(param)
ESD *param;
{
int erc;
ESD *tesd;
char switches[8];

	if((tesd = esdalloc(256)) == (ESD *)0)
		return(eNoMemory);

	get_switches(param,switches,sizeof(switches));

	if(erc = gstr(param,tesd,1))
	{
		esdfree(tesd);
		return(erc);
	}
	pputs(tesd->pb);
	if(!strchr(switches,'n'))	/* if no -n */
		pputs("\n");
	esdfree(tesd);
	return(0);

}	/* end of pcmd_echo */

/*+-------------------------------------------------------------------------
	pcmd_exec(param)
--------------------------------------------------------------------------*/
int
pcmd_exec(param)
ESD *param;
{
	int erc = 0;
	ESD *tesd = (ESD *)0;

	if(!(tesd = esdalloc(64)))
		return(eNoMemory);
	if(erc = gstr(param,tesd,1))
		goto RETURN;

	/* reset indices */
	tesd->index = 0;
	tesd->old_index = 0;

	if(proctrace)
		pprintf("executing: <%s>\n",tesd->pb);
	if(erc = execute_esd(tesd))
	{
		esdshow(tesd,"error executing dynamic statement:");
		proc_error(erc);
		erc = eFATAL_ALREADY;
	}

RETURN:
	if(tesd)
		esdfree(tesd);
	return(erc);

}	/* end of pcmd_exec */

/*+-------------------------------------------------------------------------
	pcmd_exit(param)
--------------------------------------------------------------------------*/
int
pcmd_exit(param)
ESD *param;
{
long int1;
ulong colors_at_entry = colors_current;

	if(!gint(param,&int1) && int1)
	{
		setcolor(colors_error);
		pprintf("[procedure terminating ecu: user code %ld]\n",int1);
		setcolor(colors_at_entry);
		if((int1 += TERMECU_USER1 - 1) > TERMECU_USERN)
		{
			int1 = TERMECU_USERN;
			pprintf("user exit code too large, using %d\r\n",
				TERMECU_USERN - TERMECU_USER1);
		}
		termecu((int)int1);
	}
	setcolor(colors_success);
	pputs("[procedure terminating ecu: normal exit]\n");
	setcolor(colors_at_entry);
	termecu(0);
}	/* end of pcmd_exit */

/*+-------------------------------------------------------------------------
	pcmd_lgets(param)

lgets [-er] <strvar> <int1> <int2> [<str>]

read string into string variable number <stvar>
waiting <int1> 1/10th secs for first char,
waiting <int2> 1/10th secs for subsequent chars,
optionally terminating read upon detection of <str>
-e echos to screen
-r completely raw, else strip CRs & NLs from either end of string
$i0 receives the length of the read
<strvar> receives the string
--------------------------------------------------------------------------*/
int
pcmd_lgets(param)
ESD *param;
{
int erc;
long int2;
long int3;
ESD *tesd1 = (ESD *)0;
ESD *svptr;
LRWT lr;
char switches[8];
ESD *esdalloc();
char ctmp;

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	get_switches(param,switches,sizeof(switches));

	skip_cmd_char(param,'$');
	if(erc = get_cmd_char(param,&ctmp))
		return(erc);
	if(to_lower(ctmp) != 's')
		return(eIllegalVarType);
	if(erc = get_svptr(param,&svptr,1))
		return(erc);

	if(erc = gint(param,&int2))
		return(erc);

	if(erc = gint(param,&int3))
		return(erc);

	if((tesd1 = esdalloc(64)) == (ESD *)0)
		return(eNoMemory);
	if(gstr(param,tesd1,1))	/* optional delimiter */
	{
		esdfree(tesd1);
		tesd1 = (ESD *)0;
	}	

	esdzero(svptr);

	lr.to1 = int2 * 100L;
	lr.to2 = int3 * 100L;
	/* allow interrupts + raw read per -r */
	lr.raw_flag = (strchr(switches,'r')) ? 0x81 : 0x80;
	lr.buffer = svptr->pb;
	lr.bufsize = svptr->maxcb;
	lr.delim = (tesd1) ? tesd1->pb : (char *)0;
	lr.echo_flag = (strchr(switches,'e') != (char *)0);
	(void)lgets_timeout(&lr);
	if(tesd1)
		esdfree(tesd1);

	svptr->cb = lr.count;
	esd_null_terminate(svptr);
	iv[0] = (long)lr.count;
	if(zero_length_read_detected)
	{
		zero_length_read_detected = 0;
		erc = eProcAttn_DCDloss;
	}
	if(proctrace)
		pprintf("lgets read %d chars\n",lr.count);
	return(erc);

}	/* end of pcmd_lgets */

/*+-------------------------------------------------------------------------
	pcmd_flush(param)
--------------------------------------------------------------------------*/
/*ARGSUSED*/
int
pcmd_flush(param)
ESD *param;
{
	if(shm->Liofd < 0)
		return(eNoLineAttached);

	lflush(2);
	if(proctrace)
		pputs("line flushed\n");
	return(0);
}	/* end of pcmd_flush */

/*+-------------------------------------------------------------------------
	pcmd_hangup(param)
--------------------------------------------------------------------------*/
/*ARGSUSED*/
int
pcmd_hangup(param)
ESD *param;
{
	if(shm->Liofd < 0)
	{
		if(proctrace)
			pputs("no line attached ... hangup ignored\n");
		DCE_now_on_hook();
		return(0);
	}

	if(proctrace)
		pputs("hanging up ... ");
	DCE_hangup();
	if(proctrace)
		pputs("line on hook\n");
	return(0);
}	/* end of pcmd_hangup */

/*+-------------------------------------------------------------------------
	pcmd_hexdump(param)

hexdump [-s] <str>
hexdump -t[s] <str1> <str>
<str> buf to dump
<str1> title (if -t)
-s short (terse) dump
--------------------------------------------------------------------------*/
int
pcmd_hexdump(param)
ESD *param;
{
int erc;
ESD *title = (ESD *)0;
ESD *buf;
char switches[8];
extern FILE *plog_fp;

	if((buf = esdalloc(256)) == (ESD *)0)
		return(eNoMemory);

	get_switches(param,switches,sizeof(switches));

	if(strchr(switches,'t'))	/* if -t */
	{
		if((title = esdalloc(256)) == (ESD *)0)
		{
			erc = eNoMemory;
			goto RETURN;
		}
		if(erc = gstr(param,title,0))
			goto RETURN;
	}

	if(erc = gstr(param,buf,1))
		goto RETURN;

	hex_dump(buf->pb,buf->cb,(title) ? title->pb : "",
		(strchr(switches,'s')) ? 1 : 0);

	if(plog_fp)
		hex_dump_fp(plog_fp,buf->pb,buf->cb,(title) ? title->pb : "",
			(strchr(switches,'s')) ? 1 : 0);

RETURN:
	esdfree(buf);
	if(title)
		esdfree(title);
	return(erc);

}	/* end of pcmd_hexdump */

/*+-------------------------------------------------------------------------
	pcmd_lbreak(param)
--------------------------------------------------------------------------*/
/*ARGSUSED*/
int
pcmd_lbreak(param)
ESD *param;
{
	if(shm->Liofd < 0)
		return(eNoLineAttached);

	lbreak();
	return(0);
}	/* end of pcmd_lbreak */

/*+-------------------------------------------------------------------------
	pcmd_logevent(param)

logevent 'cmd'
--------------------------------------------------------------------------*/
int
pcmd_logevent(param)
ESD *param;
{
int erc;
ESD *eventstr;
char switches[8];

	if((eventstr = esdalloc(256)) == (ESD *)0)
		return(eNoMemory);

	get_switches(param,switches,sizeof(switches));

/* a hack */
	strcpy(eventstr->pb,"PROC ");
	eventstr->pb += 5;
	eventstr->maxcb -= 5;

	if(erc = gstr(param,eventstr,0))
	{
		eventstr->pb -= 5;		/* be nice */
		eventstr->maxcb += 5;	/* or surely this will haunt us one day */
		esdfree(eventstr);
		return(erc);
	}

/* rehack */
	eventstr->pb -= 5;
	eventstr->maxcb += 5;
	eventstr->cb += 5;

	ecu_log_event(getpid(),eventstr->pb);
	esdfree(eventstr);
	return(0);

}	/* end of eventstr_logevent */

/*+-------------------------------------------------------------------------
	pcmd_lookfor(param)

lookfor [-e] [quiet | <str>] [<int>]

-e echo to screen while looking
quiet means look for quiet
<str> means look for string
<int> number 1/10ths secs (default 5.0 second) for timeout

in case of lookfor <str>, $i0 plugged 1 if found, else 0
--------------------------------------------------------------------------*/
int
pcmd_lookfor(param)
ESD *param;
{
int erc;
char switches[8];
char *cptr = (char *)0;
ESD *tesd = (ESD *)0;
ulong decisecs = 50; /* default wait is 5 seconds */
int echo_flag;
char s8[8];
long start_secs;


	if(shm->Liofd < 0)
		return(eNoLineAttached);

	get_switches(param,switches,sizeof(switches));
	echo_flag = (strchr(switches,'e') != (char *)0);

	if(!get_alpha_zstr(param,s8,sizeof(s8)))
	{
		if(strcmp(s8,"quiet"))
			return(eSyntaxError);
	} 
	else
	{
		if((tesd = esdalloc(64)) == (ESD *)0)
			return(eNoMemory);
		if(erc = gstr(param,tesd,0))
			goto RETURN;
		if(!tesd->cb)
		{
			pputs("lookfor null string\n");
			erc = eFATAL_ALREADY;
			goto RETURN;
		}
		cptr = tesd->pb;
	}

	if(erc = gint(param,&decisecs))
	{
		/* if something there non-integer */
		if(!end_of_cmd(param))
		{
			erc = eSyntaxError;
			goto RETURN;
		}
	}
	erc = 0;

	if(proctrace)
		time(&start_secs);

	if(cptr)
	{
		iv[0] = (long)llookfor(cptr,decisecs * 100L,echo_flag);
		if(proctrace)
			pprintf("lookfor set $i00 = %ld\n",iv[0]);
	}
	else
		lquiet(decisecs * 100L,echo_flag);

	if(proctrace)
		pprintf("waited %ld secs\n",time((long *)0) - start_secs);

RETURN:
	if(tesd)
		esdfree(tesd);
	if(zero_length_read_detected)
	{
		zero_length_read_detected = 0;
		erc = eProcAttn_DCDloss;
	}
	return(erc);

}	/* end of pcmd_lookfor */

/*+-------------------------------------------------------------------------
	pcmd_nap(param)
nap [-m] <int>
<int> number 1/10ths secs, except if -m, nap <int> milliseconds
--------------------------------------------------------------------------*/
int
pcmd_nap(param)
ESD *param;
{
int erc;
char switches[8];
ulong interval;

	get_switches(param,switches,sizeof(switches));

	if(erc = gint(param,&interval))
		return(erc);
	if(interval)
	{
		if(!strchr(switches,'m'))
			interval *= 100L;
		if(interval < hzmsec)		/* SCO nap bug */
			interval = hzmsec;		/* SCO nap bug */
		if(proctrace && (interval > 100))	/* short naps hurt by pprintf */
			pprintf("nap %ld msec\n",interval);
		if(Nap(interval) == -1)		/* EINTR is the only error returned ... */
		{							/* but check anyway */
			if(errno == EINTR)
				erc = eCONINT;
		}
	}
	return(erc);
}	/* end of pcmd_nap */

/*+-------------------------------------------------------------------------
	pcmd_nice(param)
--------------------------------------------------------------------------*/
int
pcmd_nice(param)
ESD *param;
{
long new_nice;
int erc;
int old_nice;
int nice();

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	if(erc = gint(param,&new_nice))
		return(erc);
	if((new_nice < 0) || (new_nice > 39))
	{
		pprintf("warning: invalid nice %ld ignored (valid range 0-39)\n",
			new_nice);
		return(0);
	}

	old_nice = nice(0) + 20;
	nice(-old_nice + (int)new_nice);

	if(proctrace)
		pprintf("nice desired %u, set to %u\n",(uint)new_nice,nice(0) + 20);
	return(0);

}	/* end of pcmd_nice */

/*+-------------------------------------------------------------------------
	pcmd_parity(param)
parity [e | o | n]
parity ['e' | 'o' | 'n']
--------------------------------------------------------------------------*/
int
pcmd_parity(param)
ESD *param;
{
int erc;
int new_parity = 0;
ESD *tesd;
char s64[64];

	if(erc = skip_cmd_break(param))
		return(erc);
	if(!(tesd = esdalloc(64)))
		return(eNoMemory);
	if(!gstr(param,tesd,0))
		new_parity = to_lower(*tesd->pb);
	else if(!get_alpha_zstr(param,s64,sizeof(s64)))
		new_parity = to_lower(s64[0]);
	else
	{
		erc = eSyntaxError;
		goto RETURN;
	}
	esdfree(tesd);

	switch(new_parity)
	{
		case 'n':
			new_parity = 0;
		case 'e':
		case 'o':
			shm->Lparity = new_parity;
			if(shm->Liofd < 0)
				lset_parity(1);
			break;
		default:
			erc = eBadParameter;
	}
	if(proctrace && !erc)
	{
		pprintf("parity set to %s\n",
			(shm->Lparity) ? ((shm->Lparity == 'e') ? "even" : "odd")
			               : "none");
	}

RETURN:
	esdfree(tesd);
	return(erc);

}	/* end of pcmd_parity */

/*+-------------------------------------------------------------------------
	pcmd_prompt(param)
--------------------------------------------------------------------------*/
int
pcmd_prompt(param)
ESD *param;
{
extern ESD *icmd_prompt;

	return(gstr(param,icmd_prompt,0));
}	/* end of pcmd_prompt */

/*+-------------------------------------------------------------------------
	pcmd_ptrace(param)
--------------------------------------------------------------------------*/
int
pcmd_ptrace(param)
ESD *param;
{
char s8[8];

	if(get_alpha_zstr(param,s8,sizeof(s8)))
		return(eSyntaxError);
	if(isdigit(s8[0]))
		proctrace = atoi(s8);
	if(!strcmp(s8,"on"))
		proctrace = 1;
	else if(!strcmp(s8,"off"))
		proctrace = 0;
	else
		return(eSyntaxError);
	return(0);
}	/* end of pcmd_ptrace */

/*+-------------------------------------------------------------------------
	pcmd_rname(param) - set remote name
--------------------------------------------------------------------------*/
int
pcmd_rname(param)
ESD *param;
{
int erc;
ESD *rname;

	if(shm->Liofd < 0)
		return(eNoLineAttached);
	if(!shm->Lconnected)
	{
		pputs("Not connected\n");
		return(eFATAL_ALREADY);
	}

	if((rname = esdalloc(sizeof(shm->Lrname) - 1)) == (ESD *)0)
		return(eNoMemory);

	if(!(erc = gstr(param,rname,0)))
	{
		strcpy(shm->Lrname,rname->pb);
		if(proctrace)
			pprintf("rname set to '%s'\n",rname->pb);
	}
	esdfree(rname);
	return(erc);

}	/* end of pcmd_rname */

/*+-------------------------------------------------------------------------
	pcmd_send(param)
send [-n] <str>
-n do not send trailing CR
-v turn on proctrace for just this statement
-p## pace characters ## msec apart
--------------------------------------------------------------------------*/
int
pcmd_send(param)
ESD *param;
{
int erc;
char *cptr;
ESD *buf;
char switches[32];
int send_cr;
int tell_it;
long pace_msec = 0L;

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	if((buf = esdalloc(256)) == (ESD *)0)
		return(eNoMemory);

	get_switches(param,switches,sizeof(switches));
	send_cr = !strchr(switches,'n');
	tell_it = !!strchr(switches,'v');
	if(cptr = strchr(switches,'p'))
		sscanf(cptr + 1,"%ld",&pace_msec);

	if(erc = gstr(param,buf,1))
	{
		esdfree(buf);
		return(erc);
	}

	if(proctrace || tell_it)
	{
		hex_dump(buf->pb,buf->cb,
			(send_cr) ? "send with CR" : "send w/o CR",1);
	}

	if(pace_msec)
		lputs_paced(pace_msec,buf->pb);
	else
		lputs(buf->pb);

	if(send_cr)
		lputc(CRET);
	if(pace_msec)
		Nap(pace_msec);

	esdfree(buf);
	return(erc);
}	/* end of pcmd_send */

/*+-------------------------------------------------------------------------
	pcmd_set(param)
--------------------------------------------------------------------------*/
int
pcmd_set(param)
ESD *param;
{
int erc;
int itmp;
ulong varnum;
uint varmax;
char vartype;
char varstr[16];
int show_status;
long *ivptr;
ESD *svptr;
char *cptr;
char *make_char_graphic();

	if(erc = skip_cmd_break(param))
		return(erc);

	do {

		/* $ is optional */
		if((erc = skip_cmd_char(param,'$')) && (erc != eSyntaxError))
			return(erc);

		/* get variable type */
		if(get_cmd_char(param,&vartype))
			return(eSyntaxError);

		/* validate variable type */
		vartype = to_lower(vartype);
		switch(vartype)
		{
			case 'i':
				varmax = IVQUAN;
				break;
			case 's':
				varmax = SVQUAN;
				break;
			default:
				return(eIllegalVarType);
		}

		if(!get_numeric_value(param,&varnum))
			goto TEST_VARNUM;
		else if(*(param->pb + param->index) == '[')
		{
			if(erc = get_subscript(param,&varnum))
				return(erc);
TEST_VARNUM:
			if((int)varnum >= varmax)
				return(eIllegalVarNumber);
			switch(vartype)
			{
				case 'i':
					ivptr = &iv[(int)varnum];
					break;
				default:
					svptr = sv[(int)varnum];
			}
		}
		else if(get_alphanum_zstr(param,varstr,sizeof(varstr)))
			return(eInvalidVarName);
		else
		{
			varnum = NAMED_VARIABLE_FLAG;
			switch(vartype)
			{
				case 'i':
					erc = find_mkvi(varstr,&ivptr,1);
					break;
				default:
					erc = find_mkvs(varstr,&svptr,1);
			}
			if(erc)
				return(erc);
		}
			
		show_status = 1;
		if(!skip_cmd_char(param,'='))	/* assignment */
		{
			switch(vartype)
			{
				case 'i':
					if(erc = gint(param,ivptr))
						return(erc);
					break;
				default:
					if(erc = gstr(param,svptr,1))
						return(erc);
					break;
			}
			if(!proctrace)
				show_status = 0;
		}
		if(show_status)
		{
			switch(vartype)
			{
				case 'i':
					if(varnum != NAMED_VARIABLE_FLAG)
						pprintf("$i%02ld = %7ld (0x%08lx,0%03lo",varnum,
							*ivptr,*ivptr,*ivptr);
					else
						pprintf("$i%s = %ld (0x%08lx,0%03lo",varstr,
							*ivptr,*ivptr,*ivptr);
					if((*ivptr >= 0) && (*ivptr <= 255))
						pprintf(",'%s'",make_char_graphic((char)*ivptr,1));
					pputs(")\n");
					break;
				default:
					if(varnum != NAMED_VARIABLE_FLAG)
						pprintf("$s%02ld = '",varnum);
					else
						pprintf("$s%s = '",varstr);
					itmp = svptr->cb;
					cptr = svptr->pb;
					while(itmp--)
						pputs(make_char_graphic(*cptr++,0));
					pputs("'\n");
					break;
			}
		}
	} while(!skip_comma(param));

	if(!end_of_cmd(param))
		return(eSyntaxError);

	return(0);
}	/* end of pcmd_set */

/*+-------------------------------------------------------------------------
	pcmd_system(param)

system [-l] 'cmd'
-l makes comm line stdin/stdout
-s keeps all fds the same

returns $i0 set to exit status of program or 0x100 if interrupted
--------------------------------------------------------------------------*/
int
pcmd_system(param)
ESD *param;
{
int erc;
ESD *cmd;
extern int last_child_wait_status;
char switches[8];

	if((cmd = esdalloc(256)) == (ESD *)0)
		return(eNoMemory);

	get_switches(param,switches,sizeof(switches));

/* a hack */
	*cmd->pb++ = (strchr(switches,'s')) ? '>' : 
		((strchr(switches,'l')) ? '$' : '!');

	cmd->maxcb--;

	if(erc = gstr(param,cmd,1))
	{
		cmd->pb--;		/* be nice */
		cmd->maxcb++;	/* or surely this will haunt us one day */
		esdfree(cmd);
		return(erc);
	}

/* rehack */
	cmd->pb--;
	cmd->cb++;
	cmd->maxcb++;

	if(proctrace)
	{
		pputs(cmd->pb + 1);
		pputs("\n");
	}

	last_child_wait_status = 0xFF00;
	shell(cmd->pb);
	iv[0] = (last_child_wait_status & 0xFF)
			? 0x100L : (long)last_child_wait_status >> 8;
	if(proctrace)
		pprintf("$i0 = %ld, (%s)\n",iv[0],
			(iv[0] == 0x100L) ? "interrupted" : "program exit status");

	esdfree(cmd);
	return(0);
}	/* end of pcmd_system */

/*+-------------------------------------------------------------------------
	get_big_endian_16(ptr)
--------------------------------------------------------------------------*/
ushort
get_big_endian_16(ptr)
register uchar *ptr;
{
register ushort uint16 = ((ushort)ptr[0] << 8) | ptr[1];
 
	return(uint16);
 
}	/* end of get_big_endian_16 */
 
/*+-------------------------------------------------------------------------
	get_big_endian_32(ptr)
--------------------------------------------------------------------------*/
ulong
get_big_endian_32(ptr)
register uchar *ptr;
{
register ulong uint32 = ((ulong)*ptr++) << 24;
	uint32 |= ((ulong)*ptr++) << 16;
	uint32 |= ((ulong)*ptr++) <<  8;
	uint32 |=  (ulong)*ptr++;
	return(uint32);
 
}	/* end of get_big_endian_32 */

/*+-------------------------------------------------------------------------
	pcmd_getf(param) - get friend memory

getf -x <int-var-spec> <offset>
where: -x ==
   -b byte
   -w word (little-endian)
   -W word (big-endian)
   -l 32-bits (little-endian)
   -L 32-bits (big-endian)
--------------------------------------------------------------------------*/
int
pcmd_getf(param)
ESD *param;
{
int erc;
char switches[8];
long *piv;
long offset;
int size;
int big_endian;

	if(erc = get_switches(param,switches,sizeof(switches)))
		return(erc);
	if((strlen(switches) != 2) || !strchr("bwWlL",switches[1]))
	{
		pputs("invalid switch\n");
		return(eFATAL_ALREADY);
	}
	size = to_lower(switches[1]);
	big_endian = isupper(switches[1]);

/*
	if(!get_svptr(param,&psv))
		return(eNotImplemented);
	else
*/
	if(!strncmp(param->pb + param->index,"$i",2))
		param->index += 2;
	if(erc = get_ivptr(param,&piv,1))
		return(erc);

	if(erc = gint(param,&offset))
		return(erc);

	if(proctrace)
		pprintf("getf %s offset=0x%lx",switches,offset);

	switch(size)
	{
		case 'b':
			if(offset > ((long)sizeof(shm->friend_space) - 1))
				goto OFFSET_TOO_LARGE;
			*piv = *(((uchar *)shm->friend_space) + (int)offset) & 0xFF;
			break;
		case 'w':
			if(offset > ((long)sizeof(shm->friend_space) - 2))
				goto OFFSET_TOO_LARGE;
			if(big_endian)
				*piv = get_big_endian_16((uchar *)shm->friend_space +
						(int)offset);
			else
				*piv = *(((ushort *)shm->friend_space) + (int)offset) & 0xFFFF;
			break;
		case 'l':
			if(offset > ((long)sizeof(shm->friend_space) - 4))
				goto OFFSET_TOO_LARGE;
			if(big_endian)
			{
				*piv = get_big_endian_32((uchar *)shm->friend_space +
						(int)offset);
			}
			else
				*piv = *((long *)((char *)shm->friend_space + (int)offset));
			break;
	}

	if(proctrace)
		pprintf(" value=%ld (%08lx)\n",*piv,*piv);
	return(0);

OFFSET_TOO_LARGE:
	if(proctrace)
		pputs("\n");
	pprintf("offset 0x%02lx too large for -%c (0x%02x bytes available)\n",
		offset,switches[1],sizeof(shm->friend_space));
	return(eFATAL_ALREADY);

}	/* end of pcmd_getf */

/*+-------------------------------------------------------------------------
	pcmd_putf(param)
--------------------------------------------------------------------------*/
/*ARGSUSED*/
int
pcmd_putf(param)
ESD *param;
{
	return(eNotImplemented);
}	/* end of pcmd_putf */

/*+-------------------------------------------------------------------------
	pcmd_xon(param)
--------------------------------------------------------------------------*/
int
pcmd_xon(param)
ESD *param;
{
int erc;
char new_xonxoff[8];
char *xon_status();

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	if(erc = get_alpha_zstr(param,new_xonxoff,sizeof(new_xonxoff)))
		return(erc);

	if(set_xon_xoff_by_arg(new_xonxoff))
		return(eBadParameter);

	if(proctrace)
		pprintf("xon/xoff flow control set to %s\n",xon_status());

	return(erc);

}	/* end of pcmd_xon */

/*+-------------------------------------------------------------------------
	pcmd_rtscts(param)
--------------------------------------------------------------------------*/
int
pcmd_rtscts(param)
ESD *param;
{
int erc;
char new_rtscts[8];

	if(shm->Liofd < 0)
		return(eNoLineAttached);

	if(erc = get_alphanum_zstr(param,new_rtscts,sizeof(new_rtscts)))
		return(erc);

#if defined(HW_FLOW_CONTROL) /* see ecu.h */
	lRTSCTS_control(yes_or_no(new_rtscts));

	if(proctrace)
		display_hw_flow_config();
#else
	if(proctrace)
		pprintf("hardware flow control not available .... rtscts ignored\n");
#endif /* RTSFLOW */

	return(erc);
}	/* end of pcmd_rtscts */

/*+-------------------------------------------------------------------------
	pcmd_setline(param) - _rc.ep setline command

This command can be used to set the initial line in _rc.ep
--------------------------------------------------------------------------*/
int
pcmd_setline(param)
ESD *param;
{
int erc;
ESD *tesd;

	if(rc_ep_has_run)
	{
		pprintf("command legal only in _rc.ep\n");
		return(eFATAL_ALREADY);
	}

	if(!(tesd = esdalloc(sizeof(shm->Lline))))
		return(eNoMemory);
	if(erc = gstr(param,tesd,0))
		goto RETURN;
	shm->Lline[0] = 0;
	if(strncmp(tesd->pb,"/dev/",5))
		strcat(shm->Lline,"/dev/");
	strncat(shm->Lline,tesd->pb,sizeof(shm->Lline) - strlen(shm->Lline));
	shm->Lline[sizeof(shm->Lline) - 1] = 0;
	if(proctrace)
		pprintf("line set to %s\n",shm->Lline);

RETURN:
	esdfree(tesd);
	return(erc);
}	/* end of pcmd_setline */

/* vi: set tabstop=4 shiftwidth=4: */
/* end of pcmd.c */
