/*+-------------------------------------------------------------------------
	scurses.c -- ecu file transfer program curses interface

  000000000011111111112222222222333333333344444444445555555550
  012345678901234567890123456789012345678901234567890123456789
00.----------------------------------------------------------.
01|  SEAlink_6____  _39____________________________________  |
02|  File ### of ###: _38__________________________________  |
03|  File position:  _8______ length: _8______               |
04|  _55____________________________________________________ | transaction
05|  _55____________________________________________________ | last rx/tx hdr
06|  Comm I/O: rx _8______  tx _8______ bytes                |
07|  Baud rate: _5___ BINARY blklen: _____ comm mode: CBREAK |
08|  Time started: session: HH:MM:SS  this file: HH:MM:SS    |
09|                elpased: HH:MM:SS       time: HH:MM:SS    |
10|  Error counts: this file:  _4__  total: _6____           |
11|  _55____________________________________________________ |  err str
12|  _55____________________________________________________ |  comment str
13|  _55____________________________________________________ |  remote info
14`----------------------------------------------------------'

  Defined functions:
	clear_area(w,row,col,len)
	clear_area_char(w,row,col,len,fillchar)
	get_elapsed_time(elapsed_secs)
	hhmmss(tod)
	report_error_count()
	report_file_byte_io(count)
	report_file_close()
	report_file_open_length(length)
	report_file_open_tod()
	report_file_rcv_started(filename,length,last_mod_time)
	report_file_send_open(filename,filestat)
	report_init(title)
	report_last_rxhdr(rptstr,error_flag)
	report_last_txhdr(rptstr,error_flag)
	report_line(baud_rate,mode)
	report_protocol_crc_type(str)
	report_rx_ind(status)
	report_rx_tx_count()
	report_rxpos(pos)
	report_str(rptstr,error_flag)
	report_top_line(topstr)
	report_transaction(str)
	report_tx_ind(status)
	report_txpos(pos)
	report_uninit()
	vanilla_ruling()

------------------------------------------------------------------------*/
/*+:EDITS:*/
/*:09-10-1992-14:00-wht@n4hgf-ECU release 3.20 */
/*:08-22-1992-15:39-wht@n4hgf-ECU release 3.20 BETA */
/*:04-24-1992-21:15-wht@n4hgf-sort out vannilla vs. at ruling */
/*:02-09-1992-16:08-root@n4hgf-ruling characters only on  SCO (tcap curses) */
/*:08-28-1991-14:08-wht@n4hgf2-SVR4 cleanup by aega84!lh */
/*:07-25-1991-12:59-wht@n4hgf-ECU release 3.10 */
/*:09-19-1990-19:36-wht@n4hgf-ecu_log_event now gets pid for log from caller */
/*:08-14-1990-20:40-wht@n4hgf-ecu3.00-flush old edit history */

#include "../ecucurses.h"
#include "../ecu_types.h"
#include "../ecu_stat.h"
#include <ctype.h>
#include <signal.h>
#include <time.h>
#include <sys/timeb.h>

#if defined(M_SYSV)
#   include <sys/machdep.h>
#else
#if defined(ISC) || defined(SVR4)
#  include <sys/at_ansi.h>
#  include <sys/kd.h>
#endif
#endif

#include "../pc_scr.h"

#ifdef USE_PROTOS
# include "protos.h"
#endif

long time();
extern char *tzname[];
struct tm *localtime();

#define WIN_LINES	15
#define WIN_COLS	60
#define WIN_TOPY	2
#define WIN_LEFTX	8

extern char curr_dir[];
extern char s128[];
extern int Filcnt;
extern int Restricted;
extern int sending_flag;	/* send == 1, receive == 0 */
extern int npaths;
extern long rxpos;
extern int log_packets;

WINDOW	*win;
int (*original_sigint_handler)();
int (*original_sigquit_handler)();
int (*original_sigterm_handler)();
int curses_installed = 0;		/* curses not yet active */
int this_file_errors = 0;
int total_errors = 0;
long current_seconds;
long start_seconds;
long elapsed_seconds;
unsigned long total_data_chars_xfered = 0L;

unsigned char sTL = at_TL;
unsigned char sTR = at_TR;
unsigned char sBL = at_BL;
unsigned char sBR = at_BR;
unsigned char sLT = at_LT;
unsigned char sRT = at_RT;
unsigned char sVR = at_VR;
unsigned char sHR = at_HR;

char *win_template[] =
{
/*00000000001111111111222222222233333333334444444444555555555 */
/*01234567890123456789012345678901234567890123456789012345678 */
/*.----------------------------------------------------------. */
  "  SEAlink                                                 ",	/* 1 */
  "  File ### of ###: _____________________________________  ",	/* 2 */
  "  File position:  ________ length: ________               ",	/* 3 */
  "                                                          ",	/* 4 */
  "  tx: ______________________  rx: ______________________  ",	/* 5 */
  "  Comm I/O: rx ________  tx ________ bytes                ",	/* 6 */
  "  Baud rate: _____ BINARY blklen: 128   comm mode: ______ ",	/* 7 */
  "  Time started: session: --:--:--  this file: --:--:--    ",	/* 8 */
  "                elapsed: --:--:--    current: --:--:--    ",	/* 9 */
  "  Error counts: this file:  ____  total: ______           ",	/* 10 */
  "                                                          ",	/* 11 */
  "                                                          ",	/* 12 */
  "                                                          ",	/* 13 */
/*`----------------------------------------------------------' */
(char *)0
};

/*+-------------------------------------------------------------------------
	vanilla_ruling() - use ordinary ruling characters
--------------------------------------------------------------------------*/
void
vanilla_ruling()
{
	sTL = vanilla_TL;
	sTR = vanilla_TR;
	sBL = vanilla_BL;
	sBR = vanilla_BR;
	sLT = vanilla_LT;
	sRT = vanilla_RT;
	sVR = vanilla_VR;
	sHR = vanilla_HR;

}	/* end of vanilla_ruling */

/*+-----------------------------------------------------------------------
	char *get_elapsed_time(elapsed_secs)
	hh:mm:ss returned
  static string address is returned
------------------------------------------------------------------------*/
char *get_elapsed_time(elapsed_secs)
long elapsed_secs;
{
	static char elapsed_time_str[10];
	long hh,mm,ss;

	hh = elapsed_secs / 3600;
	elapsed_secs -= hh * 3600;
	mm = elapsed_secs / 60L;
	elapsed_secs -= mm * 60L;
	ss = elapsed_secs;

	sprintf(elapsed_time_str,"%02ld:%02ld:%02ld",hh,mm,ss);
	return(elapsed_time_str);
}	/* end of get_elapsed_time */

/*+-----------------------------------------------------------------------
	char *hhmmss(tod) - get time of day in form "hh:mm:ss"

  static string address is returned
  if tod != (char *)0, time is returned there too
------------------------------------------------------------------------*/
char *
hhmmss(tod)
char *tod;
{
	long cur_time = 0;
	struct tm *lt;			/* local time */
	static char tod_str[32];

	cur_time = time((long *)0);
	lt = localtime(&cur_time);

	sprintf(tod_str,"%02d:%02d:%02d",lt->tm_hour,lt->tm_min,lt->tm_sec);

	if(tod != (char *)0)
		strcpy(tod,tod_str);

	return(tod_str);

}	/* end of hhmmss */

/*+-------------------------------------------------------------------------
	clear_area(w,row,col,len)
--------------------------------------------------------------------------*/
clear_area(w,row,col,len)
WINDOW	*w;
int row;
int col;
int len;
{
	wmove(w,row,col);
	while(len-- > 0)
		waddch(w,' ');
	wmove(w,row,col);

}	/* end of clear_area */

/*+-------------------------------------------------------------------------
	clear_area_char(w,row,col,len,fillchar)
--------------------------------------------------------------------------*/
clear_area_char(w,row,col,len,fillchar)
WINDOW	*w;
int row;
int col;
int len;
char fillchar;
{
	wmove(w,row,col);
	while(len-- > 0)
		waddch(w,fillchar);
	wmove(w,row,col);

}	/* end of clear_area_char */

/*+-------------------------------------------------------------------------
	report_top_line(topstr)
   top line: row 1 col 18 length 39
--------------------------------------------------------------------------*/
void
report_top_line(topstr)
char *topstr;
{
char s42[42];
	clear_area(win,1,18,39);
	if(strlen(topstr) < 39)
		waddstr(win,topstr);
	else
	{
		strncpy(s42,topstr,39);
		s42[39] = 0;
		waddstr(win,s42);
	}
}	/* end of report_top_line */

/*+-------------------------------------------------------------------------
	report_protocol_crc_type(str)

  protocol crc type:  row 1 col 10 length 6
--------------------------------------------------------------------------*/
report_protocol_crc_type(str)
register char *str;
{
char s8[8];

	if(strlen(str) > 6)
	{
		strncpy(s8,str,6);
		s8[6] = 0;
		str = s8;
	}
	clear_area(win,1,10,6);
	waddstr(win,str);
	wrefresh(win);
	if(log_packets)
	{
		write(log_packets,"chk:  ",6);
		write(log_packets,str,strlen(str));
		write(log_packets,"\n",1);
	}

}	/* end of report_protocol_crc_type */

/*+-------------------------------------------------------------------------
	report_error_count()
	DOES NOT PERFORM A REFRESH CYCLE
  this file: row 10 col 29 len 4
  total:     row 10 col 42 len 6
--------------------------------------------------------------------------*/
void
report_error_count()
{
	char tmp[16];

	clear_area(win,10,29,4);
	sprintf(tmp,"%4d",this_file_errors);
	waddstr(win,tmp);

	clear_area(win,10,42,6);
	sprintf(tmp,"%6d",total_errors);
	waddstr(win,tmp);
	wrefresh(win);

}	/* end of report_error_count */

/*+-------------------------------------------------------------------------
	report_uninit()
--------------------------------------------------------------------------*/
void
report_uninit()
{
float total = (float)total_data_chars_xfered;

	if(curses_installed)
	{
		if((total_data_chars_xfered != 0L) && (elapsed_seconds != 0L))
		{
			sprintf(s128,"Data xfer rate ~= %6.0f chars/sec",
			    total / (float)elapsed_seconds);
			if(log_packets)
			{
				write(log_packets,"info: ",6);
				write(log_packets,s128,strlen(s128));
				write(log_packets,"\n",1);
			}
			report_top_line(s128);
		}
		report_file_byte_io(0L);
		report_rx_tx_count();
		wmove(win,WIN_LINES - 1,WIN_COLS - 1);
		wrefresh(win);
		endwin();
		fprintf(stderr,"\r\n\r\n\r\n");
		fflush(stderr);
		curses_installed = 0;
	}

}	/* end of report_uninit */

/*+-------------------------------------------------------------------------
	report_init(title)
--------------------------------------------------------------------------*/
void
report_init(title)
char *title;
{
register int itmp;
#if defined(CONS_GET)
int monitor_type;
#endif

	if(curses_installed)
		return;

#if defined(M_SYSV)
	if(ioctl(0,CONS_GET,&monitor_type) < 0)	/* not multiscreen */
#ifdef M_SYSV	/* SCO non AT console */
		vanilla_ruling();
#endif
#else
	vanilla_ruling();
#endif

	initscr();
	crmode();
	noecho();
	nonl();
	clear();
	curses_installed = 1;
	win = newwin(WIN_LINES,WIN_COLS,WIN_TOPY,WIN_LEFTX);
	box(win,sVR,sHR);
#ifndef SVR4
	wmove(win,0,0); waddch(win,sTL);
	wmove(win,win->_maxy - 1,0); waddch(win,sBL);
	wmove(win,win->_maxy - 1,win->_maxx - 1); waddch(win,sBR);
	wmove(win,0,win->_maxx - 1); waddch(win,sTR);
#endif
	wmove(win,0,2);
	wstandout(win);
	waddch(win,'[');
	waddch(win,' ');
	waddstr(win,title);
	waddch(win,' ');
	waddch(win,']');
	wstandend(win);
#ifdef SVR4
	whline(win, (unsigned long)(sHR & 0x00ff), 2);
	wmove(win,0, 8 + strlen(title));
#else
	waddch(win,sHR);
	waddch(win,sHR);
#endif
	waddstr(win," dir: ");
	waddstr(win,curr_dir);
	waddch(win,' ');

	itmp = 0;
	while(1)
	{
		if(win_template[itmp] == (char *)0)
			break;
		wmove(win,itmp + 1,1);
		waddstr(win,win_template[itmp++]);
	}
	if(sending_flag)
	{
		clear_area(win,2,15,3);
		sprintf(s128,"%3d",npaths);
		waddstr(win,s128);
	}
	else	/* ecurz */
	{
		clear_area(win,2,11,8);	/* clear "of ###" */
		waddstr(win,":");
	}

	clear_area(win,1,11,47);
	report_error_count();
	clear_area(win,8,26,8);		/* starting time */
	waddstr(win,hhmmss((char *)0));
	start_seconds = time((long *)0);
	current_seconds = start_seconds;

	wrefresh(win);

}	/* end of report_init */

/*+-------------------------------------------------------------------------
	report_rx_ind(status)
--------------------------------------------------------------------------*/
void
report_rx_ind(status)
int status;
{
	wmove(win,1,54);
	waddch(win,(status) ? 'R' : ' ');
	wmove(win,1,54);
	wrefresh(win);
}	/* end of report_rx_ind */

/*+-------------------------------------------------------------------------
	report_tx_ind(status)
--------------------------------------------------------------------------*/
void
report_tx_ind(status)
int status;
{
	wmove(win,1,56);
	waddch(win,(status) ? 'T' : ' ');
	wmove(win,1,56);
	wrefresh(win);
}	/* end of report_tx_ind */

/*+-------------------------------------------------------------------------
	report_rx_tx_count()

  rx char count: row 6 col 16 length 8 unsigned long
  tx char count: row 6 col 29 length 8 unsigned long
  now time of day: row 1 col 50 length 8 hh:mm:ss
  This procedure may be counted upon to perform wrefresh(win)

elapsed time row 9 col 26 length 8
current tod row 9 col 47 length 8
--------------------------------------------------------------------------*/
report_rx_tx_count()
{
	extern unsigned long rx_char_count;
	extern unsigned long tx_char_count;

	register char *cptr;

	sprintf(s128,"%8ld",rx_char_count);
	wmove(win,6,16);
	waddstr(win,s128);
	sprintf(s128,"%8ld",tx_char_count);
	wmove(win,6,29);
	waddstr(win,s128);

	/* now time of day */
	clear_area(win,9,47,8);
	cptr = hhmmss((char *)0);
	waddstr(win,cptr);
	current_seconds = time((long *)0);
	elapsed_seconds = current_seconds - start_seconds;
	cptr = get_elapsed_time(elapsed_seconds);
	clear_area(win,9,26,8);
	waddstr(win,cptr);
	wrefresh(win);		/* calling procs expect this to occur always */

}	/* end of report_rx_tx_count */

/*+-------------------------------------------------------------------------
	report_line(baud_rate,mode)
--------------------------------------------------------------------------*/
void
report_line(baud_rate,mode)
unsigned baud_rate;
char *mode;
{
char s16[16];
	wmove(win,7,14);
	sprintf(s16,"%5u",baud_rate);
	waddstr(win,s16);
	clear_area(win,7,52,6);
	waddstr(win,mode);
	wrefresh(win);
}	/* end of report_line */

/*+-------------------------------------------------------------------------
	report_rxpos(pos) row 3 col 19 len 8
--------------------------------------------------------------------------*/
void
report_rxpos(pos)
long pos;
{
char tmp[16];
char refr;

	if(rdchk(0))
	{
		read(0,&refr,1);
		if(refr == 0x0C || refr == 0x012)	/* ^L or ^R */
		{
			write(2,"\033[2J",4);
			Nap((long)60);
			touchwin(stdscr);
			wrefresh(stdscr);
			touchwin(win);
			wrefresh(win);
		}
	}

	if((pos > 99999999L) || (pos < 0L))
		return;

	sprintf(tmp,"%8lu",pos);
	wmove(win,3,19);
	waddstr(win,tmp);
	wrefresh(win);
	report_rx_tx_count();	/* which will do a refresh */
}	/* end of report_rxpos */

/*+-------------------------------------------------------------------------
	report_txpos(pos)
--------------------------------------------------------------------------*/
void
report_txpos(pos)
long pos;
{
	report_rxpos(pos);
}	/* end of report_txpos */

/*+-------------------------------------------------------------------------
	report_last_txhdr(rptstr,error_flag)
	5,7,22
--------------------------------------------------------------------------*/
void
report_last_txhdr(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s24[24];

	if(log_packets)
	{
		write(log_packets,"tx:   ",6);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

	if(strlen(rptstr) > 22)
	{
		strncpy(s24,rptstr,22);
		s24[23] = 0;
		rptstr = s24;
	}
	clear_area(win,5,7,22);
	waddstr(win,rptstr);
	wrefresh(win);

	if(error_flag)
	{
		++this_file_errors;
		++total_errors;
		report_error_count();
	}
}	/* end of report_last_txhdr */

/*+-------------------------------------------------------------------------
	report_last_rxhdr(rptstr,error_flag)
	5,35,22
--------------------------------------------------------------------------*/
void
report_last_rxhdr(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s24[24];
extern int log_packets;

	if(log_packets)
	{
		write(log_packets,"rx: ",4);
		write(log_packets,(error_flag) ? "E " : "  ",2);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

	if(strlen(rptstr) > 22)
	{
		strncpy(s24,rptstr,22);
		s24[23] = 0;
		rptstr = s24;
	}
	clear_area(win,5,35,22);
	waddstr(win,rptstr);
	wrefresh(win);

	if(error_flag)
	{
		++this_file_errors;
		++total_errors;
		report_error_count();
	}

}	/* end of report_last_rxhdr */

/*+-------------------------------------------------------------------------
	report_str(rptstr,error_flag) row 11/12 col 3 len 55

  error_flag == 0 for status/progress message
             == 1 for bump error count, unless rptstr is null
                  in which case, merely clear error string area
             == 2 write string on bottom line (not an error)
             == 3 write string on transaction line (not an error)
             == -1 use error line but do not bump error count
--------------------------------------------------------------------------*/
void
report_str(rptstr,error_flag)
register char *rptstr;
int error_flag;
{
char s60[60];
extern int log_packets;

	if(strlen(rptstr) > 55)
	{
		strncpy(s60,rptstr,55);
		s60[56] = 0;
		rptstr = s60;
	}

	switch(error_flag)
	{
		case 0:
			clear_area(win,12,3,55);
			break;
		case 1:
			++this_file_errors;
			++total_errors;
			report_error_count();
		case -1:
			clear_area(win,11,3,55);
			break;
		case 2:
			clear_area(win,13,3,55);
			break;
		case 3:
			clear_area(win,4,3,55);
			break;
	}

	waddstr(win,rptstr);
	wrefresh(win);

	if(log_packets)
	{
		write(log_packets,"info: ",6);
		sprintf(s60,"%2d ",error_flag);
		write(log_packets,s60,3);
		write(log_packets,rptstr,strlen(rptstr));
		write(log_packets,"\n",1);
	}

}	/* end of report_str */

/*+-------------------------------------------------------------------------
	report_transaction()
--------------------------------------------------------------------------*/
void
report_transaction(str)
char *str;
{
	report_str(str,3);
}	/* end of report_transaction */

/*+-------------------------------------------------------------------------
	report_file_open_tod() -- time of start of this file

  this file open time: row 8 col 47 length 8
--------------------------------------------------------------------------*/
void
report_file_open_tod()
{
	clear_area(win,8,47,8);
	waddstr(win,hhmmss((char *)0));
	wrefresh(win);
}	/* end of report_file_open_tod */

/*+-------------------------------------------------------------------------
	report_file_open_length(long_length)
  length:   row 3 col 36 len  8
--------------------------------------------------------------------------*/
report_file_open_length(length)
long length;
{
	clear_area(win,3,36,8);
	if(length <= 0)
		waddstr(win,"unknown");
	else
	{
		sprintf(s128,"%8lu",length);
		waddstr(win,s128);
	}
	wrefresh(win);
}	/* end of report_file_open_length */

/*+-------------------------------------------------------------------------
	report_file_send_open(filename,filestat)

  filename: row 2 col 20 len 38
  number:   row 2 col 8 len 3
  length:   row 3 col 36 len  8
  mode:     row 3 col 46 len 10
  time of start of this file: row 4 col 47 length 8 hh:mm:ss
--------------------------------------------------------------------------*/
void
report_file_send_open(filename,filestat)
char *filename;
struct stat *filestat;
{
char s50[50];
register char *cptr = filename;

	if(log_packets)
	{
		write(log_packets,"file: ",6);
		write(log_packets,filename,strlen(filename));
		write(log_packets,"\n",1);
	}

	/* number */
	clear_area(win,2,8,3);
	sprintf(s50,"%3d",Filcnt);
	waddstr(win,s50);

	/* filename */
	if(strlen(filename) > 38)
	{
		strncpy(s50,filename,38);
		s50[39] = 0;
		cptr = s50;
	}
	clear_area(win,2,20,38);
	waddstr(win,cptr);

#if defined(LOG_XFER)
	sprintf(s128,"sending %s",filename);
	ecu_log_event(getppid(),s128);
#endif

	/* length */
	report_file_open_length(filestat->st_size);

	/* time of start of this file */
	report_file_open_tod();

	this_file_errors = 0;
	report_error_count();
}	/* end of report_file_send_open */

/*+-------------------------------------------------------------------------
	report_file_rcv_started(filename,length,last_mod_time)

  filename: row 2 col 7 len 50
  length:   row 3 col 36 len  8 if not xmodem
  time of start of this file: row 4 col 47 length 8 hh:mm:ss
--------------------------------------------------------------------------*/
report_file_rcv_started(filename,length,last_mod_time)
char *filename;
long length;					/* if < 0, "UNKNOWN" */
long last_mod_time;			/* not currently displayed */
{
register char *cptr;
char s50[50];

	if(log_packets)
	{
		write(log_packets,"file: ",6);
		write(log_packets,filename,strlen(filename));
		write(log_packets,"\n",1);
	}

	/* filename */
	if(strlen(filename) > 38)
	{
		strncpy(s50,filename,38);
		s50[39] = 0;
		cptr = s50;
	}
	else
		cptr = filename;

#if defined(LOG_XFER)
	sprintf(s128,"receiving %s",filename);
	ecu_log_event(getppid(),s128);
#endif

	clear_area(win,2,20,38);
	waddstr(win,cptr);

	/* file number */
	clear_area(win,2,8,3);
	sprintf(s50,"%3d",Filcnt);	/* rz uses as file number 1-n */
	waddstr(win,s50);

/* if remote sender provides a file count, display it */
	if(npaths)
	{
		clear_area(win,2,12,7);	/* clear "of ###" */
		sprintf(s50,"of %3d:",npaths);
		waddstr(win,s50);
	}

	/* length */
	report_file_open_length(length);

	/* time of start of this file */
	report_file_open_tod();

	this_file_errors = 0;
	report_error_count();
}	/* end of report_file_rcv_started */

/*+-------------------------------------------------------------------------
	report_file_close()
--------------------------------------------------------------------------*/
void report_file_close()
{
	report_str("End of file",0);
	wrefresh(win);

}	/* end of report_file_close */

/*+-------------------------------------------------------------------------
	report_file_byte_io(count)
--------------------------------------------------------------------------*/
report_file_byte_io(count)
long count;
{

	total_data_chars_xfered += (long)count;
	if(total_data_chars_xfered)
	{
		sprintf(s128,"Total file bytes transferred: %lu",
			total_data_chars_xfered);
		report_str(s128,-1);
	}

}	/* end of report_file_byte_io */

/* end of scurses.c */
/* vi: set tabstop=4 shiftwidth=4: */
