/*
 * qfax.h - definitions for Renaissoft Qfax 1.0
 * (c) 1994 Robert LeBlanc and Renaissoft
 */

#ifndef _QFAX_H
#define _QFAX_H
#include <stdio.h>
#include "config.h"

#define LINELEN      1024
#define COMMENTLEN   4096
#define PHONELEN       20
#define ACLEN           4
#define ALIASLEN       20
#define SHORTLEN       40
#define LONGLEN        80


/*
   Font: Defines the attributes of a Ghostscript-compatible font.
*/

typedef struct {
                 char name[LONGLEN];  /* font name */
		 int  size;           /* point size */
} Font;

/*
   FontSet: Defines the set of fonts used on the fax cover page.
*/

typedef struct {
                 Font logo;           /* logo font */
		 Font address;        /* address block font */
		 Font title;          /* title font */
		 Font main;           /* important header font */
		 Font label;          /* other header font */
} FontSet;

/*
   LabelSet: Defines the set of labels used on the fax cover page.
*/

typedef struct {
                 char title[LONGLEN];         /* title label */
		 char tocompany[SHORTLEN];    /* to-company label */
		 char to[SHORTLEN];           /* to label */
		 char tovoice[SHORTLEN];      /* to-voice label */
		 char tofax[SHORTLEN];        /* to-fax label */
		 char from[SHORTLEN];         /* from label */
		 char email[SHORTLEN];        /* from-email label */
		 char date[SHORTLEN];         /* todays-date label */
		 char pages[SHORTLEN];        /* page-count label */
		 char regarding[SHORTLEN];    /* regarding label */
		 char comments[SHORTLEN];     /* comments label */
		 char headerstart[LONGLEN];   /* start-of-comment label */
		 char headerend[LONGLEN];     /* end-of-comment label */
} LabelSet;

/*
   FromCompany: Defines the company sending the fax.
*/

typedef struct {
                 char logoname[SHORTLEN]; /* short name of company */
                 char fullname[LONGLEN];  /* full name of company */
                 char street[LONGLEN];    /* street address */
		 char city[LONGLEN];      /* city, state/province */
		 char postcode[SHORTLEN]; /* postal/zip code */
		 char phone[PHONELEN];    /* phone number */
		 char fax[PHONELEN];      /* fax number */
		 char domain[LONGLEN];    /* e-mail domain name */
} FromCompany;

/*
   FromPerson: Defines the individual sender of the fax.
*/

typedef struct {
                 char username[ALIASLEN]; /* sender's local e-mail id */
		 char fullname[LONGLEN];  /* sender's full name */
} FromPerson;

/*
   ToCompany: Defines the company to receive the fax.
*/

typedef struct {
                 char alias[ALIASLEN];    /* company's db alias */
		 char fullname[LONGLEN];  /* full name of company */
		 char phone[PHONELEN];    /* phone number */
		 char fax[PHONELEN];      /* fax number */
} ToCompany;

/*
   ToPerson: Defines the individual recipient of the fax.
*/

typedef struct {
                 char alias[ALIASLEN];    /* recipient's db alias */
		 char fullname[LONGLEN];  /* recipient's full name */
} ToPerson;

/*
   Fax: Defines the complete fax information structure.
*/

typedef struct {
                 int         pwidth;               /* page width */
		 int         pheight;              /* page height */
		 int         pages;                /* # of pages */
		 FontSet     fonts;                /* font info */
		 LabelSet    labels;               /* label info */
		 FromCompany fcompany;             /* from-company info */
		 FromPerson  fperson;              /* from info */
		 ToCompany   tcompany;             /* to-company info */
		 ToPerson    tperson;              /* to info */
		 char        date[LONGLEN];        /* today's date */
		 char        subject[LONGLEN];     /* fax subject */
		 char        comments[COMMENTLEN]; /* cover page comments */
		 char        tstamp[SHORTLEN];     /* time stamp */
} Fax;


/*
 *  Function prototypes.
 */

/* read config data from fax.rc */
void read_config(Fax *f);

/* look up the recipient in a fax phonebook */
int read_db(Fax *f, char *dbfile);

/* make a G3 fax cover page using mail header info, db info, etc. */
void make_cover(Fax *f);

/* reorder the fax pages so that the cover is page 001 */
void insert_cover(Fax *f);

/* put the fax pages in the outgoing spool, ready for Qrun */
void queue_fax(Fax *f);

/* get rid of intermediate files we created */
void cleanup(Fax *f);

/* send a message by e-mail to the (local) sender of the fax */
void mail_user(char *user, char *subject, char *message);

/* get a newline-terminated line from a file */
char *getsline(FILE *p);

/* get the next non-commment/non-blank line from a file */
char *getline(FILE *p);

/* get the second field item from a string */
char *getfield(char *line);

/* escape '(' and ')' to make a string PostScript-safe */
char *psfix(char *str);

/* Return the time and date in the format of the "date" command */
char *timestring(void);

/* Creates a list of all the faxes in the spool directory */
int make_cmdlist(char cmdlist[MAXFAXES][LINELEN]);

/* Extracts the fax prefix from a filename */
char *make_prefix(char *filename);

/* Extracts command info from a *.cmd file */
void get_cmdinfo(char *cmdfile, char *dialstring, char *recipient,
		 char *subject);

/* Deletes a fax from the spool directory */
void delete_fax(char *prefix);

/* Extracts the sender's username from the fax filename prefix */
char *get_sender(char *prefix);

/* Exit gracefully on an error, deleting any files we've created */
void shutdown(Fax *f);

/* Looks up a name and company in the phonebook databases */
void lookup_db(Fax *f);

/* Return the time and date in the form of a text string */
char *time_stamp(void);

#endif

/*
 *    End of qfax.h
 */
