/*      xcxmdm.c -- XMODEM Protocol module for XC
        This file uses 4-character tabstops
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <utime.h>
#include <sys/stat.h>
#include <signal.h>
#include <setjmp.h>
#include "xc.h"

#define FALLBACK 16
#define CPMEOF  032		/* ^Z */
#define OK       0
#define TIMEOUT -1		/* -1 is returned by readbyte() upon timeout */
#define ERROR   -2
#define WCEOT   -3
#define RETRYMAX 10
#define Maxtime 10
#define SECSIZ  1025
#define Resume_Not_Allowed      1
#define RUpdcrc(c,crc) (crc_xmodem_tab[((crc>>8)&0xff)]^(crc<<8)^c)&0xffff
#define TUpdcrc(c,crc) (crc_xmodem_tab[((crc>>8)^c)&0xff]^(crc<<8))

short int badline = FALSE;
short int g_flag  = FALSE;
short int y_flag  = FALSE;
short int y_done  = FALSE;
short int smallhdr = FALSE;

/* define storage for info from decoding ymodem rx header */
struct yinfo {
	char *fnam ;
	long int fsiz ;
	time_t fdat ;
	unsigned long fprm ;
}  phdr ;

/* crc_xmodem_tab calculated by Mark G. Mendel, Network Systems Corporation */
ushort crc_xmodem_tab[256] =
{
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50a5, 0x60c6, 0x70e7,
    0x8108, 0x9129, 0xa14a, 0xb16b, 0xc18c, 0xd1ad, 0xe1ce, 0xf1ef,
    0x1231, 0x0210, 0x3273, 0x2252, 0x52b5, 0x4294, 0x72f7, 0x62d6,
    0x9339, 0x8318, 0xb37b, 0xa35a, 0xd3bd, 0xc39c, 0xf3ff, 0xe3de,
    0x2462, 0x3443, 0x0420, 0x1401, 0x64e6, 0x74c7, 0x44a4, 0x5485,
    0xa56a, 0xb54b, 0x8528, 0x9509, 0xe5ee, 0xf5cf, 0xc5ac, 0xd58d,
    0x3653, 0x2672, 0x1611, 0x0630, 0x76d7, 0x66f6, 0x5695, 0x46b4,
    0xb75b, 0xa77a, 0x9719, 0x8738, 0xf7df, 0xe7fe, 0xd79d, 0xc7bc,
    0x48c4, 0x58e5, 0x6886, 0x78a7, 0x0840, 0x1861, 0x2802, 0x3823,
    0xc9cc, 0xd9ed, 0xe98e, 0xf9af, 0x8948, 0x9969, 0xa90a, 0xb92b,
    0x5af5, 0x4ad4, 0x7ab7, 0x6a96, 0x1a71, 0x0a50, 0x3a33, 0x2a12,
    0xdbfd, 0xcbdc, 0xfbbf, 0xeb9e, 0x9b79, 0x8b58, 0xbb3b, 0xab1a,
    0x6ca6, 0x7c87, 0x4ce4, 0x5cc5, 0x2c22, 0x3c03, 0x0c60, 0x1c41,
    0xedae, 0xfd8f, 0xcdec, 0xddcd, 0xad2a, 0xbd0b, 0x8d68, 0x9d49,
    0x7e97, 0x6eb6, 0x5ed5, 0x4ef4, 0x3e13, 0x2e32, 0x1e51, 0x0e70,
    0xff9f, 0xefbe, 0xdfdd, 0xcffc, 0xbf1b, 0xaf3a, 0x9f59, 0x8f78,
    0x9188, 0x81a9, 0xb1ca, 0xa1eb, 0xd10c, 0xc12d, 0xf14e, 0xe16f,
    0x1080, 0x00a1, 0x30c2, 0x20e3, 0x5004, 0x4025, 0x7046, 0x6067,
    0x83b9, 0x9398, 0xa3fb, 0xb3da, 0xc33d, 0xd31c, 0xe37f, 0xf35e,
    0x02b1, 0x1290, 0x22f3, 0x32d2, 0x4235, 0x5214, 0x6277, 0x7256,
    0xb5ea, 0xa5cb, 0x95a8, 0x8589, 0xf56e, 0xe54f, 0xd52c, 0xc50d,
    0x34e2, 0x24c3, 0x14a0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
    0xa7db, 0xb7fa, 0x8799, 0x97b8, 0xe75f, 0xf77e, 0xc71d, 0xd73c,
    0x26d3, 0x36f2, 0x0691, 0x16b0, 0x6657, 0x7676, 0x4615, 0x5634,
    0xd94c, 0xc96d, 0xf90e, 0xe92f, 0x99c8, 0x89e9, 0xb98a, 0xa9ab,
    0x5844, 0x4865, 0x7806, 0x6827, 0x18c0, 0x08e1, 0x3882, 0x28a3,
    0xcb7d, 0xdb5c, 0xeb3f, 0xfb1e, 0x8bf9, 0x9bd8, 0xabbb, 0xbb9a,
    0x4a75, 0x5a54, 0x6a37, 0x7a16, 0x0af1, 0x1ad0, 0x2ab3, 0x3a92,
    0xfd2e, 0xed0f, 0xdd6c, 0xcd4d, 0xbdaa, 0xad8b, 0x9de8, 0x8dc9,
    0x7c26, 0x6c07, 0x5c64, 0x4c45, 0x3ca2, 0x2c83, 0x1ce0, 0x0cc1,
    0xef1f, 0xff3e, 0xcf5d, 0xdf7c, 0xaf9b, 0xbfba, 0x8fd9, 0x9ff8,
    0x6e17, 0x7e36, 0x4e55, 0x5e74, 0x2e93, 0x3eb2, 0x0ed1, 0x1ef0
};

static short crcheck = TRUE;	/* CRC check enabled? */
static FILE *xfp;		/* buffered file pointer */
static short firstsec,		/* first sector of file or not? */
  ksecsize,			/* for Xmodem or old Ymodem packet */
  stx_soh,			/* SOH = Xmodem ; STX = old Ymodem */
  amdone = FALSE,		/* flag: xfer almost finished */
  sync,				/* flag for negotiating crc or checksum */
  textmode = FALSE;		/* Text translations enabled? */

static char wcbuf[SECSIZ];	/* Ward Christensen sector buffer */
static jmp_buf our_env;
static RETSIGTYPE (*oldvec)();
static char *p, sendchar;
static char syncem(void);
static int yparse(void);
void set_dp( const char *p, const time_t dt, const unsigned long prm);

/*      send 10 CAN's to try to get the other end to shut up */
static void
canit ()
{
    int i;

    for (i = 0; i < 20; i++)
	sendbyte (CAN);
    for (i = 0; i < 20; i++)
	sendbyte (8); /* backspaces to clear CAN characters */
}

static RETSIGTYPE
xmsigint(junk)
int junk;
{
    signal (SIGINT, SIG_IGN);	/* Ignore subsequent DEL's */
    show_abort ();
    canit ();			/* Abort the transmission */
    longjmp (our_env, 1);
}

/*      fill the CP/M sector buffer from the UNIX file
        do text adjustments if necessary
        return 1 if more sectors are to be read, or 0 if this is the last
*/
static int
getsec ()
{
    int i;
    register c = 0 ;
    static int j, fbinit, fback;

    if ( firstsec && y_flag ) /* build ymodem header */
      {
	return y_done? 0 : 1 ;
      }
    else if (amdone == FALSE)
      {
	  i = 0;
	  j = 8;
	  fbinit = fback = TRUE;
	  while (i < ksecsize && (c = getc (xfp)) != EOF)
	    {
		if (textmode && c == '\n')
		  {
		      wcbuf[i++] = '\r';
		      if (i >= ksecsize)
			{	/* handle a newline on the last byte */
			    ungetc (c, xfp);	/* of the sector */
			    return (1);
			}
		  }
		wcbuf[i++] = c;
	    }
	  /* make sure that an extra blank sector is not sent */
	  if (c != EOF && (c = getc (xfp)) != EOF)
	    {
		ungetc (c, xfp);
		return (1);
	    }
	  /* fill up the last sector with ^Z's if text mode or 0's if binary mode */

	  j = i;
	  while (i < ksecsize)
	      wcbuf[i++] = textmode ? CPMEOF : '\0'; 
	  amdone = TRUE;
	  fback = FALSE;

	  if ( y_flag || stx_soh == SOH || j > 896)	/* done, send whole sector */
	      return (0);

	  j += 127;
	  j >>= 7;		/* calculate number of 128 bytes sectors to send */
	  fbinit = TRUE;

      }				/* end of "amdone" */

    if (fbinit)
      {
	  p = wcbuf;		/* save old buffer base address */
	  p -= 128;
	  stx_soh = SOH;	/* switch to 128 byte blocks    */
	  ksecsize = 128;
	  fbinit = FALSE;

      }				/* end of "fbinit" */
    p += 128;
    memcpy (wcbuf, p, 128);

    if (--j > 0)		/* more packets remain */
	return (1);
    else
      {
	  if (fback)
	    {
		amdone = FALSE;
		return (1);
	    }
	  else
	      return (0);
      }
}

/*      wcgetsec() inputs an XMODEM "sector".
        This routine returns the sector number encountered, or ERROR if a valid
        sector is not received or CAN received; or WCEOT if EOT sector.

        Maxtime is the timeout for the first character, set to 10 seconds for
        retries. No ACK is sent if the sector is received ok. This must be
        done by the caller when it is ready to receive the next sector.
*/
static int
wcgetsec (maxtime)
     unsigned maxtime;
{
    register unsigned oldcrc;
    register checksum, j, c = 0;
    int sectcurr, sectcomp, attempts;

    if (sync == FALSE)
      {				/* negotiate crc or checksum error detection */
	  c = syncem ();
	  if (crcheck == FALSE)
	      show (1, "ATTENTION: sender demands CHECKSUM error detection ");
	  else
	      show (1, "CRC packet validation enabled   ");
      }

    for (attempts = 0; attempts < RETRYMAX; attempts++)
      {
	  if (sync == TRUE)
	    {
		do
		  {
		      c = readbyte (maxtime);
		  }
		while (c != STX && c != SOH && c != EOT && c != CAN && c != TIMEOUT);
	    }
	  else
	      sync = TRUE;

	  switch (c)
	    {
	    case STX:
	    case SOH:
		if (c == SOH)
		    ksecsize = 128;	/* xmodem packet size  */
		else
		    ksecsize = 1024;	/* ymodem packet size  */
		sectcurr = readbyte (3);
		sectcomp = readbyte (3);
		if ((sectcurr + sectcomp) == 0xff)
		  {
		      oldcrc = checksum = 0;
		      for (j = 0; j < ksecsize; j++)
			{
			    if ((c = readbyte (3)) == TIMEOUT)
				goto timeout;
			    wcbuf[j] = c;
			    oldcrc = RUpdcrc (c, oldcrc);
			    checksum += c;
			}
		      if ((c = readbyte (3)) < 0)
			  goto timeout;
		      if (crcheck)
			{
			    oldcrc = RUpdcrc (c, oldcrc);
			    if ((c = readbyte (3)) == TIMEOUT)
				goto timeout;
			    if (RUpdcrc (c, oldcrc))
			      {
				  show(2,"CRC error     ");
				  if(g_flag) {
					show(2,"G-mode cancel ");
					goto gquit ;
				  }
				  break;
			      }
			}
		      else if (((checksum - c) & 0xff) != 0)
			{
			    show(2,"Checksum error");
			    break;
			}
		      return (sectcurr);
		  } 
		else  /* packet header garbled */
		  {
		    sprintf (Msg, "Packet number garbled %03d %03d",
			     sectcurr, sectcomp);
		    show (2,Msg);
		    if(g_flag) {
			show(2,"G-mode cancel");
			goto gquit ;
		    }
		  }
		break;
	    case EOT:
		if (readbyte (3) == TIMEOUT)
		    return (WCEOT);
		break;
	    case CAN:
		show(2,"Sender CANcelled");
		return (ERROR);
	    case TIMEOUT:
	      timeout:
		show(2,"Timeout");
		break;
	    }
	  show(2,"Trying again on this sector");
	  purge ();
	  maxtime = 6,
	  sendbyte (NAK);
      }
    show(2,"Retry count exceeded");
gquit:
    canit ();
    return (ERROR);
}

static int
putsec ()
{
    int i;
    register c;

    /* for y_modem: only write the exact number of chars */
    if ( y_flag && 0 < phdr.fsiz ) {
	phdr.fsiz -= ksecsize ;
	if ( phdr.fsiz < 0 )
		ksecsize += phdr.fsiz ;
    }

    for (i = 0; i < ksecsize; i++)
      {
	  c = wcbuf[i];
	  if (textmode)
	    {
		if (c == CPMEOF)
		    return (1);
		if (c == '\r')
		    continue;
	    }
	  putc (c, xfp);
      }
    return (0);
}

/* Receive a file using XMODEM protocol */
static int
wcrx ()
{
    register sectnum, sectcurr;

    if ( !y_flag ){
    	/* open file for xmodem reception */
    	strcpy (Name, word);
    	xfp = QueryCreate (Resume_Not_Allowed) ;
    	if ( xfp == NULL)
		return (ERROR); 
    }

    firstsec = TRUE;
    sectnum = y_flag ? -1 : 0 ;
    sendchar = g_flag? WANTCRG : WANTCRC ;
    sync = FALSE;
    y_done = FALSE;
    crcheck = TRUE;
    show(2,"Sync...");

    while (TRUE)
      {
	  if (badline)
	      purge ();
	  if ( g_flag == FALSE && sync )
	      sendbyte (sendchar);
	  if ( y_done )
	      { cl_line(); return OK; }
	  if ( sectnum == 0 && y_flag )
	      sendbyte((crcheck ? crcheck : NAK));
	  sectcurr = wcgetsec (Maxtime);

	  if (sectcurr == ((sectnum + 1) & 0xff))
	    {
		if ( firstsec ){
			if ( y_flag ){	/* open file for rx */
				if (yparse() == ERROR ) return ERROR;
			}
			else{ 
				fprintf(stderr,"\r%s\n\r", Name );
			}
			firstsec = FALSE ;
		}
		sectnum++;
	        if (sectnum == 1)
	            show (1,(ksecsize == 128 ? "Sender using 128 byte packets   ":"Sender using 1K byte packets    "));
		if ( sectnum > 0 )
		    putsec ();
		fprintf (tfp, "Received sector #%d\r", sectnum);
		sendchar = ACK;
		continue;
	    }

	  if (sectcurr == (sectnum & 0xff))
	    {
		sprintf (Msg, "Received duplicate sector #%d", sectnum);
		show(2,Msg);
		sendchar = ACK;
		continue;
	    }

	  fclose(xfp);
	  if( y_flag ) set_dp( phdr.fnam, phdr.fdat, phdr.fprm );

	  if (sectcurr == WCEOT)
	    {
		show( 1,"File received OK                ");
		sendbyte (ACK);
		return (OK);
	    }

	  if (sectcurr == ERROR)
	      return (ERROR);

	  sprintf (Msg, "Sync error ... expected %d(%d), got %d",
		   (sectnum + 1) & 0xff, sectnum, sectcurr);
	  show(1,Msg);
	  return (ERROR);
      }
}
/*      wcputsec outputs a Ward Christensen type sector.
        it returns OK or ERROR
*/

static int
wcputsec (sectnum)
     int sectnum;
{
    register ushort oldcrc;
    int checksum, j, c, attempts, errct = 0;

    oldcrc = checksum = 0;
    for (j = 0; j < (smallhdr ? 128 : ksecsize); j++)
      {
	  c = wcbuf[j];
	  oldcrc = TUpdcrc (c, oldcrc);
	  checksum += c;
      }

    for (attempts = 0; attempts < RETRYMAX; attempts++)
      {
	  purge();
	  sendbyte ( smallhdr ? SOH : stx_soh);
	  sendbyte (sectnum);
	  sendbyte (-sectnum - 1);
	  for (j = 0; j < ( smallhdr ? 128 : ksecsize); j++)
	      sendbyte (wcbuf[j]);
	  if ( firstsec == TRUE || badline == TRUE)
	      purge ();
	  if (crcheck)
	    {
		sendbyte ((int) (oldcrc >> 8));
		sendbyte ((int) oldcrc);
	    }
	  else
	      sendbyte (checksum);
	
	/* check for response */
	if( firstsec && y_flag )
	{
		c = readbyte(10);
		/*if( c == (g_flag ? WANTCRG : ACK ))*/
		if( c == ACK ){
		   firstsec = smallhdr = FALSE ;
		   if( y_done ) return OK ;
		   c = readbyte(20);
		   if ( crcheck && c == crcheck )
		   	return OK ;
		} 
		
	} else {
	  c = readbyte( g_flag ? -1 : 10);
	}

	 /* detect false CAN, require 2 in a row */
	  if ( c == CAN )
	  	c = readbyte( g_flag ? 1 : 10);
	  if ( g_flag && c != CAN && ( !firstsec || !y_flag )) 
	  	return OK ;

	  switch ( c )
	    {
	    case CAN:
		show(2,"Receiver CANcelled");
		return (ERROR);
	    case ACK:
		firstsec = FALSE;
		return (OK);
	    case NAK:
		show(2,"Got a NAK on sector acknowledge ");
		break;
	    case TIMEOUT:
		show(2,"Timeout on sector acknowledge   ");
		break;
	    default:
		sprintf(Msg,"Got %#x for sector acknowledge ", c);
		show(2,Msg);
		if( c == crcheck && crcheck ) break ; /* imitate NAK */
		do
		  {
		      if ((c = readbyte (3)) == CAN)
			{
			    show(2,"Receiver CANcelled");
			    return (ERROR);
			}
		  }
		while (c != TIMEOUT);
		break;
	    }
	  errct++;
	  if ( !y_flag && (errct == 3) && (badline == FALSE || stx_soh == STX))
	    {
		show (2, "ATTEMPTING RECOVERY ");
		return (FALLBACK);
	    }
	  
      }
    show(2,"Retry count exceeded");
    return (ERROR);
}

/* Transmit a file using XMODEM protocol */
static int
wctx ()
{
    register c ;
    int sectnum, eoflg, attempts;
    int tmpsize, tmpstxh, tmpbdln, terr, ee;

    if (!(xfp = fopen (word, "r")) && !y_done)
      {
	  sprintf (Msg, "Can't open '%s'", word);
	  show(1, Msg); 
	  return (ERROR);
      }
    firstsec = TRUE;
    attempts = 0;
    amdone = FALSE;
    tmpsize = ksecsize;		/* saves sector size for later restoration */
    tmpstxh = stx_soh;		/* saves sector size for later restoration */
    tmpbdln = badline;
    cl_line();
    show(1,"Sync...");

    while ( (c = readbyte(30) ) != NAK && c != WANTCRC && c != WANTCRG && c != CAN)
	if (c == TIMEOUT && ++attempts > RETRYMAX)
	  {
	      show(1,"Receiver not responding");
	      if(!y_done)
	      fclose (xfp);
	      return (ERROR);
	  }
    if (c == CAN)
      {
	  show(1,"Receiver CANcelled");
	  if(!y_done)
	  fclose (xfp);
	  return (ERROR);
      }
    g_flag = FALSE ;	
    if ( c == WANTCRC )
    	crcheck = WANTCRC; 
    else if ( c == WANTCRG )
    	{ crcheck = WANTCRG; g_flag = TRUE ; } 
    else
    	crcheck = FALSE ;
    
    if( crcheck )
	show (1, "CRC packet validation enabled   ");
    else
	show (1, "ATTENTION: CHECKSUM packet validation requested");
    sectnum = y_flag ? 0 : 1;

    do
      {
	  eoflg = getsec ();
	  fprintf (tfp, "Transmitting sector #%d\r", sectnum);

	  if ((terr = wcputsec (sectnum)) == ERROR)
	    {
		if(!y_done)
		fclose (xfp);
		return (ERROR);
	    }
	  else if (terr == FALLBACK)
	    {
		if (stx_soh == STX)
		  {
		      amdone = TRUE;
		  }
		else
		    badline = TRUE;
		sectnum--;
	    }
	  sectnum++;
      }
    while (eoflg);

    ksecsize = tmpsize;
    stx_soh = tmpstxh;
    badline = tmpbdln;
    attempts = 0;
    if (!y_done){ /*  if xmodem,  or if more ymodem packets */ 
    fclose (xfp);
    do{
    sendbyte (EOT);
    ee=readbyte (10) ;
    } while ( ee != ACK && attempts++ < RETRYMAX) ;
    
    if (attempts >= RETRYMAX)
      {
	  show(1,"Receiver not responding to completion");
	  return (ERROR);
      }
    }
    if ( !y_flag || y_done )
    show(1,"Transmission complete           ");
    return (OK);
}

static int
setmode (c)
     int c;
{
    g_flag = FALSE ;
    switch (tolower (c))
      {
      case 'x':
	  textmode = FALSE;	/* the usual case */
      case 't':
	  ksecsize = 128;	/* default for xmodem */
	  stx_soh = SOH;	/* default for xmodem */
	  if (c == 't')
	      textmode = TRUE;
	  break;
      case 'g':
	  g_flag = TRUE ;
      case 'b':
      case 'o':
	  textmode = FALSE;
	  ksecsize = 1024;	/* old ymodem sector size */
	  stx_soh = STX;	/* transmit flag for 1k byte packet size */
	  break;
      case ' ':
	  break;

      default:
	  return FAILURE;
      }

    sprintf (Msg, "%s %s file transfer mode",(y_flag ?"YMODEM":"XMODEM"),(textmode?"Text":"Binary"));
    show(1,Msg);

    return SUCCESS;
}

/*      Put received WC sector into a UNIX file
        using text translations if neccesary.
*/

void
xreceive (c)
     int c;
{
    oldvec = signal (SIGINT, xmsigint);

    if (setmode (c))
      {
	  if (!setjmp (our_env))
	    {
		if (wcrx () == ERROR)
		    canit ();
		/* return; why isn't signal restored first? *??*/
	    }
      }

    signal (SIGINT, oldvec);
}

void
xsend (c)
     int c;
{
    oldvec = signal (SIGINT, xmsigint);

    if (setmode (c))
      {
	  g_flag = FALSE ; /* only receiver can set g_flag */
	  sprintf(Msg,"Sending %s",ksecsize == 128 ?"128 byte small packet   "
	          : "1K byte packets XMODEM  ") ; show(1,Msg) ;
	  if (!setjmp (our_env))
	    {
		if (wctx () == ERROR)
		  {
		      sprintf (Msg, "Error transmitting file %s", word);
		      show(1,Msg) ;
		      purge(); 
		      /* return; why isn't signal restored? *??*/
		  }
	    }
      }

    signal (SIGINT, oldvec);
}


/* automatically negotiate crc or checksum reception */

char
syncem (void)
{
    int ct = 2;
    char c;

    purge ();

    while (ct)
      {	  /* try to negotiate a CRC transfer */
	  sendbyte ( g_flag ? WANTCRG : WANTCRC );
	  c = readbyte (10);
	  if (c != CAN && c != SOH && c != STX)
	      ct--;
	  else
	    {
		sendchar = ACK;
		crcheck = g_flag ? WANTCRG : WANTCRC ;
		return (c);
	    }
	    if ( g_flag ) { ct = 2, g_flag = FALSE ;} /* try again without "G" */
      }
    ct = 2;
    purge ();
    while (ct)
      {				/* maybe we can initiate a checksum transfer */
	  sendbyte (NAK);
	  c = readbyte (10);
	  if (c != CAN && c != SOH && c != STX)
	      ct--;
	  else
	    {
		sendchar = ACK;
		crcheck = FALSE;
		return (c);
	    }
      }
    purge ();
    return (TIMEOUT);

}

/******* Here is where most of the YMODEM code starts ******/

/* normalize file name, assume *filn is at least 128 chars */
char *
getfname( char *filn, int strpflg )
{
	char *wtp, *utp ;

	wtp = filn ;
	utp = wtp + 128 ; /*!! debug: fix the 128 thingy */

	while ( *wtp == 0x20 ) { 	/* strip leading spaces */
		wtp++ ;
		if ( wtp == utp )
		    return 0 ; 		/* just spaces, no file name */
	}
	if ( ! filn[0] || ! wtp[0] )
		return 0 ; 		/* empty string */
	if ( *wtp == '.' )
		while( *++wtp == '/' ) 
			; /* increment pointer beyond '/' character */ 

	/* normalize string:  strip away path, etc */ 

	if((utp = strrchr( wtp, '/' )) != NULL) {
		if ( strpflg ) {   
			wtp = utp + 1 ;	/* strip path */
		}
		if ( utp[1] == 0 ) {
			return 0 ;  /* no file name */
		}
	}
	return wtp ;
}


int
build_hdr()
{
	int retcode = 0, i = 0;
	struct stat sbuf ;
	char *fname, *tpr = wcbuf ;

        while ( i < SECSIZ ) tpr[i] = 0, i++ ;  /* clear buffer */
	getword();
	fname = getfname( word, TRUE );

	if ( fname )
	{
		if ( stat( word, &sbuf ))
		{
			perror( fname );
			fprintf( stderr, "\rCan't stat()  %s\n\r", fname );
			retcode = -1 ;
		}
		else
		{
			smallhdr = TRUE ;
			strcpy( wcbuf, fname );
			while( *tpr ) tpr++ ; tpr++ ; 
			sprintf(tpr,"%ld %lo %o", sbuf.st_size, sbuf.st_mtime, sbuf.st_mode);
			fprintf(stderr,"%s %ld %-25o\n\r", fname, sbuf.st_size, sbuf.st_mode);
			while( *tpr ) tpr++ ;
			if ( 128 < (tpr - wcbuf) ) smallhdr = FALSE ;
		}	
	}
	else 
	{
		y_done = smallhdr = TRUE ;
	}
	return retcode ;
}



void
ysend( int e )
{
    y_done = FALSE ;
    oldvec = signal (SIGINT, xmsigint);

    if (setmode ('b'))
      {
	
	  g_flag = FALSE ; /* only receiver can set g_flag */
	  sprintf (Msg, "Sending %s", ksecsize == 128 ? "128 byte small packet YMODEM"
	          : "1K byte packets YMODEM  ") ; show(1,Msg) ;
	  if (!setjmp (our_env))
	    {
		while( !y_done )
		{
			if( !build_hdr() ) 
			{
				if (wctx () == ERROR)
		  		{
		      		sprintf (Msg, "Error transmitting file %s", word);
		      		show(1,Msg) ;
		      		purge(); 
				y_done = TRUE ;
		      		/* return; why isn't signal restored? *??*/
		  		}
			}
		}
	    }
      }
    signal (SIGINT, oldvec);
    g_flag = smallhdr = FALSE ;
}

/* re-set the date and perms of a file received via ymodem */
void
set_dp( const char *name_fl, const time_t time_fl, const unsigned long perm_fl )
{
	struct utimbuf utb ;

	if ( name_fl != NULL ) {
		if ( time_fl != 0 ) {
			utb.actime = utb.modtime = time_fl ;
			if ( utime( name_fl, &utb ))
				perror(name_fl);
		}
		if ( perm_fl != 0 ) {
			if ( chmod( name_fl, (mode_t) perm_fl ))
				perror(name_fl);
		}
	}

}


/* parse ymodem header for file name and attributes */

static int
yparse()
{
	int i = 0 ;
	char *wtp, *utp ;

	phdr.fnam = NULL ;
	phdr.fsiz = 0 ;
	phdr.fdat = 0 ;
	phdr.fprm = 0 ;

	while ( i < 128 && wcbuf[i] ) i++ ;	/*!! fix the 128 thingy!*/
	if ( i == 0 || i >= ksecsize ) {
		y_done = TRUE ;
		return OK ;
	}
	/* need some error checking here: */
        utp = getfname( wcbuf, TRUE );	
	strcpy ( Name, utp );	
	phdr.fnam = Name ;
	if ( (xfp = QueryCreate (Resume_Not_Allowed)) == NULL) {
		return (ERROR);			/* can't open file */
	}
	while ( *utp != 0 ) utp++ ; utp++ ;	/* point to start of len*/ 
	phdr.fsiz = strtol( utp, &wtp, 10 ); 	/* get file length 	*/
	if ( *wtp == 0x20 ) { 			/* file date is included */
		phdr.fdat = strtol( wtp, &utp, 8 );
		if ( *utp == 0x20 ){
			phdr.fprm = strtoul( utp, &wtp, 8 ) & 07777 ;
			*wtp = 0 ; /* terminate string */
		} 
	}
	cl_line();
	sprintf(Msg,"%s %ld %s",phdr.fnam, phdr.fsiz,(phdr.fprm ? utp : " ")); 
	fprintf(stderr,"\r%s\n\r",Msg);
	return OK ;

}


void
yreceive( int e )
{
    y_done = FALSE ;
    oldvec = signal (SIGINT, xmsigint);
    
    purge(); /*!!*/
    if (setmode ('b'))
    {
	  if (!setjmp (our_env))
	    {
    		g_flag = e == 'g' ? TRUE : FALSE ; 
		while( y_done == FALSE )
		{
			if (wcrx () == ERROR)
			{
			cl_line();
			sprintf (Msg, "Error receiving file %s", word);
			show(1,Msg) ;
			purge(); 
			y_done = TRUE ;
			/* return; why isn't signal restored? *??*/
			}
		}
	    }
      }
    signal (SIGINT, oldvec);
    g_flag = smallhdr = FALSE ;
}
