/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbrdmol.c
 *
 * SCCSINFO:		@(#)dbrdmol.c	1.10 6/6/94
 *
 * ORIGINAL AUTHOR(S):  ???, 1987-02-17
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	This file contains ReadMolecule, ReadLinks, ReceiveMolecule and
 *	ReceiveCount
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <netinet/in.h>

#include "lincks.h"
#include "dbserver.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_dbrdmol.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_molfile.h"
#include "f_dbserrors.h"
#include "f_packunpack.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern PACKEDBUFF linkbuff;
extern MOLECULE the_molecule;

extern UID uid;                   /* dbserv.c */

/* External data structures */
extern TABENTRY *filetable;	/* dbserv.c */

#ifdef INETD_HACK		/* dbserv.c */
extern int socketwr;
extern int socketrd;
#else
#define socketwr socketno
#define socketrd socketno
extern int socketno;		
#endif

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int ReadBuff P_(( PACKEDBUFF *pbuff, off_t pos, int size, int fd ));
static int ReceiveBuff P_(( PACKEDBUFF *pbuff, int size, int s ));
static int ReceiveLinks P_(( LINKGROUP **linkroot, int size, int s ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: int ReadMolecule(A_ENTRY *access)
 *
 * Reads the molecule referenced by A_ENTRY from the
 * data file
 * Uses MolAccess, GetMolfd, ReadMolFile, ReadLinks,
 * ReadBuff and Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ReadMolecule(access)
  A_ENTRY *access;
{
  FILEMOL *molfileinfo;
  int index;

  /* Get file descriptor corresponding to molecule file name */
  if ((index = GetMolfd(access->a_filename)) < 0)
    return (FAIL);

  /* Get molecule file entry */
  if ((molfileinfo = ReadMolFile(access, index)) == NULL)
    return (FAIL);

  /* Get blinks from data file */
  if (ReadLinks(&the_molecule.blinks, (off_t) molfileinfo->blink_filepos,
		molfileinfo->blink_size,
		filetable[index].datfile.fd) < 0)
    return (FAIL);

  /* Get flinks from data file */
  if (ReadLinks(&the_molecule.flinks, (off_t) molfileinfo->flink_filepos,
		molfileinfo->flink_size,
		filetable[index].datfile.fd) < 0)
    return (FAIL);

  /* Get attribute buffer from data file */
  if (ReadBuff(the_molecule.attributes, (off_t) molfileinfo->attr_filepos,
	       molfileinfo->attr_size,
	       filetable[index].datfile.fd) < 0)
    return (FAIL);

  /* Get image buffer from data file */
  if (ReadBuff(the_molecule.image, (off_t)molfileinfo->image_filepos,
	       molfileinfo->image_size,
	       filetable[index].datfile.fd) < 0)
    return (FAIL);

  /* All went well: return molecule */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int ReadLinks(LINKGROUP **linkroot, off_t pos, int size, int fd)
 *
 * Scans the specified file starting at pos for the
 * molecule link structure with size size
 * Uses UnpackGroup and Error
 *
 * Modifications:
 *      1993-07-29 Martin Sjolin. 
 *                 Added code to support locking on hpux
 */
int ReadLinks(linkroot, pos, size, fd)
  LINKGROUP **linkroot;
  off_t pos;
  int size;
  int fd;
{
  /* Short cut if size is 0 */
  if (size == 0)
    return (SUCCESS);

  /* Allocate new memory for links structure buffer */
  if ((linkbuff.buff = malloc((ALLOC_T)size)) == NULL) {
    Error(ER_MALLOC, "ReadLinks: no more memory");
    return (FAIL);
  }
  linkbuff.size = size;

#ifndef NO_LOCKS

  /* Set lock on data file */
#if defined(DEBUG) && !defined(HAVE_FLOCK)
  /* try to get a lock, if no sucess, get holder and log it */
  if( read_lock( fd, pos, L_SET, size) == -1) {
    struct flock lock;

    lock.l_type   = F_RDLCK;	/* F_RDLCK, F_WRLCK or F_UNLCK */
    lock.l_start  = pos;	/* byte offset relative to where */
    lock.l_whence = L_SET;	/* SEEK_SET, SEEK_CUR, SEEK_END */
    lock.l_len    = size;	/* length of lock, 0 => to EOF */
    
    if (fcntl( fd, F_GETLK, &lock) < 0) {
      Error(ER_SYSERR,"ReadLinks: Failed to get lock holder");
      return(FAIL);
    }

    if ( lock.l_type == F_UNLCK )
      LogMessage(uid,"dbs: Strange, no holder, but failed setlk.");
    else
      LogMessage(uid,"dbs: Readlinks - lock type %d holder %d",
		 lock.l_type, (int) lock.l_pid);
    
    /* And after extracting out the information, we try again */
    if( readw_lock( fd, pos, L_SET, size) == -1) {
      Error(ER_FLOCK, "ReadLinks: could not lock data file");
      return (FAIL);
    }
  }
#else /* DEBUG && !HAVE_FLOCK */
  if ((readlockit(fd, pos, (off_t)size)) == -1) {
    Error(ER_FLOCK, "ReadLinks: could not lock data file");
    return (FAIL);
  }
#endif /* n DEBUG && !HAVE_FLOCK */

#endif	/* n NO_LOCKS */

  /* Seek to file position specified */
  if (lseek(fd, pos, L_SET) != pos) {
    Error(ER_LSEEK, "ReadLinks: could not seek to file position (%d)", errno);
    unlockit(fd, pos, (off_t)size);
    return (FAIL);
  }
  /* Get links structure from file */
  if (read(fd, linkbuff.buff, (IOSIZE_T) size) != size) {
    Error(ER_READ, "ReadLinks: unexpected EOF (%d)", errno);
    unlockit(fd, pos, (off_t)size);
    return (FAIL);
  }

  unlockit(fd, pos, (off_t)size);

  /* Unpack buffer into link structure */
  return (UnpackGroup(linkroot, &linkbuff));
}

/*  */
/**********************************************************************
 * Function: static int ReadBuff(PACKEDBUFF *pbuff,off_t pos,int size,int fd)
 *
 * Reads the data file at the position specified by pos
 * into a buffer structure
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReadBuff(pbuff, pos, size, fd)
  PACKEDBUFF *pbuff;
  off_t pos;
  int size;
  int fd;
{

  /* Short cut if size is 0 */
  if (!size)
    return (SUCCESS);

  if ((pbuff->buff = (char *)malloc((ALLOC_T)size + 1)) == NULL) {
    Error(ER_MALLOC, "ReadBuff: no more memory");
    return FAIL;
  }
  pbuff->size = size;

  /* Set lock on data file */
  if ((readlockit(fd, pos, (off_t)size)) == -1) {
    Error(ER_FLOCK, "ReadBuff: could not lock data file");
    return FAIL;
  }
  /* Seek to file position specified */
  if (lseek(fd, pos, L_SET) != pos) {
    Error(ER_LSEEK, "ReadBuff: could not seek to file position (%d)", errno);
    unlockit(fd, pos, (off_t)size);
    return (FAIL);
  }
  /* Get links structure from file */
  if (read(fd, pbuff->buff, (IOSIZE_T) size) != size) {
    Error(ER_READ, "ReadBuff: unexpected EOF (%d)", errno);
    unlockit(fd, pos, (off_t)size);
    return (FAIL);
  }
  /* Unlock data file */
  unlockit(fd, pos, (off_t)size);

  return (SUCCESS);
}


/*  */
/**********************************************************************
 * Function: int ReceiveMolecule()
 *
 * Reads a molecule from the socket connection
 * Uses ReceiveCount, ReceiveBuff and ReceiveLinks
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ReceiveMolecule()
{
  int count;

  /* Receive links buffer */
  if ((ReceiveCount((t_32bits *)&count, socketrd) < 0) ||
      (ReceiveLinks(&the_molecule.flinks, count, socketrd) < 0))
    return (FAIL);

  /* Receive attributes buffer */
  if ((ReceiveCount((t_32bits *)&count, socketrd) < 0) ||
      (ReceiveBuff(the_molecule.attributes, count, socketrd) < 0))
    return (FAIL);

  /* Receive image buffer */
  if ((ReceiveCount((t_32bits *)&count, socketrd) < 0) ||
      (ReceiveBuff(the_molecule.image, count, socketrd) < 0))
    return (FAIL);

  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: static int ReceiveLinks(LINKGROUP **linkroot, int size, int s)
 *
 * Reads from the specified socket for the
 * molecule link structure with size size
 * Uses UnpackWSGroup and Error
 *
 * Modifications:
 *      1994-03-14 Martin Sj|lin. Used UnpackWSGroup since we are
 *                 working against the database ...
 *      <list mods with name and date>
 */
static int ReceiveLinks(linkroot, size, s)
  LINKGROUP **linkroot;
  register int size;
  int s;
{
  char *p;
  int count;

  /* Short cut if size is 0 */
  if (!size)
    return (SUCCESS);

  /* Alloc memory for links structure buffer */
  if ((linkbuff.buff = malloc((ALLOC_T)size)) == NULL) {
    Error(ER_MALLOC, "ReceiveLinks: no more memory");
    return (FAIL);
  }
  linkbuff.size = size;

  /* Read from socket into buffer */
  p = linkbuff.buff;
  while (size > 0) {
    /* Read from socket */
    if ((count = read(s, p, (IOSIZE_T) size)) < 0) {
      Error(ER_READ, "ReceiveLinks: Socket read failure (%d)", errno);
      return (FAIL);
    }
    if (!count && size) {
      Error(ER_READ, "ReceiveLinks: unexpected EOF on socket");
      return (FAIL);
    }
    /* Adjust byte count and position pointer */
    size -= count;
    p += count;
  }

  /* Unpack buffer into link structure */
  return (UnpackWSGroup(linkroot, &linkbuff));
}

/*  */
/**********************************************************************
 * Function: static int ReceiveBuff(PACKEDBUFF *pbuff, int size, int s)
 *
 * Reads a buffer of specified size from the socket
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int ReceiveBuff(pbuff, size, s)
  PACKEDBUFF *pbuff;
  register int size;
  int s;
{
  register int count;
  register char *buffp;

  /* Short cut if size is 0 */
  if (!size)
    return (SUCCESS);

  /* Allocate new memory for buffer */
  if ((pbuff->buff = malloc((ALLOC_T)size)) == NULL) {
    Error(ER_MALLOC, "ReceiveBuff: no more memory");
    return FAIL;
  }
  pbuff->size = size;

  /* Prepare to receive */
  buffp = pbuff->buff;

  /* Get buffer contents from socket*/
  while (size > 0) {
    if ((count = read(s, buffp, (IOSIZE_T) size)) < 0) {
      Error(ER_READ, "ReceiveBuff: socket read failure (%d)", errno);
      return (FAIL);
    }
    if ((count == 0) && (size != 0)) {
      Error(ER_READ, "ReceiveBuff: unexpected EOF");
      return (FAIL);
    }
    /* Adjust byte count and position pointer */
    size -= count;
    buffp += count;
  }

  /* Return status */
  return (SUCCESS);
}

/*  */
/**********************************************************************
 * Function: int ReceiveCount(t_32bits *val, int s)
 *
 * Reads an integer from the socket connection
 * Uses Error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int ReceiveCount(val, s)
  t_32bits *val;
  int s;
{
  register int size, count;
  register char *buffp;
           t_u32bits netint;

  /* Prepare to read */
  size = sizeof(netint);  /* cf with code in dbwrmol.c:SendCount */
  buffp = (char *)&netint;

  /* Get integer from socket */
  while (size > 0) {
    if ((count = read(s, buffp, (IOSIZE_T) size)) < 0) {
      Error(ER_READ, "ReceiveCount: could not read from socket (%d)", errno);
      return (FAIL);
    }
    if (!count && size) {
      Error(ER_READ, "ReceiveCount: EOF on socket");
      return (FAIL);
    }
    /* Adjust byte counter and buffer pointer */
    size -= count;
    buffp += count;
  }

  /* and convert value ... */
  *val = ntohl(netint);
  
  /* Return status */
  return (SUCCESS);
}
