/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	aimcmdstream.c
 *
 * SCCSINFO:		@(#)aimcmdstream.c	1.8 5/3/94
 *
 * ORIGINAL AUTHOR(S):  1992-05-20, Ralph R\"onnquist
 *
 * DESCRIPTION:
 *	The functions in this file relate to handling of a command
 *	stream.  Commands can be added onto the command stream using
 *	newlastcmd and the stream of commands processed by do_commands.
 *	These formerly resided in aimopmove.c.
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"
#include "liblincks.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE
 *********************************************************************/
#include "f_aimcmdstream.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* for debug output */
char *indextbl[] =
{
  "INTERNAL", "ATTRGROUPTAG", "ATTRFIELDTAG", "ATTRVALUE", "LINKGROUPTAG",
  "LINKFIELDTAG", "LINKITEM", "IMAGE", "SAMEASPARENT"
};

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimsubr.h"
#include "f_aimeditops.h"
#include "f_aimpropagate.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
typedef struct aim_editcmd {
  struct aim_editcmd *next;
  int cmd;			/* CREATEOP, CHANGEOP, REMOVEOP, LINKOP */
  label obj;			/* Object affected */
  int index;			/* Which field is affected. */
  char *group;			/* Access group tag (when needed) */
  char *field;			/* Access field tag (when needed) */
  int pos;			/* Link item position number (when needed) */
  attrval val;			/* New contents or tag text (when needed) */
  label lbl;			/* New label value (when needed) */
} *editcmd;

typedef struct mapfndata {
  int unique;
  char *tag;
} mapfndatarec;

#define LINK_ERROR \
"LINK returned ERR_PARAMS.\n\
The object was not linked in.\n\
This is most likely a GPD error.\n"

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int do_change P_(( label *obj, int index, char *group, char
                           *field, int pos, attrval *val, label *lbl,
                           char **oldtag ));
static int do_create P_(( label *obj, int index, char *group, char
                           *field, int pos, attrval *val, label *lbl ));
static int do_remove_infonode P_(( label *obj, int index, char *group,
                                    char *field, int pos, char **oldtag ));
static void free_editcmd P_(( editcmd cmd ));
static int mapfn P_(( struct mapfndata *p, char *tag, int count, int len ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
#ifdef DEBUG
/* for debug output */
static char *cmdtbl[] =
{"Create", "Change", "Remove", "Link", "Copy"};
#endif	/* DEBUG */

/*
 * The edit command stream.
 */
static editcmd CS_head = (editcmd) NULL;
static editcmd CS_tail = (editcmd) NULL;

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int mapfn(char *x,char *tag,int count,int len)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int mapfn(p,tag,count,len)
  struct mapfndata *p;
  char *tag;
  int count;
  int len;
{
  if ((count == 0) || (strcmp(p->tag,tag) != 0))
    return GVS_CONTINUE;
  (void)fprintf(stderr, "The tag \"%s\" is not unique.  Creation will fail.\n",
		p->tag);
  p->unique = 0;
  return !GVS_CONTINUE;
}

/* ^L */
/*********************************************************************
 * Function: int do_commands()
 *
 * Processes the edit command stream and calls the appropriate edit
 * operations.
 *
 * Modification:
 *	<list mod's with name and date>
 */
int do_commands()
{
  label obj;
  editcmd cmd;
  char *oldtag;
  int flag;
  int retval = 0;

  if (!CS_head)
    return 0;

  for (cmd = CS_head; cmd != (editcmd) NULL; cmd = CS_head) {
    if ((CS_head = CS_head->next) == (editcmd)NULL)
      CS_tail = (editcmd)NULL;

    (void)copylbl(&obj, &(cmd->obj));
    /*
     * Here we "discover" the liblincks' intrinsic changes, i.e.
     * changes to attributes SYSTEM:Owner and SYSTEM:Created, and
     * the link SYSTEM:Parent. These are discovered only for "real"
     * changes.
     */
    flag = (!EDITED(&obj)) && (cmd->cmd >= 0);

#ifdef DEBUG
    /*
     * Here we make a trace printout prior to executing the command.
     */
    (void)printf("%s %s in <%d:%d> ",
	   cmdtbl[(cmd->cmd >= 0) ? cmd->cmd : -(cmd->cmd + 1)],
	   indextbl[cmd->index],
	   cmd->obj.vs, cmd->obj.inst);

    switch (cmd->index) {
    case IMAGE:
      (void)printf("to \n\t\"%s\": ",
	     (cmd->val.attvalue) ? cmd->val.attvalue : "(null image)");
      break;
    case LINKITEM:
      (void)printf("to <%d:%d> pos %d, \"%s:%s\": ",
	     cmd->lbl.vs, cmd->lbl.inst,
	     cmd->pos,
	     (cmd->group) ? cmd->group : "(null grp)", 
	     (cmd->field) ? cmd->field : "(null fld)");
      break;
    case ATTRVALUE:
    (void)printf("\"%s:%s\" to \n\t\"%s\": ",
	   (cmd->group) ? cmd->group : "(null grp)", 
	   (cmd->field) ? cmd->field : "(null fld)",
	   (cmd->val.attvalue) ? cmd->val.attvalue : "(null)");
      break;
    case ATTRFIELDTAG:
    case LINKFIELDTAG:
      (void)printf("\"%s:%s\" to \"%s:%s\": ", 
	     (cmd->group) ? cmd->group : "(null grp)",
	     (cmd->field) ? cmd->field : "(null fld)",
	     (cmd->group) ? cmd->group : "(null grp)",
	     cmd->val.attvalue);
      break;
    case ATTRGROUPTAG:
    case LINKGROUPTAG:
      (void)printf("\"%s\" to \"%s\": ", 
	     (cmd->group) ? cmd->group : "(null grp)",
	     cmd->val.attvalue);
      break;
    default:
      break;
    }

    (void)printf(" \n\t=> ");
#endif	/* DEBUG */

    switch (cmd->cmd) {
    case COPYOP:
      oldtag = (char *)NULL;
      retval = CPO_COPYOBJ(&obj,&(cmd->lbl));
      flag = TRUE;
      break;
    case CREATEOP:
      oldtag = (char *)NULL;
      retval = do_create(&obj, cmd->index, cmd->group, cmd->field,
				 cmd->pos, &(cmd->val), &(cmd->lbl));
      break;
    case CHANGEOP:
      retval = do_change(&obj, cmd->index, cmd->group, cmd->field,
				 cmd->pos, &(cmd->val), &(cmd->lbl), &oldtag);
      break;
    case REMOVEOP:
      retval = do_remove_infonode(&obj, cmd->index, cmd->group,
					  cmd->field, cmd->pos, &oldtag);
      break;
    case LINKOP:
      retval = LINK(&obj, cmd->group, cmd->field, &(cmd->lbl), cmd->pos);
      if (retval == ERR_PARAMS)
        short_error(LINK_ERROR);
      cmd->cmd = CREATEOP;
      break;
    default:
      /*
       * Execution is inhibited by using -(op+1) as command code.
       */
      retval = 0;
      if (cmd->cmd < 0) {
#ifdef DEBUG
	(void)printf("(done before) ");
#endif	/* DEBUG */
	cmd->cmd = -(cmd->cmd + 1);
      } else {
#ifdef DEBUG
	(void)printf("(cmd = %d?)\n", cmd->cmd);
#endif	/* DEBUG */
	cmd->cmd = -1;
      }
      break;
    }
#ifdef DEBUG
    (void)printf("(%s)\n", (retval == 0) ? "success" : "an error");
#endif	/* DEBUG */

    if (!retval && flag) {
      attrval v;
      label lbl;

      if (!GA_GETATTR(&obj,"SYSTEM","Owner",&v)) {
	newlastcmd(-(CHANGEOP+1),&obj,ATTRVALUE,"SYSTEM","Owner",0,
		   &v,(label*)NULL);
	free(v.attvalue);
      }
      if (!GA_GETATTR(&obj,"SYSTEM","Created",&v)) {
	newlastcmd(-(CHANGEOP+1),&obj,ATTRVALUE,"SYSTEM","Created",0,
		   &v,(label*)NULL);
	free(v.attvalue);
      }
      if (!GLI_GETLINKITEM(&obj,"SYSTEM","Parent",1,&lbl))
	newlastcmd(-(LINKOP+1),&obj,LINKITEM,"SYSTEM","Parent",1,
		   (attrval*)NULL,&lbl);
    }

    if (!retval && (cmd->cmd >= 0)) 
      propagate(cmd->cmd, &(cmd->obj), cmd->index,
		cmd->group, cmd->field, cmd->pos,
		&(cmd->val), &(cmd->lbl), oldtag);
    free_editcmd(cmd);
  }
  CS_head = CS_tail = (editcmd) NULL;
  return 1;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int do_create(obj, index, group, field, pos, val, lbl)
 *  label *obj;		(in)  object to change
 *  int index;		(in)  Information item index
 *  char *group;	(in)  group tag (when needed)
 *  char *field;	(in)  field tag (when needed)
 *  int pos;		(in)  position (when needed)
 *  attrval *val;	(in)  new value
 *  label *lbl;		(out) holds new label when LINKITEM
 * 
 * Executes a create operation.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int do_create(obj, index, group, field, pos, val, lbl)
  label *obj;	/* (in)	 object to change */
  int index;	/* (in)	 Information item index */
  char *group;	/* (in)  group tag (when needed) */
  char *field;	/* (in)	 field tag (when needed) */
  int pos;	/* (in)  position (when needed) */
  attrval *val;	/* (in)  new value */
  label *lbl;	/* (out) holds new label when LINKITEM */
{
  static attrval dummyval = { 0, (char *)NULL };
  static struct mapfndata p;

  p.unique = TRUE;
  p.tag = val->attvalue;

  switch (index) {
  case ATTRGROUPTAG:
    if (GAGN_GETATTRGROUPNAMES(obj,mapfn,(void *)&p) ||
	(!p.unique) ||
	(SA_SETATTR(obj,val->attvalue,"A",&dummyval)) ||
	(RAGV_REMOVEATTRGROUPVAL(obj,val->attvalue)))
      return FAIL;
    break;
  case ATTRFIELDTAG:
    switch (GAN_GETATTRNAMES(obj,group,mapfn,(void *)&p)) {
    case SUCCESS:
    case ERR_GROUP:
      if ((!p.unique) ||
	  (SA_SETATTR(obj,group,val->attvalue,&dummyval)) ||
	  (RAV_REMOVEATTRVAL(obj,group,val->attvalue)))
	return FAIL;
      break;
    default:
      return FAIL;
    }
    break;
  case LINKGROUPTAG:
    if (GLGN_GETLINKGROUPNAMES(obj,mapfn,(void *)&p) ||
	(!p.unique) ||
	(LINK(obj,val->attvalue,"A",obj,1)) ||
	(RLGV_REMOVELINKGROUPVAL(obj,val->attvalue)))
      return FAIL;
    break;
  case LINKFIELDTAG:
    switch (GLN_GETLINKNAMES(obj,group,mapfn,(void *)&p)) {
    case SUCCESS:
    case ERR_GROUP:
      if ((!p.unique) ||
	  (LINK(obj,group,val->attvalue,obj,1)) ||
	  (RLV_REMOVELINKVAL(obj,group,val->attvalue)))
	return FAIL;
      break;
    default:
      return FAIL;
    }
    break;
  case LINKITEM:
  case ATTRVALUE:
  case IMAGE:
  case SAMEASPARENT:
    /* These cannot be created, only changed. */
    return FAIL;
  }

  return SUCCESS;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int do_change(obj,index,group,field,pos,val,lbl,oldtag)
 *  label *obj;	 (in)	object to change
 *  int index;	 (in)	Information item index
 *  char *group; (in)	group tag (when needed)
 *  char *field; (in)	field tag (when needed)
 *  int pos;	 (in)	position (when needed)
 *  attrval *val;(in)	new value
 *  label *lbl;	 (out)  holds new label when LINKITEM
 *  char **oldtag;(out) holds old value for tag change
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static int do_change(obj, index, group, field, pos, val, lbl,oldtag)
  label *obj;	/* (in)	 object to change */
  int index;	/* (in)	 Information item index */
  char *group;	/* (in)  group tag (when needed) */
  char *field;	/* (in)	 field tag (when needed) */
  int pos;	/* (in)  position (when needed) */
  attrval *val;	/* (in)  new value */
  label *lbl;	/* (out) holds new label when LINKITEM */
  char **oldtag;/* (out) holds old value for tag change */
{
  static struct mapfndata p;

  p.unique = TRUE;
  p.tag = val->attvalue;

  switch (index) {
  case ATTRGROUPTAG:
    *oldtag = group;
    if (GAGN_GETATTRGROUPNAMES(obj,mapfn,(void *)&p) ||
	(!p.unique) ||
	CAGN_CHANGEATTRGROUPNAME(obj,group,val->attvalue))
      return FAIL;
    break;
  case ATTRFIELDTAG:
    *oldtag = field;
    if (GAN_GETATTRNAMES(obj,group,mapfn,(void *)&p) ||
	(!p.unique) ||
	CAN_CHANGEATTRNAME(obj,group,field,val->attvalue))
      return FAIL;
    break;
  case LINKGROUPTAG:
    *oldtag = group;
    if (GLGN_GETLINKGROUPNAMES(obj,mapfn,(void *)&p) ||
	(!p.unique) ||
	CLGN_CHANGELINKGROUPNAME(obj,group,val->attvalue))
      return FAIL;
    break;
  case LINKFIELDTAG:
    *oldtag = field;
    if (GLN_GETLINKNAMES(obj,group,mapfn,(void *)&p) ||
	(!p.unique) ||
	CLN_CHANGELINKNAME(obj,group,field,val->attvalue))
      return FAIL;
    break;
  case IMAGE:
    if (SI_SETIMAGE(obj,val))
      return FAIL;
    break;
  case ATTRVALUE:
    if (SA_SETATTR(obj, group, field, val))
      return FAIL;
    break;
  case LINKITEM:
  case SAMEASPARENT:
  default:
    /* These cannot be changed */
    return FAIL;
  }
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int do_remove_infonode(obj,index,group,field,pos,oldtag)
 *  label *obj;	 (in)	object to change
 *  int index;	 (in)	Information item index
 *  char *group; (in)	group tag (when needed)
 *  char *field; (in)	field tag (when needed)
 *  int pos;	 (in)	position (when needed)
 *  char **oldtag;(out) holds old value for tag change
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static int do_remove_infonode(obj, index, group, field, pos, oldtag)
  label *obj;	/* (in)	 object to change */
  int index;	/* (in)	 Information item index */
  char *group;	/* (in)  group tag (when needed) */
  char *field;	/* (in)	 field tag (when needed) */
  int pos;	/* (in)  position (when needed) */
  char **oldtag;/* (out) holds old value for tag change */
{
  attrval v;
  *oldtag = (char*)NULL;

  switch(index) {
  case ATTRGROUPTAG:
    *oldtag = group;
    if (RAGT_REMOVEATTRGROUPTAG(obj,group))
      return FAIL;
    break;
  case ATTRFIELDTAG:
    *oldtag = field;
    if (RAT_REMOVEATTRTAG(obj,group,field))
      return FAIL;
    break;
  case LINKGROUPTAG:
    *oldtag = group;
    if (RLGT_REMOVELINKGROUPTAG(obj,group))
      return FAIL;
    break;
  case LINKFIELDTAG:
    *oldtag = field;
    if (RLT_REMOVELINKTAG(obj,group,field))
      return FAIL;
    break;
  case LINKITEM:
    if (ULA_UNLINKAT(obj,group,field,pos))
      return FAIL;
    break;
  case ATTRVALUE:
    if (RAV_REMOVEATTRVAL(obj,group,field))
      return FAIL;
    break;
  case IMAGE:
    v.attsize = 0;
    v.attvalue = (char *)NULL;
    if (SI_SETIMAGE(obj,&v))
      return FAIL;
    break;
  case SAMEASPARENT:
  default:
    return FAIL;
  }
  return SUCCESS;
}

/* ^L */
/*********************************************************************
 * Function: void newlastcmd(EIGHT PARAMETERS)
 * Parameters:
 *	int cmd;	CREATEOP, CHANGEOP, REMOVEOP, LINKOP
 *	label* obj;	Object affected
 *	int index;	Which field is affected.
 *	char *group;	Access group tag (when needed)
 *	char *field;	Access field tag (when needed)
 *	int pos;	Link item position number (when needed)
 *	attrval *val;	New contents or tag text (when needed)
 *	label *lbl;	New label value (when needed)
 *
 * Creates a new command record at end of current list of commands.
 *
 * Modification:
 *	<list mod's with name and date>
 */
void newlastcmd(cmd, obj, index, group, field, pos, val, lbl)
  int cmd;			/* CREATEOP, CHANGEOP, REMOVEOP, LINKOP */
  label *obj;			/* Object affected */
  int index;			/* Which field is affected. */
  char *group;			/* Access group tag (when needed) */
  char *field;			/* Access field tag (when needed) */
  int pos;			/* Link item position number (when needed) */
  attrval *val;			/* New contents or tag text (when needed) */
  label *lbl;			/* New label value (when needed) */
{
  /* Make a new entry at end of cmdstrm. */
  editcmd new;

  new = (editcmd) malloc(sizeof(struct aim_editcmd));
  if (new == NULL)
    fatal_error("%s\n", "malloc failed in newlastcmd.");

  new->cmd = cmd;
  new->next = (editcmd) NULL;
  (void)copylbl(&(new->obj), obj);
  if (cmd == COPYOP) {
    new->index = IMAGE;
    new->group = (char *)NULL;
    new->field = (char *)NULL;
    new->pos = 0;
    if (GI_GETIMAGE(lbl,&(new->val))) {
      new->val.attsize = 0;
      new->val.attvalue = (char *)NULL;
    }
    (void)copylbl(&(new->lbl), lbl);
  } else {
    new->index = index;
    new->group = strdup(group);
    new->field = strdup(field);
    new->pos = pos;
    if ((val != (attrval *) NULL) && ((new->val.attsize = val->attsize) > 0)) {
      new->val.attvalue = malloc((ALLOC_T)val->attsize);
      (void)strncpy(new->val.attvalue, val->attvalue, val->attsize);
    } else {
      new->val.attsize = 0;
      new->val.attvalue = (char *)NULL;
    }
    if (lbl != (label *) NULL)
      (void)copylbl(&(new->lbl), lbl);
    else {
      new->lbl.vs = -1;
      new->lbl.inst = -1;
    }
  }
  if (CS_head == (editcmd) NULL)
    CS_head = new;
  if (CS_tail != (editcmd) NULL)
    CS_tail->next = new;
  CS_tail = new;
}

/*  */
/*********************************************************************
 * Function: static void free_editcmd(editcmd cmd)
 *
 * Reclaims memory.
 *
 * Modification:
 *      <list mod's with name and date>
 */
static void free_editcmd(cmd)
  editcmd cmd;
{
  if (cmd->group != (char *)NULL)
    free(cmd->group);
  if (cmd->field != (char *)NULL)
    free(cmd->field);
  if (cmd->val.attvalue != (char *)NULL)
    free(cmd->val.attvalue);
  free((FREEPTR *)cmd);
}

/*  */
/*********************************************************************
 * Function: void revise_command_stream(int present,int really)
 *
 * Remove operations on object 'present' and redirect linking to it
 * to use <'really':-1> instead.
 *
 * Modification:
 *      <list mod's with name and date>
 */
void revise_command_stream(present, really)
  int present;
  int really;
{
  editcmd *p;
  editcmd cmd;

  p = &CS_head;
  while ((*p)) {
    if ((*p)->obj.vs == present) {
      cmd = (*p);
      (*p) = cmd->next;
      free_editcmd(cmd);
    } else {
      if ((*p)->lbl.vs == present) {
        (*p)->lbl.vs = really;
        (*p)->lbl.inst = -1;
      }
      p = &((*p)->next);
    }
  }
}
