/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	aimdbdump.c
 *
 * SCCSINFO:		@(#)aimdbdump.c	1.7 5/17/94
 *
 * ORIGINAL AUTHOR(S):  1993-02-04, Ralph R{\"o}nnquist.
 *
 * MODIFICATIONS:
 *      1994-05-12 Martin Sjlin. Added check code for exporting
 *                 binary information in uudecoded format.
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	Implements the selective dump facility.
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimdbdump.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
extern char *sys_errlist[];	/* errno.h */
extern int sys_nerr;		/* errno.h */
extern int errno;		/* errno.h */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimsubr.h"
#include "f_aimstruct.h"
#include "f_uu.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern reference_structure *refstructs;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define DUMP_HEADER "## A Database Dump.\n"
#define LAST_DIGIT 10
#define FIRST_DIGIT 4

typedef struct _link_desc {
  struct _link_desc *next;
  char *group;
  char *field;
  int dumped;
  int plural;
} link_desc;

typedef struct _name_list {
  struct _name_list *next;
  char *name;
  label lbl;
  int subtree;
  link_desc *links;
} name_list;

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/

static struct {
  label obj;
  reference_structure *refstructs;
  name_list *names;
  char gen_name[LAST_DIGIT + 2];
  attrval file_name;
  FILE *fd;
} dump_desc;

typedef struct access {
  char *group;
  char *field;
} access_struct;

static name_list *global_p = NULL;
  
/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void add_link_desc P_(( label *obj, char *group, char *field, int
                                plural ));
static void add_object_name P_(( char *name, label *lbl ));
static int check_links P_(( label *lbl, char *group, char *field, int
                             plural ));
static void collect_dump_nodes P_(( identry *idp ));
static int first_field_map P_(( char *extra, char *field, int count, int
                                 len ));
static int first_item_map P_(( char *gpd, label *item, int count, int
                                len ));
static void free_dump_stuff P_(( void ));
static char *generate_object_name P_(( void ));
static int name_dump_node P_(( label *lbl ));
static int raw_attr_field P_(( char *group, char *field, int count, int
                                len ));
static int raw_attr_group P_(( char *extra, char *group, int count, int
                                len ));
static int raw_dumped P_(( label *lbl ));
static int raw_link_field P_(( char *group, char *field, int count, int
                                len ));
static int raw_link_group P_(( char *extra, char *group, int count, int
                                len ));
static int raw_link_item P_(( access_struct *access, label *item, int
                               count, int len ));
static int second_field_map P_(( char *extra, char *field, int count,
                                  int len ));
static int subtree_avoid_map P_(( char *extra, label *item, int count,
                                   int len ));
static int subtree_field_map P_(( char *group, char *field, int count,
                                   int len ));
static int subtree_group_map P_(( char *extra, char *group, int count,
                                   int len ));
static int subtree_item_map P_(( char *extra, label *item, int count,
                                  int len ));
static void write_attr_value P_(( label *obj, char *group, char *field,
                                   attrval *val ));
static void write_node_contents P_(( identry *idp ));
static void write_node_symbol P_(( label *lbl ));
static void write_string P_(( char *s ));
static void write_t2l_file P_(( void ));

/*  */
/**********************************************************************
 * Function: static void add_object_name(char *name,label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void add_object_name(name, lbl)
  char *name;
  label *lbl;
{
  name_list *p;

  (void)fprintf(stderr,".");
  (void)fflush(stderr);
  p = (name_list *) malloc(sizeof(name_list));
  p->next = dump_desc.names;
  p->links = NULL;
  p->name = strdup(name);
  p->subtree = 0;
  (void)copylbl(&(p->lbl), lbl);
  dump_desc.names = p;
}

/*  */
/**********************************************************************
 * Function: static char *generate_object_name()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static char *generate_object_name()
{
  int i;

  for (i = LAST_DIGIT; i >= FIRST_DIGIT; i--) {
    if (dump_desc.gen_name[i]++ < '9')
      break;
    dump_desc.gen_name[i] = '0';
  }
  return (&(dump_desc.gen_name[0]));
}

/*  */
/**********************************************************************
 * Function: static int name_dump_node(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int name_dump_node(lbl)
  label *lbl;
{
  name_list *p;

  if (BOUND(lbl)) {
    (void)printf("\nWarning: bound label being dumped.\n");
  }
  for (p = dump_desc.names; p; p = p->next) {
    if (p->lbl.vs == lbl->vs) {
      if (p->lbl.inst != lbl->inst)
	return 1;
      return 0;
    }
  }
  add_object_name(generate_object_name(), lbl);
  return 0;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int subtree_avoid_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	label *item
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int subtree_avoid_map(extra, item, count, len)
  char *extra;
  label *item;
  int count;
  int len;
{
  name_list *p;

  if (count == 0)
    return 0;

  for (p = dump_desc.names; p; p = p->next)
    if (p->lbl.vs == item->vs)
      break;

  if (!p) {
    add_object_name(generate_object_name(), item);
    p = dump_desc.names;
  }

  if (!p->subtree)
    p->subtree = 2;

  return 0;

}
/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int subtree_item_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	label *item
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int subtree_item_map(extra, item, count, len)
  char *extra;
  label *item;
  int count;
  int len;
{
  name_list *p;
  name_list *in_global_p = global_p;

  if (count == 0)
    return 0;

  for (p = dump_desc.names; p; p = p->next)
    if (p->lbl.vs == item->vs)
      break;

  if (!p) {
    add_object_name(generate_object_name(), item);
    p = dump_desc.names;
  }

  if (!p->subtree) {
    p->subtree = 1;
    global_p = p;
    (void)GLGN_GETLINKGROUPNAMES(&(p->lbl),subtree_group_map,(char *)NULL);
  }
  global_p = in_global_p;
  return 0;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int subtree_group_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	char *group
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int subtree_group_map(extra, group, count, len)
  char *extra;
  char *group;
  int count;
  int len;
{
  char *s;
  int err;

  if (count == 0)
    return 0;

  s = strdup(group);
  err = GLN_GETLINKNAMES(&(global_p->lbl), s, subtree_field_map, s);
  free(s);
  return (err);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int subtree_field_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	char *field
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int subtree_field_map(group, field, count, len)
  char *group;
  char *field;
  int count;
  int len;
{
  char *s;
  int err;

  if (count == 0)
    return 0;

  s = strdup(field);
  err = GLV_GETLINKVAL(&(global_p->lbl), group, field,
		       subtree_item_map,(char *)NULL);
  free(s);
  return (err);
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int first_item_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	label *item
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int first_item_map(gpd, item, count, len)
  char *gpd;
  label *item;
  int count;
  int len;
{
  reference_structure *ref;
  label obj;

  if (count == 0)
    return 0;

  if ((ref = buildref(copylbl(&obj, item), gpd, (reference_structure *) NULL))) {
    refstructs = ref->next;
    ref->next = dump_desc.refstructs;
    dump_desc.refstructs = ref;
  }
  return 0;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int first_field_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	char *field
 *	int count
 *	int len
 *
 * Expand each link within `field' as `field' and determine and name all
 * elementary objects uniquely.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int first_field_map(extra, field, count, len)
  char *extra;
  char *field;
  int count;
  int len;
{
  char *s;
  int err = SUCCESS;
  int (*fn)() = first_item_map;

  if (count == 0)
    return 0;

  s = strdup(field);
  if (strcmp(s,"Raw") == 0)
    fn = subtree_item_map;

  if (strcmp(s,"Avoid") != 0)
    err = GLV_GETLINKVAL(&(dump_desc.obj), "DUMP", s, fn, s);

  free(s);
  return (err);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int second_field_map(FOUR PARAMETERS)
 * Parameters:
 *	char *extra
 *	char *field
 *	int count
 *	int len
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int second_field_map(extra, field, count, len)
  char *extra;
  char *field;
  int count;
  int len;
{
  label lbl;

  if (count == 0)
    return 0;

  if (!GLI_GETLINKITEM(&(dump_desc.obj), "DUMP NAMES", field, 1, &lbl)) {
    add_object_name(field, &lbl);
  }
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void collect_dump_nodes(identry *idp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void collect_dump_nodes(idp)
  identry *idp;
{
  infonode *inp;
  int i;

  for (; idp; idp = idp->next) {
    if (!(idp->value) || (idp->value->iflg & NONEXIST_MSK))
      continue;
    if (idp->value->index == LINKITEM) {
      add_link_desc(&(idp->value->obj), grouptag(idp), fieldtag(idp),
		    (idp->tmpl->value->pos == -1));
      for (i = 1, inp = idp->value; inp; i++, inp = inp->next)
	if (name_dump_node(&(inp->lbl)))
	  (void)printf("\nWarning: version mismatch (%s:%s:%d)\n",
		       grouptag(idp), fieldtag(idp), i);
    }
    for (inp = idp->value; inp; inp = inp->next)
      collect_dump_nodes(inp->subnodes);
  }
}

/*  */
/**********************************************************************
 * Function: static void free_dump_stuff()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void free_dump_stuff()
{
  reference_structure *ref;
  link_desc *x;
  name_list *p;

  for (ref = dump_desc.refstructs; ref; ref = dump_desc.refstructs) {
    dump_desc.refstructs = ref->next;
    free_ref_cont(ref);
    free((FREEPTR *)ref);
  }
  for (p = dump_desc.names; p; p = dump_desc.names) {
    dump_desc.names = p->next;
    for (x = p->links; x; x = p->links) {
      p->links = x->next;
      free((FREEPTR *)x->group);
      free((FREEPTR *)x->field);
      free((FREEPTR *)x);
    }
    free(p->name);
    free((FREEPTR *)p);
  }
  if (dump_desc.file_name.attvalue)
    free(dump_desc.file_name.attvalue);

  dump_desc.file_name.attvalue = NULL;
  dump_desc.file_name.attsize = 0;
}


/*  */
/**********************************************************************
 * Function: static void write_string(char *s)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void write_string(s)
  char *s;
{
  (void)fprintf(dump_desc.fd, "\"");
  if (s)
    for (; *s; s++) {
      if ((*s == '"') || (*s == '\n') || (*s == '\\'))
	(void)fprintf(dump_desc.fd, "\\");
      (void)fprintf(dump_desc.fd, "%c", *s);
    }
  (void)fprintf(dump_desc.fd, "\"");
}


/*  */
/**********************************************************************
 * Function: static void write_node_symbol(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void write_node_symbol(lbl)
  label *lbl;
{
  name_list *p;

  for (p = dump_desc.names; p; p = p->next) {
    if (p->lbl.vs == lbl->vs)
      break;
  }
  if (!p) {
    (void)name_dump_node(lbl);
    (void)fprintf(stderr, "\n%s<%d:%d> (%s)%s\n",
		  "Warning: the node ",
		  lbl->vs, lbl->inst,
		  dump_desc.names->name,
		  " is improperly defined.");
    write_node_symbol(lbl);
  } else {
    write_string(p->name);
  }
}

/*  */
/**********************************************************************
 * Function: static int raw_dumped(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_dumped(lbl)
  label *lbl;
{
  name_list *p;

  for (p = dump_desc.names; p; p = p->next)
    if (p->lbl.vs == lbl->vs)
      return (p->subtree == 1);
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void write_attr_value(FOUR PARAMETERS)
 * label *obj;
 * char *group;
 * char *field;
 * attrval *val;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void write_attr_value(obj,group,field,val)
  label *obj;
  char *group;
  char *field;
  attrval *val;
{
  (void)fprintf(dump_desc.fd, ".group ");
  write_node_symbol(obj);
  (void)fprintf(dump_desc.fd, " ");
  write_string(group);
  (void)fprintf(dump_desc.fd, " ");
  write_string(field);
  (void)fprintf(dump_desc.fd, ": ");
  write_string(val->attvalue);
  (void)fprintf(dump_desc.fd, "\n");
}
     

/*  */
/**********************************************************************
 * Function: static void write_node_contents(identry *idp)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void write_node_contents(idp)
  identry *idp;
{
  infonode *inp;

  for (; idp; idp = idp->next) {
    if (!idp->value || (idp->value->iflg & NONEXIST_MSK))
      continue;
    if (!raw_dumped(&(idp->value->obj))) {
      inp = idp->value;
      switch (idp->value->index) {
      case ATTRVALUE:
	write_attr_value(&(idp->value->obj),grouptag(idp),
			 fieldtag(idp), &(idp->value->val));
	break;
      case LINKITEM:
	if ((!strcmp(grouptag(idp), "SYSTEM")) &&
	    (!strcmp(fieldtag(idp), "Parent")))
	  break;
	if (check_links(&(idp->value->obj), grouptag(idp), fieldtag(idp),
			(idp->tmpl->value->pos == -1))) {
	  (void)fprintf(stderr, "\n%s\n%s %s <%d:%d>\n",
			"Commenting out duplicate link directive:",
			grouptag(idp), fieldtag(idp), idp->value->obj.vs,
			idp->value->obj.vs);
	  (void)fprintf(dump_desc.fd, "# ");
	}
	(void)fprintf(dump_desc.fd, ".links ");
	write_node_symbol(&(idp->value->obj));
	(void)fprintf(dump_desc.fd, " ");
	write_string(grouptag(idp));
	(void)fprintf(dump_desc.fd, " ");
	write_string(fieldtag(idp));
	(void)fprintf(dump_desc.fd, "= ");
	for (inp = idp->value; inp; inp = inp->next) {
	  write_node_symbol(&(inp->lbl));
	  if (inp->next)
	    (void)fprintf(dump_desc.fd, ", ");
	}
	(void)fprintf(dump_desc.fd, "\n");
	break;
	
	break;
      default:
	break;
      }
    }
    for (inp = idp->value; inp; inp = inp->next)
      write_node_contents(inp->subnodes);
  }
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int raw_attr_field(FOUR PARAMETERS)
 * char *group;
 * char *field;
 * int count;
 * int len;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_attr_field(group,field,count,len)
  char *group;
  char *field;
  int count;
  int len;
{
  attrval val;

  if (count == 0)
    return 0;

  if (!GA_GETATTR(&(global_p->lbl),group,field,&val)) {
    write_attr_value(&(global_p->lbl),group,field,&val);
    free(val.attvalue);
  }
  
  return 0;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int raw_attr_group(FOUR PARAMETERS)
 * char *extra;
 * char *group;
 * int count;
 * int len;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_attr_group(extra, group, count, len)
  char *extra;
  char *group;
  int count;
  int len;
{
  (void)GAN_GETATTRNAMES(&(global_p->lbl),group,raw_attr_field,group);
  return 0;
}

/*  */
/**********************************************************************
 * Function: static int raw_link_item(FOUR PARAMETERS)
 * access_struct *access;
 * label *item;
 * int count;
 * int len;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_link_item(access,item,count,len)
  access_struct *access;
  label *item;
  int count;
  int len;
{
  if (count == 0) {
    if (len > 0) {
      (void)fprintf(dump_desc.fd, ".links ");
      write_node_symbol(&(global_p->lbl));
      (void)fprintf(dump_desc.fd, " ");
      write_string(access->group);
      (void)fprintf(dump_desc.fd, " ");
      write_string(access->field);
      (void)fprintf(dump_desc.fd, "= ");
    }
    return 0;
  }
  write_node_symbol(item);
  if (count < len)
    (void)fprintf(dump_desc.fd, ", ");
  else
    (void)fprintf(dump_desc.fd, "\n");
  return 0;
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int raw_link_field(FOUR PARAMETERS)
 * char *group;
 * char *field;
 * int count;
 * int len;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_link_field(group,field,count,len)
  char *group;
  char *field;
  int count;
  int len;
{
  static access_struct access;

  if (count == 0)
    return 0;

  if (!strcmp(group,"SYSTEM") && !strcmp(field,"Parent"))
    return 0;

  access.group = group;
  access.field = field;

  (void)GLV_GETLINKVAL(&(global_p->lbl), group, field,
		       raw_link_item, (void *)&access);
  return 0;
}
/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int raw_link_group(FOUR PARAMETERS)
 * char *extra;
 * char *group;
 * int count;
 * int len;
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int raw_link_group(extra, group, count, len)
  char *extra;
  char *group;
  int count;
  int len;
{
  (void)GLN_GETLINKNAMES(&(global_p->lbl),group,raw_link_field,group);
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void write_t2l_file()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void write_t2l_file()
{
  reference_structure *ref;
  name_list *p;
  attrval val;

  (void)fprintf(dump_desc.fd, DUMP_HEADER);
  if (!GA_GETATTR(&(dump_desc.obj), "SYSTEM", "Created", &val) && val.attvalue) {
    (void)fprintf(dump_desc.fd, 
		  "# Dump description created %s\n", val.attvalue);
    free(val.attvalue);
  }
  if (!GA_GETATTR(&(dump_desc.obj), "DUMP", "preamble", &val) && val.attvalue) {
    (void)fprintf(dump_desc.fd, "%s\n\n", val.attvalue);
    free(val.attvalue);
  }
  (void)fprintf(dump_desc.fd, "\n######### Nodes #########\n");
  for (p = dump_desc.names; p; p = p->next) {
    int typeisstring = TRUE;

    if (GI_GETIMAGE(&(p->lbl), &val))
      val.attvalue = NULL;

    /* check if it is a string, ended by a zero */
    if (val.attsize > 0 && val.attvalue != (char *) NULL) 
      if (val.attvalue[val.attsize-1] != '\0' || strlen(val.attvalue) != val.attsize-1) 
	typeisstring = FALSE;

    (void)fprintf(dump_desc.fd, typeisstring ? ".object " : ".binary ");
    write_string(p->name);

    if (typeisstring) {
      write_string(val.attvalue);
      (void)fprintf(dump_desc.fd, "\n  ");
    } 
    /* limit in libparser.a */
    else if (val.attsize < PARSERINBUFLIMIT/5) {
      char *uu;
      if ((uu = uuencode(val.attsize, val.attvalue, NULL)) == NULL)
	(void) fprintf(stderr, 
		       "Failed to convert binary information into uudecoded for %s\n",
		       p->name);
      else {
	write_string(uu);
	free((FREEPTR *) uu);
      }
      (void)fprintf(dump_desc.fd, "\n  ");
    } /* if typeisstring and short */
    else {
      char filename[64];
      int fd;
      size_t size = val.attsize;
      char *buf = val.attvalue;
      int len;

      /* create a file with the stuff ... */
      (void) sprintf(filename,"%s.image", p->name);
      if ((fd = open(filename, O_WRONLY|O_TRUNC|O_CREAT, S_IRUSR|S_IWUSR|S_IRGRP)) < 0) {
	(void) fprintf(stderr,
		       "Dumpfirst: Failed to open '%s' for writing - %s\n", 
		       filename,
		       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	(void) fprintf(dump_desc.fd,
		       " \"(empty - error opening '%s' for writing - %s)\"\n", 
		       filename,
		       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
      }
      else {
	(void) fprintf(stderr, 
		       "Dumpfirst: binary value in %s stored in file '%s'\n",
		       p->name, filename);
	(void) fprintf(dump_desc.fd," < \"%s\"\n", filename);    
	for(; size > 0; size -= len, buf += len)
	  if ((len = write(fd, buf, size)) < 0) {
	    (void) fprintf(stderr, 
			   "Dumpfirst: Error writing to '%s' - %s\n", 
			   filename,
			   sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	    break;
	  } /* write */
	(void) close(fd);

      } /* ok, opening file */
    } /* a binary, but big */

    if (val.attvalue)
      free(val.attvalue);
  } /* for */

  (void)fprintf(dump_desc.fd, "\n######### Raw Node Contents #########\n");
  for (global_p = dump_desc.names; global_p; global_p = global_p->next) {
    if (global_p->subtree == 1) {
      (void)GAGN_GETATTRGROUPNAMES(&(global_p->lbl),raw_attr_group,(void *)NULL);
      (void)GLGN_GETLINKGROUPNAMES(&(global_p->lbl),raw_link_group,(void *)NULL);
    }
  }

  (void)fprintf(dump_desc.fd, "\n######### Node Contents #########\n\n");
  for (ref = dump_desc.refstructs; ref; ref = ref->next) {
    (void)fprintf(dump_desc.fd, "# Dumping ");
    write_node_symbol(&(ref->target->value->obj));
    (void)fprintf(dump_desc.fd, " in the view ");
    write_string(ref->target->idname);
    (void)fprintf(dump_desc.fd, "\n");
    write_node_contents(ref->target);
    (void)fprintf(dump_desc.fd, "\n##########################\n");
  }

  if (!GA_GETATTR(&(dump_desc.obj), "DUMP", "postamble", &val) && val.attvalue) {
    (void)fprintf(dump_desc.fd, "%s\n\n", val.attvalue);
    free(val.attvalue);
  }
}

/*  */
/**********************************************************************
 * Function: int make_t2l_file(label *desc)
 *
 * Produces a t2lincks-able dump file according to the given dump file
 * description object.
 *
 * The link group "DUMP" tells which reference structures to dump, except
 * the field "Raw", which holds sub-trees to dump completely, and the
 * field "Avoid" which holds nodes to not dump raw.
 *
 * The link value "DUMP" tells which reference structures to dump.
 *
 * The link group "DUMP NAMES" holds user defined node names.
 *
 * The attribute "DUMP":"file name" holds the dump file name.
 *
 * The attribute "DUMP":"preamble" holds preamble text (literally).
 *
 * Modifications:
 *      <list mods with name and date>
 */
int make_t2l_file(desc)
  label *desc;
{
  int err;
  reference_structure *ref;

  (void)copylbl(&(dump_desc.obj), desc);
  (void)strcpy(&(dump_desc.gen_name[0]), "node0000000");
  dump_desc.file_name.attvalue = NULL;
  dump_desc.file_name.attsize = 0;

  /*
   * Build the initial node name list from user's definitions
   */
  switch (err = GLN_GETLINKNAMES(desc, "DUMP NAMES",
				 second_field_map, (void *)NULL)) {
  case ERR_GROUP:
  case SUCCESS:
    break;
  default:
    free_dump_stuff();
    return (err);
  }

  /*
   * Build the avoidance list (if any).
   */
  switch (err = GLV_GETLINKVAL(desc, "DUMP", "Avoid",
			       subtree_avoid_map, (void *)NULL)) {
  case ERR_GROUP:
  case ERR_FIELD:
  case SUCCESS:
    break;
  default:
    free_dump_stuff();
    return (err);
  }

  
  /*
   * Build a (hidden) list of reference structures for the things
   * to dump.
   */
  if ((err = GLN_GETLINKNAMES(desc, "DUMP", first_field_map, (void *)NULL)))
    return err;

  for (ref = dump_desc.refstructs; ref; ref = ref->next) {
    (void)name_dump_node(&(ref->target->value->obj));
    collect_dump_nodes(ref->target);
  }

  /*
   * Now we're ready for the actual dumping.
   */

  if ((err = GA_GETATTR(desc, "DUMP", "file name", &(dump_desc.file_name)))) {
    free_dump_stuff();
    return err;
  }
  if ((dump_desc.fd = fopen(dump_desc.file_name.attvalue, "w")) == NULL) {
    (void)printf("Error: couldn't open %s\n", dump_desc.file_name.attvalue);
    free_dump_stuff();
    return FAIL;
  }
  write_t2l_file();

  (void)fclose(dump_desc.fd);
  free_dump_stuff();
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void add_link_desc(FOUR PARAMETERS)
 * Parameters:
 *	label *obj
 *	char *group
 *	char *field
 *	int plural
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void add_link_desc(obj, group, field, plural)
  label *obj;
  char *group;
  char *field;
  int plural;
{
  name_list *node;
  link_desc *p;

  if (!obj || !group || !field)
    return;

  for (node = dump_desc.names; node; node = node->next)
    if (node->lbl.vs == obj->vs)
      break;

  if (!node)
    return;

  for (p = node->links; p; p = p->next) {
    if ((!strcmp(p->field, field)) && (!strcmp(p->group, group))) {
      p->plural |= plural;
      return;
    }
  }

  p = (link_desc *) malloc(sizeof(link_desc));
  p->plural = plural;
  p->next = node->links;
  p->group = strdup(group);
  p->field = strdup(field);
  p->dumped = 0;
  node->links = p;
}

/*  */
/**********************************************************************
 * Function: static void check_links(FOUR PARAMETERS)
 * Parameters:
 *	label *lbl
 *	char *group
 *	char *field
 *	int plural
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int check_links(lbl, group, field, plural)
  label *lbl;
  char *group;
  char *field;
  int plural;
{
  name_list *n;
  link_desc *l;

  for (n = dump_desc.names; n; n = n->next) {
    if (n->lbl.vs == lbl->vs)
      break;
  }

  if (!n)
    return 0;

  for (l = n->links; l; l = l->next) {
    if ((!strcmp(l->field, field)) && (!strcmp(l->group, group))) {
      if (l->dumped)
	return 1;
      if (!plural && l->plural)
	return 1;
      l->dumped = 1;
      return 0;
    }
  }
  return 0;
}
