/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME:         aimpropagate.c
 *
 * SCCSINFO:            @(#)aimpropagate.c	1.10 5/17/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1990-04-18
 *
 * MODIFICATIONS:
 *      1994-05-17 Martin Sjlin. Moved free_pqe_element into
 *                 aimpropagate since it is independent of the
 *                 uim_propagate.c
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains the functions that handle
 * propagation into reference structures after database
 * updates. Entry points are
 * 	- propagate() generic propagation entry after
 * 	reference structure update.
 * 	- linkpropagate() to set a link and propagate after
 * 	setting.void blddisplay(rsp)
 * 	- propagatedisplay() to call blddisplay on all reference
 * 	structures.
 *
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimpropagate.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_blddisplay.h"
#include "f_xstuff.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern reference_structure *refstructs;	/* list of all ref structures. */
extern int global_unbind;
/* liblincks.a */
extern int LL_uid;		/* for printing error messages */
extern int sys_nerr;		/* in propagation routines */
extern char *sys_errlist[];
extern int errno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/*
 * The "swcase" macro defines case indices for a pair of infonode
 * index. The idea is that swcase(x,y) denotes a switch case when
 * the present identry value is of kind x (e.g. LINKFIELDTAG) and
 * the current update is of kind y.
 */
#define swcase(x,y)	((x)*8+(y))
/*
 * The "EXPOBJ" macro determines which of the "obj" and "lbl" fields
 * of the identry owner that contains the label for which the identry
 * carries information.
 */
#define EXPOBJ(idp)	((idp->owner->index == LINKITEM) ?\
			 &(idp->owner->lbl) : &(idp->owner->obj))

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int ChangePropagate P_(( identry *idp ));
static int CopyPropagate P_(( identry *idp ));
static void createaccesstag P_(( identry *idp, char *tag ));
static int CreatePropagate P_(( identry *idp ));
static void Dopropagate P_(( identry *idp ));
static int ensureaccesstag P_(( identry *idp, char *tag ));
static int eqstr P_(( char *s1, char *s2 ));
static int is_implicitaccess P_(( identry *idp ));
static int maybeaffected P_(( identry *idp ));
static void RemovePropagate P_(( identry *idp ));
static int renewattrvalue P_(( identry *idp ));
static void renewimage P_(( infonode *inp, int flg ));
static void renewplural P_(( identry *idp ));
static void renewtags P_(( identry *idp ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/*
 * The variables below contain information about what the current update
 * is (was). Some information is valid only sametimes (like, the chgpos
 * is valid only for changes to link items).
 */
static label *chgobj;	/* Which database object was changed. */
static int chgix;	/* Which item within the object	*/
static int chghow;	/* How? (CREATEOP, CHANGEOP, or REMOVEOP) */
static char *chggrp;	/* Group tag for affected group (if any) */
static char *chgfld;	/* Field tag for affected field (if any) */
static int chgpos;	/* Affected position (if any) */
static label *chglbl;	/* New label (when link item is created) */
static char *chgoldv;	/* Old infonode value (used for tag changes) */
static char *chgnewv;	/* New infonode value (used for tag changes) */
/*
 * The variable rsp is used by the propagate function to pin-point
 * the current reference structure (i.e. the one currently being
 * investigated. Apart from in the propagate function, the variable
 * rsp is used at calls to 'expandtarget'.
 */
static reference_structure *rsp;

static int global_propagate_called = 0;

/*  */
/*******************************************************************
 * Function: static int maybeaffected(identry *idp)
 *
 * Determines whether the value field of the target identry idp may
 * be affected by the current update because it carries (or may carry)
 * a value from the changed object.
 * - The top of the reference structure is never affected.
 * - If the carried values are of LINKITEM type and the affected item
 *   is an IMAGE, then the values may be affected.
 * - If the carried values are from the object being changed, then
 * they may be affected, otherwise not.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int maybeaffected(idp)
  identry *idp;
{
  label *obj;

  if (idp->owner == (infonode *) NULL) {
    if ((chgix == IMAGE) && (chgobj->vs == idp->value->obj.vs))
      return TRUE;
    return FALSE;
  }

  if (idp->tmpl->value->index == LINKITEM && chgix == IMAGE)
    return TRUE;

  obj =
    (idp->owner->index == LINKITEM) ?
      &(idp->owner->lbl) : &(idp->owner->obj);

  if (BOUND(obj))
    return FALSE;

  return (chgobj->vs == obj->vs);
}

/*  */
/*******************************************************************
 * Function: static void createaccesstag(identry *idp, char *tag)
 *
 * Creates an infonode to carry a newly created tag. Following the
 * NODE level, a new tag is always inserted first in the tag list.
 * Expands upon the new tag according to present reference structure.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void createaccesstag(idp, tag)
  identry *idp;
  char *tag;
{
  infonode *inp;

  /* Check for place holder and remove it. */

  inp = idp->value;
  if ((inp != (infonode *) NULL) && (inp->iflg & NONEXIST_MSK)) {
    unlink_infonode(inp);
    free_infonode(rsp, inp, 0);
  }
  inp = createinfonode(idp, chgobj);
  inp->next = idp->value;
  idp->value = inp;
  inp->val.attsize = strlen(tag) + 1;
  inp->val.attvalue = strdup(tag);

  expandtarget(rsp, inp);
}

/*  */
/*******************************************************************
 * Function: static int ensureaccesstag(identry *idp, char *tag)
 *
 * Handles implicit creation of access information. The effect here
 * is to ensure that the 'tag' is within the 'idp->value' list. If
 * not, a carrier is created and expanded upon (createaccesstag).
 * [Should re-fetch the whole object list??]
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int ensureaccesstag(idp, tag)
  identry *idp;
  char *tag;
{
  infonode *inp;

  for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
    if (eqstr(inp->val.attvalue, tag))
      return 0;

  createaccesstag(idp, tag);
  return 1;
}

/*  */
/*******************************************************************
 * Function: static int CreatePropagate(identry *idp)
 *
 * Determine the appropriate create-propagation action when knowing
 * that the idp value may be affected.
 * Returns 1 if the first idp->value thing has been newly expanded
 * (may happen for implicit access tag creations)
 * Otherwise 0.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int CreatePropagate(idp)
  identry *idp;
{
  infonode *inp, **pinp;
  label boundobj;

  inp = idp->value;

  switch (swcase(idp->tmpl->value->index, chgix)) {

  case swcase(ATTRFIELDTAG, ATTRFIELDTAG):
  case swcase(LINKFIELDTAG, LINKFIELDTAG):
    /*
     * This case handles the value "all field tags" when such a field
     * tag is created.
     */
    if (!eqstr(grouptag(idp), chggrp))
      break;
    /* CONTINUED in case below */
  case swcase(ATTRGROUPTAG, ATTRGROUPTAG):
  case swcase(LINKGROUPTAG, LINKGROUPTAG):
    /*
     * This case handles the value "all group tags" when such a group
     * tag is created.
     */
    createaccesstag(idp, chgnewv);
    break;

  case swcase(LINKITEM, LINKITEM):
    /*
     * This case handles the value "link item(s)" when a linkitem is
     * created.
     */
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgfld))
      break;

    if (idp->tmpl->value->pos == -1) {
      /*
       * The idp->value is plural and holds all link items under a given
       * group and field tag. The action is therefore to insert an
       * infonode for the new link item.
       */
      /* Check for place holder and remove it. */

      if ((inp != (infonode *) NULL) && (inp->iflg & NONEXIST_MSK)) {
	unlink_infonode(inp);
	free_infonode(rsp, inp, 0);
      }
      inp = createinfonode(idp, chgobj);
      (void)copylbl(&(inp->lbl), chglbl);
      /* 1992-11-30: make sure you update new binding tables when you
       * add a new link */
      (void)GBTC_GETBTCOMP(&(rsp->bt),
                     copylbl(&boundobj,&(inp->obj)), !rsp->flags.bound, 0);
      inp->pos = chgpos;

      for (pinp = &(idp->value); *pinp != (infonode *) NULL; pinp = &((*pinp)->next))
	if ((*pinp)->pos >= chgpos)
	  break;

      inp->next = (*pinp);
      (*pinp) = inp;

      while (inp->next != (infonode *) NULL) {
	inp = inp->next;
	(inp->pos)++;
      }

      inp = (*pinp);

    } else {

      if (chgpos > idp->tmpl->value->pos)
	break;

      /* The idp->value is singular and holds a link item that occurs
	 after the created one. The action is therefore to re-make the
	 infonode contents.
	 */

      if (inp->val.attsize > 0)
	free(inp->val.attvalue);
      inp->val.attsize = 0;
      inp->val.attvalue = NULL;
      free_subnodes(rsp, inp, 0);
      (void)GBTC_GETBTCOMP(&(rsp->bt),
                     copylbl(&boundobj,&(inp->obj)), !rsp->flags.bound, 0);
      (void)GLI_GETLINKITEM(&(boundobj), chggrp, chgfld, inp->pos, &(inp->lbl));
    }


    /* NOTE: continuation for both plural and singular branches. */

   (void)GBTC_GETBTCOMP(&(rsp->bt),
                copylbl(&boundobj,&(inp->lbl)), !rsp->flags.bound, 0);
    (void)GI_GETIMAGE(&(boundobj), &(inp->val));
    expandtarget(rsp,inp);

    break;


  case swcase(LINKGROUPTAG, LINKITEM):
  case swcase(LINKGROUPTAG, LINKFIELDTAG):
  case swcase(ATTRGROUPTAG, ATTRFIELDTAG):
  case swcase(ATTRGROUPTAG, ATTRVALUE):
    /* This case handles implicit creation of group access information. */

    return ensureaccesstag(idp, chggrp);
    break;


  case swcase(LINKFIELDTAG, LINKITEM):
  case swcase(ATTRFIELDTAG, ATTRVALUE):
    /* This case handles implicit creation of field access information. */

    if (eqstr(grouptag(idp), chggrp))
      return ensureaccesstag(idp, chgfld);
    break;

  default:
    break;
  }

  return 0;
}


/*  */
/*******************************************************************
 * Function: static void renewimage(infonode *inp, int flg)
 *
 * Release present value and get the image from object 'obj' (if
 * flg=0) or 'lbl' (otherwise).
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void renewimage(inp, flg)
  infonode *inp;
  int flg;
{
  char *p;
  label boundobj;

  p = inp->val.attvalue;
  inp->val.attvalue = NULL;
  inp->val.attsize = 0;
  inp->changed = 1;

  if (flg) {
    (void)GBTC_GETBTCOMP(&(rsp->bt),
                copylbl(&boundobj,&(inp->lbl)), !rsp->flags.bound, 0);
    (void)GI_GETIMAGE(&(boundobj), &(inp->val));
  }
  else {
    (void)GBTC_GETBTCOMP(&(rsp->bt),
                copylbl(&boundobj,&(inp->obj)), !rsp->flags.bound, 0);
    (void)GI_GETIMAGE(&(boundobj), &(inp->val));
  }

  if ((propagation_queue_add(rsp, inp, (Widget) NULL, (formentry *) NULL,
			     PQE_OP_CHANGEWIDGET)) == FAIL) {
    fatal_error("propagation_queue_add failed in renewimage\n");
  }
  if (p != NULL)
    free(p);
}


/*  */
/*******************************************************************
 * Function: static int renewattrvalue(identry *idp)
 *
 * Release present value and get the indicated attribute value.
 *
 * Note: a return value of 1 means that propagation will not inspect
 * (propagate into) the substructure.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int renewattrvalue(idp)
  identry *idp;
{
  char *p;
  label boundobj;
  infonode *np = idp->value;

  if (np->iflg & NONEXIST_MSK) {	/* if not visible */
    unlink_infonode(np);
    free_infonode(rsp, np, 0);
    np = createinfonode(idp, chgobj);
    idp->value = np;
  }

  p = np->val.attvalue;
  np->val.attvalue = NULL;
  np->val.attsize = 0;

  np->group = grouptag(np->head);
  np->field = fieldtag(np->head);
  np->changed = 1;
  (void)GBTC_GETBTCOMP(&(rsp->bt),
                copylbl(&boundobj,&(np->obj)), !rsp->flags.bound, 0);
  (void)GA_GETATTR(&(boundobj), np->group, np->field, &(np->val));

  if (np -> widp) {
    if ((propagation_queue_add(rsp, np, (Widget) NULL, (formentry *) NULL,
			     PQE_OP_CHANGEWIDGET)) == FAIL) {
      fatal_error("propagation_queue_add failed in renewattrvalue\n");
    }
  }
  if (p != NULL)
    free(p);

  if (!np->subnodes) {
    expandtarget(rsp, np); 
    return 1;
  }
  return 0;
}


/*  */
/*******************************************************************
 * Function: static void renewplural(identry *idp)
 *
 * Release and rebuild identry value list.
 * [ralro FIX: the rebuilding should compare actual database
 * information with the present list, so as to not rebuild
 * unchanged parts.]
 * >called when group tag has changed and value is any but ATTRVAL
 * >called when field tag has changed and value is any but ATTRVAL
 * >called when link group tag is removed and value is link item
 * >called when link field tag is removed and value is link item
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void renewplural(idp)
  identry *idp;
{
  infonode *inp;
  label obj;
  int saved = global_unbind;

  if (idp->owner) {
    if (idp->owner->index == LINKITEM)
      (void)copylbl(&obj,&(idp->owner->lbl));
    else
      (void)copylbl(&obj,&(idp->owner->obj));
  } else {
    global_unbind = 1;
    (void)copylbl(&obj,&(idp->value->obj));
    obj.inst = -1;
  }
  free_value(rsp, idp, 0);
  idp->value = makeinfonodes(idp, &obj);
  if (idp->value == (infonode *) NULL) {
    /* Create a place holder */
    add_def_values(idp, &obj, rsp);
    idp->value = createinfonode(idp, &obj);
    setplaceholder(idp->value,(int)NONEXIST_MSK);
  } else {
    for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
      expandtarget(rsp,inp);
  }
  global_unbind = saved;
}


/*  */
/*******************************************************************
 * Function: static void renewtags(identry *idp)
 *
 * Rebuild a tag list from knowing that some tag has changed.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void renewtags(idp)
  identry *idp;
{
  infonode *inp;
  char *p;

  for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
    if (eqstr(inp->val.attvalue, chgoldv)) {
      p = inp->val.attvalue;	/* By malloc-ing before */
      inp->val.attsize = strlen(chgnewv) + 1;	/* free-ing, the new string */
      inp->val.attvalue = strdup(chgnewv);	/* must have a new address. */
      inp->changed = 1;
      if ((propagation_queue_add(rsp, inp, (Widget) NULL, (formentry *) NULL,
				 PQE_OP_CHANGEWIDGET)) == FAIL) {
	fatal_error("propagation_queue_add failed in renewtags\n");
      }
      free(p);
      break;
    }
}

/*  */
/*******************************************************************
 * Function: static int ChangePropagate(identry *idp)
 *
 * Determine the appropriate change-propagation action given the
 * knowledge that the idp value is obtained from the changed
 * database object.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int ChangePropagate(idp)
  identry *idp;
{
  infonode *kind, *inp;
  char *tmp;
  label boundobj;

  kind = idp->tmpl->value;
  inp = idp->value;

  if (kind->index == SAMEASPARENT) {
    if (inp->val.attvalue != inp->parent->val.attvalue) {
      inp->val.attsize = inp->parent->val.attsize;
      inp->val.attvalue = inp->parent->val.attvalue;
      inp->changed = 1;
      if ((propagation_queue_add(rsp, inp, (Widget) NULL, (formentry *) NULL,
				 PQE_OP_CHANGEWIDGET)) == FAIL) {
	fatal_error("propagation_queue_add failed in ChangePropagate\n");
      }
    }
    return 0;
  }
  switch (swcase(kind->index, chgix)) {

  case swcase(ATTRGROUPTAG, ATTRGROUPTAG):
  case swcase(LINKGROUPTAG, LINKGROUPTAG):
    renewtags(idp);
    break;

  case swcase(ATTRVALUE, ATTRGROUPTAG):
    if (eqstr(kind->group, chgoldv) || eqstr(kind->group, chgnewv))
      return (renewattrvalue(idp));
    if (eqstr(tmp = grouptag(idp), chgnewv))
      for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
	inp->group = tmp;
    break;

  case swcase(ATTRFIELDTAG, ATTRGROUPTAG):
  case swcase(LINKFIELDTAG, LINKGROUPTAG):
  case swcase(LINKITEM, LINKGROUPTAG):
    if (eqstr(kind->group, chgoldv) || eqstr(kind->group, chgnewv)) {
      renewplural(idp);
      break;
    }
    if (eqstr(tmp = grouptag(idp), chgnewv))
      for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
	inp->group = tmp;

    break;

  case swcase(ATTRFIELDTAG, ATTRFIELDTAG):
  case swcase(LINKFIELDTAG, LINKFIELDTAG):
    if (!eqstr(grouptag(idp), chggrp))
      break;
    renewtags(idp);

    break;

  case swcase(ATTRVALUE, ATTRFIELDTAG):
    if (!eqstr(grouptag(idp), chggrp))
      break;

    if (eqstr(kind->field, chgoldv) || eqstr(kind->field, chgnewv))
      return (renewattrvalue(idp));
    if (eqstr(tmp = fieldtag(idp), chgnewv))
      for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
	inp->field = tmp;

    break;

  case swcase(LINKITEM, LINKFIELDTAG):

    if (!eqstr(grouptag(idp), chggrp))
      break;

    if (eqstr(kind->field, chgoldv) || eqstr(kind->field, chgnewv)) {
      renewplural(idp);
      break;
    }
    if (eqstr(tmp = fieldtag(idp), chgnewv))
      for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
	inp->field = tmp;

    break;

  case swcase(ATTRVALUE, ATTRVALUE):
    if (!eqstr(inp->group, chggrp) || !eqstr(inp->field, chgfld))
      break;
    return (renewattrvalue(idp));

  case swcase(LINKITEM, LINKITEM):
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgfld))
      break;
    for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
      if (inp->pos == chgpos) {

        (void)GBTC_GETBTCOMP(&(rsp->bt),
                copylbl(&boundobj,&(inp->obj)), !rsp->flags.bound, 0);
	(void)GLI_GETLINKITEM(&(boundobj), chggrp, chgfld, chgpos, &(inp->lbl));
	renewimage(inp, TRUE);
      }
    break;

  case swcase(IMAGE, IMAGE):
    renewimage(inp, FALSE);
    break;

  case swcase(LINKITEM, IMAGE):
    for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next)
      if (inp->lbl.vs == chgobj->vs) {
	renewimage(inp, TRUE);
      }
    break;

  case swcase(LINKGROUPTAG, LINKITEM):
  case swcase(LINKGROUPTAG, LINKFIELDTAG):
  case swcase(ATTRGROUPTAG, ATTRFIELDTAG):
  case swcase(ATTRGROUPTAG, ATTRVALUE):
    /* This case handles implicit creation of group access information. */

    return ensureaccesstag(idp, chggrp);
    break;


  case swcase(LINKFIELDTAG, LINKITEM):
  case swcase(ATTRFIELDTAG, ATTRVALUE):
    /* This case handles implicit creation of field access information. */

    if (eqstr(grouptag(idp), chggrp))
      return ensureaccesstag(idp, chgfld);
    break;

  default:
    break;
  }

  return 0;
}


/*  */
/*******************************************************************
 * Function: static void RemovePropagate(identry *idp)
 *
 * Determine the appropriate remove-propagation action given the
 * knowledge that the idp value is obtained from the changed
 * database object.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void RemovePropagate(idp)
  identry *idp;
{
  infonode *kind, *inp, *p;
  int flag = NONEXIST_MSK;

  inp = idp->value;
  kind = idp->tmpl->value;

  switch (swcase(kind->index, chgix)) {

  case swcase(ATTRFIELDTAG, ATTRFIELDTAG):
  case swcase(LINKFIELDTAG, LINKFIELDTAG):
    if (!eqstr(grouptag(idp), chggrp))
      break;

  case swcase(ATTRGROUPTAG, ATTRGROUPTAG):
  case swcase(LINKGROUPTAG, LINKGROUPTAG):
    while (inp != (infonode *) NULL) {
      p = inp->next;
      trace("(Checking tag %s)\n", inp->val.attvalue);
      if (eqstr(inp->val.attvalue, chgoldv)) {
	unlink_infonode(inp);
	free_infonode(rsp, inp, 0);
	break;
      }
      inp = p;
    }
    break;

  case swcase(ATTRFIELDTAG, ATTRGROUPTAG):
  case swcase(LINKFIELDTAG, LINKGROUPTAG):
    if (!eqstr(grouptag(idp), chgoldv))
      break;
    free_value(rsp, idp, 0);
    break;

  case swcase(ATTRVALUE, ATTRGROUPTAG):
    if (!eqstr(grouptag(idp), chgoldv))
      break;
    unlink_infonode(inp);
    free_infonode(rsp, inp, 0);
    break;

  case swcase(ATTRVALUE, ATTRFIELDTAG):
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgoldv))
      break;
    unlink_infonode(inp);
    free_infonode(rsp, inp, 0);
    break;

  case swcase(ATTRVALUE, ATTRVALUE):
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgfld))
      break;
    unlink_infonode(inp);
    flag = 0;
    free_infonode(rsp, inp, 0);
    break;
  case swcase(IMAGE, IMAGE):
    unlink_infonode(inp);
    flag = 0;
    free_infonode(rsp, inp, 0);
    break;

  case swcase(LINKITEM, LINKGROUPTAG):
    if (!eqstr(grouptag(idp), chgoldv))
      break;
    renewplural(idp);
    break;

  case swcase(LINKITEM, LINKFIELDTAG):
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgoldv))
      break;
    renewplural(idp);
    break;

  case swcase(LINKITEM, LINKITEM):
    if (!eqstr(grouptag(idp), chggrp) || !eqstr(fieldtag(idp), chgfld))
      break;
    while (inp != (infonode *) NULL) {
      p = inp->next;
      if (inp->pos == chgpos) {
	unlink_infonode(inp);
	free_infonode(rsp, inp, 0);
      } else {
	if (inp->pos > chgpos) {
	  (inp->pos)--;
	}
      }
      inp = p;
    }
    break;

  default:
    break;
  }

  if (idp->value == (infonode *) NULL) {
    /* Create a place holder */
    add_def_values(idp, EXPOBJ(idp), rsp);
    idp->value = createinfonode(idp, EXPOBJ(idp));
    setplaceholder(idp->value, flag);
  }
}

/*  */
/*******************************************************************
 * Function: static int CopyPropagate(identry *idp)
 *
 * Determine the appropriate copy-propagation action given the
 * knowledge that the idp value may be affected by the copying.
 *
 * Modifications:
 *	<list mods with name and date>
 */
static int CopyPropagate(idp)
  identry *idp;
{
  infonode *kind, *inp;

  inp = idp->value;
  kind = idp->tmpl->value;

  if (kind->index == LINKITEM) {
    for (inp = idp->value; inp != (infonode *) NULL; inp = inp->next) {
      if (inp->lbl.vs == chgobj->vs) {
	renewimage(inp, TRUE);
	free_subnodes(rsp, inp, FALSE);
	expandtarget(rsp, inp);
      }
    }
  } else {
    renewplural(idp);
  }
  return 1;
}

/*  */
/*******************************************************************
 * Function: static void Dopropagate(identry *idp)
 *
 * This function traverses a target structure and calls the
 * appropriate propagation handler when needed.
 *
 * [The skipfirst flag is set within a Create or Change propagation
 * when the affected item involves an implicit creation of access
 * information. Then the first infonode of the idp->value has been
 * handled through an expandtarget call for a newly added access
 * tag (see ensureaccesstag)]
 *
 * Modifications:
 *	<list mods with name and date>
 */
static void Dopropagate(idp)
  identry *idp;
{
  int skipfirst;
  infonode *inp;
  infonode *owner;

  if (idp == (identry *)NULL)
    return;
  owner = idp->owner;

  while (idp != (identry *)NULL) {
    skipfirst = 0;
    if (maybeaffected(idp)) {
      if (is_implicitaccess(idp)) {
	renewplural(idp);
	idp = idp->next;
	continue;
      }
      switch (chghow) {
      case COPYOP:
	if(CopyPropagate(idp)) {
	  idp = idp->next;
	  continue;
	}
	break;
      case CREATEOP:
	skipfirst = CreatePropagate(idp);
	break;
      case CHANGEOP:
	skipfirst = ChangePropagate(idp);
	break;
      case REMOVEOP:
	RemovePropagate(idp);
	break;
      default:
	/* This `should' never happen, but.... */
	break;
      }
    }
    if ((inp = idp->value) && skipfirst)
      inp = inp->next;
    while (inp != (infonode *) NULL) {
      Dopropagate(inp->subnodes);
      inp = inp->next;
    }

    idp = idp->next;
  }
  if (owner)
    owner->changed = 0;
}


/*  */
/*******************************************************************
 * Function: void propagatedisplay()
 *
 * This function calls 'blddisplay' on each of the reference
 * structures to ensure that all windows are updated.  Currently,
 * there is no marking of which reference structures have been
 * affected by a change so all of them must be traversed.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void propagatedisplay()
{
  if (global_propagate_called == 0)
    return;
  for (rsp = refstructs; rsp; rsp = rsp->next) 
    blddisplay(rsp);

  global_propagate_called = 0;
}

/*  */
/*******************************************************************
 * Function: void set_global_propagate_called()
 *
 * Modifications:
 *	<list mods with name and date>
 */
void set_global_propagate_called()
{
  global_propagate_called = 1;
}


/*  */
/*******************************************************************
 * Function: void propagate(identry *idp,int how,infonode *oldinfo,infonode *newinfo)
 *
 * This function is the general interface function called after a
 * database update. It maps through the list of reference
 * structures and propagates the given database update into
 * incremental changes to the reference structures.
 *
 * Modifications:
 *	<list mods with name and date>
 */
void propagate(cmd, obj, index, group, field, pos, val, lbl, oldtag)
  int cmd;			/* CREATEOP, CHANGEOP, REMOVEOP, LINKOP */
  label *obj;			/* Object affected (before liblincks call) */
  int index;			/* Which field is affected. */
  char *group;			/* Access group tag (when needed) */
  char *field;			/* Access field tag (when needed) */
  int pos;			/* Link item position number (when needed) */
  attrval *val;			/* New contents or tag text (when needed) */
  label *lbl;			/* New label value (when needed) */
  char *oldtag;			/* Old tag value (when appropriate) */
{
  label bt;
  label root;
  label gpd;
  int whole;

  global_propagate_called = 1;

  chghow = cmd;			/* What kind of operation */
  chgobj = obj;			/* Object that changed */
  chgix = index;		/* Which kind of item changed */
  chggrp = group;
  chgfld = field;
  chgpos = pos;
  chglbl = lbl;
  chgnewv = (val != (attrval*)NULL) ? val->attvalue : (char *)NULL;
  chgoldv = oldtag;

  for (rsp = refstructs; rsp != (reference_structure *)NULL; rsp = rsp->next) {
    if (!rsp->flags.bound) {
      whole = 0;
      (void)GC_GETCURRENT(copylbl(&bt,&(rsp->bt)));
      (void)GC_GETCURRENT(copylbl(&root,&(rsp->root)));
      (void)GC_GETCURRENT(copylbl(&gpd,&(rsp->gpd)));
      if (bt.inst != rsp->bt.inst && !INWORKSPACE(&bt)) {
#ifdef DEBUG
	(void)printf("%s <%d:%d>\n%s\n", 
		     "Non-current binding table version",
		     rsp->bt.vs, rsp->bt.inst,
		     "Rebuilding view from scratch.");
#endif	/* DEBUG */
	whole = 1;
      } else {
	if (root.inst != rsp->root.inst && !INWORKSPACE(&root)) {
#ifdef DEBUG
	    (void)printf("%s <%d:%d>\n%s\n", 
			 "Non-current root version",
			 rsp->root.vs, rsp->root.inst,
			 "Rebuilding view from scratch.");
#endif	/* DEBUG */
	  whole = 1;
	} else {
	  if (gpd.inst != rsp->gpd.inst && !INWORKSPACE(&gpd)) {
#ifdef DEBUG
	    (void)printf("%s <%d:%d>\n%s\n", 
			 "Non-current gpd version",
			 rsp->gpd.vs, rsp->gpd.inst,
			 "Rebuilding view from scratch.");
#endif	/* DEBUG */
	    whole = 1;
	  }
	}
      }
      if (whole) {
	(void)UBL_UNBINDLABEL(&root);
	(void)buildref(&root,rsp->template->idname,rsp);
      } else {
	Dopropagate(rsp->target);
      }
    }
  }
}

/*  */
/*******************************************************************
 * Function: int propagation_queue_add(FIVE PARAMETERS)
 * Parameters:
 *	reference_structure *ref
 *	infonode *inptr
 *	Widget widget
 *	formentry *format
 *	int operation
 *
 * Modifications:
 *	<list mods with name and date>
 *
 * This function adds a propagation queue element to the queue (at the end)
 * associated with a reference structure.  This element indicates that the
 * widget associated with an infonode requires some widget-level
 * propagation function.  It is only called internally by AIM level
 * functions and should not be called by others.
 *
 * Arguments:
 *   (1) pointer to a reference structure with which the queue is
 *       associated.
 *   (2) infonode pointer, which indicates the infonode with which the
 *       operation is associated.
 *   (3) Widget pointer, which will only be non-null in the cases where
 *       the element will be for making a new widget.  Even in this case,
 *       it can be null if the widget is to be put at the very first
 *       position.
 *   (4) which operation is to be performed.  One of PQE_OP_MAKEWIDGET,
 *       PQE_OP_CHANGEWIDGET, or PQE_OP_REMOVEWIDGET
 *
 * Returns SUCCESS or FAILURE (only on malloc error).
 *
 * Assumes that reference structure pointer and the infonode
 * pointer are non-null and will give a fatal error should they not
 * be.  Assumes a valid operation, with a fatal error should it not
 * be.
 *
 * malloc's memory which will be free'ed in propagation_queue_process()
 *
 * Modified:
 *	1991-Dec-05 by David Partain to add formatting information
 * 	to the propagation queue element so that it can be dealt
 *	with at the higher level.
 *
 * 	<list mods here with date and name>
 */
int propagation_queue_add(ref_struct, in_infonode, widget, format, operation)
  reference_structure *ref_struct;	/* which ref struct */
  infonode *in_infonode;	/* which infonode */
  Widget widget;		/* which widget */
  formentry *format;		/* formats associated with this widget */
  int operation;		/* which operation */
{
  propagation_q_element *pqe_element = NULL;

  /* check the parameters just in case */
  if ((ref_struct == (reference_structure *)NULL) ||
    ((in_infonode == (infonode *) NULL) && (operation != PQE_OP_REMOVEWIDGET))) {
    fatal_error("propagation_queue_add: null mandatory parameters\n");
  }
  /* check the operation NOTE:  PQE_OP_COMPLETED not permitted initially */
  if ((operation != PQE_OP_MAKEWIDGET) &&
      (operation != PQE_OP_CHANGEWIDGET) &&
      (operation != PQE_OP_REMOVEWIDGET)) {
    fatal_error("propagation_queue_add: invalid operation\n");
  }
  /* check to make sure the queue is not corrupted */
  if ((!ref_struct->pqueue.head) != (!ref_struct->pqueue.tail)) {
    fatal_error("propagation_queue_add: corrupted queue\n");
  }
  /*
   * get the struct to put on the queue. this will be free'ed within
   * the function propagation_queue_process().
   */
  if ((pqe_element = (propagation_q_element *)
       malloc(sizeof(propagation_q_element))) == NULL) {
    LogMess(LL_uid,
	"aim: memory allocation error: pqe_element propagation_queue_add, %s",
	    sys_errlist[(errno > sys_nerr) ? 0 : errno]);
    return FAIL;
  }
  /* initialize structure */
  pqe_element->pqe_next = NULL;
  pqe_element->pqe_prev = NULL;
  pqe_element->pqe_format = NULL;
  pqe_element->pqe_widget = widget;	/* legally can be null */
  pqe_element->pqe_infonode = in_infonode;	/* will be null for remove */
  pqe_element->pqe_operation = operation;

  /*
   * add formats associated with a particular infonode if not a remove op.
   */
  if (operation != PQE_OP_REMOVEWIDGET) {
    if (format != (formentry *) NULL) {
      pqe_element->pqe_format = format;
    }
  }
  /* FINALLY... add the element to the queue */
  /* the queue is empty */
  if (!ref_struct->pqueue.tail) {
    ref_struct->pqueue.head = ref_struct->pqueue.tail = pqe_element;
  } else {
    ref_struct->pqueue.tail->pqe_next = pqe_element;
    pqe_element->pqe_prev = ref_struct->pqueue.tail;
    ref_struct->pqueue.tail = pqe_element;
  }
  return SUCCESS;
}

/*  */
/**********************************************************************
 * Function: static int eqstr(char *s1, char *s2)
 *
 * returns whether two strings are lexicographically equal.
 * note that it returns true if both are null.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int eqstr(s1, s2)
  char *s1;
  char *s2;
{
  if (!s1 || !s2)
    return (s1 == s2);
  else
    return (strcmp(s1, s2) == 0);
}

/*  */
/**********************************************************************
 * Function: static int is_implicitaccess(identry *idp)
 *
 * Determines if a function gets its access information implicitly
 * from its parent.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int is_implicitaccess(idp)
  identry *idp;
{
  if ((idp->owner == (infonode *) NULL) || (idp->owner->changed == 0))
    return 0;

  switch (idp->tmpl->value->index) {
  case ATTRVALUE:
  case LINKITEM:
    if (eqstr(idp->tmpl->value->group, "%VALUE%") ||
	eqstr(idp->tmpl->value->field, "%VALUE%"))
      return 1;
    return 0;
  case ATTRFIELDTAG:
  case LINKFIELDTAG:
    if (eqstr(idp->tmpl->value->group, "%VALUE%"))
      return 1;
    return 0;
  case ATTRGROUPTAG:
  case LINKGROUPTAG:
  case IMAGE:
  case SAMEASPARENT:
  case INTERNAL:
  default:
    return 0;
  }
}
/*  */
/**********************************************************************
 * Function: void free_pqe_element(propagation_q_element *pqe_element)
 *
 * Frees up all of the memory associated with a propagation queue
 * element.
 *
 * Modifications:
 *	<list mods here with date and name>
 */
void free_pqe_element(pqe_element)
  propagation_q_element *pqe_element;
{
  formentry *tmp_format1 = (formentry *)NULL;
  formentry *tmp_format2 = (formentry *)NULL;

  if (pqe_element == (propagation_q_element *) NULL) return;
  tmp_format1 = pqe_element -> pqe_format;
  while (tmp_format1 != (formentry *)NULL) {
    tmp_format2 = tmp_format1;
    tmp_format1 = tmp_format1 -> next;
    free((FREEPTR *) tmp_format2 -> formval);	/* formatting string */
    free((FREEPTR *) tmp_format2 -> formid);	/* formatting string */
    free((FREEPTR *) tmp_format2);			/* rest of structure */
  }
  free((FREEPTR *) pqe_element);
}
