/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	aimstruct.c
 *
 * SCCSINFO:		@(#)aimstruct.c	1.9 5/17/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1990-01-30
 *
 * MODIFICATIONS:
 *      1993-11-05 Martin Sjlin. Moved previousinfonode here from aim-
 *            command to have all external references in X non 
 *            dependents files.
 *      1994-05-17 Martin Sjlin. Moved currentinfonode here from
 *            aimstart .... (since previousinfonode is here ;-)
 *      <list mods with name and date>
 *
 * DESCRIPTION:
 *
 * This file contains sub-routines and primitives that operates directly
 * on the aim structures.
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_aimstruct.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
infonode *previousinfonode = (infonode *)NULL;
infonode *currentinfonode  = (infonode *)NULL;

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimsubr.h"
#include "f_xstuff.h"
#include "f_aimpropagate.h"
#include "f_uim_propagate.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern infonode *currentinfonode;	/* aimstart.c */
extern int global_unbind;		/* aimbuildref.c */
extern reference_structure *refstructs;	/* List of reference structures. */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* none */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void check_prop_queue P_(( reference_structure *ref, infonode
                                   *inp ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* none */

/*  */
/**********************************************************************
 * Function: void free_subnodes(reference_structure *ref_struct,infonode *inp,int flg)
 *
 * Release the memory occupied by the subnodes of an infonode. The
 * 'flg' marks template sub-tree, in which case also the strings in
 * the 'group' and 'field' fields are released.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_subnodes(ref_struct, inp, flg)
  reference_structure *ref_struct;
  infonode *inp;
  int flg;
{
  identry *p;

  while (inp->subnodes) {
    p = inp->subnodes;
    inp->subnodes = p->next;
    free_identry(ref_struct, p, flg);
  }
}

/*  */
/**********************************************************************
 * Function: void free_infonode(reference_structure *ref_struct,infonode *inp,int flg)
 *
 * Release the memory occupied by the sub-tree rooted in an infonode.
 * The 'flg' marks template sub-tree, in which case also the strings
 * in the 'group' and 'field' fields are released.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_infonode(ref_struct, inp, flg)
  reference_structure *ref_struct;
  infonode *inp;
  int flg;
{
  /*
   * if we're about to whack the current or previous infonode
   * global pointers (used primarily by the structure editing
   * functions) then we must reset it or watch xlincks dump core
   */
  if (inp == currentinfonode)
    currentinfonode = (infonode *)NULL;
  if (inp == previousinfonode)
    previousinfonode = (infonode *)NULL;
  free_subnodes(ref_struct, inp, flg);

  if (flg && (inp->group != NULL)) {
    free(inp->group);
    inp->group = NULL;
  }
  if (flg && (inp->field != NULL)) {
    free(inp->field);
    inp->field = NULL;
  }
  switch (inp->index) {
  case SAMEASPARENT:		/* The parent "owns" the value */
    break;
  case INTERNAL:		/* The template "owns" the value */
    if (!flg)
      break;
  default:
    if (inp->val.attsize != 0) {
      free(inp->val.attvalue);
      inp->val.attvalue = NULL;
    }
  }				/* switch */

  /* add element to propagation queue to get widget destroyed */
  if (inp->widp != (Widget) NULL) {
    check_prop_queue(ref_struct, inp);
    if ((propagation_queue_add(ref_struct, (infonode *)NULL, inp->widp,
			     (formentry *) NULL, PQE_OP_REMOVEWIDGET)) == FAIL) {
      fatal_error("propagation_queue_add failed in free_infonode\n");
    }
  }

  {
    formentry *tmp, *next;
    for (tmp = inp->formats; tmp ; tmp = next) {
      next = tmp->next;
      free((FREEPTR *)tmp->formid);
      free((FREEPTR *)tmp->formval);
      free((FREEPTR *)tmp);
    }
  }

  if (inp != (infonode *)NULL) {
    free((FREEPTR *)inp);
    inp = (infonode *)NULL;
  }
}

/*  */
/**********************************************************************
 * Function: void free_value(reference_structure *ref_struct,identry *idp,int flg)
 *
 * Release the memory occupied by the value of an identry. The
 * 'flg' marks template sub-tree, in which case also the strings in
 * the 'group' and 'field' fields are released.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_value(ref_struct, idp, flg)
  reference_structure *ref_struct;
  identry *idp;
  int flg;
{
  infonode *p;

  if (!flg || !(idp->flags.is_recursive)) {
    while (idp->value) {
      p = idp->value;
      idp->value = p->next;
      free_infonode(ref_struct, p, flg);
    }
  }
  idp->value = (infonode *)NULL;

}

/*  */
/**********************************************************************
 * Function: void free_identry(reference_structure *ref_struct,identry *idp,int flg)
 *
 * Release the memory occupied by the sub-tree rooted in an identry.
 * The 'flg' marks template sub-tree, in which case also the string
 * names for the identries are released.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_identry(ref_struct, idp, flg)
  reference_structure *ref_struct;
  identry *idp;
  int flg;
{
  if (!idp) { /* shouldn't happen */
    (void)fprintf(stderr, "free_identry: null identry idp.\n");
    (void)fprintf(stderr, "This SHOULDN'T be happening!\n");
    (void)fprintf(stderr, "Returning without trying to free anything.\n");
    return;
  }

  free_value(ref_struct, idp, flg);
  if (flg) {
    free(idp->idname);
    idp->idname = NULL;
  }
  if (idp) {
    free((FREEPTR *)idp);
    idp = (identry *)NULL;
  }
}

/*  */
/**********************************************************************
 * Function: infonode *newinfonode()
 *
 * Creates a new infonode record and resets all its fields.
 *
 * Modifications:
 *      <list mods with name and date>
 */
infonode *newinfonode()
{
  infonode *p;

  p = (infonode *) malloc(sizeof(infonode));
  if (p == (infonode *) NULL) {
    fatal_error("%s\n", "malloc failed in newinfonode");
  }
  p->next = p->parent = (infonode *) NULL;
  p->head = p->subnodes = (identry *) NULL;
  p->formats = (formentry *) NULL;
  p->selobj.vs = p->selobj.inst = -1;
  p->iflg = SUBLEVEL;
  p->obj.vs = p->obj.inst = -1;
  p->index = INTERNAL;
  p->group = p->field = (char *)NULL;
  p->pos = -1;
  p->lbl.vs = p->lbl.inst = -1;
  p->val.attsize = 0;
  p->val.attvalue = (char *)NULL;
  p->widp = 0;
  p->printed = 0;
  p->changed = 0;
  return (p);
}

/*  */
/**********************************************************************
 * Function: infonode *createinfonode(identry *idp,label *obj)
 *
 * Create a new infonode to fit as value for the identry idp taking
 * value from object obj. Fill in all but kind-dependent information.
 *
 * Modifications:
 *      <list mods with name and date>
 */
infonode *createinfonode(idp, obj)
  identry *idp;
  label *obj;
{
  infonode *p;

  p = newinfonode();
  p->parent = idp->owner;
  p->head = idp;
  p->formats = (formentry *) NULL;
  (void)copylbl(&(p->selobj), &(idp->tmpl->value->selobj));
  p->iflg |= idp->tmpl->value->iflg & LEVEL_MSK;
  (void)copylbl(&(p->obj), obj);
  if (global_unbind)
    (void)UBL_UNBINDLABEL(&(p->obj));
  p->index = idp->tmpl->value->index;
  p->group = grouptag(idp);
  p->field = fieldtag(idp);
  p->pos = idp->tmpl->value->pos;
  return p;
}


/*  */
/**********************************************************************
 * Function: identry *newidentry()
 *
 * Creates a new identry record and resets all its fields
 *
 * Modifications:
 *      <list mods with name and date>
 */
identry *newidentry()
{
  identry *p;

  p = (identry *) malloc(sizeof(identry));
  if (p == (identry *) NULL) {
    fatal_error("%s\n", "malloc failed in newidentry");
  }
  p->next = (identry *) NULL;
  p->owner = p->value = (infonode *) NULL;
  p->flags.is_recursive = 0;
  p->idname = (char *)NULL;
  p->tmpl = (identry *) NULL;
  return (p);
}

/*  */
/**********************************************************************
 * Function: void free_ref_cont(reference_structure *ref)
 *
 * Input:        ref = The reference structure to free.
 *
 * Note: don't free the widl's here because that will close down
 * history graph windows which should not be closed when a reference
 * structure is reused.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_ref_cont(ref)
  reference_structure *ref;
{

  if (ref == NULL)
    return;

  free_identry(ref, ref->target, 0);
  free_identry(ref, ref->template, 1);
  ref->target = NULL;
  ref->template = NULL;
}

/*  */
/**********************************************************************
 * Function: void free_ref(reference_structure *ref)
 *
 * Input:        ref = The reference structure to free.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_ref(ref)
  reference_structure *ref;
{
  reference_structure **p;
  popd_list *pdl, *pdl2;
  propagation_q_element *pqe, *pqe2;

  if (ref == NULL)
    return;

  free_ref_cont(ref);

  free_widls(ref);

  /* free the pop down list of previous reference structures */
  for (pdl = ref->pdl; pdl; pdl = pdl2) {
    pdl2 = pdl->next;
    free((FREEPTR *)pdl->tag);
    free((FREEPTR *)pdl);
  }

  for (pqe = ref->pqueue.head; pqe; pqe = pqe2) {
    pqe2 = pqe->pqe_next;
    free_pqe_element(pqe);
  }

  for (p = &refstructs; (*p) != ref; p = &((*p)->next));
  (*p) = (*p)->next;
  free((FREEPTR *)ref);
  
}

/*  */
/**********************************************************************
 * Function: void free_widls(reference_structure *ref)
 *
 * free all the widgets associated with a reference structure.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void free_widls(ref)
  reference_structure *ref;
{
  destroy_list *widl, *widl2;

  /* free all widgets tied to this window: refstruct dumps, hist graphs */
  for (widl = ref -> widl; widl != NULL; widl = widl2) {
    widl2 = widl -> next;
    uim_destroy((Widget)(widl->widget));
    free((FREEPTR *)widl);
  }
  ref->widl = NULL;
}

/*  */
/**********************************************************************
 * Function: void remove_widl(Widget widget, reference_structure **ref)
 *
 *  remove and free a particular destroy_list structure from a widl
 *  list.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void remove_widl(widget, ref)
  Widget widget;
  reference_structure *ref;
{
  destroy_list **p, *q;

  for (p = &(ref->widl); *p != NULL; p = &((*p)->next)) {
    if ((*p)->widget == widget) {
      uim_destroy((Widget)((*p)->widget));
      q = *p;
      *p = q->next;
      free((FREEPTR *)q);
      break;
    }
  }
}

/*  */
/**********************************************************************
 * Function: void unlink_infonode(infonode *inp)
 *
 *  Un-link an infonode from the structure it appears in.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void unlink_infonode(inp)
  infonode *inp;
{
  infonode **p;

  for (p = &(inp->head->value); (*p != inp); p = &((*p)->next));

  *p = inp->next;

  inp->parent = (infonode *)NULL;
  inp->next = (infonode *)NULL;
  inp->head = (identry *)NULL;
}

/*  */
/**********************************************************************
 * Function: void determinenode(infonode *inp,label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void determinenode(inp, lbl)
  infonode *inp;
  label *lbl;
{
  if (inp->index == LINKITEM) {
    (void)copylbl(lbl, &(inp->lbl));
  } else {
    (void)copylbl(lbl, &(inp->obj));
  }
}

/*  */
/**********************************************************************
 * Function: static void check_prop_queue(TWO PARAMETERS)
 * Parameters:
 *	reference_structure *ref
 *	infonode *inp
 *
 * This function is here so that we don't remove and free an infonode
 * from free_infonode, then try to change it in the propagation queue
 * processing.  this only happens if you change an item, don't move out,
 * then remove it.  surprised we hadn't seen it before.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void check_prop_queue(ref, inp)
  reference_structure *ref;
  infonode *inp;
{
  propagation_q_element *pqe, *pqe2;

  for (pqe = ref->pqueue.head; pqe; /* nothing */ ) {
    if (pqe->pqe_infonode == inp) {
      pqe2 = pqe;
      if ((ref->pqueue.head == ref->pqueue.tail) && (ref->pqueue.head == pqe2))
	ref->pqueue.head = ref->pqueue.tail = NULL;
      else if (ref->pqueue.head == pqe2)
	ref->pqueue.head = pqe2->pqe_next;
      else if (ref->pqueue.tail == pqe2)
	ref->pqueue.tail = pqe2->pqe_prev;
      else
        pqe2->pqe_prev = pqe2->pqe_next;
      pqe = pqe->pqe_next;
      free_pqe_element(pqe2);
    } else {
      pqe = pqe->pqe_next;
    }
  }
}
