/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	gpdmenu.c
 *
 * SCCSINFO:		@(#)gpdmenu.c	1.11 5/30/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R\"onnquist, 1992-03-11
 *
 * MODIFICATIONS:
 *      1993-11-02 Martin Sjlin - whichref is moved to aimsubr.c and 
 *                 changed external section accordingly.
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	Popup menu for GPD selection.  This replaces the old gpdmenu
 *	stuff that used the HP widget set, which is non-standard.  This
 *	uses the Athena Widget Set, which is what is used with all of
 *	xlincks.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * void create_GPD_menu(Widget parent)
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/IntrinsicP.h>
#include <X11/Xaw/List.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Viewport.h>

#include "aimtypes.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
Widget gpdmenu;

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_gpdmenu.h"

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_aimbuildref.h"
#include "f_aimstruct.h"
#include "f_aimsubr.h"
#include "f_blddisplay.h"

/* libshared */
extern char *strdup( /* char *incoming */ );

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern Widget toplevel;			/* aimstart.c */
extern label userroot, sysroot;		/* aimstart.c */
extern infonode *currentinfonode;	/* aimstart.c */
extern Widget command_menu;		/* aimcommand.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define MAXMENUSIZE	1000
#define MENULINEWIDTH	40

/* extra width to be thrown on a window to account for possible scrollbar */
#define SCROLL_BAR_WIDTH 40

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void activate_menu P_(( Widget w, XtPointer client, XtPointer
                                call ));
static int build_entry P_(( label *maplbl, char *tag, int count, int
                             length ));
static void execute_menu_choice P_(( Widget w, XtPointer client,
                                      XtPointer call ));
static void expand_from_selection P_(( int i ));
static void make_menu_entry P_(( char *tag, label *GPDobj, int hdr ));
static int scan_GPD_map P_(( void *extra, label *item, int count, int
                              length ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
static Dimension shellheight;
static Dimension shellwidth;
static Dimension menulinewidth = MENULINEWIDTH;

static char grp[] = "SYSTEM";
static char fld[] = "GPDmaplist";

static label nolabel = {-1,-1};

static Widget list;

static char *(theList[MAXMENUSIZE]);
static struct {
  char *tag;
  label gpd;
  } theTable[MAXMENUSIZE];
static int listCount = 0;


/*  */
/**********************************************************************
 * Function: static void make_menu_entry(char *tag,label *GPDobj,int hdr)
 * 
 * Adds tag and GPDobj to theTable and a corresponding meny selection
 * line to theList.
 *
 * Currently the tag is used for menu selection line, but one could
 * imagine a combination of the tag and the image of the GPDobj (which
 * would load them all...).
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void make_menu_entry(tag,GPDobj,hdr)
     char *tag;
     label *GPDobj;
     int hdr;
{
  int pad;
  char *p;

  if (listCount == MAXMENUSIZE) return;	/* The list is full ... */

  p = theList[listCount] = (char *)malloc((ALLOC_T)menulinewidth);

  p[menulinewidth-1] = '\0';

  pad = strlen(tag);
  if (pad > (int)menulinewidth - 5)
    pad = menulinewidth - 5;
  if (hdr) {
    /* A header is centered over a dashed line */
    (void)memset(p,'-',menulinewidth-1);
    p +=  (((int)menulinewidth - pad)/2)-1;
    *(p++) = ' ';
    (void)memcpy(p,tag,pad);
    (void)memset(p+pad,' ',1);
  } else {
    /* A non-headers is right justified over a blank line */
    (void)memset(p,' ',menulinewidth-1);
    (void)memcpy(p,tag,pad);
  }

  theTable[listCount].tag = strdup(tag);
  (void) copylbl(&(theTable[listCount++].gpd),GPDobj);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int build_entry(label *maplbl,char *tag,int count,int length)
 *
 * Retrieves the GPDmap:?tag:1 label of the maplbl object, which is the
 * information needed to build a menu line entry. This funtion is used
 * as mapping function over the link fields of a GPD map object.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int build_entry(maplbl,tag,count,length)
     label *maplbl;
     char *tag;
     int count;
     int length;
{
  label lbl;
  
  if (count == 0) return GVS_CONTINUE;

  if (GLI_GETLINKITEM(maplbl,"GPDmap",tag,1,&lbl) == SUCCESS) {
    make_menu_entry(tag,&lbl,FALSE);
  }
  return GVS_CONTINUE;
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int scan_GPD_map(void *extra,label *item,int count,int length)
 * 
 * Map through a GPD map to build menu lines for all its GPDs. This
 * function is itself a map function over the items within a GPD map
 * list.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int scan_GPD_map(extra,item,count,length)
     void *extra;
     label *item;
     int count;
     int length;
{
  attrval v;

  if (count == 0) return GVS_CONTINUE;

  if ((GI_GETIMAGE(item,&v) == SUCCESS) && (v.attvalue != 0)) {
    make_menu_entry(v.attvalue,&nolabel,TRUE);
    free(v.attvalue);
  } else {
    make_menu_entry("(GPD map without image)",&nolabel,TRUE);
  }

  (void)GLN_GETLINKNAMES(item,"GPDmap",build_entry,(void *)item);
  return GVS_CONTINUE;
}


/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void activate_menu(Widget w,XtPointer client,XtPointer call)
 * 
 * Recomputes the list of GPDs for menu selection and hands it to the
 * list widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void activate_menu(w,client,call)
     Widget w;			/* Parameters are unused */
     XtPointer client;
     XtPointer call;
{
  Position x;
  Position y;
  Position root_x;
  Position root_y;
  int height, width;
  
  /* Recompute the list contents. */
  while (--listCount >= 0) {
    if (theList[listCount]) free(theList[listCount]);
    if (theTable[listCount].tag) free(theTable[listCount].tag);
  }

  listCount = 0;

  make_menu_entry("(Select here to close the window)*",&nolabel,FALSE);

  /* Map through user object GPD map list */
  (void)GLV_GETLINKVAL(&userroot,grp,fld,scan_GPD_map,(void *)NULL);
  /* Map through system object GPD map list */
  (void)GLV_GETLINKVAL(&sysroot,grp,fld,scan_GPD_map,(void *)NULL);

  XawListChange(list,theList,listCount,0,True);

  /* Position the GPD menu close to the command menu.
   * (command_menu is a global in aimcommand.c)
   * GPD menu height given by shellheight (600 as a resource)
   * GPD menu width given by based on shellwidth (280 as a resource)
   * Ensure the whole list to be on screen.
   */

  XtVaGetValues(command_menu, XtNx, &x, XtNy, &y, NULL);
  XtTranslateCoords(command_menu,x,y,&root_x,&root_y);

  height = DisplayHeight(XtDisplay(list),0) - shellheight - 4;
  width = DisplayWidth(XtDisplay(list),0) - shellwidth;

  if (root_x < 1)  root_x = 1;
  if (root_x > width) root_x = width;
  if (root_y < 1) root_y = 1;
  if (root_y > height) root_y = height;

  /* move the gpd menu widget to be on top of the command menu */
  XtMoveWidget(gpdmenu,root_x,root_y);

}


/*  */
/**********************************************************************
 * Function: static void expand_from_selection(int i)
 * 
 * Expand current infonode using the selected GPD.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void expand_from_selection(i)
     int i;
{
  label lbl;
  reference_structure *r;

  if (theTable[i].gpd.vs == -1)
    return;

  if ((!currentinfonode) || (currentinfonode->iflg & NONEXIST_MSK))
    return;

  determinenode(currentinfonode, &lbl);

  r = whichref(currentinfonode, (identry *)NULL, 0);
  r = buildrefgpd(&lbl,&theTable[i].gpd,theTable[i].tag, 
		  (reference_structure *)NULL);
  if (r)
    blddisplay(r);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static void execute_menu_choice(Widget w,XtPointer client,XtPointer call)
 * 
 * Recomputes the list of GPDs for menu selection and hands it to the
 * list widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void execute_menu_choice(w,client,call)
     Widget w;			/* Parameters are unused */
     XtPointer client;
     XtPointer call;
{
  XawListReturnStruct *p;

  p = XawListShowCurrent(list);
  if ((p->list_index > 0) && (theTable[p->list_index].gpd.vs == -1)) {
    /* Bad selection... */
    XawListUnhighlight(list);
    XBell(XtDisplay(list),50);
  } else {
    if (p->list_index == 0) {
      /* Chosing the close button */
      XtPopdown((Widget)gpdmenu);
    } else {
      if (p->list_index != XAW_LIST_NONE) {
	expand_from_selection(p->list_index);
      }
    }
  }
  XtFree((char *)p);
}


/*  */
/**********************************************************************
 * Function: void create_GPD_menu(Widget parent)
 * 
 * Creates the GPD menu window, but invisible and without contents.
 * The menu is supposed to pop up when the middle button is held down
 * over the 'expand' button and the menu contents is then produced
 * immediately before the popup.
 * Argument 'parent' is the 'expand' button widget.
 *
 * Modifications:
 *      <list mods with name and date>
 */
void create_GPD_menu(parent)
  Widget parent;
{
  Widget    scroll;
  XtArgVal  font;
  Dimension fontwidth;

  gpdmenu = XtVaCreatePopupShell
    ("gpdmenu", wmShellWidgetClass, parent,
     XtNtitle, (XtArgVal) "GPD Selection Menu",
     NULL);

  scroll  = XtVaCreateManagedWidget
    ("scroll", viewportWidgetClass, (Widget)gpdmenu,
     XtNallowVert, (Boolean)True,
     NULL);

  list = XtVaCreateManagedWidget
    ("list", listWidgetClass, (Widget)scroll,
     XtNlist, (XtArgVal) theList,
     XtNforceColumns, True,
     NULL);

  /* get size from resources db */
  XtVaGetValues(gpdmenu,
		XtNheight,(Dimension *)&shellheight,
		XtNwidth, (Dimension *)&shellwidth, 
		NULL);
  XtVaGetValues(list,
		XtNfont, &font, 
		NULL);

  fontwidth = XTextWidth((XFontStruct *) font, "m", 1);
  menulinewidth = shellwidth/fontwidth;
  make_menu_entry("dummy",&nolabel,FALSE);

  XtAddCallback(gpdmenu, XtNpopupCallback, activate_menu, NULL);
  XtAddCallback(list, XtNcallback, execute_menu_choice, NULL);

/* 
 * NOTE: the list is not mapped to display. This will happen through
 * the XtMenuPopup("gpdmenu") protocol, which is an addition to
 * the 'expand' button translation. Further, the list is rebuilt prior
 * to display, so you will always have the right values.
 */

}
