/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* 
 * MODULE NAME: 	ltextsrc.c
 *
 * SCCSINFO:		@(#)ltextsrc.c	1.9 6/9/94
 *
 * ORIGINAL AUTHOR(S):  ???????
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#ifdef NEED_X11_XAW_TEXT_H
#include <X11/Xaw/Text.h>
#endif /* NEED_X11_XAW_TEXT_H */
#include <X11/Xaw/XawInit.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xmu/CharSet.h>
#include "ltextsrcP.h"
#include "ltextP.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_logws.h"
#include "f_layout.h"
#include "f_ltextaction.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_uid;		/* liblincks.a */
extern char *sys_errlist[];	/* liblincks.a */
extern int sys_nerr;		/* liblincks.a */
extern int errno;		/* liblincks.a */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define HALF_PIECE (src->xltext_src.piece_size/2)

#define LOGMALLOCMESS(function) \
  LogMess(LL_uid, "%s in libxl: XtMalloc failed.  Error %s", \
          function, sys_errlist[(errno > sys_nerr) ? 0 : errno]); \
  (void)fprintf(stderr, "%s in libxl: XtMalloc failed.\n", function)

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static Piece *AllocNewPiece P_(( XlTextSrcObject src, Piece *prev ));
static void BreakPiece P_(( XlTextSrcObject src, Piece * piece ));
static void ClassInitialize P_(( void ));
static void Destroy P_(( Widget w ));
static Piece *FindPiece P_(( XlTextSrcObject src, XawTextPosition
                              position, XawTextPosition *first ));
static void FreeAllPieces P_(( XlTextSrcObject src ));
static void GetValuesHook P_(( Widget w, ArgList args, Cardinal *
                                num_args ));
static void Initialize P_(( Widget request, Widget new ));
static void InitString P_(( XlTextSrcObject src ));
static void LoadPieces P_(( XlTextSrcObject src, char *string ));
static void MyStrncpy P_(( char * s1, char * s2, int n ));
static XawTextPosition ReadText P_(( Widget w, XawTextPosition pos,
                                      XawTextBlock *text, int length ));
static void RemoveOldStringOrFile P_(( XlTextSrcObject src ));
static void RemovePiece P_(( XlTextSrcObject src, Piece *piece ));
static int ReplaceText P_(( Widget w, XawTextPosition startPos,
                             XawTextPosition endPos, XawTextBlock *text ));
static XawTextPosition Scan P_(( Widget w, XawTextPosition position,
                                  XawTextScanType type,
                                  XawTextScanDirection dir, int count,
                                  XtBoolean include ));
static XawTextPosition Search P_(( Widget w, XawTextPosition position,
                                    XawTextScanDirection dir,
                                    XawTextBlock *text ));
static XtBoolean SetValues P_(( Widget current, Widget request, Widget new ));
static String StorePiecesInString P_(( XlTextSrcObject src ));
static XtBoolean XlTextSave P_(( Widget w ));

/*********************************************************************
 * INTERNAL (STATIC) DATA: 
 *********************************************************************/
#define offset(field) XtOffset(XlTextSrcObject, xltext_src.field)
static XtResource resources[] = {
    {XtNstring, XtCString, XtRString, sizeof (char *),
       offset(string), XtRString, NULL},
    {XtNdataCompression, XtCDataCompression, XtRBoolean, sizeof (Boolean),
       offset(data_compression), XtRImmediate, (XtPointer) TRUE},
    {XtNpieceSize, XtCPieceSize, XtRInt, sizeof (XawTextPosition),
       offset(piece_size), XtRImmediate, (XtPointer) BUFSIZ},
    {XtNcallback, XtCCallback, XtRCallback, sizeof(XtPointer), 
       offset(callback), XtRCallback, (XtPointer)NULL},
#undef offset
};


XlTextSrcClassRec XltextsrcClassRec = {
  {
/* Core_class fields */	
    /* superclass		*/	(WidgetClass) &textSrcClassRec,
    /* class_name		*/	"XlTextSrc",
    /* widget_size		*/	sizeof(XlTextSrcRec),
    /* class_initialize   	*/	ClassInitialize,
    /* class_part_initialize	*/	NULL,
    /* class_inited       	*/	FALSE,
    /* initialize	  	*/	(XtInitProc)Initialize,
    /* initialize_hook		*/	NULL,
    /* realize		  	*/	NULL,
    /* actions		  	*/	NULL,
    /* num_actions	  	*/	0,
    /* resources	  	*/	resources,
    /* num_resources	  	*/	XtNumber(resources),
    /* xrm_class	  	*/	NULLQUARK,
    /* compress_motion	  	*/	FALSE,
    /* compress_exposure  	*/	FALSE,
    /* compress_enterleave	*/	FALSE,
    /* visible_interest	  	*/	FALSE,
    /* destroy		  	*/	Destroy,
    /* resize		  	*/	NULL,
    /* expose		  	*/	NULL,
    /* set_values	  	*/	(XtSetValuesFunc)SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	NULL,
    /* get_values_hook		*/	GetValuesHook,
    /* accept_focus	 	*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private   	*/	NULL,
    /* tm_table		   	*/	NULL,
    /* query_geometry		*/	NULL,
    /* display_accelerator	*/	NULL,
    /* extension		*/	NULL
    },
    {
/* textsrc_class fields */
    /* Read                     */      ReadText,
    /* Replace                  */      ReplaceText,
    /* Scan                     */      Scan,
    /* Search                   */      Search,
    /* SetSelection             */      XtInheritSetSelection,
    /* ConvertSelection         */      XtInheritConvertSelection
    },
/* xltextsrc_class fields */
    {
    /* Keep the compiler happy */       0
    }
};

WidgetClass XltextSrcObjectClass = (WidgetClass)&XltextsrcClassRec;

/*  */
/* Semi-Public Interfaces */

/**********************************************************************
 * Function: static void ClassInitialize()
 * 
 * Class Initialize routine, called only once.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void ClassInitialize()
{
  XawInitializeWidgetSet();
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static void Initialize(Widget request, Widget new)
 * 
 * Initializes the simple menu widget
 *
 * Arguments:
 *	request	- the widget requested by the argument list.
 *	new	- the new widget with both resource and non resource values.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Initialize(request, new)
  Widget request;
  Widget new;
{
  XlTextSrcObject src = (XlTextSrcObject) new;

/*
 * Set correct flags (override resources) depending upon widget class.
 */
  src->xltext_src.allocated_string = FALSE;

  InitString(src);
  LoadPieces(src, (char *)NULL);

}

/*  */
/**********************************************************************
 * Function: static XawTextPosition ReadText(FOUR PARAMETERS)
 * Parameters:
 *	Widget w		- the AsciiSource widget
 *	XawTextPosition pos	- position of the text to retrieve
 * 	XawTextBlock text	- text block that will contain returned text
 *	int length		- maximum number of characters to read
 *
 * This function reads the source
 *
 * Returns: The number of characters read into the buffer.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static XawTextPosition ReadText(w, pos, text, length)
  Widget w;
  XawTextPosition pos;
  XawTextBlock *text;	
  int length;		
{
  XlTextSrcObject src = (XlTextSrcObject) w;
  XawTextPosition count, start;
  Piece * piece = FindPiece(src, pos, &start);
    
  text->firstPos = pos;
  text->ptr = piece->text + (pos - start);
  count = piece->used - (pos - start);
  text->length = (length > count) ? count : length;
  return(pos + text->length);
}

/*  */
/*ARGSUSED*/
/**********************************************************************
 * Function: static int ReplaceText (FOUR PARAMETERS)
 *	Widget w		- the XlTextSource widget
 *	XawTextPosition startPos- ends of text that will be removed
 *	XawTextPosition endPos	- ends of text that will be removed
 *	XawTextBlock *text	- new text to be inserted into
 *				  buffer at startPos
 * 
 * Replaces a block of text with new text.
 *
 * Returns: XawEditError or XawEditDone.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static int ReplaceText (w, startPos, endPos, text)
  Widget w;
  XawTextPosition startPos;
  XawTextPosition endPos;
  XawTextBlock *text;
{
  XlTextSrcObject src = (XlTextSrcObject) w;
  XlTextWidget pw = (XlTextWidget)XtParent(w);
  Piece *start_piece, *end_piece, *temp_piece;
  XawTextPosition start_first, end_first;
  int length, firstPos;

  if (src->text_src.edit_mode == XawtextRead) {
    /*
     * Editing a read only source is discouraged.
     */
    return(XawEditError);
  }

  start_piece = FindPiece(src, startPos, &start_first);
  end_piece = FindPiece(src, endPos, &end_first);

  pw->xltext.change_cnt += text->length + (endPos - startPos);

  /* 
   * Remove the old stuff. 
   */

  if (start_piece != end_piece) {
    temp_piece = start_piece->next;
    /*
     * If empty and not the only piece then remove it. 
     */

    start_piece->used = startPos - start_first;

    if ((!start_piece->used) && (start_piece->next || start_piece->prev))
      RemovePiece(src, start_piece);

    while (temp_piece != end_piece) {
      temp_piece = temp_piece->next;
      RemovePiece(src, temp_piece->prev);
    }
    end_piece->used -= endPos - end_first;
    if (end_piece->used != 0)
      MyStrncpy(end_piece->text, (end_piece->text + endPos - end_first),
		(int) end_piece->used);
  } else {
    /* We are fully in one piece. */
    start_piece->used -= endPos - startPos;
    if (start_piece->used == 0) {
      if (start_piece->next || start_piece->prev)
	RemovePiece(src, start_piece);
    } else {
      MyStrncpy(start_piece->text + (startPos - start_first),
		start_piece->text + (endPos - start_first),
		(int) (start_piece->used - (startPos - start_first)) );
    }
  }

  src->xltext_src.length += text->length - (endPos - startPos);

  if (text->length != 0) {
    /* 
     * Put in the New Stuff.
     */
    
    start_piece = FindPiece(src, startPos, &start_first);
    
    length = text->length;
    firstPos = text->firstPos;
    
    while (length > 0) {
      char * ptr;
      int fill = length;
      
      if (start_piece->used == src->xltext_src.piece_size) {
	BreakPiece(src, start_piece);
	start_piece = FindPiece(src, startPos, &start_first);
      }

      fill = src->xltext_src.piece_size - start_piece->used;
      if (length < fill)
	fill = length;
      
      ptr = start_piece->text + (startPos - start_first);
      MyStrncpy(ptr + fill, ptr, 
		(int) (start_piece->used - (startPos - start_first)));
      (void)strncpy(ptr, text->ptr + firstPos, (int)fill);
      
      startPos += fill;
      firstPos += fill;
      start_piece->used += fill;
      length -= fill;
    }
  }

  return(XawEditDone);
}

/*  */
/**********************************************************************
 * Function: static XawTextPosition Scan (SIX PARAMETERS)
 * Parameters:
 *	Widget w		- the XltextSource widget
 *	XawTextPosition position- the position to start scanning
 *	XawTextScanType type	- type of thing to scan for
 *	XawTextScanDirection dir- direction to scan
 *	int count		- which occurance if this thing to search for
 *	XtBoolean include	- whether or not to include the character
 *				  found in the position that is returned.
 *
 * Scans the text source for the number and type of item specified.
 *
 * Returns: the position of the item found.
 *
 * Note: While there are only 'n' characters in the file there are n+1 
 *       possible cursor positions (one before the first character and
 *       one after the last character.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static XawTextPosition Scan (w, position, type, dir, count, include)
  Widget w;
  XawTextPosition position;
  XawTextScanType type;
  XawTextScanDirection dir;
  int count;
  XtBoolean include;
{
  XlTextSrcObject src = (XlTextSrcObject) w;
  register int inc;
  Piece * piece;
  XawTextPosition first;
  XawTextPosition first_eol_position = 0; 
  register char * ptr;

  if (type == XawstAll) {	/* Optomize this common case. */
    if (dir == XawsdRight)
      return(src->xltext_src.length);
    return(0);			/* else. */
  }

  if (position > src->xltext_src.length)
    position = src->xltext_src.length;

  if ( dir == XawsdRight ) {
    if (position == src->xltext_src.length)
/*
 * Scanning right from src->xltext_src.length???
 */
      return(src->xltext_src.length);
    inc = 1;
  }
  else {
    if (position == 0)
      return(0);		/* Scanning left from 0??? */
    inc = -1;
    position--;
  }

  piece = FindPiece(src, position, &first);

/*
 * If the buffer is empty then return 0. 
 */

  if ( piece->used == 0 ) return(0); 

  ptr = (position - first) + piece->text;

  switch (type) {
  case XawstEOL: 
  case XawstParagraph: 
  case XawstWhiteSpace: 
    for ( ; count > 0 ; count-- ) {
      Boolean non_space = FALSE, first_eol = TRUE;
      while (TRUE) {
	register unsigned char c = *ptr;

	ptr += inc;
	position += inc;
	
	if (type == XawstWhiteSpace) {
	  if (isspace(c)) {
	    if (non_space) 
	      break;
	  }
	  else
	    non_space = TRUE;
	}
	else if (type == XawstEOL) {
	  if (c == '\n') break;
	}
	else { /* XawstParagraph */	/* gcc says we should handle
					   XawstAll as well */
	  if (first_eol) {
	    if (c == '\n') {
	      first_eol_position = position;
	      first_eol = FALSE;
	    }
	  }
	  else
	    if ( c == '\n') 
	      break;
	    else if ( !isspace(c) )
	      first_eol = TRUE;
	}
	      

	if ( ptr < piece->text ) {
	  piece = piece->prev;
	  if (piece == NULL)	/* Begining of text. */
	    return(0);
	  ptr = piece->text + piece->used - 1;
	}
	else if ( ptr >= (piece->text + piece->used) ) {
	  piece = piece->next;
	  if (piece == NULL)	/* End of text. */
	    return(src->xltext_src.length);
	  ptr = piece->text;
	}
      }
    }
    if (!include) {
      if ( type == XawstParagraph)
	position = first_eol_position;
      position -= inc;
    }
    break;
  case XawstPositions: 
    position += count * inc;
    break;
  case XawstAll: /* Already handled prior to the switch */
    break;
  }

  if ( dir == XawsdLeft )
    position++;

  if (position >= src->xltext_src.length)
    return(src->xltext_src.length);
  if (position < 0)
    return(0);

  return(position);
}

/*  */
/**********************************************************************
 * Function: static XawTextPosition Search(FOUR PARAMETERS)
 * Parameters:
 *	Widget w		- the AsciiSource Widget
 *	XawTextPosition position- the position to start scanning
 *	XawTextScanDirection dir- direction to scan
 *	XawTextBlock *text	- the text block to search for
 * 
 * Searchs the text source for the text block passed.
 *
 * Returns: the position of the item found
 *
 * Modifications:
 *      <list mods with name and date>
 */
static XawTextPosition Search(w, position, dir, text)
  Widget w;
  XawTextPosition position;
  XawTextScanDirection dir;
  XawTextBlock *text;
{
  XlTextSrcObject src = (XlTextSrcObject) w;
  register int inc, count = 0;
  register char * ptr;
  Piece * piece;
  char * buf;
  XawTextPosition first;

  if ( dir == XawsdRight )
    inc = 1;
  else {
    inc = -1;
    if (position == 0)
      return(XawTextSearchError);	/* scanning left from 0??? */
    position--;
  }

  if ((buf = XtMalloc((unsigned)sizeof(unsigned char) * text->length)) == NULL){
    LOGMALLOCMESS("Search");
  }

  (void)strncpy(buf, (text->ptr + text->firstPos), (int)text->length);
  piece = FindPiece(src, position, &first);
  ptr = (position - first) + piece->text;

  while (TRUE) {
    if (*ptr == ((dir == XawsdRight) ? *(buf + count) 
		                     : *(buf + text->length - count - 1)) ) {
      if (count == (text->length - 1))
	break;
      else
	count++;
    }
    else {
      if (count != 0) {
	position -=inc * count;
	ptr -= inc * count;
      }
      count = 0;
    }

    ptr += inc;
    position += inc;
    
    while ( ptr < piece->text ) {
      piece = piece->prev;
      if (piece == NULL) {	/* Begining of text. */
	XtFree((char *)buf);
	return(XawTextSearchError);
      }
      ptr = piece->text + piece->used - 1;
    }
   
    while ( ptr >= (piece->text + piece->used) ) {
      piece = piece->next;
      if (piece == NULL) {	/* End of text. */
	XtFree((char *)buf);
	return(XawTextSearchError);
      }
      ptr = piece->text;
    }
  }

  XtFree((char *)buf);
  if (dir == XawsdLeft)
    return(position);
  return(position - (text->length - 1));
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static XtBoolean SetValues(Widget current, request, new)
 * 
 * Sets the values for the XlTextSource.
 *
 * Arguments:
 *	Widget current	- current state of the widget.
 *	Widget request	- what was requested.
 *	Widget new	- what the widget will become.
 * Returns: True if redisplay is needed.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static XtBoolean SetValues(current, request, new)
  Widget current;
  Widget request;
  Widget new;
{
  XlTextSrcObject src =     (XlTextSrcObject) new;
  XlTextSrcObject old_src = (XlTextSrcObject) current;
  TextWidget ctx = (TextWidget)XtParent(new);
  Boolean caret;
  XawTextPosition insertPos = ((TextWidget)(XtParent(new)))->text.insertPos;

  if (old_src->xltext_src.string != src->xltext_src.string) {
    RemoveOldStringOrFile(old_src);        /* remove old info. */
    src->xltext_src.allocated_string = FALSE;

    /* Init new info. */
    InitString(src);

    /* load new info into internal buffers. */
    LoadPieces(src, (char *)NULL); 

    /* Tell text widget what happened. */
    caret = ctx->text.display_caret;
    ctx->text.display_caret = 0;
    XawTextSetSource((Widget)ctx, new, (XawTextPosition) 0);
    ctx->text.display_caret = caret;
    ctx->text.insertPos = insertPos;
    src->xltext_src.string = NULL;
    _XlTextCheckResize((TextWidget)XtParent(new));
  }

  return(FALSE);
}

/*  */
/**********************************************************************
 * Function: static void GetValuesHook(Widget w,ArgList args,Cardinal *num_args)
 *
 * This is a get values hook routine that sets the
 * values specific to the xltext source.
 *
 * Arguments: 
 *	Widget w - the AsciiSource Widget.
 *	ArgList args - the argument list.
 *	Cardinal *num_args - the number of args.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void GetValuesHook(w, args, num_args)
  Widget w;
  ArgList args;
  Cardinal * num_args;
{
  XlTextSrcObject src = (XlTextSrcObject) w;
  register int i;

  for (i = 0; i < *num_args ; i++ ) 
    if (streq(args[i].name, XtNstring)) {
      if (XlTextSave(w))	/* If save sucessful. */
	*((char **) args[i].value) = src->xltext_src.string;
      break;
    }
}    

/*  */
/**********************************************************************
 * Function: static void Destroy(Widget w)
 * 
 * Destroys an ascii source (frees all data)
 *
 * Arguments:
 *	Widget src	- the Ascii source Widget to free.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Destroy (w)
  Widget w;
{
  RemoveOldStringOrFile((XlTextSrcObject) w);
}

/*  */
/* Private Functions.*/
/**********************************************************************
 * Function: static XtBoolean XlTextSave(Widget w)
 * 
 * Saves all the pieces into a file or string as required.
 *
 * Arguments:
 *	Widget w	- the asciiSrc Widget.
 *
 * Returns: TRUE if the save was successful.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static XtBoolean XlTextSave(w)
  Widget w;
{
  XlTextSrcObject src = (XlTextSrcObject) w;

/*
 * If using the string in place then there is no need to play games
 * to get the internal info into a readable string.
 */


  if (src->xltext_src.allocated_string == TRUE) 
    XtFree((char *)src->xltext_src.string);

  src->xltext_src.string = StorePiecesInString(src);
  src->xltext_src.allocated_string = TRUE;
    
  return(TRUE);
}

/*  */
/**********************************************************************
 * Function: static void RemoveOldStringOrFile(XlTextSrcObject src)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void RemoveOldStringOrFile(src) 
  XlTextSrcObject src;
{
  FreeAllPieces(src);

  if (src->xltext_src.allocated_string) 
    XtFree((char *)src->xltext_src.string);
}

/*  */
/**********************************************************************
 * Function: static String StorePiecesInString(XlTextSrcObject src)
 * 
 * store the pieces in memory into a standard xltext string.
 *
 * Arguments:
 *	XlTextSrcObject src	- the xltext pointer data.
 *
 * Returns: String.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static String StorePiecesInString(src)
  XlTextSrcObject src;
{
  String string;
  XawTextPosition first;
  Piece * piece;

  if ((string=
       XtMalloc((Cardinal)(sizeof(unsigned char)*(src->xltext_src.length+1))))
       ==NULL) {
    LOGMALLOCMESS("RemoveOldStringOrFile");
  }

  for (first = 0, piece = src->xltext_src.first_piece ; piece != NULL; 
       first += piece->used, piece = piece->next) 
    (void)strncpy(string + first, piece->text, (size_t)piece->used);

  string[src->xltext_src.length] = '\0';	/* NULL terminate this sucker. */

/*
 * This will refill all pieces to capacity. 
 */

  if (src->xltext_src.data_compression) {	
    FreeAllPieces(src);
    LoadPieces(src, string);
  }

  return(string);
}

/*  */
/**********************************************************************
 * Function: static void InitString(XlTextSrcObject src)
 * 
 * Initializes the string or filexgo.
 *
 * Arguments:
 *	XlTextSrcObject src	- the AsciiSource.
 *
 * Returns: none - May exit though.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void InitString(src)
  XlTextSrcObject src;
{
    if (src->xltext_src.string == NULL)
      src->xltext_src.length = 0;
    else 
      src->xltext_src.length = strlen(src->xltext_src.string);
}

/*  */
/**********************************************************************
 * Function: static void LoadPieces(XlTextSrcObject src, char *string)
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void LoadPieces(src, string)
  XlTextSrcObject src;
  char *string;
{
  char *local_str, *ptr;
  register Piece * piece = NULL;
  XawTextPosition left;

  if (string == NULL) {
    local_str = src->xltext_src.string;
  }
  else
    local_str = string;

  ptr = local_str;
  left = src->xltext_src.length;

  do {
    piece = AllocNewPiece(src, piece);

    if ((piece->text = 
	 XtMalloc((Cardinal)src->xltext_src.piece_size * 
		  sizeof(unsigned char))) == NULL) {
      LOGMALLOCMESS("LoadPieces");
    }
    piece->used = Min(left, src->xltext_src.piece_size);
    if (piece->used != 0)
      (void)strncpy(piece->text, ptr, (int)piece->used);

    left -= piece->used;
    ptr += piece->used;
  } while (left > 0);

}

/*  */
/**********************************************************************
 * Function: static Piece *AllocNewPiece(XlTextSrcObject src, Piece *prev)
 * 
 * Allocates a new piece of memory.
 *
 * Arguments:
 *	XlTextSrcObject src	- The AsciiSrc Widget.
 *	Piece *prev		- the piece just before this one, or NULL.
 *
 * Returns: the allocated piece.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static Piece *AllocNewPiece(src, prev)
  XlTextSrcObject src;
  Piece *prev;
{
  Piece * piece = XtNew(Piece);

  if (prev == NULL) {
    src->xltext_src.first_piece = piece;
    piece->next = NULL;
  }
  else {
    if (prev->next != NULL)
      (prev->next)->prev = piece;
    piece->next = prev->next;
    prev->next = piece;
  }
  
  piece->prev = prev;

  return(piece);
}

/*  */
/**********************************************************************
 * Function: static void FreeAllPieces(XlTextSrcObject src)
 * 
 * Frees all the pieces
 *
 * Arguments:
 *	XlTextSrcObject src	- The AsciiSrc Widget.
 *
 * Returns: none.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void FreeAllPieces(src)
  XlTextSrcObject src;
{
  Piece * next, * first = src->xltext_src.first_piece;

  if (first->prev != NULL)
    (void)printf("Programmer Botch in FreeAllPieces, there may be a memory leak.\n");

  for ( ; first != NULL ; first = next ) {
    next = first->next;
    RemovePiece(src, first);
  }
}

/*  */
/**********************************************************************
 * Function: static void RemovePiece(XlTextSrcObject src, Piece *piece)
 * 
 * Removes a piece from the list.
 *
 * Arguments: 
 *	XlTextSrcObject src	- The AsciiSrc Widget
 *	Piece *piece		- the piece to remove.
 *
 * Returns: none.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void RemovePiece(src, piece)
  XlTextSrcObject src;
  Piece *piece;
{
  if (piece->prev == NULL)
    src->xltext_src.first_piece = piece->next;
  else
    (piece->prev)->next = piece->next;

  if (piece->next != NULL)
    (piece->next)->prev = piece->prev;

  if (piece->text)
    XtFree((char *)piece->text);

  XtFree((char *)piece);
}

/*  */
/**********************************************************************
 * Function: static Piece *FindPiece(THREE PARAMETERS)
 * Parameters:
 *	XlTextSrcObject src	- The AsciiSrc Widget
 *	XawTextPosition position- the position that we are searching for
 *	XawTextPosition *first	- the position of the first character
 *				  in this piece
 *
 * Finds the piece containing the position indicated.
 * 
 * Returns: piece - the piece that contains this position.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static Piece *FindPiece(src, position, first)
  XlTextSrcObject src;
  XawTextPosition position;
  XawTextPosition *first;
{
  Piece * old_piece = NULL;
  Piece * piece = src->xltext_src.first_piece;
  XawTextPosition temp;

  for ( temp = 0 ; piece != NULL ; temp += piece->used, piece = piece->next ) {
    *first = temp;
    old_piece = piece;

    if ((temp + piece->used) > position) 
      return(piece);
  }
  return(old_piece);	  /* if we run off the end the return the last piece */
}

/*  */
/**********************************************************************
 * Function: static void MyStrncpy(char *s1, char *s2, int n)
 * 
 * Just like string copy, but slower and will always work on 
 * overlapping strings.
 *
 * Arguments: (same as strncpy):
 * 	s1, s2 - strings to copy (2->1).
 *	n - the number of chars to copy.
 * Returns: s1.
 * 
 * Modifications:
 *      <list mods with name and date>
 */
static void MyStrncpy(s1, s2, n)
  char * s1, * s2;
  int n;
{
  char *temp;

  if ((temp = XtMalloc((unsigned)sizeof(unsigned char) * n)) == NULL) {
    LOGMALLOCMESS("MyStrncpy");
  }

  (void)strncpy(temp, s2, n);
  (void)strncpy(s1, temp, n);
  XtFree((char *)temp);
}

/*  */
/**********************************************************************
 * Function: static void BreakPiece(XlTextSrcObject src, Piece *piece)
 * 
 * Breaks a full piece into two new pieces.
 *
 * Arguments:
 *	src	- The AsciiSrc Widget.
 *	piece	- the piece to break.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void BreakPiece(src, piece)
  XlTextSrcObject src;
  Piece * piece;
{
  Piece * new = AllocNewPiece(src, piece);
  
  if ((new->text=XtMalloc((Cardinal)src->xltext_src.piece_size*sizeof(unsigned char)))==NULL) {
    LOGMALLOCMESS("BreakPiece");
  }

  (void)strncpy(new->text, piece->text + HALF_PIECE,
	  (size_t)(src->xltext_src.piece_size - HALF_PIECE));
  piece->used = HALF_PIECE;
  new->used = src->xltext_src.piece_size - HALF_PIECE; 
}
