/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	linckshandler.c
 *
 * SCCSINFO:		@(#)linckshandler.c	1.17 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Hans Gustavsson, PEN pum group. 1992-05-26
 *
 * MODIFICATIONS:
 *      1993-08-12 Martin Sjlin, Rewrote retrieval code from dbs to
 *		get a bunch of pen and thus change user level interface
 *		into using a mapfn. (see usfuns.c)
 *	1993-08-26 Martin Sjlin. To enable better handling of the
 *		extra oob byte(s) (two) needed when we have a buggy
 *		select we need to read two bytes from the stream.
 *      1993-11-24 Martin Sjlin. SCO does not seem to support
 *              FIONREAD ioctl on pipes. Added code to use non blocking
 *              i/o on pipe. Using POSIX semantics, not sure if SCO have
 *              it or not, might add in O_NDELAY, too.
 *      1994-01-01 Martin Sjlin. register_pen_function returns the
 *              previously registered function.
 *      1994-01-07 Martin Sjlin. unified interface for callback function
 *              for pen using the same interface as RCB_. Also return an
 *              int for now ...
 *      1994-01-13 Martin Sjlin. Restructured callback interface
 *              for PEN - we now have two levels, one lowlevel
 *              corresponding to the original PEN interface and
 *              a second corresponding to the high level events
 *              we receive from the DBS. This will remove most of
 *              duplicated code in pen_history.c as well as the 
 *              low level constant copied from dbcodes.h!
 *	1994-01-14 Martin Sjlin. Added PEN_GET_HANDLE to get the
 *		pipe handle used when adding it to XtAppSelect (?)
 *
 * DESCRIPTION:
 *  This module handels the signal SIGURG and the read of
 *  ``out-of-band'' data from the socket.
 *
 *********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************
 * int InitSignals(int s)
 * int (*register_pen_function(int what, int (* func_to_call)()), void *extra)()
 * void POLL_oob(void *extra)
 * int PEN_GET_HANDLE();
 */

/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#include <sys/ioctl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <netdb.h>
#if !(defined(HAVE_F_SETOWN)||!defined(HAVE_FLOCK)||defined(NO_FIONREAD_PIPE))
#include <fcntl.h>
#endif

#ifdef HAVE_SYS_SOCKIO_H
#include <sys/sockio.h>
#endif /* HAVE_SYS_SOCKIO_H */
#if !defined(FIONREAD) && defined(HAVE_SYS_FILIO_H) 
#include <sys/filio.h>
#endif /* n FIONREAD && HAVE_SYS_FILIO_H */
#if !defined(SIGURG) && defined(HAVE_STROPTS_H)
#include <stropts.h>                      /* to get options for streamio */
#endif /* n SIGURG && STROPTS_H */

#include "dbcodes.h"
#include "sunlincks.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_usfuns.h"
#include "f_misc.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern int LL_socketno;

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
/* None */

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static int default_PEN P_(( void *extra, int index, char *msg ));
static int default_PEN_oob P_(( void *extra, int oob, int atoob ));
static RETSIGTYPE Handler P_(( int sig ));
static int PEN_mapfn P_(( void *extra, int index, int type, char *msg,
                           int count, int len ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
typedef struct {
  int (* function)(/* void *extra, int index, char *msg */);
  void *extra;
  } t_callback;

static t_callback callback_func[RCB_PEN_POLL+1] = {
  {default_PEN_oob, NULL},		/* low level callback */
#ifdef PENDEBUG
  {default_PEN, "Pen Resolve: index %d msg %s\n"}, /* RESOLVE */
  {default_PEN, "Pen Check: index %d msg %s\n" }, /* CHECK */
  {default_PEN, "Pen Poll: index %d msg %s\n" }, /* POLL */
#else
  {default_PEN, NULL},			/* RESOLVE */
  {default_PEN, NULL},			/* CHECK */
  {default_PEN, NULL}			/* POLL */
#endif /* n PENDEBUG */
};

static t_callback callback_default[RCB_PEN_POLL+1] = {
  {default_PEN_oob, NULL},		/* low level callback */
#ifdef PENDEBUG 
  {default_PEN, "Pen Resolve: index %d msg %s\n"}, /* RESOLVE */
  {default_PEN, "Pen Check: index %d msg %s\n" }, /* CHECK */
  {default_PEN, "Pen Poll: index %d msg %s\n" }, /* POLL */
#else
  {default_PEN, NULL},			/* RESOLVE */
  {default_PEN, NULL},			/* CHECK */
  {default_PEN, NULL}			/* POLL */
#endif /* n PENDEBUG */
};

static int PEN_pipe[2];
static int PEN_inited = 0;

#ifdef OOBSELECTBUG
static int oobxbyte = 0;
#endif

/*  */
/**********************************************************************
 * Function: int InitSignals(int s)
 *
 * Sets up liblincks signals SIGURG
 *
 * Modifications:
 *      Sat Oct  9 19:28:05 1993 Martin Sjlin. Added check for SIGURG
 *                   and if not defined try SIGPOLL instead (SCO system).
 *      Thu Nov 11 10:20:05 1993 Martin Sjolin -  SCO uses SIGUSR1 instead
 *                   of SIGURG for OOB and SIGPOLL for async I/O.
 *      <list mods with name and date>
 */
int InitSignals(s)
  int s;
{
#ifndef HAVE_F_SETOWN
  int pid = getpid();
#endif
 /* Register signals to take care of */
#ifdef SIGURG
  if (Signal(SIGURG, Handler) == BADSIG) {
    perror("InitSignals: Cannot register signal SIGURG");
    return (FAIL);
  }
#else
#ifdef SIGUSR1_FOR_OOB
  if (Signal(SIGUSR1, Handler) == BADSIG) {
    perror("InitSignals: Cannot register signal SIGUSR1");
    return (FAIL);
  }
#else 
  if (Signal(SIGPOLL, Handler) == BADSIG) {
    perror("InitSignals: Cannot register signal SIGPOLL");
    return (FAIL);
  }
#endif /* n SIGUSR1_FOR_OOB */
#endif
#ifdef HAVE_F_SETOWN
  if (fcntl(s, F_SETOWN, getpid()) < 0) {
#else
#if defined(SIOCSPGRP) 
  if (ioctl(s, SIOCSPGRP, &pid) < 0) {  /* socket(7) */
#else /* SIOCSPGRP */  
  /* Experimental!. No confirmation that this works. msj */
  /*
   * If we do not have a complete emulation of sockets on top
   * of streams, and since the SIOCSPGRP might be missing too, we 
   * check if we have
   *  - S_RDBAND (any band > 0, check streamio(7) under Solaris ) 
   *    to set on socket, else
   *  - assume that urgent data is implemented as hi priority message
   *    and enable signal.
   * It is very possible that I am wrong for the first case, and that
   * we should enable both S_RDBAND and S_HIPRI, but we will find out
   * later. [msj]
   */
#if defined(S_RDBAND)
  if (ioctl(s, I_SETSIG, S_RDBAND) < 0) { /* streamio(7) */
#else 
  if (ioctl(s, I_SETSIG, S_HIPRI) < 0)  { /* streamio(7) */
#endif /* n S_RDBAND */
#endif /* n SIOCSPGRP */
#endif /* n HAVE_F_SETOWN */
    perror("InitSignals: Cannot set pid/ioctl for OOB.");
    return (FAIL);
  }
  return (SUCCESS);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static RETSIGTYPE Handler(int sig, int code, struct sigcontext *scp)
 *
 * Takes care of system signals and performs appropriate
 * actions on them.
 *
 * Uses a function registered with register_pen_function()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static RETSIGTYPE Handler(sig)
  int sig;
{
  int atoob = 0;
  int savederrno = errno;
  char oob;

  switch (sig) {
  case SIGALRM:
    break;

#ifdef SIGURG
  case SIGURG:
#else
#ifdef SIGUSR1_FOR_OOB
  case SIGUSR1:
#else
  case SIGPOLL:
#endif /* n SIGUSR1_FOR_OOB */
#endif /* n SIGURG */
#ifndef HAVE_RELIABLE_SIGNALS
    /* Reinstall the handler */
    if (Signal(sig, Handler) == BADSIG) 
      perror("Handler: Cannot register signal SIGURG/SIGUSR1/SIGPOLL");
#endif /* n HAVE_RELIABLE_SIGNALS */
#ifdef PENDEBUG
    (void)printf("Handler socket number = %d\n",LL_socketno);
#endif /* PENDEBUG */
#if defined(OOBSELECTBUG) || defined(OOBXLINUXBUG)
    if (ioctl( LL_socketno, SIOCATMARK, &atoob) < 0)
      perror("Handler: ioctl for SIOCATMARK");
#endif
#ifdef OOBLINUXBUG
    if (atoob == 1)  {
#endif /* OOBLINUXBUG */
    if (recv(LL_socketno, &oob, 1, MSG_OOB) != 1) {
      if (errno != EINVAL) {
	perror("Handler");
	break;
      }
      /*
       * The error value EINVAL is not documented for recv in SunOS
       * manuals. We assume it is a case of OOB signal overrun and
       * act accordingly.
       */
#ifdef OOBSELECTBUG
      /* According to my doc(s), EINVAL signals that no OOB data
       * are currently available in the stream, but possibly later
       * in the stream - ioctl SIOCATMARK tells that this is the
       * case. atoob != 0 then the stream has been read up to
       * place where the oob data starts and thus it is safe
       * to read the following scrap byte.
       */
      if ( atoob == 0 )
	perror("Handler: OOB Sent, but not yet in buffer.");
#endif
#ifdef PENDEBUG
      perror("Handler: overrun");
#endif
      oob = OOB_MESSAGE_CODE;
    } 
#if defined(OOBLINUXBUG)
    else {
      oob = OOB_MESSAGE_CODE;
#ifdef PENDEBUG
      (void)fprintf(stderr,"Handler: Couldn't read OOB (Linux)\n");
#endif /* PENDEBUG */
    } /* atoob == 0 */
    }
#endif /* OOBLINUXBUG */
#if defined(OOBSELECTBUG) 
    else {
      int  size = 0;
      byte     buff[2];  

    /* this is to get around a bug in SunOS/BSD and Solaris 2.x handling of
     * OOB data. After reading an OOB byte, you need to read at least one
     * byte from the "in-band-data" (a scrap byte) which we send after
     * the oob data. This MIGHT mess up the normal stream, but since we
     * are using TCP_NODELAY on the socket it looks like it is working 
     * and the response when logging out is quick (again).
     *
     * if we received an oob earlier, then we can read these extra 
     * bytes (SPECBYTE and RCOOB) and  clear the exception status 
     * in the select in LL_ReceiveStatus.
     *
     * check if data is in stream - if not we can assume that we will not
     * get the extra bytes after the oob ... also, and yet more important
     * we do want to 'hang' in any recv here
     */
#ifdef PENDEBUG
      fprintf( stderr, "oobxbyte = %d, atoob = %d\n", oobxbyte, atoob);
#endif /* PENDEBUG */
      if ( atoob == 1 ) {
	if (ioctl(LL_socketno, FIONREAD, &size) < 0) 
	  perror("Handler: At mark: ioctl when checking bytes in buffer.");
	else if ( size < sizeof(buff) ) {
#ifdef PENDEBUG
	  (void)fprintf(stderr, 
			"Handler: not enough data (%d) to read scrap.\n", 
			size);
#endif /* PENDEBUG */
	  oobxbyte++;			/* count number of missed occasions */

        /* we got the extra bytes and we "are at mark" */
	} 
	else if (recv(LL_socketno,&buff[0],sizeof(buff),0) == sizeof(buff)) {
	  if (buff[0] != SPECBYTE || buff[1] != RCOOB) 
	    (void) fprintf( stderr,
			   "Handler: Have a messed up datastream with %d %d\n",
			   (int)buff[0], (int)buff[1]);
	}
	else 
	  perror("Handler: Failed to read extra bytes after OOB");
      } /* atoob == 1 */
#ifdef PENDEBUG
      else
	(void)fprintf( stderr, 
		      "Handler: not at OOB(mark, handled in ReceiveStatus.\n");
#endif /* PENDEBUG */
    } /* oobselectbug */
#endif /* OOBSELECTBUG && n OOBSELECTBUGX */

    if (oob != OOB_MESSAGE_CODE) {
      perror("Handler: Unknown oob code");
      break;
    }
    /* For now, we ignore the return value, but later ... */
    (void) (*callback_func[RCB_PEN].function)
      (callback_func[RCB_PEN].extra, (int) oob, atoob);
    break;

  default:
    perror("Handler: Cannot handle signal");
  }

 errno = savederrno;
}

/*  */
/**********************************************************************
 * Function: int (*register_pen_function(int what, int (* func_to_call)()), void *extra)()
 *
 * This function has as an argument the function that should
 * be called by the library if parallel editing notification is
 * required. It takes a char argument.
 *
 * Modifications:
 *      1994-01-01 Martin Sjlin. Return previous value and if called
 *                 with a NULL value restore initial value setup in this
 *                 module.
 *      1994-01-07 Martin Sjlin. Unified callback function registering
 *                 function to all look alike (RCB_).
 *      1994-01-13 Martin Sjlin. See comment in header.
 *      <list mods with name and date>
 */
int (*register_pen_function(what,func_to_call,extra))()
  int what;
  int (*func_to_call) (/* void *extra, int index, char *msg */);
  void *extra;
{
  int (*old)(/* void *extra, int index, char *msg */) = NULL;

  if (RCB_PEN <= what && what <= RCB_PEN_POLL) {
    /* "high" level event based interface */
    int index = what - RCB_PEN;		/* offset by 0 (=> RCB_PEN) */
    old = callback_func[index].function;
    if (func_to_call) {
      callback_func[index].function = func_to_call;
      callback_func[index].extra    = extra;
    } else 
      callback_func[index] = callback_default[index];
  }
  return(old);
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: int PEN_GET_HANDLE()
 * 
 * Modifications:
 *      <list mods with name and date>
 */
int PEN_GET_HANDLE()
{
  if (PEN_inited == 0) {
    if (pipe(PEN_pipe) != 0) {
      perror("PEN_GET_HANDLE: Can't open pipe\n");
      return(-1);
    }
#ifdef PENDEBUG
    (void)fprintf(stderr,"PEN_GET_HANDLE: Open PEN pipe\n");
#endif /* PENDEBUG */
    PEN_inited = 1;
  } 
  return(PEN_pipe[0]);			/* return read end */
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int default_PEN_oob(void *extra, int oob, int atoob)
 * 
 * put an event in the event_queue
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int default_PEN_oob(extra, oob, atoob)
  void *extra;
  int oob;
  int atoob;
{
  char c = (unsigned char) oob;

  if (PEN_inited == 0) {
    if (pipe(PEN_pipe) != 0) {
      perror("default_PEN_oob: Can't open pipe\n");
      return(-1);
    } /* pipe */
#ifdef PENDEBUG
    (void)fprintf( stderr,"default_PEN_oob: Open PEN pipe\n");
#endif /* PENDEBUG */
    PEN_inited = 1;
  } /* PEN_inited */

  if( write(PEN_pipe[1], &c, sizeof(c)) != 1) 
    perror("default_PEN_oob: Write failed");
#ifdef PENDEBUG
  fprintf( stderr, "default_PEN_oob: Wrote pen %d on pipe\n", (int) oob);
#endif /* PENDEBUG */
 return(0);
}

/*  */
/**********************************************************************
 * Function: void POLL_oob(void *do_poll)
 * 
 * Polls the PEN_pipe to manage OOB signals from dbs. This function is
 * called from almost every liblincks/usfuns.c function.
 *
 * If this routine is not setup ( when default_PEN_oob is called the 
 * first time), just exit. Else, we flush the pipe and count the 
 * number of characters (it does not really care as we read all messages
 * from the dbs), then retrieve all messages use PEN_RETRIEVE_MESSAGE
 * and our private mapping function.                
 *
 * Modifications:
 *      1993-08-12 Martin Sjlin -  rewrote user level function to use
 *               a mapfn function as we retrieve a whole bunch of pens
 *               at once. Implied changes here.
 *
 */
void POLL_oob(do_poll)
  void *do_poll;
{
  int  polling =  *((int *) do_poll);
  int  pens;
  int size = 0;
  int  count;
  int  st;
#ifdef NO_FIONREAD_PIPE
  int  flags;
#endif /* NO_FIONREAD_PIPE */
# define BUFSIZE 512
  static char buff[BUFSIZE]; /* skip the contents */

  /* if argument zero, do not poll */
  if (polling == 0)
    return;

  if (!PEN_inited)
    return;

#ifdef NO_FIONREAD_PIPE
  /* It seems like you cannot do a FIONREAD on a pipe under SCO !*/ 
  if ((flags = fcntl(PEN_pipe[0], F_GETFL, 0)) < 0) {
    perror("POLL_oob: fcntl(1) failed to get on pipe");
    return;
  }
# ifdef O_NONBLOCK
  if (fcntl(PEN_pipe[0], F_SETFL, flags | O_NONBLOCK) < 0) {
# else
  if (fcntl(PEN_pipe[0], F_SETFL, flags | O_NDELAY) < 0) {
# endif /* n O_NONBLOCK */
    perror("POLL_oob: fcntl(2) failed to set on pipe");
    return;
  }
  size = 64;				/* try some large number */
#else
  /* we get number character available in pipe and read/skip them */
    if ( ioctl(PEN_pipe[0],FIONREAD,&size) < 0) 
      perror("POLL_oob: ioctl failed");
    if ( size <= 0)
      return;
#endif /* n NO_FIONREAD_PIPE */

  /* OK, skip number of character in pipe */
  for(pens = 0; pens < size; pens += count) {
    count = size < BUFSIZE ? size : BUFSIZE;
#ifdef NO_FIONREAD_PIPE
    if ((count = read( PEN_pipe[0], buff, (IOSIZE_T)count)) > 0)
      continue;
    else
# ifdef O_NONBLOCK
    if (count == -1 && errno == EAGAIN) break;
# else /* to handle SysV(O_NDELAY) (0) and 4.2BSD(O_NDELAY) (-1) */
    if (count <=  0 && errno == EWOULDBLOCK) break;
# endif /* n O_NONBLOCK */
    else {
      perror("POLL_oob: non-blocking read on pipe failed");
      goto exit;
    } 
#else /* NO_FIONREAD_PIPE */
    if ( read( PEN_pipe[0], buff, (IOSIZE_T)count) != count ) {
      perror("POLL_oob: read failed");
      goto exit;
    }
#endif /* n NO_FIONREAD_PIPE */
  } /* for */

  if ( (st = PEN_RETRIEVE_MESSAGE(PEN_mapfn, (char *)&pens)) != SUCCESS ) 
    (void)fprintf(stderr,"POLL_obb: Can't retrieve/treat messages (%d)\n", st);

 exit:
#ifdef NO_FIONREAD_PIPE
  if (fcntl(PEN_pipe[0], F_SETFL, flags) < 0) 
    perror("POLL_oob: fcntl(3) failed to set on pipe");
#endif /* NO_FIONREAD_PIPE */
  return;
}

/*  */
/**********************************************************************
 * Function: static int PEN_mapfn(void *extra, int index, int type, char *msg, int count, int len)
 *
 * Is called once for every pen message - for parameter description
 * see usfuns.c. Is always called at least once, for setup. Returns
 * nonzero value and it will no longer be call during processing 
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int PEN_mapfn(extra, index, type, msg, count, len)
 void *extra;
 int   index;
 int   type;
 char *msg; 
 int   count; 
 int   len;
{
#ifdef PENDEBUG
  int pens = * (int *) extra;
#endif /* PENDEBUG */

  /* setup time ... */
  if ( count == 0 ) {
#ifdef PENDEBUG
    (void)fprintf( stderr, 
		  "PEN_mapfn: Number of OOB in pipe: %d from dbs %d\n", 
		  pens, len);
#endif /* PENDEBUG */
  } else {
#ifdef PENDEBUG
      (void) fprintf( stderr,
		     "PEN_mapfn: PEN type %d index %d msg %s.\n",
		     type, index, msg != NULL ? msg : "");
#endif /* PENDEBUG */
    if (type < PEN_RESOLVE || type > PEN_POLL)  
      /* print out a nice error message ... */
      (void) fprintf( stderr,
		     "PEN_mapfn: Unknown PEN type %d index %d msg %s.\n",
		     type, index, msg != NULL ? msg : "");
    else 
      /* since PEN_RESOLVE starts at 0, but it is placed in position 1 */
      (void) (*callback_func[type+1].function)
	(callback_func[type+1].extra, index, msg);

    if ( msg )
      (void) free(msg); /* allocated in rdwrmol.c in ReceiveImage */
  } /* count != 0 */

  /* and we like to see all of them ... */
  return(0);
}

/*  */
#ifdef PENDEBUG
/* ARGSUSED */
#endif
/**********************************************************************
 * static int default_PEN(void *extra, int index, char *msg);
 *
 * Default pen handler, prints out information if we have enabled
 * PENDEBUG, else just returns.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int default_PEN(extra, index, msg)
  void *extra;
  int   index;
  char *msg;
{
#ifdef PENDEBUG
  (void)fprintf( stderr, (char *) extra, index, 
		msg != NULL && msg[0] != '\0' ? msg : "");
#endif /* PENDEBUG */
  return(1);
}


