/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	files.c
 *
 * SCCSINFO:		@(#)files.c	1.7 6/7/94
 *
 * ORIGINAL AUTHOR(S):  Jens M Hansen, ????
 *
 * MODIFICATIONS:
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *
 *  File containing all functions of CUTOFF FILES module.
 *  FILES consists of a set of general file functions.
 *  All names of FILES functions contain the word "File".
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#if !(defined(HAVE_F_SETOWN)||!defined(HAVE_FLOCK)||defined(NO_FIONREAD_PIPE))
#include <fcntl.h>
#endif

#include "cutoff.h"
#include "monitor.h"
#include "xconfig.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
#include "f_files.h"

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_cutoffmain.h"
#include "f_pointers.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define NAMESTABLESIZE 64

#define PROTMOD 0644
#define BUFSIZE 16384

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void LoadNameFiles P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static struct {
  char filename[MAXPATHLEN];
  char arch_filename[MAXPATHLEN];
} molnames[NAMESTABLESIZE];

static struct {
  char filename[MAXPATHLEN];
  char arch_filename[MAXPATHLEN];
} datnames[NAMESTABLESIZE];

static int numnames = 0;

/*  */
/**********************************************************************
 * Function: int datfileOfIndex(int which, int index, int archive)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int datfileOfIndex(which, index, archive)
  int which;
  int index;
  int archive;
{
  int fd = -1;
  static int prevarch = -999;
  static int prevnew = -999;
  static int previndex = -999;
  int reuse = 0;

  LoadNameFiles();

  if (which == CLOSE) {
    if ((prevarch != -999) && ((close(prevarch)) == -1))
      fatal_error(FATAL_IO, "datfileOfIndex", (char *)NULL);
    if ((prevnew != -999) && ((close(prevnew)) == -1))
      fatal_error(FATAL_IO, "datfileOfIndex", (char *)NULL);
    prevarch = prevnew = previndex = -999;
    return(0);
  }

  if ((previndex == index) && (previndex != -999))
    reuse = 1;
  else {
    previndex = index;
    if ((prevarch != -999) && ((close(prevarch)) == -1))
      fatal_error(FATAL_IO, "datfileOfIndex", (char *)NULL);
    if ((prevnew != -999) && ((close(prevnew)) == -1))
      fatal_error(FATAL_IO, "datfileOfIndex", (char *)NULL);
    prevarch = prevnew = -999;
  }

  switch (which) {
  case DAT:
    if (archive) {
      if (reuse && prevarch != -999)
	fd = prevarch;
      else {
#ifdef DEBUG
	(void)fprintf(stderr, "opening %s\n", datnames[index].arch_filename);
#endif
        if ((fd = open(datnames[index].arch_filename, O_RDONLY)) == -1)
         fatal_error(FATAL_OPEN,"datfileOfIndex",datnames[index].arch_filename);
	prevarch = fd;
      }
    } else {
      if (reuse && prevnew != -999)
	fd = prevnew;
      else {
#ifdef DEBUG
	(void)fprintf(stderr, "opening %s\n", datnames[index].filename);
#endif
        if ((fd = open(datnames[index].filename, O_RDWR)) == -1)
          fatal_error(FATAL_OPEN, "datfileOfIndex", datnames[index].filename);
	prevnew = fd;
      }
    }
    break;
  case MOL:
  case CLOSE:
  default:
    (void)fprintf(stderr, "datfileOfIndex:  strange cases\n");
    break;
  }
  return fd;
}
/*  */
/**********************************************************************
 * Function: int molfileOfIndex(int which, int index, int archive)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int molfileOfIndex(which, index, archive)
  int which;
  int index;
  int archive;
{
  int fd = -1;
  static int prevarch = -999;
  static int prevnew = -999;
  static int previndex = -999;
  int reuse = 0;

  LoadNameFiles();

  if (which == CLOSE) {
    if ((prevarch != -999) && ((close(prevarch)) == -1))
      fatal_error(FATAL_IO, "molfileOfIndex", (char *)NULL);
    if ((prevnew != -999) && ((close(prevnew)) == -1))
      fatal_error(FATAL_IO, "molfileOfIndex", (char *)NULL);
    prevarch = prevnew = previndex = -999;
    return(0);
  }

  if ((previndex == index) && (previndex != -999))
    reuse = 1;
  else {
    previndex = index;
    if ((prevarch != -999) && ((close(prevarch)) == -1))
      fatal_error(FATAL_IO, "molfileOfIndex", (char *)NULL);
    if ((prevnew != -999) && ((close(prevnew)) == -1))
      fatal_error(FATAL_IO, "molfileOfIndex", (char *)NULL);
    prevarch = prevnew = -999;
  }

  switch (which) {
  case MOL:
    if (archive) {
      if (reuse && prevarch != -999)
	fd = prevarch;
      else {
#ifdef DEBUG
	(void)fprintf(stderr, "opening %s\n", molnames[index].arch_filename);
#endif
	reset_cache();
        if ((fd = open(molnames[index].arch_filename, O_RDONLY)) == -1)
         fatal_error(FATAL_OPEN,"molfileOfIndex",molnames[index].arch_filename);
	prevarch = fd;
      }
    } else {
      if (reuse && prevnew != -999)
	fd = prevnew;
      else {
#ifdef DEBUG
	(void)fprintf(stderr, "opening %s\n", molnames[index].filename);
#endif
	reset_cache();
        if ((fd = open(molnames[index].filename, O_RDWR)) == -1)
          fatal_error(FATAL_OPEN, "molfileOfIndex", molnames[index].filename);
	prevnew = fd;
      }
    }
    break;
  case DAT:
  case CLOSE:
  default:
    (void)fprintf(stderr, "molfileOfIndex:  strange cases\n");
    break;
  }
  return fd;
}

/*  */
/**********************************************************************
 * Function: off_t LengthOfFile(int fd)
 *
 * fetch length of file with descriptor fd
 *
 * returns length on success,
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
off_t LengthOfFile(fd)
  int fd;
{
  struct stat buf;

  if ((fstat(fd, &buf)) != 0)
    fatal_error(FATAL_STAT, "LengthOfFile", (char *)NULL);

  return buf.st_size;
}

/*  */
/**********************************************************************
 * Function: void truncate_old_files()
 *
 * Modifications:
 *      <list mods with name and date>
 */
void truncate_old_files()
{
  int i;
  int fd;

  (void)datfileOfIndex(CLOSE, 0, 0);
  (void)molfileOfIndex(CLOSE, 0, 0);

  for (i = 0; i < numnames; i++) {
    if ((fd = open(datnames[i].filename, O_RDWR | O_TRUNC)) == -1)
      fatal_error(FATAL_OPEN, "truncate_old_files", datnames[i].filename);
    if ((close(fd)) == -1)
      fatal_error(FATAL_IO, "truncate_old_files", (char *)NULL);
    if ((fd = open(molnames[i].filename, O_RDWR | O_TRUNC)) == -1)
      fatal_error(FATAL_OPEN, "truncate_old_files", molnames[i].filename);
    if ((close(fd)) == -1)
      fatal_error(FATAL_IO, "truncate_old_files", (char *)NULL);
  }
}

/*  */
/**********************************************************************
 * Function: static void LoadNameFiles()
 *
 * Fetch name of molecule/data file names.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void LoadNameFiles()
{
  int i;
  FILE *namesfp;
  static int done = 0;

  if (done == 1)
    return;
  else
    done = 1;

  if (!(namesfp = fopen(MOLNAMES, "r")))
    fatal_error(FATAL_OPEN, "LoadNameFiles", MOLNAMES);

  while (fscanf(namesfp, "%s\n", molnames[numnames].filename) == 1) {
    (void)sprintf(molnames[numnames].arch_filename, "Archive/%s",
		  molnames[numnames].filename);
    numnames++;
  }
  (void)fclose(namesfp);

  if (!(namesfp = fopen(DATANAMES, "r")))
    fatal_error(FATAL_OPEN, "LoadNameFiles", DATANAMES);
  i = 0;
  while (fscanf(namesfp, "%s\n", datnames[i].filename) == 1) {
    (void)sprintf(datnames[i].arch_filename, "Archive/%s",
		  datnames[i].filename);
    i++;
  }
  if (i != numnames)
    fatal_error(FATAL_GEN, "LoadNameFiles","molecule and data names mismatch");
}


/*  */
/**********************************************************************
 * Function: int CopyFile(char *fn)
 *
 * copy file with name fn to archive
 *
 * returns SUCCESS on success,
 * FAIL on error
 *
 * Modifications:
 *      <list mods with name and date>
 */
int CopyFile(fn)
  char *fn;
{
  int srcfd, dstfd;		/* fd of source and destination files */
  static char buf[BUFSIZE];	/* file buffer */
  int i;

  /* prepend archive header to file name */
  (void)sprintf(buf, "%s%s", HEADER, fn);

  /* open file to be copied from */
  if ((srcfd = open(fn, O_RDONLY)) == -1) 
    fatal_error(FATAL_OPEN, "CopyFile", fn);

  /* create file to be copied to */
  if ((dstfd = open(buf, O_WRONLY | O_CREAT | O_TRUNC , PROTMOD)) == -1)
    fatal_error(FATAL_CREATE, "CopyFile", buf);

  /* copy from source to destination */
  while ((i = read(srcfd, buf, (IOSIZE_T)BUFSIZE)) > 0) {
    if (i < 0 || write(dstfd, buf, (IOSIZE_T)i) != i) {
      fatal_error(FATAL_IO, "CopyFile", (char *)NULL);
      return FAIL;
    }
  }
  if (close(srcfd) < 0 || close(dstfd) < 0) 
    fatal_error(FATAL_CLOSE, "CopyFile", (char *)NULL);

  return SUCCESS;
}
