/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	dbdump.c
 *
 * SCCSINFO:		@(#)dbdump.c	1.13 6/6/94
 *
 * ORIGINAL AUTHOR(S):  Ralph R|nnquist, 1991-02-07
 *
 * MODIFICATIONS:
 *      1994-05-12 Martin Sjlin. When dumping image values, if the length
 *		of the value is equal to the string length and we have no
 *		zero in the in, dump it as a string, else we dump it is as
 *		uuencoded string using the 'new' t2lincks syntax of
 *			.binary <name> "uu-encoded-string"
 *		Added Printuuval().
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 *	The dbdump program attempts to read a database and
 *	produce (on stdout) a t2lincks-loadable description
 *	of the current database state.
 *	NOTE: Bound labels are not treated accurately!!
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */

#if !(defined(HAVE_F_SETOWN)||!defined(HAVE_FLOCK)||defined(NO_FIONREAD_PIPE))
#include <fcntl.h>
#endif

#include "liblincks.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
int main( /* int argc, char **argv */ );

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_uu.h"
#ifdef NEED_STRDUP
extern char *strdup( /* char *incoming */ );
#endif	/* NEED_STRDUP */

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *optarg;
extern int optind;
extern int opterr;
extern int global_comhistoff;

extern char *sys_errlist[];	/* errno.h */
extern int sys_nerr;		/* errno.h */
extern int errno;		/* errno.h */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
typedef struct {
  label lbl;
  char *group;
  char *field;
} mapaccess;

typedef struct x_entry {
  struct x_entry *next, *link;
  label lbl;
} DUMPENTRY;

#define MAXDEPTH	100
#define STRLEN		124
#define PRINTNODE(x)	if (x) (void)fprintf(outfile, "N%d",x); else (void) fprintf(outfile, "\"System Root\"")

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void Addtodumplist P_(( label *lbl ));
static void arginit P_(( int argc, char **argv ));
static int Attrfieldfn P_(( mapaccess *mp, char *fld, int c, int n ));
static int Attrgroupfn P_(( mapaccess *mp, char *grp, int c, int n ));
static void Builddump P_(( void ));
static FILE *do_login P_(( void ));
static void Dumpfirst P_(( label *lbl ));
static void Dumplinks P_(( DUMPENTRY *p ));
static int Fielddumpfn P_(( mapaccess *mp, char *fld, int c, int n ));
static int Fieldmapfn P_(( mapaccess *mp, char *fld, int c, int n ));
static int Groupdumpfn P_(( mapaccess *mp, char *grp, int c, int n ));
static int Groupmapfn P_(( mapaccess *mp, char *grp, int c, int n ));
static int Itemdumpfn P_(( mapaccess *mp, label *item, int c, int n ));
static int Itemmapfn P_(( mapaccess *mp, label *item, int c, int n ));
static void Printstr P_(( char *p, int n ));
static void Printval P_(( attrval *v ));
static void Printuuval P_(( attrval *v ));
static void print_version P_(( void ));
static void usage P_(( void ));

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
/* The savelist consists of all links to handle at a level.  */
static DUMPENTRY *savelist;

static int level = 0;
static int maxdepth = 0;

/*
 * Each dumplst entry is an object instance that 
 * should be dumped. The list is kept sorted by lbl.vs.
 */
static DUMPENTRY *dumplst = NULL;

static FILE *outfile = NULL;

static char dbdir[MAXPATHLEN];
static char username[STRLEN];
static char password[STRLEN];
static char filename[MAXPATHLEN];
static int noninteractive = 0;

/*  */
/**********************************************************************
 * Function: static void Printstr(char *p, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Printstr(p, n)
  char *p;
  int n;
{
  (void)fprintf(outfile, "\"");
  while (n-- > 0) {
    if (*p != '\0') {
      if ((*p == '"') || (*p == '\n') || (*p == '\\'))
	(void)fprintf(outfile, "\\");
      (void)fprintf(outfile, "%c", *(p++));
    }
  }
  (void)fprintf(outfile, "\"");
}

/*  */
/**********************************************************************
 * Function: static void Printuuval(attrval *v)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Printuuval(v)
  attrval *v;
{
  char *uu;
  if ((uu = uuencode(v->attsize, v->attvalue, NULL)) == NULL)
    (void) fprintf(stderr, 
		   "Failed to convert binary information into uudecoded\n");
  else {
    (void)fprintf(outfile, " ");
    Printstr(uu, strlen(uu));
    free((FREEPTR *) uu);
    (void)fprintf(outfile, "\n");
  }
}

/*  */
/**********************************************************************
 * Function: static void Printval(attrval *v)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Printval(v)
  attrval *v;
{
  (void)fprintf(outfile, " ");
  Printstr(v->attvalue, v->attsize);
  (void)fprintf(outfile, "\n");
}


/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Attrfieldfn(mapaccess *mp, char *fld, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Attrfieldfn(mp, fld, c, n)
  mapaccess *mp;
  char *fld;
  int c;
  int n;
{
  attrval v;
  if (c == 0)
    return 0;

  if (GA_GETATTR(&(mp->lbl), mp->group, fld, &v) == SUCCESS) {
    (void)fprintf(outfile, "\t");
    Printstr(fld, strlen(fld));
    (void)fprintf(outfile, " :");
    Printval(&v);
    if (v.attsize > 0)
      free(v.attvalue);
  }
  return 0;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Attrgroupfn(mapaccess *mp, char *grp, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Attrgroupfn(mp, grp, c, n)
  mapaccess *mp;
  char *grp;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  (void)fprintf(outfile, ".group ");
  PRINTNODE(mp->lbl.vs);
  (void)fprintf(outfile, " ");
  Printstr(grp, strlen(grp));
  (void)fprintf(outfile, "\n");

  mp->group = strdup(grp);
  if ((GAN_GETATTRNAMES(&(mp->lbl), grp, Attrfieldfn, (void *)mp)) != SUCCESS)
    (void)fprintf(stderr, "Attrgroupfn: GAN_GETATTRNAMES failed\n");
  free(mp->group);
  mp->group = 0;

  (void)fprintf(outfile, "\n");
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void Dumpfirst(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Dumpfirst(lbl)
  label *lbl;
{
  mapaccess mp;
  attrval v;

  (void)fprintf(stdout, "+ Node N%d\n", lbl->vs);
  if (lbl->vs == 0) {
    (void)fprintf(outfile,"########################################\n");
    (void)fprintf(outfile,"# object \"System Root\" is predefined.\n");
  } else {
    int typestring = TRUE;

    v.attsize = 0;
    if ((GI_GETIMAGE(lbl, &v)) != SUCCESS)
      (void)fprintf(stderr, "Dumpfirst: GI_GETIMAGE failed\n");

    /* check if it is a string, ended by a zero */
    if (v.attsize > 0 && v.attvalue != (char *) NULL) 
      if (v.attvalue[v.attsize-1] != '\0' || strlen(v.attvalue) != v.attsize-1) 
	typestring = FALSE;

    (void)fprintf(outfile,
		  "########################################\n.%s ", 
		  typestring ? "object" : "binary");
    PRINTNODE(lbl->vs);

    if (typestring)
      Printval(&v);
    /* since the parser only handle line up to INBUFLIMIT chars */
    else if (v.attsize < PARSERINBUFLIMIT/5)	
      Printuuval(&v);
    else {
      char filename[MAXPATHLEN];
      int fd;
      size_t size = v.attsize;
      char *p = v.attvalue;
      int len;

      /* create a file with the stuff ... */
      (void) sprintf(filename,"N%d.image", lbl->vs);
      if ((fd = open(filename,
		     O_WRONLY|O_TRUNC|O_CREAT, 
		     S_IRUSR|S_IWUSR|S_IRGRP)) < 0) {
	(void) fprintf(stderr,
		       "Dumpfirst: Failed to open '%s' for writing - %s\n", 
		       filename,
		       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	(void) fprintf(outfile,
		       " \"(empty - error opening '%s' for writing - %s)\"\n", 
		       filename,
		       sys_errlist[(errno > sys_nerr) ? 0 : errno]);
      }
      else {
	(void) fprintf(outfile," < \"%s\"\n", filename);    
	for(; size > 0; size -= len, p += len)
	  if ((len = write(fd, p, size)) < 0) {
	    (void) fprintf(stderr, 
			   "Dumpfirst: Error writing to '%s' - %s\n", 
			   filename,
			   sys_errlist[(errno > sys_nerr) ? 0 : errno]);
	    break;
	  } /* write */
	(void) close(fd);
      } /* error opening file for writing */
    } /* to long binary file ... */
      
    if (v.attsize > 0)
      free(v.attvalue);
  }
  mp.lbl.vs = lbl->vs;
  mp.lbl.inst = lbl->inst;
  if ((GAGN_GETATTRGROUPNAMES(lbl, Attrgroupfn, (void *)&mp)) != SUCCESS)
    (void)fprintf(stderr, "Dumpfirst: GAGN_GETATTRGROUPNAMES failed\n");
}

/*  */
/**********************************************************************
 * Function: static void Addtodumplist(label *lbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Addtodumplist(lbl)
  label *lbl;
{
  DUMPENTRY **at, *p;

  at = &dumplst;

  while ((*at) && ((*at)->lbl.vs < lbl->vs))
    at = &((*at)->next);

  if ((*at) && ((*at)->lbl.vs == lbl->vs))
    return;

  p = (DUMPENTRY *) malloc(sizeof(DUMPENTRY));
  p->lbl.vs = lbl->vs;
  p->lbl.inst = lbl->inst;
  p->link = savelist;
  savelist = p;
  p->next = (*at);
  *at = p;

  Dumpfirst(lbl);
  return;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Itemmapfn(mapaccess *mp, label *item, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Itemmapfn(mp, item, c, n)
  mapaccess *mp;
  label *item;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  Addtodumplist(item);
  return 0;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Fieldmapfn(mapaccess *mp, char *fld, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Fieldmapfn(mp, fld, c, n)
  mapaccess *mp;
  char *fld;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  mp->field = strdup(fld);
  if ((GLV_GETLINKVAL(&(mp->lbl), mp->group,
		      fld, Itemmapfn, (void *)mp)) != SUCCESS)
    (void)fprintf(stderr, "Fieldmapfn: GLV_GETLINKVAL failed\n");
  free(mp->field);
  mp->field = 0;
  return 0;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Groupmapfn(mapaccess *mp, char *grp, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Groupmapfn(mp, grp, c, n)
  mapaccess *mp;
  char *grp;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  mp->group = strdup(grp);
  if ((GLN_GETLINKNAMES(&(mp->lbl), grp, Fieldmapfn, (void *)mp)) != SUCCESS)
    (void)fprintf(stderr, "Groupmapfn: GLN_GETLINKNAMES failed\n");
  free(mp->group);
  mp->group = 0;
  return 0;
}

/*  */
/**********************************************************************
 * Function: static void Builddump()
 *
 * Process the current savelist and make a new one for the next level.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Builddump()
{
  mapaccess mp;
  DUMPENTRY *oldsave;

  while (savelist) {
    oldsave = savelist;
    savelist = 0;
    level++;
    (void)fprintf(stderr, "Beginning level %d\n", level);
    (void)fprintf(outfile, "### Beginning level %d ###\n", level);
    while (oldsave) {
      if (level >= maxdepth) {
	(void)fprintf(stderr, "## Exceeding maximal depth for object ");
	(void)fprintf(stderr, "N%d ##\n", oldsave->lbl.vs);
	(void)fprintf(outfile, "#### Exceeding maximal depth for object ");
	PRINTNODE(oldsave->lbl.vs);
	(void)fprintf(outfile, "####.\n");
	oldsave->lbl.inst = oldsave->lbl.vs;
	oldsave->lbl.vs = -1;
      } else {
	mp.lbl.vs = oldsave->lbl.vs;
	mp.lbl.inst = oldsave->lbl.inst;
	mp.group = 0;
	mp.field = 0;
	if ((GLGN_GETLINKGROUPNAMES(&(oldsave->lbl), Groupmapfn,
				    (void *)&mp)) != SUCCESS)
	  (void)fprintf(stderr, "Builddump: GLGN_GETLINKGROUPNAMES failed\n");
      }
      oldsave = oldsave->link;
    }
  }
}

/*  */
/********************************************************************/
/* Dumping links */

/* ARGSUSED */
/**********************************************************************
 * Function: static int Itemdumpfn(mapaccess *mp, label *item, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Itemdumpfn(mp, item, c, n)
  mapaccess *mp;
  label *item;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  if (c == 1) {
    (void)fprintf(outfile, "\t");
    Printstr(mp->field, strlen(mp->field));
    (void)fprintf(outfile, " = ");
  } else {
    (void)fprintf(outfile, ", ");
  };
  PRINTNODE(item->vs);
  return 0;
}

/*  */
/* ARGSUSED */
/**********************************************************************
 * Function: static int Fielddumpfn(mapaccess *mp, char *fld, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Fielddumpfn(mp, fld, c, n)
  mapaccess *mp;
  char *fld;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  if (strcmp(fld, "Parent") == 0 && strcmp(mp->group, "SYSTEM") == 0)
    return 0;

  mp->field = strdup(fld);
  if ((GLV_GETLINKVAL(&(mp->lbl), mp->group, fld, 
		 Itemdumpfn, (void *)mp)) != SUCCESS)
    (void)fprintf(stderr, "Fielddumpfn: GLV_GETLINKVAL failed\n");
  free(mp->field);
  mp->field = 0;

  (void)fprintf(outfile, "\n");
  return 0;
}

/*  */
/**********************************************************************
 * Function: static int Groupdumpfn(mapaccess *mp, char *grp, int c, int n)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int Groupdumpfn(mp, grp, c, n)
  mapaccess *mp;
  char *grp;
  int c;
  int n;
{
  if (c == 0)
    return 0;

  (void)fprintf(outfile, ".links ");
  PRINTNODE(mp->lbl.vs);
  (void)fprintf(outfile, " ");
  Printstr(grp, strlen(grp));
  (void)fprintf(outfile, "\n");

  mp->group = strdup(grp);
  if ((GLN_GETLINKNAMES(&(mp->lbl), grp, Fielddumpfn, (void *)mp)) != SUCCESS)
    (void)fprintf(stderr, "Groupdumpfn: GLN_GETLINKNAMES failed\n");
  free(mp->group);
  mp->group = 0;

  if (c == n)
    (void)fprintf(outfile, "########################################\n\n");
  else
    (void)fprintf(outfile, "\n");

  return 0;
}

/*  */
/**********************************************************************
 * Function: static void Dumplinks(DUMPENTRY *p)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Dumplinks(p)
  DUMPENTRY *p;
{
  mapaccess mp;

  (void)fprintf(stderr, "*** Dumping the links ***\n");
  (void)fprintf(outfile, "######## Here come all links: #########\n\n");
  while (p != 0) {
    if ((mp.lbl.vs = p->lbl.vs) != -1) {
      mp.lbl.inst = p->lbl.inst;
      mp.group = 0;
      mp.field = 0;
      if ((GLGN_GETLINKGROUPNAMES(&(p->lbl), Groupdumpfn,
				  (void *)&mp)) != SUCCESS)
	(void)fprintf(stderr, "Dumplinks: GLGN_GETLINKGROUPNAMES failed\n");

    }
    p = p->next;
  }
}

/*  */
/**********************************************************************
 * Function: int main(int argc, char **argv)
 *
 * Modifications:
 *      <list mods with name and date>
 */
int main(argc, argv)
  int argc;
  char **argv;
{
  label root;

  global_comhistoff = 1;

  arginit(argc, argv);
  outfile = do_login();

  if (GSR_GETSYSTEMROOT(&root) != SUCCESS) {
    (void)fprintf(stderr, "Error: Couldn't get System Root !!!\n");
    exit(1);
  }
  Addtodumplist(&root);
  Builddump();
  Dumplinks(dumplst);

  (void)LOGOUT();
  return (0);
}

/*  */
/**********************************************************************
 * Function: static void arginit(int argc, char **argv)
 *
 * processes the command line options.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void arginit(argc, argv)
  int argc;
  char **argv;
{
  int c = 0, user = 0, pword = 0;
  char *db;

  (void)memset((char *)dbdir, 0, (int)sizeof(dbdir));
  (void)memset((char *)username, 0, (int)sizeof(username));
  (void)memset((char *)password, 0, (int)sizeof(password));
  (void)memset((char *)filename, 0, (int)sizeof(filename));

  while ((c = getopt(argc, argv, "f:u:p:d:h")) != EOF) {
    switch (c) {
    case 'u':
      (void)strcpy(username, optarg);
      user = 1;
      noninteractive = 1;
      break;
    case 'p':
      (void)strcpy(password, optarg);
      pword = 1;
      break;
    case 'f':
      (void)strcpy(filename, optarg);
      break;
    case 'd':
      maxdepth = atoi(optarg);
      break;
    case 'h':
      print_version();
      usage();
      break;
    default:
      break;
    }
  }
  if (pword != user) {
    (void)fprintf(stderr, "Must use both -u and -p or neither.\n");
    usage();
  }
  if (maxdepth == 0)
    maxdepth = MAXDEPTH;

  switch (argc - optind) {
  case 0:
    if ((db = (char *)getenv("LINCKSDBDIR")) == (char *)NULL)
      usage();
    (void)strcpy(dbdir, db);
    break;
  case 1:
    (void)strcpy(dbdir, argv[argc - 1]);
    break;
  default:
    usage();
    break;
  } 
}

/*  */
/**********************************************************************
 * Function: static void usage()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void usage()
{
  (void)fprintf(stderr, "%s\n%s\n",
			"Usage: dbdump [-h] [-u username -p password]", 
			"       [-f outfile] [-d maxdepth] database-dir");
  exit(1);
}

/*  */
/**********************************************************************
 * Function: static FILE *do_login()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static FILE *do_login()
{
  FILE *f;

  if (noninteractive) {
    if (NOPROMPT_LOGIN(username, password, dbdir) != SUCCESS) {
      (void)fprintf(stderr, "NOPROMPT_LOGIN: login failure\n");
      exit(-1);
    }
  } else {
    if (LOGIN(dbdir) != SUCCESS) {
      (void)fprintf(stderr, "LOGIN: login failure\n");
      exit(-1);
    }
  }

  if (filename[0] == '\0') {
    (void)fprintf(stderr, "No output file.  Writing to stdout.\n");
    if ((f = fdopen(fileno(stdout), "w")) == (FILE *) NULL) {
      perror("Failed to open stdout");
      exit(-1);
    }
  } else {
    if ((f = fopen(filename, "wr")) == (FILE *) NULL) {
      (void)fprintf(stderr, "Failed to open file %s: ", filename);
      perror("");
      exit(-1);
    }
  }
  return f;
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"

/*  */
/**********************************************************************
 * Function: static void print_version()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tdbdump version %s (%s)\n%s", VERSION, DATE,
                VERSION_INFO);

}
