/* 
 * Linkoping Intelligent Communication of Knowledge System (LINCKS)
 *      Copyright (C) 1993, 1994 Lin Padgham, Ralph Rnnquist
 *       Department of Computer and Information Sciences
 *		University of Linkoping, Sweden
 *		    581 83 Linkoping, Sweden
 *		       lincks@ida.liu.se
 *
 * These collective LINCKS programs are free software; you can 
 * redistribute them and/or modify them under the terms of the GNU
 * General Public License as published by the Free Software Foundation,
 * version 2 of the License.
 *
 * These programs are distributed in the hope that they will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with the programs; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * MODULE NAME: 	t2lincks.c
 *
 * SCCSINFO:		@(#)t2lincks.c	1.18 6/2/94
 *
 * ORIGINAL AUTHOR(S):  1990-03-16 by Ralph R|nnquist.
 *
 * MODIFICATIONS:
 *       1994-05-11 Martin Sjlin. Extended the .image syntax
 *                  to be able to read input from a file, as well
 *                  converting an uudecode string into binary data ...
 *       1994-05-25 Martin Sjlin. Added a %USER% variable set to
 *                  login user name ...
 *	<list mods with name and date>
 *
 * DESCRIPTION:
 */
/*********************************************************************
 * INCLUDES:
 *********************************************************************/
#include "config.h"	/* includes system dependent includes */
#include "parser.h"
#include "liblincks.h"
#include "libshared.h"

/*********************************************************************
 * EXTERNALLY-CALLABLE ROUTINES FOUND IN THIS MODULE:
 *********************************************************************/
void main( /* int argc, char **argv */ );

/*********************************************************************
 * EXTERNALLY-AVAILABLE	DATA FOUND IN THIS MODULE:
 *********************************************************************/
/* none */

/*********************************************************************
 * EXTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
#include "f_uu.h"

/*********************************************************************
 * EXTERNAL DATA STRUCTURES USED BY THIS MODULE:
 *********************************************************************/
extern char *optarg;
extern int optind;
extern int opterr;
extern int global_comhistoff;
extern char LL_loginuser[];		/* dbcomm.c */

/*********************************************************************
 * LOCAL DEFINES, STRUCTS, TYPEDEFS, ETC.:
 *********************************************************************/
#define new(tp)	(tp *) malloc(sizeof(tp))

typedef struct x_cmd_rec {
  struct x_cmd_rec *next;
  char *group;
  char *field;
  int pos;
  int ix;
  attrval val;
  label lbl;
} cmd_rec;

typedef struct x_symbol {
  struct x_symbol *s_next;
  char *pname;
  label obj;
  cmd_rec *cmd;
} objectsymbol;

typedef struct x_path {
  struct x_path *pnext;
  char *group, *field;
  int pos;
  label lbl;
} path;

#define GRMGOAL	"text2lincks"
#define STRLEN 64

/*********************************************************************
 * INTERNAL FUNCTIONS USED BY THIS MODULE:
 *********************************************************************/
static void arginit P_(( int argc, char **argv ));
static void copylbl P_(( label *tolbl, label *fromlbl ));
static FILE *do_login P_(( void ));
static void fatal_error P_(( char *string, char *string2 ));
static objectsymbol *Findobjectsymbol P_(( char *s ));
static void flush_cmd P_(( objectsymbol *os ));
static void Makeobjectsymbol P_(( char *objid ));
static int peepatline P_(( char **str ));
static void print_version P_(( void ));
static void put_cmd P_(( char *val, label *lbl, int pos, int bound ));
static void usage P_(( void ));

/* NOTE: "#define ACTION(fn) int fn()" in parser.h */
static ACTION(syntax_error);
static ACTION(buildpath);
static ACTION(setobject);
static ACTION(flushobj);
static ACTION(newobject);
static ACTION(newbinaryobject);
static ACTION(newfileobject);
static ACTION(beginlinks);
static ACTION(beginattrs);
static ACTION(defattr);
static ACTION(ltag);
static ACTION(boundlink);
static ACTION(unboundlink);
static ACTION(deflinks);
static ACTION(defuser);

/*********************************************************************
 * INTERNAL (STATIC) DATA:
 *********************************************************************/
static objectsymbol *table;
static objectsymbol *curobj;
static char *grp = (char *)NULL;
static char *fld = (char *)NULL;
static char *global_lastinput = NULL;

static actionentry T2LACTIONS[] =
{
  {"buildpath", buildpath},
  {"newobject", newobject},
  {"newfileobject", newfileobject},
  {"newbinaryobject", newbinaryobject},
  {"flushobj", flushobj},
  {"setobject", setobject},
  {"beginlinks", beginlinks},
  {"beginattrs", beginattrs},
  {"defattr", defattr},
  {"ltag", ltag},
  {"boundlink", boundlink},
  {"unboundlink", unboundlink},
  {"deflinks", deflinks},
  {"syntax_error", syntax_error},
  {"defuser", defuser},
  {0, 0}
};

static char *T2LSYNTAX[] =
{
  "<link> ::= <id> : <id> : <integer> => noop",

  "<path> ::= [ <link> while + ] => buildpath",

  "<text2lincks> ::=	<peepatline>		=> noop	!\
			.flush <id>             => flushobj !\
			.object <id> <string>   => newobject	!\
			.binary <id> <string>   => newbinaryobject !\
                        .binary <id> < <string> => newfileobject !\
			.set <id> from <id> + <path> => setobject !\
			.links <id> <id>	=> beginlinks	!\
			.group <id> <id>	=> beginattrs	!\
			<id> : <id>		=> defattr	!\
			<ltag> = [ <binding> <id> while , ] => deflinks",

  "<binding> ::= current => boundlink ! <empty> => unboundlink",

  "<t2lmeta> ::=	# <catchline>		=> drop1 !\
			<eoln>			=> noop !\
			! <metacmd>		=> noop !\
			<catchline>		=> syntax_error",


  "<ltag> ::=		<id>			=> ltag",

  "<id>	::= %USER% => defuser ! <identifier> => noop ! <integer> => noop ! <string> => noop",

  (char *)NULL
};

static path *lst;
static char dbdir[MAXPATHLEN];
static char filename[MAXPATHLEN];
static char username[STRLEN];
static char password[STRLEN];
static int noninteractive = 0;
static int parse_only = 0;

/*  */
/**********************************************************************
 * Function: void main(int argc, char **argv)
 *
 * Modifications:
 *      <list mods with name and date>
 */
void main(argc, argv)
  int argc;
  char **argv;
{
  FILE *file;
  int i;
  int retval = 0;

  global_comhistoff = 1;
  arginit(argc, argv);
  file = do_login();
  init_parser();
  metagrammar();
  defineactions(T2LACTIONS);
  bindprimary("peepatline",peepatline);
  definegrammar(T2LSYNTAX);
  (void)setmeta("t2lmeta");

  table = new(objectsymbol);
  table->pname = strdup("System Root");
  table->s_next = (objectsymbol *) NULL;
  table->cmd = (cmd_rec *)NULL;
  if (GSR_GETSYSTEMROOT(&(table->obj)))
    fatal_error("couldn't retrieve system root", NULL);

  (void)usefile(file);
  interact(GRMGOAL);

  if (parse_only) {
    (void)printf("\nt2lincks parse completed successfully.\n");
    (void)LOGOUT();
    exit(0);
  }

  while (table != (objectsymbol *) NULL) {
    objectsymbol *prev = table;
    flush_cmd(table);
    if (BOUND(&(table->obj)) && EDITED(&(table->obj))) {
      (void)printf("Storing object %s in database...", table->pname);
      (void)fflush(stdout);
      (void)printf("(%d)", SO_STOREOBJ(&(table->obj)));
      (void)printf(" done.\n");
    }
    (void)RO_RELEASEOBJ(&(table->obj));
    if (table->pname != NULL) {
      free((FREEPTR *)table->pname);
    }
    table = table->s_next;
    (void)free(prev);
  }

  for (i = 0; i < 3; i++) {
    switch ((retval = LOGOUT())) {
    case FAIL:
    case ERR_RCBAD:
      (void)fprintf(stderr, "%s %s\n%s\n",
		    "LOGOUT returned ", 
		    (retval == FAIL) ? "FAIL." : "ERR_RCBAD.",
		    "Sleeping then trying again.");
      sleep(5);
      break;
    case SUCCESS:
    default:
      exit(0);
    }
  }
  exit(1);				/* else we fail! */
}

/*  */
/**********************************************************************
 * Function: static void copylbl( label *tolbl, label *fromlbl)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void copylbl(tolbl, fromlbl)
  label *tolbl, *fromlbl;
{
  tolbl->vs = fromlbl->vs;
  tolbl->inst = fromlbl->inst;
}

/*  */
/**********************************************************************
 * Function: static void fatal_error()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void fatal_error(string, string2)
  char *string;
  char *string2;
{
  (void)fprintf(stderr, "fatal_error: %s %s\n", string, 
			(string2) ? string2 : ".");
  if (global_lastinput) {
    (void)fprintf(stderr,
		  "********** Last read input follows **********\n%s\n",
		  global_lastinput);
  } else {
    (void)fprintf(stderr,
		  "The error occurred before any input was read\n");
  }
  (void)LOGOUT();
  exit(1);
}

/*  */
/**********************************************************************
 * Function: static objectsymbol *Findobjectsymbol(char *s)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static objectsymbol *Findobjectsymbol(s)
  char *s;
{
  objectsymbol *sp;

  for (sp = table; sp != (objectsymbol *) NULL; sp = sp->s_next)
    if (strcmp(s, sp->pname) == 0)
      return sp;
  return (objectsymbol *) NULL;
}

/*  */
/**********************************************************************
 * Function: static void Makeobjectsymbol(char *objid)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void Makeobjectsymbol(objid)
  char *objid;
{
  if ((curobj = Findobjectsymbol(objid)) == (objectsymbol *) NULL) {
    curobj = new(objectsymbol);
    curobj->s_next = table;
    table = curobj;
    curobj->pname = objid;
    curobj->cmd = (cmd_rec *)NULL;
  } else {
    free(objid);
    flush_cmd(curobj);
    if (EDITED(&(curobj->obj))) {
      (void)printf("Flushing object %s into database...", curobj->pname);
      (void)fflush(stdout);
      (void)printf("(%d)", SO_STOREOBJ(&(curobj->obj)));
      (void)printf(" done.\n");
    } else {
      (void)printf("Reusing object %s.\n", curobj->pname);
    }
    (void)RO_RELEASEOBJ(&(curobj->obj));
  }
}

/*  */
/**********************************************************************
 * Function: static int peepatline(char **str)
 *
 * A primary which consumes nothing and always fails.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static int peepatline(str)
  char **str;
{
  global_lastinput = *str;
  Failif(1);
}

/*  */
/**********************************************************************
 * Function: static ACTION(syntax_error)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(syntax_error)
{
  fatal_error("Unintelligble", "input line");
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(buildpath)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(buildpath)
{
  int n;
  path *lp;

  lst = (path *) NULL;
  for (n = popint(); n > 0; n--) {
    lp = new(path);
    lp->pos = popint();
    lp->field = popdata();
    lp->group = popdata();
    lp->pnext = lst;
    lst = lp;
  }
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(setobject)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(setobject)
{
  char *oldid;
  objectsymbol *oldobj;
  path *lp;

  oldid = popdata();
  if (!(oldobj = Findobjectsymbol(oldid)))
    fatal_error("Undefined object symbol: ", oldid);
  if (!parse_only)
    flush_cmd(oldobj);
  free(oldid);

  Makeobjectsymbol(popdata());

  copylbl(&(curobj->obj), &(oldobj->obj));
  while (lst != (path *) NULL) {
    if (GLI_GETLINKITEM(&(curobj->obj),
			lst->group,
			lst->field,
			lst->pos,
			&(lst->lbl)) != SUCCESS) {
      (void)fprintf(stderr, "Failed to follow %s:%s:%d - ",
		    lst->group,lst->field,lst->pos);
      fatal_error("no such link", NULL);
    }
    copylbl(&(curobj->obj), &(lst->lbl));
    lp = lst;
    lst = lst->pnext;
    free(lp->field);
    free(lp->group);
    free((FREEPTR *)lp);
  }
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static  ACTION(flushobj)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(flushobj)
{
  objectsymbol *obj;
  char *id = popdata();

  if (!(obj = Findobjectsymbol(id)))
    fatal_error("Undefined object symbol: ", id);
  if (!parse_only)
    Makeobjectsymbol(id);
  else
    free(id);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static  ACTION(newobject)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(newobject)
{
  attrval imp;

  imp.attvalue = popdata();
  if (imp.attvalue == (char *)NULL)
    imp.attsize = 0;
  else
    imp.attsize = strlen(imp.attvalue) + 1;

  Makeobjectsymbol(popdata());

  if (CO_CREATEOBJ(&(curobj->obj)) ||
      SI_SETIMAGE(&(curobj->obj), &imp))
    fatal_error("Couldn't create object ", imp.attvalue);
  free((FREEPTR *)imp.attvalue);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static  ACTION(newbinaryobject)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(newbinaryobject)
{
  attrval imp;

  imp.attvalue = popdata();
  if (imp.attvalue == (char *)NULL)
    imp.attsize = 0;
  else {
    char *p;
    if ((p = (char *) strdup(imp.attvalue)) == NULL) 
      fatal_error("Failed to allocate memory for uudecoding of ",imp.attvalue);
    if((imp.attsize = uudecode(imp.attvalue, p)) == 0) 
      fatal_error("Failed to uudecode ", imp.attvalue);
    free((FREEPTR *) imp.attvalue);
    imp.attvalue = p;
  }

  Makeobjectsymbol(popdata());

  if (CO_CREATEOBJ(&(curobj->obj)) ||
      SI_SETIMAGE(&(curobj->obj), &imp))
    fatal_error("Couldn't create object ", imp.attvalue);
  free((FREEPTR *)imp.attvalue);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static  ACTION(newfileobject)
 *
 * Open the specified file, stat to get size, read in into
 * memory, and store the object. User is responsible for adding
 * an ending zero for textual files.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(newfileobject)
{
  int fd;
  char *filename;
  struct stat statinfo;
  attrval imp;

  filename = popdata();
  Makeobjectsymbol(popdata());

  if ((fd = open(filename, O_RDONLY)) < 0) {
    fatal_error("failed to open file for reading input ", filename);
  }
  
  if ((fstat(fd, &statinfo)) < 0) {
    fatal_error("failed to stat file to get size of ", filename);
  }

  imp.attsize = statinfo.st_size;

  if(statinfo.st_size == 0) 
    imp.attvalue = NULL;
  else {
    if ((imp.attvalue = malloc(imp.attsize)) == NULL)
      fatal_error("failed to allocated memory for contents in ", filename);
    else
    {
      size_t size = imp.attsize;
      int len = 0;
      char *p = imp.attvalue;
      for(; size > 0; p += len, size -= len) {
	if ((len = read(fd, p, size)) < 0) {
	  perror("");
	  fatal_error("failed when reading from ", filename);
	}
      }
    }
  }

  if (CO_CREATEOBJ(&(curobj->obj)) ||
      SI_SETIMAGE(&(curobj->obj), &imp))
    fatal_error("Couldn't create object ", imp.attvalue);
  if (imp.attvalue != NULL)
    free((FREEPTR *)imp.attvalue);
  END_ACTION;
}


/*  */
/**********************************************************************
 * Function: static ACTION(beginlinks)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(beginlinks)
{
  char *objid;

  if (grp != (char *)NULL)
    free(grp);
  grp = popdata();
  objid = popdata();
  if (!(curobj = Findobjectsymbol(objid)))
    fatal_error("Undefined object symbol:", objid);
  free(objid);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(beginattrs)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(beginattrs)
{
  (void)beginlinks();			/* Action is the same */
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(defuser)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(defuser)
{
  pushdata((char *) strdup(LL_loginuser));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(defattr)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(defattr)
{
  char *s;

  s = popdata();
  if (fld != (char *)NULL)
    free(fld);
  fld = popdata();
  put_cmd(s, (label *)NULL, 0, 0);
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(ltag)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(ltag)
{
  if (fld != (char *)NULL)
    free(fld);
  fld = popdata();
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(boundlink)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(boundlink)
{
  pushdata(strdup("1"));
  END_ACTION;
}

/* ^L */
/**********************************************************************
 * Function: static ACTION(unboundlink)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(unboundlink)
{
  pushdata(strdup("0"));
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static ACTION(deflinks)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static ACTION(deflinks)
{
  int n;
  int bound;
  char *objid;
  objectsymbol *p;

  for (n = popint(); n > 0; n--) {
    objid = popdata();
    bound = popint();
    if (!(p = Findobjectsymbol(objid)))
      fatal_error("Undefined object symbol:",objid);
    free(objid);
    put_cmd((char *)NULL, &(p->obj), n, bound);
  }
  END_ACTION;
}

/*  */
/**********************************************************************
 * Function: static void put_cmd(char *val,label *lbl,int pos,int bound)
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void put_cmd(val,lbl,pos,bound)
  char *val;
  label *lbl;
  int pos;
  int bound;
{
  cmd_rec *cmd;

  if (!(cmd = new(cmd_rec)))
    fatal_error("put_cmd:", "out of memory");
  cmd->next = curobj->cmd;
  cmd->group = strdup(grp);
  cmd->field = strdup(fld);
  cmd->pos = pos;
  cmd->val.attvalue = (char *)NULL;
  cmd->val.attsize = 0;
  cmd->lbl.vs = -1;
  cmd->lbl.inst = -1;
  cmd->ix = 0;
  if (val) {
    cmd->ix = 1;
    cmd->val.attvalue = val; /* NOTE: not copied */
    cmd->val.attsize = strlen(val)+1;
  }
  if (lbl) {
    cmd->ix = 2;
    copylbl(&(cmd->lbl),lbl);
    if (!bound)
      (void) UBL_UNBINDLABEL(&(cmd->lbl));
  }
  curobj->cmd = cmd;
}

/*  */
/**********************************************************************
 * Function: static void flush_cmd(objectsymbol *os)
 *
 * Modifications:
 *      Martin Sjlin. Added free of cmd->val.attvalue if non NULL
 *      <list mods with name and date>
 */
static void flush_cmd(os)
  objectsymbol *os;
{
  cmd_rec *cmd;

  for (cmd = os->cmd; cmd; cmd = os->cmd) {
    os->cmd = cmd->next;
    switch (cmd->ix) {
    case 1:
      if (SA_SETATTR(&(os->obj), cmd->group, cmd->field, &(cmd->val)))
	fatal_error("Couldn't set attribute:", cmd->field);
      break;
    case 2:
      if (LINK(&(os->obj), cmd->group, cmd->field, &(cmd->lbl), cmd->pos))
	fatal_error("Couldn't set link:",fld);
      break;
    default:
      break;
    }
    free(cmd->group);
    free(cmd->field);
    if (cmd->val.attvalue != NULL)
      free((FREEPTR *) cmd->val.attvalue);
    free((FREEPTR *)cmd);
  }
}

/*  */
/**********************************************************************
 * Function: static void arginit(int argc, char **argv)
 *
 * processes the command line options.
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void arginit(argc, argv)
  int argc;
  char **argv;
{
  int c = 0, user = 0, pword = 0;
  char *db;

  (void)memset((char *)dbdir, 0, (int)sizeof(dbdir));
  (void)memset((char *)filename, 0, (int)sizeof(filename));
  (void)memset((char *)username, 0, (int)sizeof(username));
  (void)memset((char *)password, 0, (int)sizeof(password));

  while ((c = getopt(argc, argv, "f:u:p:vhc")) != EOF) {
    switch (c) {
    case 'u':
      (void)strcpy(username, optarg);
      user = 1;
      noninteractive = 1;
      break;
    case 'p':
      (void)strcpy(password, optarg);
      pword = 1;
      break;
    case 'f':
      (void)strcpy(filename, optarg);
      break;
    case 'h':
      print_version();
      usage();
      break;
    case 'c':
      parse_only = 1;
      break;
    case 'v':
      traceinput(1);
      print_version();
      break;
    default:
      break;
    }
  }
  if (pword != user) {
    (void)fprintf(stderr, "Must use both -u and -p or neither.\n");
    usage();
  }

  switch (argc - optind) {
  case 0:
    if ((db = (char *)getenv("LINCKSDBDIR")) == (char *)NULL)
      usage();
    (void)strcpy(dbdir, db);
    break;
  case 1:
    (void)strcpy(dbdir, argv[argc - 1]);
    break;
  default:
    usage();
    break;
  } 
}

/*  */
/**********************************************************************
 * Function: static void usage()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void usage()
{
  (void)fprintf(stderr, "%s\n%s\n",
			"Usage: t2lincks [-h] [-v] [-u username -p password]", 
			"       [-f t2lincks-file] [-c] database-dir");
  exit(1);
}

/*  */
/**********************************************************************
 * Function: static FILE *do_login()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static FILE *do_login()
{
  FILE *f;

  if (noninteractive) {
    if (NOPROMPT_LOGIN(username, password, dbdir) != SUCCESS) {
      (void)fprintf(stderr, "NOPROMPT_LOGIN: login failure\n");
      exit(-1);
    }
    (void)promptfile(NULL);
  } else {
    if (LOGIN(dbdir) != SUCCESS) {
      (void)fprintf(stderr, "LOGIN: login failure\n");
      exit(-1);
    }
  }

  if (filename[0] == '\0') {
    (void)fprintf(stderr, "No input file.  Reading from stdin.\n");
    if ((f = fdopen(fileno(stdin), "r")) == (FILE *) NULL) {
      perror("Failed to open stdin");
      exit(-1);
    }
  } else {
    if ((f = fopen(filename, "r")) == (FILE *) NULL) {
      (void)fprintf(stderr, "Failed to open file %s: ", filename);
      perror("");
      exit(-1);
    }
  }

  return f;
}

#define VERSION "1.3"
#define DATE "1994-06-01"
#define VERSION_INFO "\n"

/*  */
/**********************************************************************
 * Function: static void print_version()
 *
 * Modifications:
 *      <list mods with name and date>
 */
static void print_version()
{
  (void)fprintf(stdout, "\tt2lincks version %s (%s)\n%s", VERSION, DATE,
		VERSION_INFO);

}


