/*-------------------------------------------------------------------------
 *
 * pgresult.cc--
 *
 *
 *   DESCRIPTION
 *      implementation of the PGresult class.
 *   PGresult encapsulates the results returned from the backend
 *   (PGresult is a replacement for the old "PortalBuffer")
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 * Copyright (c) 1994, Regents of the University of California
 *
 *
 * IDENTIFICATION
 *    /usr/local/devel/pglite/cvs/src/libpq++/pgresult.cc,v 1.2 1995/03/11 01:01:29 jolly Exp
 *
 *-------------------------------------------------------------------------
 */
#include <stdlib.h>
#include <stdio.h>
#include "libpq++.H"

// the tuples array in a PGresGroup  has to grow to accommodate the tuples
// returned.  Each time, we grow by this much:
#define TUPARR_GROW_BY 100

// keep this in same order as ExecStatusType in libpq++.H
char* pgresStatus[] = {
  "PGRES_FATAL_ERROR",
  "PGRES_BAD_CONNECTION",
  "PGRES_BAD_QUERY",
  "PGRES_EMPTY_QUERY",
  "PGRES_NO_RESPONSE",
  "PGRES_BAD_RESPONSE",
  "PGRES_NONFATAL_ERROR",
  "PGRES_COMMAND_OK",
  "PGRES_TUPLES_OK"
};

PGresult::PGresult(ExecStatusType status)
{
  ntups = 0;
  ngrps = 0;
  pname = NULL;
  resultStatus = status;
}

PGresult::PGresult(FILE* f, PGconn* connection, 
		   char* portalName)
{
  char buffer[MAX_MESSAGE_LEN];
  int id;
  PGresGroup* group = NULL;

  int numTuples;
  int i,j;
  int done = 0;

  conn = connection;
  Pfin = f;

  groups = NULL;

  pname = dupstr(portalName);

  // this constructor should only be called when the 
  // id of the stream is 'T' to start with
  id = 'T';

  // temp for counting
  numTuples  = 0;

  // ntups and ngrps are member data fields
  ntups = 0;
  ngrps = 0;

  // process the data stream until we're finished
  while(!done) {
    switch (id) {
    case 'T': // a new tuple group
      // if this is not the first group, record the number of tuples
      // in the previous group
      if (group != NULL) {
	group->numTuples = numTuples;
	ntups += numTuples;
	numTuples = 0;
      }
      group = addGroup();

      int nfields;
      // the next two bytes are the number of fields 
      if (pqGetInt(&nfields, 2, Pfin) == 1) {
	sprintf(conn->errorMessage,
		"could not get the number of fields from the 'T' message\n");
	resultStatus = PGRES_BAD_RESPONSE;
	return;
      }
      else
	group->numAttributes = nfields;

      if (nfields > 0) {
	group->attDescs = (PGresAttDesc*) malloc(nfields * sizeof(PGresAttDesc));
	// get type info
	for (i=0;i<nfields;i++) {
	  char typName[MAX_MESSAGE_LEN];
	  int adtid;
	  int adtsize;
	  
	  if ( pqGets(typName, MAX_MESSAGE_LEN, Pfin) ||
	       pqGetInt(&adtid, 4, Pfin) ||
	       pqGetInt(&adtsize, 2, Pfin)) {
	    sprintf(conn->errorMessage,
		    "error reading type information from the 'T' message\n");
	    resultStatus = PGRES_BAD_RESPONSE;
	    return;
	    }

	  
	  group->attDescs[i].name = dupstr(typName);
	  group->attDescs[i].adtid = adtid;
	  group->attDescs[i].adtsize = adtsize;
	}
      }
      else
	group->attDescs = NULL;
      break;
    case 'B': // a tuple in binary format
    case 'D': // a tuple in ASCII format
	if (group == NULL) {
	    sprintf(conn->errorMessage,
		    "type information ('T' messages) missing from backend\n");
	    resultStatus = PGRES_BAD_RESPONSE;
	    return;
	  }
      PGresAttValue* newTup;
      newTup = getTuple(group, (id == 'B'));
      if (newTup == NULL) {
	resultStatus = PGRES_BAD_RESPONSE;
	return;
	}

      addTuple(group,newTup);
      numTuples++;
      break;
    case 'A': // async portal, not supported
      sprintf(conn->errorMessage,
	      "Asynchronous portals not supported");
      resultStatus = PGRES_NONFATAL_ERROR;
      return;
      break;
    case 'C': // end of portal tuple stream
      group->numTuples = numTuples;
      ntups += numTuples;
      char command[MAX_MESSAGE_LEN];
      pqGets(command,MAX_MESSAGE_LEN, Pfin); // read the command tag
      done = 1;
      break;
    case 'E': // errors
      if (pqGets(conn->errorMessage, MAX_ERRMSG_LEN, Pfin) == 1) {
	sprintf(conn->errorMessage,
		"Error return detected from backend, but error message cannot be read");
      }
      resultStatus = PGRES_FATAL_ERROR;
      return;
      break;
    case 'N': // notices from the backend
      if (pqGets(conn->errorMessage, MAX_ERRMSG_LEN, Pfin) == 1) {
	sprintf(conn->errorMessage,
	"Error return detected from backend, but error message cannot be read");
    }
    else
      fprintf(stderr, "%s\n", conn->errorMessage);
      break;
    default: // uh-oh
	// this should never happen but frequently does when the 
	// backend dumps core
      sprintf(conn->errorMessage,"FATAL:  unexpected results from the backend, it probably dumped core.");
      fprintf(stderr, conn->errorMessage);
      resultStatus = PGRES_FATAL_ERROR;
      return;
      break;
    }
    if (!done)
      id = getc(Pfin);
  } // while (1)

  resultStatus = PGRES_TUPLES_OK;
  return;
}

PGresult::~PGresult()
{
  int i,j,k;

  if (pname)
    free(pname);

  for (i=0;i<ngrps;i++) {
    for (j=0;j<groups[i].numAttributes;j++) {
	if (groups[i].attDescs[j].name)
	    free(groups[i].attDescs[j].name);
    }
    free(groups[i].attDescs);
    for (j=0;j<groups[i].numTuples;j++)  {
	for (k=0;k<groups[i].numAttributes;k++) {
	    if (groups[i].tuples[j][k])
		free (groups[i].tuples[j][k]);
	}
	free (groups[i].tuples[j]);
    }
}
}

//
// add a new element to the groups array
// side-effects:  changes groups and ngrps
//                

PGresGroup*
PGresult::addGroup()
{
  int i;

  ngrps++;
  if (groups)
 // realloc because shallow copying of the PGresGroup structure is ok
    groups = (PGresGroup*) realloc(groups, ngrps);
  else
    groups = (PGresGroup*) malloc(ngrps * sizeof(PGresGroup));

  groups[ngrps-1].numTuples = 0;
  groups[ngrps-1].numAttributes = 0;
  groups[ngrps-1].attDescs = NULL;
  groups[ngrps-1].tuples = NULL;
  groups[ngrps-1].tupArrSize = 0;
  
  return &(groups[ngrps-1]);
}

// get the next tuple from the stream
// and add it to the current group
PGresAttValue*
PGresult::getTuple(PGresGroup* group, int binary)
{
  char bitmap[MAX_FIELDS]; // the backend sends us a bitmap of 
                           // which attributes are null
  int bitmap_index = 0;
  int i;
  int nbytes;              // the number of bytes in bitmap 
  int len;
  char 	bmap;		   //  One byte of the bitmap */
  int 	bitcnt = 0; 	   // number of bits examined in current byte */
  int 	vlen;		   // length of the current field value

  PGresAttValue* res;

  int nfields = group->numAttributes;

  res = (PGresAttValue*) malloc(nfields * sizeof(PGresAttValue));

  nbytes = nfields / BYTELEN;
  if ( (nfields % BYTELEN) > 0)
    nbytes++;
  
  if (pqGets(bitmap, nbytes, Pfin) == 1){
      sprintf(conn->errorMessage,
	      "Error reading null-values bitmap from tuple data stream\n");
      return NULL;
    }

  bmap = bitmap[bitmap_index];
  
  for (i=0;i<nfields;i++) {
    if (!(bmap & 0200))
	// if the field value is absent, nullify it
	res[i] = NULL;
    else {
      // get the value length (the first four bytes are for length)
      pqGetInt(&vlen, 4, Pfin);
      if (binary == 0)
	vlen -= 4;
      // allocate storage for the value, add 1 for '\0';
      res[i] = (PGresAttValue) malloc(vlen + 1);
      // read in the value;
      pqGets((char*)res[i], vlen, Pfin);
      res[i][vlen] = '\0';
    }
    // get the appropriate bitmap
    bitcnt++;
    if (bitcnt == BYTELEN) {
      bitmap_index++;
      bmap = bitmap[bitmap_index];
      bitcnt = 0;
    } else
      bmap <<= 1;
  }

  return res;
}

//
// add a tuple a group
// updates the internal fields of the group

void
PGresult::addTuple(PGresGroup* group, PGresAttValue* tup)
{
  int i;
  PGresAttValue* *newTuples;
  int newsize;

    
  if (group->numTuples == group->tupArrSize) {
    // grow the array
    group->tupArrSize += TUPARR_GROW_BY;
    
    if (group->numTuples == 0)
      group->tuples = (PGresAttValue**) malloc(group->tupArrSize * sizeof(PGresAttValue*));
    else
    // we can use realloc because shallow copying of the structure is okay
      group->tuples = (PGresAttValue**) realloc(group->tuples, group->tupArrSize * sizeof(PGresAttValue*));
    }

  group->tuples[group->numTuples] = tup;
  group->numTuples++;
  
}

