%{
/*----------------------------------------------------------------------*
** File: pql.l
** 
** 
** Project:	PQL
** =====================================================================*
** Description:	PQL (Subset II) - lexical analyser
**
**  Author:	Bjoern Lemke
**  E-Mail:	lemke@lf.net
**
** Copyright (c) 1994 Bjoern Lemke.
**----------------------------------------------------------------------*/

/* include this first, FD_SETSIZE must be defined */
#include "eng.h"

#include <fcntl.h>
#include <string.h>
#include <readline/readline.h>
#include <readline/history.h>

#include "pql_conf.h"
#include "pql_parse.h"

#define MAXPQLBUF 1000
#define IBUFSIZE 100

enum PqlMode { FILEMODE, INTERACTMODE, STDINMODE };
enum IMode  { FILEIN, LINEIN, STDIN, ARGIN };

char histfile[NAMELEN];
char actcomp[3];
char actname[NAMELEN];
int actintval;
int actkey;
int acttype;
float actfloatval;
double actdoubleval;
char actstrval[STRINGLEN];
char actvar[NAMELEN];
char tpath[NAMELEN];

/* command line argument vars */
int actarg, gargc;
char *argptr, **gargv;


int lineno;
void yyerror(char *s);

/* input vars */
int pqlmode;
int imode=0;
int hidemode;
int ifd;
int num=0;
char buffer[IBUFSIZE];
char *bufptr=buffer;
int endofinput=0;

/* got input from pqlstring */
char pqlbuf[MAXPQLBUF];
char *pqlptr;

#ifdef FLEX
#undef YY_INPUT
#define YY_INPUT(b, r, m) \
{ \
   int c=myinput(); \
   r = (c == 0) ? YY_NULL : (b[0] = c,1); \
}
#undef unput
#define unput myunput
#endif /* FLEX */

#ifdef ATT_LEX
#undef input
#define input myinput
#undef unput
#define unput myunput
#endif /* ATT_LEX */

int expand(char *string);
int getnum(char *num);
extern char *getvar(char*);

%}

%%

all	{ return ALL; } /* literal keywords */ 
and	{ return AND; }
avg	{ return AVG; } 
as      { return AS; }
min 	{ return MIN; }
max 	{ return MAX; }
sum     { return SUM; }
count	{ return COUNT; }
create  { return CREATE; }
any	{ return ANY; }
asc	{ return ASC; }
by	{ return BY; }
count	{ return COUNT; }
delete	{ return DELETE; }
desc	{ return DESC; }
distinct { return DISTINCT; }
env     { return ENV; }
exit    { return EXIT; }
exists	{ return EXISTS; }
extend  { return EXTEND; }
from	{ return FROM; }
get     { return GET; }
group	{ return GROUP; }
having	{ return HAVING; }
in	{ return IN; }
insert	{ return INSERT; }
into	{ return INTO; }
invoke  { return INVOKE; }
is      { return IS; }
like	{ return LIKE; }
list    { return LIST; }
not	{ return NOT; }
NULL	{ return NULLVAL; }
or	{ return OR; }
order   { return ORDER; }
print   { return PRINT; }
select	{ return SELECT; }
set	{ return SET; }
shrink  { return SHRINK; }
stats   { return STATS; }
to	{ return TO; }
update	{ return UPDATE; }
values	{ return VALUES; }
where	{ return WHERE; }
begin   { return TABEGIN; }
commit  { return TACOMMIT; }
abort   { return TAABORT; }
reset   { return RESET; }

char(acter)?	{ acttype=T_CHAR; return FTYPE; }
int(eger)?	{ acttype=T_LONG; return BTYPE; }
float	        { acttype=T_FLOAT; return BTYPE; }
double          { acttype=T_DOUBLE; return BTYPE; }
primary         { actkey=PKEY; return KEY; }
ordinary        { actkey=ORDY; return KEY; }

"="	| /* comparison */
"<>"	|
"<"	|
">"	|
"<="	|
">="	{ strcpy(actcomp, yytext); return COMPARISON; }


[-+*/:(),.;]	{ return yytext[0]; } /* special symbols */

[A-Za-z/][A-Za-z0-9_/]*	{ strcpy(actname, yytext); return NAME; } 


([+-])?[0-9]+"."[0-9]+ { sscanf(yytext, "%f", &actfloatval);
                                            return FLOATNUM; } /* (restricted) */

([+-])?([0-9]+)"."([0-9]+)("f"|"F") { sscanf(yytext, "%lf", &actdoubleval); return DOUBLENUM; } /* (restricted) */


([+-])?([0-9])|([1-9][0-9]+)	{ actintval=atoi(yytext); return INTNUM; } 


'[^'\n]*' { 
	int c = myinput();
	int i=0;

	unput(c);
	
	if (c !='\'') {

		while (yytext[i+1]!='\'') { /* cut off " ' " */
			yytext[i]=yytext[i+1];
			i++;
		}
		yytext[i]='\0';
		strcpy(actstrval, yytext);
                if (expand(actstrval) == -1)
                  return(1);
		return(STRING);
	}
	else
	 	yymore();
	}

\n	{ lineno++; } /* whitespaces */
[ \t\r]	;


"#".*	; /* comments */ 

((\$[0-9])|(\$[1-9][0-9]))  { actarg=atoi(yytext+1);
                             if (actarg >= gargc) {
				fprintf(stderr, "PQL-ERROR: illegal command line argument scanned\n");
				return(1);
			     }
		             argptr=gargv[actarg];
			     hidemode = imode;
                             imode = ARGIN; }

.	{ yyerror("invalid character");  /* anything else */
          return(1); }
%%

void yyerror(char *s)
{
   fprintf(stderr, "PQL-ERROR: Line %d: %s at \"%s\"\n", lineno, s, yytext);
}

main(int argc, char **argv)
{
  char *sign;
  /* char *inbuf; */
  char *inbuf = NULL;
  int offset;
  int complete;
  int i, c;
  char inputfile[NAMELEN], *hptr, hstring[NAMELEN];

  umask(0);

  loadvars();
  init_htable();

  sprintf(histfile, "%s/%s", getenv("HOME"), HISTFILE);

  /* propagte argv and argc */
  gargv = argv;
  gargc = argc;

  if (isatty(fileno(stdin))) {
     pqlmode=INTERACTMODE;
     if (read_history(histfile) != 0) 
       perror("Cannot open history file");
  } else {
     pqlmode=STDINMODE;
  }

  for (i=1; i < argc; i++) {
     if (argv[i][0] == '-') {
       switch (argv[i][1]) {
       case 'f':
	 if (pqlmode == FILEMODE) {
	 	fprintf(stderr, "PQL-ERROR: multiple input modes specified\n");
		exit(1);
	 }
         i++;
         if (i >= argc) {
	        fprintf(stderr, "PQL-ERROR: illegal command line argument scanned\n");
		exit(1);
	 }
	 strcpy(inputfile, argv[i]);
	 pqlmode = FILEMODE;
	 break;
       case 'h':
	 fprintf(stderr, "usage: %s [-f inputfile] [arg1, arg2 ..]\n", argv[0]);
	 exit(0);
       default:
	 fprintf(stderr, "unknown option %c, use -h for help\n", argv[i][1]);
	 exit(0); 
       }
     }
   }

  switch (pqlmode) {
  case FILEMODE:
    /* read input from file */
    if ((ifd = open(inputfile, O_RDONLY)) == -1) {
      perror("open");
      exit(1);
    }
    imode = FILEIN;
    while ((c=myinput()) != 0) {
         myunput(c);
         if(yyparse() == 1) {
           fprintf(stderr, "PQL-ERROR: transaction aborted\n");
	   exit(1);
         }
	 pql_cleanup();     
    }
    break;
  case INTERACTMODE:     /* read input interactive */
    
    rl_readline_name = "pql:";
    imode = LINEIN;

    while(1) {    
      pqlptr=pqlbuf; /* reset scanner */
      complete=0;
      if (inbuf != NULL) free(inbuf);
      inbuf = readline("PQL > ");
      if (inbuf && *inbuf) {
	add_history(inbuf);
	offset=0;
	
	/* buffering until ';' is found */
	do {
	  sign  = inbuf;
	  strcpy(pqlbuf+offset, inbuf);
         
	  while (*sign != ';' && *sign != '\0') {
	    offset++;
	    sign++;
	  }
	  if (*sign == ';') {
#ifdef FLEX
            yyrestart(yyin);
#endif FLEX
	    if(yyparse() == 1)
	      fprintf(stderr, "PQL-ERROR: PQL operation failed\n");      
       	    pql_cleanup();
	    complete=1;
	  } else {
	    strcpy(pqlbuf+offset, " ");
	    offset++;
            if (inbuf != NULL) free(inbuf);
	    inbuf = readline("+ ");
	  }
	} while (!complete);
      }
    }
    break;
  default:
    /* read input from stdin */
    ifd = 0;
    imode = STDIN;
    while ((c=myinput()) != 0) {
      if (c != '\n' && c != '\t' && c != ' ') { /* strip off whitespaces */
	myunput(c);
	if(yyparse() == 1) {
          fprintf(stderr, "PQL-ERROR: transaction aborted\n");
	  exit(1);
	}
	pql_cleanup();
      }
    }
    break;    
  }
}

int myinput()  
{
  switch (imode) {
  case LINEIN:
    return(*pqlptr++);
  case FILEIN:
  case STDIN:
    if (bufptr < buffer + num) {
      return((int)*bufptr++);
    } else {
      num = read(ifd, buffer, IBUFSIZE);
      if (num == 0) {
	endofinput = 1;
	return(0);
      }
      bufptr=buffer;
      return((int)*bufptr++);
    }
  case ARGIN:
    if (*(argptr+1) == '\0') /* return to original imode */
	imode=hidemode;
    return(*argptr++);
  }
}

int myunput(int c)
{
  if (c == 0) /* ignore */
    return(0);
  switch (imode) {
  case LINEIN:
    *--pqlptr = c;
    break;
  case FILEIN:
  case STDIN:
    if (bufptr > buffer)
      *--bufptr = c;
    else {
      bufptr = buffer + num - 1;
      *bufptr = c;
    }
    break;
  case ARGIN:
    *--argptr = c;
    break;
  }
  return(0);
}
	  
int expand(char *string)
{
  char buf[STRINGLEN];
  char *evar;
  char *scan;
  int idx;
  scan = string;
  evar = buf;

  while (*scan != '\0') {
    if (*scan == '$') {
      scan++;
      idx = getnum(scan);
      if (idx >= gargc) {
	 fprintf(stderr, "PQL-ERROR: illegal command line argument scanned\n");
	 exit(1);
      }
      if (idx < 10) 
	scan++;
      else 
	scan = scan+2; /* we support cla < 100 */
      strcpy(evar, gargv[idx]); /* expand here */
      evar = evar + strlen(gargv[idx]);
    } else {
      *evar = *scan;
      evar++;
      scan++;
    }
    if (evar-buf >= STRINGLEN || scan-string >= STRINGLEN) {
      fprintf(stderr, "PQL-ERROR: string buffer exceeded.\n");
      return(-1);
    }
  }
  *evar='\0';
  strcpy(string, buf);
  return(0);
}


int getnum(char *num)
{
  char numbuf[5];
  char *nptr;

  nptr = numbuf;

  while (*num >= '0' && *num <= '9') {
    *nptr=*num;
    nptr++;
    num++;
  }
  *nptr='\0';
  
  return(atoi(numbuf));
}
