
/* qddb/Lib/LibQddb/BuildDB.c
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* EXPORTED:
 *	void Qddb_BuildDatabase(Schema *)
 */

static void ResetNextNumber _ANSI_ARGS_((char *));

void Qddb_BuildDatabase(RelationSchema)
    Schema		*RelationSchema;
{
    char		*RelationName = RelationSchema->RelationName;
    int			RelationFile, TmpFile;
    char		RelationFN[MAXFILENAMELEN], ChangeFN[MAXFILENAMELEN];
    char		TmpFN[MAXFILENAMELEN], ChangeTmpFN[MAXFILENAMELEN];
    char		NewTmpFN[MAXFILENAMELEN];
    char		buf[MAXFILENAMELEN], dirnam[MAXFILENAMELEN];
    char		KeyFN[MAXFILENAMELEN], FirstLine[128];
    char		*TmpBuf;
    Entry 		ThisEntry = (Entry)NULL;
    FILE		*KeyFilePtr, *fopen();
    int			Start, EntryNumber, tmpint;
    size_t		Length;
    DIR			*dp, *opendir();
    struct dirent	*dirp;

    strcpy(RelationFN, RelationName);
    strcat(RelationFN, "/Database");
    RelationFile = RelationSchema->database_fd;
    lseek(RelationFile, (off_t)0, 0);
    strcpy(ChangeFN, RelationName);
    strcat(ChangeFN, "/Changes/");
    strcpy(TmpFN, RelationName);
    strcat(TmpFN, "/.Rtmp");
    if ((TmpFile = Open(TmpFN, O_RDWR|O_CREAT|O_TRUNC, 0644)) == -1) {
	fprintf(stderr, "Cannot create file %s\n", TmpFN);
	exit(1);
    }
    strcpy(KeyFN, RelationName);
    strcat(KeyFN, "/Database.key");
    if ((KeyFilePtr = fopen(KeyFN, "r")) == NULL) {
	fprintf(stderr, "Relation %s does not have a key file.\n", RelationName);
	exit(1);
    }
    Qddb_InitReducedAttrTable(RelationSchema);
    /* Read the Database, noting each invalid entry.  An invalid entry
     * number X is:
     *	1. A changed entry if the file Changes/X exists.
     *	2. A deleted entry otherwise.
     * The algorithm for updates is a simple sequential pass over
     * the Database file, copying changes as it goes and making additions
     * at the end of the file.
     */
    EntryNumber = 1;
    while (fscanf(KeyFilePtr, "%d %d\n", &Start, &tmpint) > 0) {
	Length = tmpint;
	Qddb_ReadEntry(RelationFile, &ThisEntry, (off_t)Start, (size_t)Length, False);
	Qddb_ReducedAttrToFullAttr(RelationSchema, ThisEntry);
	if (EntryInvalid(ThisEntry) == True) {
	    int				ThisEntryNumber;
	    int				fd;
	    
	    ThisEntryNumber = GetEntryNumber(ThisEntry);
	    sprintf(ChangeTmpFN, "%s/%d", ChangeFN, ThisEntryNumber);
	    if ((fd=EntryChanged(ChangeTmpFN, &Length)) != -1) {
		/* Read the entire changed entry.
		 */
		Qddb_ReadEntry(fd, &ThisEntry, (off_t)0, (size_t)Length, False);
		close(fd);
	    } else {
		if (access(ChangeTmpFN, F_OK) != -1) {
		    fprintf(stderr, "Error: %s file unreadable or unwritable\n", ChangeTmpFN);
		    fflush(stderr);
		    exit(1);
		}
		fprintf(stderr, "Deleting %d\n", ThisEntryNumber);
		continue;
	    }
	}
	sprintf(FirstLine, "\n%%0 V %d", EntryNumber++);
	TmpBuf = ThisEntry[0];
	ThisEntry[0] = FirstLine;
	Qddb_FullAttrToReducedAttr(RelationSchema, ThisEntry+1);
	WriteEntry(RelationSchema, TmpFile, ThisEntry);
	ThisEntry[0] = TmpBuf;
    }
    /* Add the entries in the Additions directory to the Database.
     */
    strcpy(ChangeFN, RelationName);
    strcat(ChangeFN, "/Additions/");
    dp = opendir(ChangeFN);
    while ((dirp = readdir(dp)) != NULL) {
	int		fd;
	size_t		size, dirlen;
	
	dirlen = NLENGTH(dirp);
	if (dirlen == 0 || dirp->d_name[0] == '.' || dirp->d_name[0] == 'N')
	    continue;
	bcopy(dirp->d_name, dirnam, dirlen);
	dirnam[dirlen] = '\0';
	sprintf(ChangeTmpFN, "%s%s", ChangeFN, dirnam);
	if (access(ChangeTmpFN, F_OK|R_OK) == -1) {
	    fprintf(stderr, "Error: %s file doesn't exist or is unreadable.\n", ChangeTmpFN);
	    fprintf(stderr, "PANIC: make files readable and try again.\n");
	    fflush(stderr);
	    exit(1);
	}
	fd = Open(ChangeTmpFN, O_RDONLY, 0);
	if ((size = SizeOfFile(fd)) == 0) {
	    continue;
	}
	Qddb_ReadEntry(fd, &ThisEntry, (off_t)0, size, False);
	sprintf(FirstLine, "\n%%0 V %d", EntryNumber++);
	TmpBuf = ThisEntry[0];
	ThisEntry[0] = FirstLine;
	Qddb_FullAttrToReducedAttr(RelationSchema, ThisEntry+1);
	WriteEntry(RelationSchema, TmpFile, ThisEntry);
	ThisEntry[0] = TmpBuf;
	close(fd);
    }
    Qddb_Free(QDDB_TYPE_ENTRY, ThisEntry);
    Write(TmpFile, "\n", 1);
    Close(TmpFile);
    Close(RelationSchema->database_fd);
    /* You need to make sure the write of the reduced attribute
     * table succeeds before moving the new Database file to
     * the permanent location.
     */
    if (RelationSchema->UseReducedAttributeIdentifiers == True) {
	if (Qddb_WriteReducedAttrTable(RelationSchema) < 0) {
	    fprintf(stderr, "Write of RedAttrIndex failed, old Database not removed\n");
	    PANIC("Error writing RedAttrIndex");
	}
    }

    /* Move the temporary file to the real location.
     */
    strcpy(NewTmpFN, RelationFN);
    strcat(NewTmpFN, ".old");
#if defined(HAVE_RENAME)
    if (rename(RelationFN, NewTmpFN) == -1) {
	PANIC("Rename failed...\nCall your System Administrator\n");
    }	    
    if (rename(TmpFN, RelationFN) == -1) {
	PANIC("Rename failed...\nCall your System Administrator\n");
    }
    if (unlink(NewTmpFN) == -1) {
	fprintf(stderr, "Failed to remove %s file.  Contact your system administrator\n", NewTmpFN);
    }
#else
    if (link(RelationFN, NewTmpFN) == -1) {
	fprintf(stderr, "Link of %s to %s failed, database not modified\n",
		RelationFN, NewTmpFN);
	PANIC("Please remedy situation and try again");
    }
    if (unlink(RelationFN) == -1) {
	fprintf(stderr, "Unlink of %s failed, database not modified\n",
		RelationFN);
	unlink(NewTmpFN);
	PANIC("Please remedy situation and try again");
    }
    if (link(TmpFN, RelationFN) == -1) {
	fprintf(stderr, "Link of %s to %s failed, database not modified\n",
		TmpFN, RelationFN);
	if (link(NewTmpFN, RelationFN) == 0) { /* try it anyway :-) */
	    fprintf(stderr, 
"This really shouldn't happen.... but here's an explanation of your problem\n\
\n\
I tried to link the temporary database file %s to the real database file\n\
%s and it failed.  But I was able to relink the old one, so your database\n\
is in exactly the same format as it was in the first place.  Contact your\n\
local Qddb guru to fix further.\n", TmpFN, RelationFN);
	} else { /* failed as expected */
	    fprintf(stderr, 
"This really shouldn't happen.... but here's an explanation of your problem\n\
\n\
I tried to link the temporary database file %s to the real database file\n\
%s and it failed.  I was not able to relink the old one, so your database\n\
is in an inconsistent state.  You must manually 'mv %s %s' to fix the\n\
problem.   Contact your local Qddb guru for further assistance.\n", 
NewTmpFN, RelationFN);
	}
	fprintf(stderr, "Please remedy situation and try again\n");
	exit(1);
    }
    if (unlink(TmpFN) == -1)
	fprintf(stderr, "Unlink of %s failed, please remove manually\n",
		TmpFN);
    unlink(NewTmpFN);
#endif
    RelationSchema->database_fd = OpenDatabase(RelationSchema->RelationName, 0);
    /* Delete all entries from Additions */
    strcpy(ChangeFN, RelationName);
    strcat(ChangeFN, "/Additions/");
    if ((dp=opendir(ChangeFN)) == NULL)
	fprintf(stderr, "Warning: could not delete Additions/[0-9]*\n");
    else {
	size_t		dirlen;
	char		buf[BUFSIZ];
	
	while ((dirp = readdir(dp)) != NULL) {
	    dirlen = NLENGTH(dirp);
	    if (dirlen == 0 || dirp->d_name[0] == '.' || dirp->d_name[0] == 'N')
		continue;
	    bcopy(dirp->d_name, dirnam, dirlen);
	    dirnam[dirlen] = '\0';
	    strcpy(buf, ChangeFN);
	    strcat(buf, dirnam);
	    if (unlink(buf) == -1) {
		fprintf(stderr, "Warning, could not remove %s\n", buf);
		fflush(stderr);
	    }
	}
    }
    closedir(dp);
    /* Delete all the Changes */
    strcpy(ChangeFN, RelationName);
    strcat(ChangeFN, "/Changes/");
    if ((dp=opendir(ChangeFN)) == NULL)
	fprintf(stderr, "Warning: could not delete Additions/[0-9]*\n");
    else {
	size_t		dirlen;

	while ((dirp = readdir(dp)) != NULL) {
	    dirlen = NLENGTH(dirp);
	    if (dirlen == 0 || dirp->d_name[0] == '.')
		continue;
	    bcopy(dirp->d_name, dirnam, dirlen);
	    dirnam[dirlen] = '\0';
	    strcpy(buf, ChangeFN);
	    strcat(buf, dirnam);
	    if (unlink(buf) == -1) {
		fprintf(stderr, "Warning, could not remove %s\n", buf);
		fflush(stderr);
	    }
	}
    }
    fclose(KeyFilePtr);
    ResetNextNumber(RelationName);
    if (RelationSchema->ReducedAttributeTable != NULL) {
	Free(RelationSchema->ReducedAttributeTable[0]);
	Free(RelationSchema->ReducedAttributeTable);
    }
    if (RelationSchema->UseReducedAttributeIdentifiers == True) {
	RelationSchema->ReducedAttributeTable = Qddb_ReadReducedAttrTable(RelationSchema);
	Qddb_FreeReducedAttrTable();
    } else {
	char			redattrFN[MAXFILENAMELEN];

	strcpy(redattrFN, RelationSchema->RelationName);
	strcat(redattrFN, "/RedAttrIndex");
	unlink(redattrFN);
    }
}

static void ResetNextNumber(Path)
    char		*Path;
{
    char		tmpPath[BUFSIZ];
    int			File;

    strcpy(tmpPath, Path);
    strcat(tmpPath, "/Additions/NextEntry");
    if ((File = Open(tmpPath, O_RDWR|O_CREAT|O_TRUNC, 0644)) == -1)
	PANIC("Cannot open Additions/NextEntry file");
#if defined(RECORDLOCKING)
    if (LockSection(File, F_WRLCK, (off_t)0, (off_t)0, True) == -1)
	return;
#endif
    write(File, "1\n", 2);
#if defined(RECORDLOCKING)
    UnlockSection(File, (off_t)0, (off_t)0);
#endif
    close(File);
}

