
%{

/* qddb/Lib/LibQddb/SchemaParse.y
 *
 * Copyright (C) 1993, 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */

#include "Qddb.h"

/* Define yyparse() to be SchemaParse() so that we can have multiple parsers
 * in a single executable.
 */
int yyerror _ANSI_ARGS_((char *));
int Qddblex();

#define yyparse		SchemaParse
#define yylex   	Qddblex

static Schema	*CurrentSchema;

void Qddb_SchemaParse_Set(schema) 
    Schema	*schema;
{
    CurrentSchema = schema;
}

Schema *Qddb_SchemaParse_Get() 
{
    return CurrentSchema;
}

static int	CurrentLevel;
static int	CurrentAttribute;
static int	CurrentAttributeNumber;
static int	CurrentAncestors[MAXIMUM_NUMBER_OF_SCHEMA_LEVELS];

static void 	FillInSchemaEntry _ANSI_ARGS_((SchemaNode *, char *, char *, char *, \
					       char *, char *, char *, Boolean));
%}
%pure_parser
%expect 1	

%token STRING CSTRING
%token ALIAS VERBOSENAME
%token LPAREN RPAREN ASTERISK COMMA DOUBLEQUOTE SINGLEQUOTE EQUALS
%token BLANKLINE NUMBER TYPE DEFAULTVALUE FORMAT DATEFORMAT
%token HASHSIZE USE CACHED HASHING REDUCED ATTRIBUTE IDENTIFIERS
%token CACHESIZE
%token SEPARATORS EXCLUDE
%token INTEGER 

%type <String>	STRING CSTRING
%type <Integer> Identifier IdentifierLparen
%type <Integer> Options INTEGER
%type <Option> AttrOption AttrOptions

%union {
    char		*String;
    int			Integer;
    double		Real;
    attroption_t	*Option;
};


%%
SchemaFile:
    OptionList Record {
    } |
    Record {
    };

OptionList:
    Options OptionList {
    } |
    {
    };

Options:
    DATEFORMAT EQUALS CSTRING {
        Free(CurrentSchema->default_date_format);
        CurrentSchema->default_date_format = $3;
    } |
    HASHSIZE EQUALS INTEGER {
	CurrentSchema->HashSize = $3;
    } |
    CACHESIZE EQUALS INTEGER {
	CurrentSchema->CacheSize = $3;
    } |
    USE CACHED HASHING {
	CurrentSchema->UseCachedHashing = True;
    } |
    USE REDUCED ATTRIBUTE IDENTIFIERS {
	CurrentSchema->UseReducedAttributeIdentifiers = True;
    };
    

Record:
    IndividualRecord {
    } |
    IndividualRecord Record {
    };

IndividualRecord:
    Identifier {
	CurrentSchema->Entries[$1].IsLeaf = True;
    } |
    IdentifierLparen Record RPAREN {
	CurrentLevel--;
	CurrentAttributeNumber = CurrentAncestors[CurrentLevel]+1;
	CurrentSchema->Entries[$1].IsLeaf = False;
#ifdef DEBUG
fprintf(stderr, "ID ( : Name=%s; Alias=%s; IsLeaf=%s;\n",
	CurrentSchema->Entries[$1].Name,
	CurrentSchema->Entries[$1].Alias,
	CurrentSchema->Entries[$1].IsLeaf==True? "True":"False");
#endif
    } |
    Identifier ASTERISK {
	CurrentSchema->Entries[$1].AsteriskPresent = True;
	CurrentSchema->Entries[$1].IsLeaf = True;
    } |
    IdentifierLparen Record RPAREN ASTERISK {
	CurrentSchema->Entries[$1].AsteriskPresent = True;
	CurrentLevel--;
	CurrentAttributeNumber = CurrentAncestors[CurrentLevel]+1;
	CurrentSchema->Entries[$1].IsLeaf = False;
    };

IdentifierLparen:
    Identifier LPAREN {
	CurrentAncestors[CurrentLevel] = CurrentSchema->Entries[$1].Number;
	CurrentLevel++;
	$$ = $1;
	CurrentAttributeNumber = 1;
    };

Identifier:
    STRING {
	$$ = CurrentAttribute;
	FillInSchemaEntry(CurrentSchema->Entries+CurrentAttribute, 
			  $1, NULL, NULL, NULL, NULL, NULL, False);
    } | 
    STRING AttrOptions {
        char		*Name, *Alias, *VerboseName, *Type, *Format, *Separators;
        attroption_t    *ptr, *tmp;
        Boolean         exclude = False;

	$$ = CurrentAttribute;
        Name = Alias = VerboseName = Type = Format = Separators = NULL;
        for (ptr = $2; ptr != NULL; ptr = ptr->next) {
	    switch (ptr->option) {
            case VERBOSENAME:
		VerboseName = ptr->value;
		break;
	    case TYPE:
		Type = ptr->value;
		break;
	    case FORMAT:
		Format = ptr->value;
		break;
	    case SEPARATORS:
		Separators = ptr->value;
		break;
	    case ALIAS:
		Alias = ptr->value;
		break;
	    case EXCLUDE:
		exclude = True;
                break;
	    default:
		fprintf(stderr, "SchemaParse: internal error: cannot recognize attribute option %d\n", 
			ptr->option);
	    }
	}
        for (ptr = $2; ptr != NULL;) {
	    tmp = ptr->next;
	    Free(ptr);
	    ptr = tmp;
	}
	FillInSchemaEntry(CurrentSchema->Entries+CurrentAttribute, 
			  $1, Alias, VerboseName, Type, Format, Separators, exclude);
    };

AttrOptions:
    AttrOption {
        $$ = $1;
    } |
    AttrOption AttrOptions {
        $$ = $1;
        $$->next = $2;
    };

AttrOption:
    EXCLUDE {
        $$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = EXCLUDE;
        $$->value = NULL;
        $$->next = NULL;
    } |
    SEPARATORS CSTRING {
	$$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = SEPARATORS;
        $$->value = $2;
        $$->next = NULL;
    } |
    FORMAT CSTRING {
	$$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = FORMAT;
        $$->value = $2;
        $$->next = NULL;
    } |
    TYPE STRING {
	$$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = TYPE;
        $$->value = $2;
        $$->next = NULL;
    } |
    ALIAS STRING {
	$$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = ALIAS;
        $$->value = $2;
        $$->next = NULL;
    } |
    VERBOSENAME CSTRING {
	$$ = (attroption_t *)Malloc(sizeof(attroption_t));
        $$->option = VERBOSENAME;
        $$->value = $2;
        $$->next = NULL;
    };
%%

static int SchemaFormatCheck(form, val)
    char		*form, *val;
{
    regexp		*RegexpVal;

    RegexpVal = regcomp(form);
    if (RegexpVal == NULL)
	return 1;
    return regexec(RegexpVal, val);
}

static void FillInSchemaEntry(TheEntry, Name, Alias, VerboseName, Type, Format, Separators, Exclude)
    SchemaNode		*TheEntry;
    char		*Name, *Alias, *VerboseName, *Type, *Format, *Separators;
    Boolean		Exclude;
{
    int			i;
#ifdef DEBUG
fprintf(stderr, "Filling :%s:%s:%s:%s:\n", Name, Alias, VerboseName, Type);
#endif
    CurrentSchema->NumberOfAttributes++;
    if (Name == NULL) {
	TheEntry->Name = Malloc(1);
	TheEntry->Name[0] = '\0';
    } else
	TheEntry->Name = Name;
    if (Alias == NULL) {
	TheEntry->Alias = Malloc(1);
	TheEntry->Alias[0] = '\0';
    } else
	TheEntry->Alias = Alias;
    if (VerboseName == NULL) {
	TheEntry->VerboseName = Malloc(1);
	TheEntry->VerboseName[0] = '\0';
    } else
	TheEntry->VerboseName = VerboseName;
    TheEntry->Level = CurrentLevel;
    for (i = 1; i < CurrentLevel; i++)
	TheEntry->AncestorNumber[i] = CurrentAncestors[i];
#ifdef DEBUG
    {int j;
for (j = 1; j < CurrentLevel; j++)
printf("%d.", TheEntry->AncestorNumber[j]);
printf("%d", CurrentAttributeNumber);
printf("\n");
 }
#endif
    CurrentAncestors[CurrentLevel] = CurrentAttributeNumber;
    TheEntry->AsteriskPresent = False;
    TheEntry->Number = CurrentAttributeNumber;
    CurrentAttribute++;
    CurrentAttributeNumber++;
    if (Type == NULL || Type[0] == '\0') {
#ifdef DEBUG
printf("No type found\n");
#endif
	TheEntry->Type = Typeless;
    } else if (StrcmpNoupper(Type, "Typeless") == 0)
	TheEntry->Type = Typeless;
    else if (StrcmpNoupper(Type, "Real") == 0)
	TheEntry->Type = Real;
    else if (StrcmpNoupper(Type, "Integer") == 0)
	TheEntry->Type = Integer;
    else if (StrcmpNoupper(Type, "String") == 0)
	TheEntry->Type = String;
    else if (StrcmpNoupper(Type, "Date") == 0)
	TheEntry->Type = Date;
    else {
	fprintf(stderr, "Syntax Error: Type '%s'\n", Type);
	fprintf(stderr, "Redeclaring '%s' to 'Typeless'\n", Type);
	TheEntry->Type = Typeless;
    }
    if (Type != NULL)
	Free(Type);
    if (Format != NULL) {
	switch (TheEntry->Type) {
	case Integer:
	    if (SchemaFormatCheck("^[\\-\\#0\\ \\+]*[0-9]*\\.?[0-9]*f$", Format) != 0) {
		fprintf(stderr, "Invalid format specifier (%s) for type Integer\n", Format);
		Free(Format);
		Format = NULL;
	    }
	    break;
	case Real:
	    if (SchemaFormatCheck("^%[\\-\\#0\\ \\+]*[0-9]*\\.?[0-9]*d$", Format) != 0) {
		fprintf(stderr, "Invalid format specifier (%s) for type Real\n", Format);
		Free(Format);
		Format = NULL;
	    }
	    break;
	case Date:
	    break;
	default:
	    ;
	}
    }
    if (Format == NULL) {
	switch (TheEntry->Type) {
	case Typeless:
	    TheEntry->Format = Malloc(3);
	    strcpy(TheEntry->Format, "%s");
	    break;
	case Real:
	    TheEntry->Format = Malloc(3);
	    strcpy(TheEntry->Format, "%f");
	    break;
	case Integer:
	    TheEntry->Format = Malloc(3);
	    strcpy(TheEntry->Format, "%d");
	    break;
	case String:
	    TheEntry->Format = Malloc(3);
	    strcpy(TheEntry->Format, "%s");
	    break;
	case Date:
	    TheEntry->Format = Malloc(strlen(CurrentSchema->default_date_format)+1);
	    strcpy(TheEntry->Format, CurrentSchema->default_date_format);
	    break;
	default:
	    TheEntry->Format = Malloc(1);
	    TheEntry->Format[0] = '\0';
	}
    } else {
	if (TheEntry->Type == String || TheEntry->Type == Typeless) {
	    fprintf(stderr, "format specifier in attribute of type string ignored\n");
	    TheEntry->Format = Malloc(3);
	    strcpy(TheEntry->Format, "%s");
	} else {
	    TheEntry->Format = Format;
	}
    }
    if (Separators == NULL) {
	switch (TheEntry->Type) {
	case Typeless:
	    TheEntry->Separators = Malloc(strlen(DEFAULT_SEPARATORS)+1);
	    strcpy(TheEntry->Separators, DEFAULT_SEPARATORS);
	    break;
	case Real:
	    TheEntry->Separators = Malloc(strlen(DEFAULT_REAL_SEPARATORS)+1);
	    strcpy(TheEntry->Separators, DEFAULT_REAL_SEPARATORS);
	    break;
	case Integer:
	    TheEntry->Separators = Malloc(strlen(DEFAULT_INTEGER_SEPARATORS)+1);
	    strcpy(TheEntry->Separators, DEFAULT_INTEGER_SEPARATORS);
	    break;
	case String:
	    TheEntry->Separators = Malloc(strlen(DEFAULT_SEPARATORS)+1);
	    strcpy(TheEntry->Separators, DEFAULT_SEPARATORS);
	    break;
	case Date:
	    TheEntry->Separators = Malloc(strlen(DEFAULT_DATE_SEPARATORS)+1);
	    strcpy(TheEntry->Separators, DEFAULT_DATE_SEPARATORS);
	    break;
	default:
	    TheEntry->Separators = Malloc(2);
	    TheEntry->Separators[0] = '\n';
	    TheEntry->Separators[1] = '\0';	    
	}
    } else {
	if (strchr(Separators, '\n') == NULL) {
	    char		*tmp = Separators;

	    Separators = Malloc(strlen(tmp)+2);
	    strcpy(Separators, tmp);
	    strcat(Separators, "\n");
	    Free(tmp);
	}
	TheEntry->Separators = Separators;
    }
    TheEntry->Exclude = Exclude;
}

void ResetSchemaParser()
{
    CurrentAttribute = 1;
    CurrentLevel = 1;
    CurrentAttributeNumber = 1;
}
