
/* TclSort.c - TCL interface routines for sorting Qddb data structures..
 *
 * Copyright (C) 1994 Herrin Software Development, Inc.
 * All rights reserved.
 *
 * This file is part of Qddb.
 *
 * Qddb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License Version 2
 * as published by the Free Software Foundation.
 *
 * Qddb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Qddb; see the file LICENSE.  If not, write to:
 *
 *	Herrin Software Development, Inc. 
 *	R&D Division
 *	41 South Highland Ave. 
 *	Prestonsburg, KY 41653 
 */


#include "tcl.h"
#include "Qddb.h"
#include "tclQddb.h"

typedef union {
    int				sortnode_idx;
    struct {
	Boolean			sortnode_ascending;
	DataTree		*sortnode_node;
    } st;
} TclQddb_SortNode;

#define idx sortnode_idx
#define ascending st.sortnode_ascending
#define node st.sortnode_node

static int sort_row_argc;
static int sort_ascending_argc;
static char **sort_ascending_argv;

static Boolean NodeAscending _ANSI_ARGS_((char *));

static int CompareSortNodes(a1, a2)
    TclQddb_SortNode		**a1, **a2;
{
    TclQddb_SortNode		*n1 = *a1, *n2 = *a2;
    int				i, comp;
    char			*t1_str, *t2_str;
    time_t			t1, t2;
    
    if (n1 == NULL && n2 == NULL)
	return 0;
    if (n1 == NULL)
	return n2[1].ascending == False? 1:-1;
    if (n2 == NULL)
	return n1[1].ascending == False? -1:1;
    for (i = 1; i <= sort_row_argc; i++) {
	if (n1[i].node == NULL && n2[i].node == NULL)
	    continue;
	if (n1[i].node == NULL)
	    return n2[i].ascending == False? 1:-1;
	if (n2[i].node == NULL)
	    return n1[i].ascending == False? -1:1;
#if defined(DIAGNOSTIC)
	if (n1[i].node->datatree_type != n2[i].node->datatree_type) {
	    fprintf(stderr, "CompareSortNodes: type(%d) != type(%d)\n", 
		    n1[i].node->datatree_type, n2[i].node->datatree_type);
	    fflush(stderr);
	    abort();
	}
#endif
	switch (n1[i].node->datatree_type) {
	case DATATREE_REAL:
	    if (n1[i].ascending == True) {
		if (n1[i].node->datatree_real < n2[i].node->datatree_real)
		    return -1;
		if (n1[i].node->datatree_real > n2[i].node->datatree_real)
		    return 1;
	    } else {
		if (n1[i].node->datatree_real < n2[i].node->datatree_real)
		    return 1;
		if (n1[i].node->datatree_real > n2[i].node->datatree_real)
		    return -1;
	    }
	    break;
	case DATATREE_INT:
	    if (n1[i].ascending == True) {
		if (n1[i].node->datatree_int < n2[i].node->datatree_int)
		    return -1;
		if (n1[i].node->datatree_int > n2[i].node->datatree_int)
		    return 1;
	    } else {
		if (n1[i].node->datatree_int < n2[i].node->datatree_int)
		    return 1;
		if (n1[i].node->datatree_int > n2[i].node->datatree_int)
		    return -1;
	    }
	    break;
	case DATATREE_STRING:
	    if (n1[i].ascending == True) {
		comp = strcasecmp(n1[i].node->datatree_string, n2[i].node->datatree_string);
		if (comp != 0)
		    return comp;
	    } else {
		comp = strcasecmp(n1[i].node->datatree_string, n2[i].node->datatree_string);
		if (comp != 0)
		    return -comp;
		break;
	    }
	    break;
	case DATATREE_DATE:
	    t1_str = Qddb_DateStringToTime(n1[i].node->datatree_date);
	    t2_str = Qddb_DateStringToTime(n2[i].node->datatree_date);
	    t1 = atoi(t1_str);
	    t2 = atoi(t2_str);
	    Free(t1_str);
	    Free(t2_str);
	    if (n1[i].ascending == True) {
		if (t1 < t2)
		    return -1;
		if (t1 > t2)
		    return 1;
	    } else {
		if (t1 < t2)
		    return 1;
		if (t1 > t2)
		    return -1;
	    }
	    break;
	default:
	    PANIC("CompareSortNodes");
	}
    }
    return 0;
}

static TclQddb_SortNode *TclQddb_LookupNodesSameTuple(interp, tuple, row, argc, argv)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*row;
    int				argc;
    char			**argv;
{
    TclQddb_Rows		*r1;
    TclQddb_SortNode		*retval;
    int				i;

    retval = (TclQddb_SortNode *)Malloc(sizeof(TclQddb_SortNode)*(argc+2));
    for (i = 0; i < argc; i++) {
	r1 = row;
	while (r1 != NULL && strcmp(r1->attr_name, argv[i]) != 0)
	    r1 = r1->next;
	if (r1 == NULL) {
	    retval[i+1].node = NULL;
	} else {
	    retval[i+1].node = TclQddb_DataTreeLookupNode(interp, tuple, r1->attr_name, r1->instance);
	}
	retval[i+1].ascending = NodeAscending(argv[i]);
    }
    retval[argc+1].node = NULL;
    return retval;
}

void TclQddb_SortRowsSameTuple(interp, tuple, rows, argc, argv, ascending_argc, ascending_argv)
    Tcl_Interp			*interp;
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		**rows;
    int				argc;
    char			**argv;
    int				ascending_argc;
    char			**ascending_argv;
{
    TclQddb_SortNode		**nodes;
    TclQddb_Rows		**nrows;
    size_t			num_rows, i;

    for (num_rows = 0; rows[num_rows] != NULL; num_rows++);
    nodes = (TclQddb_SortNode **)Malloc(sizeof(TclQddb_SortNode *)*num_rows);
    sort_ascending_argc = ascending_argc;
    sort_ascending_argv = ascending_argv;
    for (i = 0; i < num_rows; i++) {
	nodes[i] = TclQddb_LookupNodesSameTuple(interp, tuple, rows[i], argc, argv);
	nodes[i][0].idx = i;
    }
    sort_row_argc = argc;
    qsort(nodes, num_rows, sizeof(TclQddb_SortNode *),
	  (int (*)(const void *, const void *))CompareSortNodes);
    nrows = (TclQddb_Rows **)Malloc(sizeof(TclQddb_Rows *)*num_rows);
    for (i = 0; i < num_rows; i++) {
	nrows[i] = rows[nodes[i][0].idx];
	Free(nodes[i]);
    }
    Free(nodes);
    for (i = 0; i < num_rows; i++) {
	rows[i] = nrows[i];
    }
    Free(nrows);
}

static Boolean NodeAscending(name)
    char			*name;
{
    int				i;

    for (i = 0; i < sort_ascending_argc; i++) {
	if (strcmp(name, sort_ascending_argv[i]) == 0) {
	    return True;
	}
    }
    return False;
}

static TclQddb_SortNode *TclQddb_LookupNodes(interp, row, argc, argv)
    Tcl_Interp			*interp;
    TclQddb_RowHeader		*row;
    int				argc;
    char			**argv;
{
    TclQddb_Tuple		*tuple;
    TclQddb_Rows		*r1;
    TclQddb_SortNode		*retval;
    int				i;

    tuple = TclQddb_GetTuple(interp, row->tuple_name);
    retval = (TclQddb_SortNode *)Malloc(sizeof(TclQddb_SortNode)*(argc+2));
    for (i = 0; i < argc; i++) {
	r1 = row->row;
	while (r1 != NULL && strcmp(r1->attr_name, argv[i]) != 0)
	    r1 = r1->next;
	if (r1 == NULL) {
	    retval[i+1].node = NULL;
	} else {
	    retval[i+1].node = TclQddb_DataTreeLookupNode(interp, tuple, r1->attr_name, r1->instance);
	}
	retval[i+1].ascending = NodeAscending(argv[i]);
    }
    retval[argc+1].node = NULL;
    return retval;
}

int TclQddb_SortRows(interp, row_argc, row_descs, argc, argv, ascending_argc, ascending_argv)
    Tcl_Interp			*interp;
    int				row_argc;
    char			**row_descs;
    int				argc;
    char			**argv;
    int				ascending_argc;
    char			**ascending_argv;
{
    TclQddb_RowHeader		**headers;
    TclQddb_RowHeader		**rows;
    char			**nrow_descs;
    TclQddb_SortNode		**nodes;
    size_t			num_rows, i;

    headers = (TclQddb_RowHeader **)Malloc(sizeof(TclQddb_RowHeader *)*(row_argc+1));
    for (i = 0; row_descs[i] != NULL; i++) {
	headers[i] = TclQddb_GetRows(interp, row_descs[i]);
	if (headers[i] == NULL) {
	    Tcl_AppendResult(interp, "Cannot find row for descriptor \"", row_descs[i], "\"", NULL);
	    Free(headers);
	    return TCL_ERROR;
	}
    }
    num_rows = i;
    headers[i] = NULL;
    rows = headers;
    nodes = (TclQddb_SortNode **)Malloc(sizeof(TclQddb_SortNode *)*num_rows);
    sort_ascending_argc = ascending_argc;
    sort_ascending_argv = ascending_argv;
    for (i = 0; i < num_rows; i++) {
	nodes[i] = TclQddb_LookupNodes(interp, rows[i], argc, argv);
	nodes[i][0].idx = i;
    }
    sort_row_argc = argc;
    qsort(nodes, num_rows, sizeof(TclQddb_SortNode *), 
	  (int (*)(const void *, const void *))CompareSortNodes);
    nrow_descs = (char **)Malloc(sizeof(char *)*num_rows);
    for (i = 0; i < num_rows; i++) {
	nrow_descs[i] = row_descs[nodes[i][0].idx];
	Free(nodes[i]);
    }
    Free(nodes);
    for (i = 0; i < num_rows; i++) {
	row_descs[i] = nrow_descs[i];
    }
    Free(nrow_descs);
    Free(headers);
    return TCL_OK;
}


int TclQddb_SortSelectRows(interp, row_argc, row_descs, val_descs, 
			   argc, argv, ascending_argc, ascending_argv)
    Tcl_Interp			*interp;
    int				row_argc;
    char			**row_descs;
    char			**val_descs;
    int				argc;
    char			**argv;
    int				ascending_argc;
    char			**ascending_argv;
{
    TclQddb_RowHeader		**headers;
    TclQddb_RowHeader		**rows;
    char			**nrow_descs, **nval_descs;
    TclQddb_SortNode		**nodes;
    size_t			num_rows, i;

    headers = (TclQddb_RowHeader **)Malloc(sizeof(TclQddb_RowHeader *)*(row_argc+1));
    for (i = 0; i < row_argc; i++) {
	headers[i] = TclQddb_GetRows(interp, row_descs[i]);
	if (headers[i] == NULL) {
	    Tcl_AppendResult(interp, "Cannot find row for descriptor \"", row_descs[i], "\"", NULL);
	    Free(headers);
	    return TCL_ERROR;
	}
    }
    num_rows = row_argc;
    headers[i] = NULL;
    rows = headers;
    nodes = (TclQddb_SortNode **)Malloc(sizeof(TclQddb_SortNode *)*num_rows);
    sort_ascending_argc = ascending_argc;
    sort_ascending_argv = ascending_argv;
    for (i = 0; i < num_rows; i++) {
	nodes[i] = TclQddb_LookupNodes(interp, rows[i], argc, argv);
	nodes[i][0].idx = i;
    }
    sort_row_argc = argc;
    qsort(nodes, num_rows, sizeof(TclQddb_SortNode *), 
	  (int (*)(const void *, const void *))CompareSortNodes);
    nrow_descs = (char **)Malloc(sizeof(char *)*num_rows);
    nval_descs = (char **)Malloc(sizeof(char *)*num_rows);
    for (i = 0; i < num_rows; i++) {
	nrow_descs[i] = row_descs[nodes[i][0].idx];
	nval_descs[i] = val_descs[nodes[i][0].idx];
	Free(nodes[i]);
    }
    Free(nodes);
    for (i = 0; i < num_rows; i++) {
	row_descs[i] = nrow_descs[i];
	val_descs[i] = nval_descs[i];
    }
    Free(nrow_descs);
    Free(nval_descs);
    Free(headers);
    return TCL_OK;
}



int TclQddb_SortQueryRows(interp, schema, row_argc, row_descs, argc, argv, ascending_argc, ascending_argv)
    Tcl_Interp			*interp;
    Schema			*schema;
    int				row_argc;
    char			**row_descs;
    int				argc;
    char			**argv;
    int				ascending_argc;
    char			**ascending_argv;
{
    char			**nrow_descs, ***headers, **val_argv;
    TclQddb_SortNode		**nodes, *thisnode;
    DataTree			*tree;
    size_t			num_rows;
    int				i, j, headers_argc;
    int				*attr_nums, len;

    headers = (char ***)Malloc(sizeof(char **)*(row_argc+1));
    for (i = 0; row_descs[i] != NULL; i++) {
	Tcl_SplitList(interp, row_descs[i], &headers_argc, headers+i);
	if (headers_argc-2 != argc) {
	    for (; i >= 0; i--)
		Free(headers[i]);
	    Free(headers);
	    Tcl_AppendResult(interp, "number of elements in query list do not match the \
number of attributes\n", NULL);
	    return TCL_ERROR;
	}
    }
    num_rows = i;
    headers[i] = NULL;
    sort_ascending_argc = ascending_argc;
    sort_ascending_argv = ascending_argv;
    attr_nums = (int *)Malloc(sizeof(int)*argc);
    for (i = 2; i < argc; i++) {
	attr_nums[i] = Qddb_ConvertAttributeNameToNumber(schema, argv[i]);
	if (attr_nums[i] == -1) {
	    for (i = 0; i < num_rows; i++)
		Free(headers[i]);
	    Free(headers);
	    Free(attr_nums);
	    Tcl_AppendResult(interp, "Cannot find attribute \"", argv[i], "\"", NULL);
	    return TCL_ERROR;
	}
    }
    nodes = (TclQddb_SortNode **)Malloc(sizeof(TclQddb_SortNode *)*num_rows);
    for (i = 0; i < num_rows; i++) {
	thisnode = nodes[i] = (TclQddb_SortNode *)Malloc(sizeof(TclQddb_SortNode)*(argc+1));
	thisnode->idx = i;
	tree = (DataTree *)Malloc(sizeof(DataTree)*argc);
	thisnode++;
	val_argv = headers[i];
	for (j = 1; j <= argc; j++, thisnode++) {
	    /* FIXME: this is nastily inefficient */
	    thisnode->ascending = NodeAscending(argv[j-1]);
	    thisnode->node = tree+j-1;
	    switch (schema->Entries[j-1].Type) {
	    case Integer:
		thisnode->node->datatree_type = DATATREE_INT;
		thisnode->node->datatree_int = atoi(val_argv[j-1]);
		break;
	    case Real:
		thisnode->node->datatree_type = DATATREE_REAL;
		thisnode->node->datatree_real = strtod(val_argv[j-1], NULL);
		break;
	    case Date:
		thisnode->node->datatree_type = DATATREE_DATE;
		thisnode->node->datatree_date = val_argv[j-1];
		break;
	    case String:
	    case Typeless:
	    default:
		thisnode->node->datatree_type = DATATREE_STRING;
		len = strlen(val_argv[j-1]);
		if (val_argv[j-1][len-1] == '}')
		    val_argv[j-1][len-1] = '\0';
		if (val_argv[j-1][0] == '{')
		    thisnode->node->datatree_string = val_argv[j-1]+1;
		else
		    thisnode->node->datatree_string = val_argv[j-1];
	    }
	}
    }
    sort_row_argc = argc;
    qsort(nodes, num_rows, sizeof(TclQddb_SortNode *), 
	  (int (*)(const void *, const void *))CompareSortNodes);
    nrow_descs = (char **)Malloc(sizeof(char *)*num_rows);
    for (i = 0; i < num_rows; i++) {
	nrow_descs[i] = row_descs[nodes[i][0].idx];
	Free(headers[i]);
	Free(nodes[i]);
    }
    Free(nodes);
    Free(headers);
    for (i = 0; i < num_rows; i++) {
	row_descs[i] = nrow_descs[i];
    }
    Free(nrow_descs);
    return TCL_OK;
}
