#
# Module for editing Canvas widgets
#

# Undos are handled thus: Each cmd that can be undone stores an entry into
# TH(Undo,Data,$w). This entry contains the entry's string before the command
# was done, and a human-readable title to prompt the user with what the cmd
# did, and the item the command occurred in.

# Undoes last command.
proc th_Canvas_undo {w} {
  global TH
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {([llength $TH(Undo,Data,$w)] == 0)} {th_beep ; return}

  set data [lindex $TH(Undo,Data,$w) 0]
  set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 1 end]
  set TH(Modified,$w) 1

  th_Canvas_replace $w [lindex $data 2] [lindex $data 0]
}

proc th_Canvas_register_undoable_cmd {w i {name ""}} {
  global TH ; set TH(Modified,$w) 1
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {[catch "set TH(Undo,Max,[winfo class $w])"]} {set TH(Undo,Max,[winfo class $w]) 0}

  set new_entry [list [lindex [$w itemconfigure $i -text] 4] $name $i]
  set TH(Undo,Data,$w) [concat [list $new_entry] $TH(Undo,Data,$w)]
  if {($TH(Undo,Max,[winfo class $w]) >= 0) && ($TH(Undo,Max,[winfo class $w]) < [llength $TH(Undo,Data,$w)])} {
    set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 0 [expr "$TH(Undo,Max,[winfo class $w]) - 1"]]
}}


# Basic insertion commands

# Inserts some characters into the entry at the cursor.
proc th_Canvas_insert {w chars} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  $w insert $i insert $chars
  global TH ; set TH(Modified,$w) 1
  th_Canvas_goto $w $i insert 0
}

# Replaces contents of entry with new_string.
proc th_Canvas_replace {w i new_string} {
  set new_index [expr [$w index $i insert] + [string length $new_string] - \
        [string length [lindex [$w itemconfigure $i -text] 4]]]
  $w dchars $i 0 end
  $w insert $i 0 $new_string
  global TH ; set TH(Modified,$w) 1
  th_Canvas_goto $w $i $new_index 0
}

proc th_Canvas_undoable_insert {w i string} {
  th_Canvas_register_undoable_cmd $w $i "Insert $string"
  th_Canvas_insert $w $string
}

proc th_Canvas_paste_selection {w} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  if {[catch {set chars [string range [lindex [$w configure $i -text] 4] \
		[$w index $i sel.first] [$w index $i sel.last]]}]} {
    if {[catch {set chars [selection get]}]} {
      set chars "" ; th_beep ; return}}
  th_Canvas_undoable_insert $w $i $chars
}

proc th_Canvas_self_insert {w {c ""}} {
  if {(![regexp . $c])} {return}
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  th_Canvas_insert $w $c
  global TH
  if {[catch "set TH(Overwrite,$w)"]} {set TH(Overwrite,$w) 0}
  if $TH(Overwrite,$w) {$w dchars $i insert}
}


# Killing text

proc th_Canvas_kill_line {w} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  catch ".th_kill delete killed.first killed.last"
  set replace_string [string range [lindex [$w itemconfigure $i -text] 4] \
		[$w index $i insert] [$w index $i end]]
  .th_kill insert 1.0 $replace_string
  set end [.th_kill index "1.0 +[string length $replace_string] c"]
  .th_kill tag add sel 1.0 $end
  .th_kill tag add killed 1.0 $end
  $w dchars $i insert end
  global TH ; set TH(Modified,$w) 1
}

proc th_Canvas_kill_range {w i start end} {
  set s [$w index $i $start]
  set e [$w index $i $end]
  if {$s > $e} {th_beep ; return}
  global TH ; set TH(Modified,$w) 1
  catch ".th_kill delete killed.first killed.last"
  set replace_string [string range [lindex [$w itemconfigure $i -text] 4] $s $e]
  .th_kill insert 1.0 $replace_string
  set k_end [.th_kill index "1.0 +[string length $replace_string] c"]
  .th_kill tag add sel 1.0 $k_end
  .th_kill tag add killed 1.0 $k_end
  $w dchars $i $s $e
}

proc th_Canvas_kill_region {w} {
  global TH
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}
  if {![catch "$w index $i sel.last"]} {th_Canvas_kill_range $w $i sel.first sel.last
  } elseif {![catch "set TH(Mark,$w,$i)"]} {
    if {[$w index $i insert] > $TH(Mark,$w,$i)} {
      th_Canvas_kill_range $w $i $TH(Mark,$w,$i) [expr [$w index $i insert] -1]
    } else {th_Canvas_kill_range $w $i insert [expr $TH(Mark,$w,$i) -1]}
  } else {th_beep}
}


# Deletion

proc th_Canvas_delete_range {w i start end {undo 1}} {
  set s [$w index $i $start]
  set e [expr [$w index $i $end] -1]
  if {$s > $e} {th_beep ; return}
  global TH ; set TH(Modified,$w) 1

  if $undo {
    th_Canvas_register_undoable_cmd $w $i "Delete [string range [lindex [$w itemconfigure $i -text] 4] $s $e]"
  }
  $w dchars $i $s $e
  th_Canvas_goto $w $i insert 0
}

proc th_Canvas_delete_char_backward {w i} {
  if {([$w index $i insert] == 0)} {th_beep ; return}
  th_Canvas_delete_range $w $i [expr [$w index $i insert] -1] insert 0
}

proc th_Canvas_delete_char_forward {w i} {
  if {([$w index $i insert] == [$w index $i end])} {th_beep ; return}
  th_Canvas_delete_range $w $i insert [expr [$w index $i insert] +1] 0
}

proc th_Canvas_delete_word_backward {w i} {
  set y [expr [$w index $i insert] -1]
  if {$y < 0} {th_beep ; return}
  set x [th_string_wordstart [lindex [$w itemconfigure $i -text] 4] $y]
  th_Canvas_delete_range $w $i $x insert
}

proc th_Canvas_delete_word_forward {w i} {
  set x [$w index $i insert]
  if {$x >= [$w index $i end]} {th_beep ; return}
  set y [expr [th_string_wordend [lindex [$w itemconfigure $i -text] 4] $x] +1]
  th_Canvas_delete_range $w $i $x $y
}

proc th_Canvas_delete_selection {w} {
  if {[catch "$w select item" i]} {th_beep ; return}
  th_Canvas_delete_range $w $i sel.first [expr [$w index $i sel.last] +1]
}


# String filtering

proc th_Canvas_filter {w filter} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  if {[catch "$w index $i sel.last"]} {
    set start [$w index $i insert]
    set end [expr [th_string_wordend [lindex [$w itemconfigure $i -text] 4] $start] -1]
    set selected 0
  } else {set start [$w index $i sel.first]
    set end [$w index $i sel.last]
    set selected 1
  }
  set word [string range [lindex [$w itemconfigure $i -text] 4] $start $end]
  set new_w [$filter $word]

  if {($word == $new_w)} {th_Canvas_goto $w $i [expr [$w index $i $end] +1] 0 ; return}

  th_Canvas_register_undoable_cmd $w $i "Change $word"

  $w dchars $i $start $end
  $w icursor $i $start
  $w insert $i $start $new_w

  if {($selected)} {
    $w select from $start
    $w select to [expr [$w index $i insert] -1]
  }
  th_Canvas_goto $w $i insert 0
}


# Transposition

proc th_Canvas_transpose_chars {w} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  set in [$w index $i insert]
  if {($in < 2) || ($in >= [$w index $i end])} {th_beep ; return}

  set c1 [string index [lindex [$w itemconfigure $i -text] 4] [expr "$in-1"]]
  set c2 [string index [lindex [$w itemconfigure $i -text] 4] [expr "$in"]]

  th_Canvas_register_undoable_cmd $w $i "Transpose $c1 $c2"

  $w dchars $i [expr $in-1] $in
  $w insert $i insert $c2
  $w insert $i insert $c1
  th_Canvas_goto $w $i insert 0
}

proc th_Canvas_transpose_words {w} {
  if {[catch "$w index [set i [$w focus]] insert"]} {th_beep ; return}

  set in [$w index $i insert]
  if {($in < 2) || ($in >= [$w index $i end])} {th_beep ; return}

  th_Canvas_register_undoable_cmd $w $i "Transpose words"

  set y [expr "[$w index $i insert]-1"]
  set c2 [string index [lindex [$w itemconfigure $i -text] 4] $y]
  $w dchars $i $y
  incr y -1

  $w icursor $i [th_string_wordstart [lindex [$w itemconfigure $i -text] 4] [expr [$w index $i insert] -2]]
  set x [$w index $i insert]
  set c1 [string range [lindex [$w itemconfigure $i -text] 4] $x $y]
  $w dchars $i $x $y

  set y [$w index $i insert]
  $w icursor $i [th_string_wordend [lindex [$w itemconfigure $i -text] 4] [$w index $i insert]]
  set x [expr "[$w index $i insert]-1"]
  set c3 [string range [lindex [$w itemconfigure $i -text] 4] $y $x]
  $w dchars $i $y $x

  $w insert $i insert $c3
  $w insert $i insert $c2
  $w insert $i insert $c1
  th_Canvas_goto $w $i [expr [$w index $i insert]+1] 0
}


