#
# Module for editing Entry widgets
#

# Undos are handled thus: Each cmd that can be undone stores an entry into
# TH(Undo,Data,$w). This entry contains the entry's string before the command
# was done, and a human-readable title to prompt the user with what the cmd
# did.

# Undoes last command.
proc th_Entry_undo {w} {
  global TH
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {([llength $TH(Undo,Data,$w)] == 0)} {th_beep ; return}

  set data [lindex $TH(Undo,Data,$w) 0]
  set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 1 end]
  set TH(Modified,$w) 1

  th_Entry_replace $w [lindex $data 0]
}

proc th_Entry_register_undoable_cmd {w {name ""}} {
  global TH ; set TH(Modified,$w) 1
  if {[catch "set TH(Undo,Data,$w)"]} {set TH(Undo,Data,$w) ""}
  if {[catch "set TH(Undo,Max,[winfo class $w])"]} {set TH(Undo,Max,[winfo class $w]) 0}

  set new_entry [list [$w get] $name]
  set tmp [concat [list $new_entry] $TH(Undo,Data,$w)] ; set TH(Undo,Data,$w) $tmp
  if {($TH(Undo,Max,[winfo class $w]) >= 0) && ($TH(Undo,Max,[winfo class $w]) < [llength $TH(Undo,Data,$w)])} {
    set TH(Undo,Data,$w) [lrange $TH(Undo,Data,$w) 0 [expr "$TH(Undo,Max,[winfo class $w]) - 1"]]
}}


# Basic insertion commands

# Inserts some characters into the entry at the cursor.
proc th_Entry_insert {w chars} {
  global TH ; set TH(Modified,$w) 1
  $w insert insert $chars
  th_Entry_goto $w insert 0
}

# Replaces contents of entry with new_string.
proc th_Entry_replace {w new_string} {
  global TH ; set TH(Modified,$w) 1
  set new_index [expr [$w index insert] + [string length $new_string] - \
        [string length [$w get]]]
  $w delete 0 end
  $w insert 0 $new_string
  th_Entry_goto $w $new_index 0
}

proc th_Entry_undoable_insert {w string} {
  th_Entry_register_undoable_cmd $w "Insert $string"
  th_Entry_insert $w $string
}

proc th_Entry_paste_selection {w} {
  if {[catch {set chars [string range [$w get] [$w index sel.first] [$w index sel.last]]}]} {
    if {[catch {set chars [selection get]}]} {
      set chars "" ; th_beep ; return}}
  th_Entry_undoable_insert $w $chars
}

proc th_Entry_self_insert {w {c ""}} {
  if {(![regexp . $c])} {return}
  th_Entry_insert $w $c
  global TH
  if {[catch "set TH(Overwrite,$w)"]} {set TH(Overwrite,$w) 0}
  if $TH(Overwrite,$w) {$w delete insert}
}


# Killing text

proc th_Entry_kill_line {w} {
  global TH ; set TH(Modified,$w) 1
  catch ".th_kill delete killed.first killed.last"
  set replace_string [string range [$w get] [$w index insert] [$w index end]] 
  .th_kill insert 1.0 $replace_string
  set end [.th_kill index "1.0 +[string length $replace_string] c"]
  .th_kill tag add sel 1.0 $end
  .th_kill tag add killed 1.0 $end
  $w delete insert end
}

proc th_Entry_kill_range {w start end} {
  set s [$w index $start]
  set e [$w index $end]
  if {$s > $e} {th_beep ; return}
  global TH ; set TH(Modified,$w) 1
  catch ".th_kill delete killed.first killed.last"
  set replace_string [string range [$w get] $s $e]
  .th_kill insert 1.0 $replace_string
  set k_end [.th_kill index "1.0 +[string length $replace_string] c"]
  .th_kill tag add sel 1.0 $k_end
  .th_kill tag add killed 1.0 $k_end
  $w delete $s $e
}

proc th_Entry_kill_region {w} {
  global TH
  if {![catch "$w index sel.last"]} {th_Entry_kill_range $w sel.first sel.last
  } elseif {![catch "set TH(Mark,$w)"]} {
    if {[$w index insert] > $TH(Mark,$w)} {
      th_Entry_kill_range $w $TH(Mark,$w) [expr [$w index insert] -1]
    } else {th_Entry_kill_range $w insert [expr $TH(Mark,$w) -1]}
  } else {th_beep}
}


# Deletion

proc th_Entry_delete_range {w start end {undo 1}} {
  set s [$w index $start]
  set e [expr [$w index $end] -1]
  if {$s > $e} {th_beep ; return}
  global TH ; set TH(Modified,$w) 1

  if $undo {
    th_Entry_register_undoable_cmd $w "Delete [string range [$w get] $s $e]"
  }
  $w delete $s $e
  th_Entry_goto $w insert 0
}

proc th_Entry_delete_char_backward {w} {
  if {([$w index insert] == 0)} {th_beep ; return}
  th_Entry_delete_range $w [expr [$w index insert] -1] insert 0
}

proc th_Entry_delete_char_forward {w} {
  if {([$w index insert] == [$w index end])} {th_beep ; return}
  th_Entry_delete_range $w insert [expr [$w index insert] +1] 0
}

proc th_Entry_delete_word_backward {w} {
  set x [th_string_wordstart [$w get] [expr [$w index insert] -1]]
  if {[$w index insert] == 0} {th_beep ; return}
  th_Entry_delete_range $w $x insert
}

proc th_Entry_delete_word_forward {w} {
  set x [$w index insert]
  if {$x >= [$w index end]} {th_beep ; return}
  set y [th_string_wordend [$w get] $x]
  th_Entry_delete_range $w $x $y
}

proc th_Entry_delete_selection {w} {
  if {[catch "$w index sel.last"]} {th_beep ; return}
  th_Entry_delete_range $w sel.first [expr [$w index sel.last] +1]
}


# String filtering

proc th_Entry_filter {w filter} {
  if {[catch "$w index sel.last"]} {
    set start [$w index insert]
    set end [expr [th_string_wordend [$w get] $start] -1]
    set selected 0
  } else {set start [$w index sel.first]
    set end [$w index sel.last]
    set selected 1
  }
  set word [string range [$w get] $start $end]
  set new_w [$filter $word]

  if {($word == $new_w)} {th_Entry_goto $w [expr [$w index $end] +1] 0 ; return}

  th_Entry_register_undoable_cmd $w "Change $word"

  $w delete $start $end
  $w icursor $start
  $w insert $start $new_w

  if {($selected)} {
    $w select from $start
    $w select to [expr [$w index insert] -1]
  }
  th_Entry_goto $w insert 0
}


# Transposition

proc th_Entry_transpose_chars {w} {
  set i [$w index insert]
  if {($i < 2) || ($i >= [$w index end])} {th_beep ; return}

  set c1 [string index [$w get] [expr "$i-1"]]
  set c2 [string index [$w get] [expr "$i"]]

  th_Entry_register_undoable_cmd $w "Transpose $c1 $c2"

  $w delete [expr "$i-1"] [expr "$i"]
  $w insert insert $c2
  $w insert insert $c1
  th_Entry_goto $w insert 0
}

proc th_Entry_transpose_words {w} {
  set i [$w index insert]
  if {($i < 2) || ($i >= [$w index end])} {th_beep ; return}

  th_Entry_register_undoable_cmd $w "Transpose words"

  set y [expr "[$w index insert]-1"]
  set c2 [string index [$w get] $y]
  $w delete $y
  incr y -1

  $w icursor [th_string_wordstart [$w get] [expr [$w index insert] -2]]
  set x [$w index insert]
  set c1 [string range [$w get] $x $y]
  $w delete $x $y

  set y [$w index insert]
  $w icursor [th_string_wordend [$w get] [$w index insert]]
  set x [expr "[$w index insert]-1"]
  set c3 [string range [$w get] $y $x]
  $w delete $y $x

  $w insert insert $c3
  $w insert insert $c2
  $w insert insert $c1
  th_Entry_goto $w [expr [$w index insert]+1] 0
}


