#
# Module for performing searches in Misc widgets
#

# Performs a search on w, informing user of success or failure.
# If direction and type are not specified, they are determined from the
# search label. Direction is forward or backward, and
# type indicates the search type, one of 'string', 'glob', or 'regexp'.
# If next is 0, stays on the same find, otherwise moves to the next one.
proc th_Misc_find {w string {next 1} {direction ""} {type ""} {wrap ""}} {
  set label "[th_frame $w].search.l"
  set text [lindex [$label configure -text] 4]
  if {$direction == ""} {set direction [lindex $text 0]}
  if {$type == ""} {set type [lindex $text 1]}

  if {[th_[winfo class $w]_find $w $string $next $direction $type]} {
    $label configure -text "$direction $type search:"
    return 1
  } else {
    $label configure -text "$direction $type search failed:"
    th_beep
    return 0
}}

# Bindings for incremental and regexp search

# Called whenever key is hit in entry during incremental search.
proc th_revise_incremental_search {w e c} {
  if {![regexp . $c]} {return}
  th_Misc_find $w [$e get] 0
}

proc th_toggle_incremental_search {w} {
  th_checkbutton_variable Search,Incremental,$w
  global TH
  set e "[th_frame $w].search.e"
  if {![winfo exists $e]} {
    th_show_entry $w search ; th_hide_entry $w search
  }
  if $TH(Search,Incremental,$w) {
    foreach binding [bind Entry] {
      if {([bind $e $binding] == "")} {
        bind $e $binding "[bind Entry $binding] ; th_revise_incremental_search $w $e %A"
  }}} else {
    foreach binding [bind Entry] {
      if {[bind Entry $binding] == [string range [bind $e $binding] 0 \
			[string length [bind Entry $binding]]]} {
        bind $e $binding ""
}}}}

proc th_Misc_searchbind {w f} {
  th_bind $f.e Cancel "+ th_Misc_search_exit $w"
  th_bind $w Cancel "+ th_Misc_search_exit $w"

  if {[string match "th_Misc_find *" [th_bind $f.e Search_Forward]]} {return}

  th_bind $f.e OK "th_Misc_find $w \[$f.e get\] ; th_Misc_search_exit $w"
  th_bind $f.e Search_Forward "th_Misc_find $w \[$f.e get\] 1 forward"
  th_bind $f.e Search_Backward "th_Misc_find $w \[$f.e get\] 1 backward"
  th_bind $f.e Toggle_Case_Sensitivity "th_checkbutton_variable Search,Case,$w"
  th_bind $f.e Toggle_Incremental_Search "th_toggle_incremental_search $w"

  global TH
  if {[winfo class $w] == "Text"} {
    th_bind $f.e Highlight_Searches "th_Text_highlight_searches $w \[$f.e get\]"}
  if {![catch "set TH(Binding,Replace_One)"]} {
    th_bind $f.e Replace_One "th_Misc_replace_setup $w"
    th_bind $f.e Replace_All "th_Misc_replace_setup $w"
  }
  if $TH(Search,Incremental,$w) {
    set TH(Search,Incremental,$w) 0 ; set TH(search,Incremetal,$w,aux) 0
    th_toggle_incremental_search $w
}}

# Starts up the search routines
proc th_Misc_search {w {direction "forward"} {type "string"}} {
  global TH
  if {[catch "set TH(Search,Incremental,$w)"]} {set TH(Search,Incremental,$w) 0}
  if {[catch "set TH(Search,Case,$w)"]} {set TH(Search,Case,$w) 0}
  if {[set f [th_show_entry $w search]] == ""} {th_beep ; return}
  $f.l configure -text "$direction $type search:"
  if $TH(Search,Incremental,$w) {$f.e delete 0 end}
  set TH(Search,Failed,$w) 0
  th_Misc_searchbind $w $f
}


# Closes down the search routines.
proc th_Misc_search_exit {w} {
  global auto_index
  set exit_proc th_[winfo class $w]_search_exit
  if {([info procs $exit_proc] != "") || \
     ([lsearch [array names auto_index] $exit_proc] >= 0)} {$exit_proc $w}
  th_hide_entry $w search ; 

  set f [th_frame $w]
  if {![catch {winfo ismapped "$f.replace.e"} result]} {
    if $result {th_Misc_replace_exit $w}}
}


### Comments on searching algorithms

# If TH(Search,Case,%W) is 1 for any widget %W, then searches in that widget
# are case-insensitive, otherwise they are case-sensitive.

# The find procuedres for verious widgets:
#
# Searches w for string, starting either from insert, or from last
# found string (indicated by the was_search tag, set by this function)
# Direction may be forward or backward, type may be string, glob, or regexp.
# Moves insert and resets tags if a match is found, returns 1 if successful.
# If the last search failed, starts from the beginning or end of the text
# If next is 0, can stay on the same search tag as before, otherwise move on.

# Glob Algorithm:
# --Extract all the nonspecial characters from the start and end of string.
#   (The special characters are []?*\). Either start and end may be empty.
# --If both start and end are nonempty, search for occurrences of start and
#   end that might work. Apply string match to all regions that begin with
#   start and end with end. (Ex: n*t finds "not", "nt", "n off t".)
# --If end is empty, search for occurrences of start, and for each one found
#   try string match on the region from start to the end of the word after the
#   start string. (Ex: n* finds any word with 'n', and selects from 'n' to end.)
# --Likewise, if start is empty, search for occurrences of end, and for each
#   one found, try string match on the region from the beginning of the word
#   containing the end string to the end of the end string. (Ex: *n finds any
#   word with 'n', and selects from beginning to 'n'.)
# --If both start and end are empty, traverses words until one satisfies the
#   string match command. (Ex: *n* finds any word with 'n', and selects it.)
#
#   This can be slow, especially if there are many instances of both start
#   and end, but their regions don't satisfy string match. If you use
#   regular characters in start and end, use as many as possible.
#
#   Due to the algorithm, searching forward is much faster than reverse
#   searching.
#
#   If you want a character to begin a word, but you want to restrict the
#   search to single words, enclose it in braces. (Ex: [n]* finds all words
#   that begin with n, selecting it, but n* finds all occurrences of 'n', and
#   selects from it to the end of its word.)
#
#   Often the program can become bogged down, when both start and end are
#   nonempty in finding large regions of text between start and end, and
#   running them through string match. TH(Search,Maxlines,$w) specifies
#   the maximum size of a region to test.

# String searching in Text widgets algorithm
#
# It seems that [$w get $start $end] is slow when start and end are far apart.
# So this routine searches first the line after start,
# then the next line after it, then the next two lines, the next four lines,
# on to the end, returning whenever the string is found.
# Since it does a line-by-line search, a string with more than one newline
# may not get found.
