/*-
 * Copyright (c) 1993, 1994
 *	The Regents of the University of California.  All rights reserved.
 * Copyright (c) 1994, 1995
 *	Keith Bostic.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef lint
static char sccsid[] = "@(#)cl_suspend.c	8.4 (Berkeley) 1/24/95";
#endif /* not lint */

#include <sys/types.h>
#include <sys/queue.h>
#include <sys/time.h>

#include <bitstring.h>
#include <curses.h>
#include <signal.h>
#include <stdio.h>
#include <termios.h>
#include <unistd.h>

#include "compat.h"
#include <db.h>
#include <regex.h>

#include "vi.h"
#include "cl.h"
#include "../sex/sex_screen.h"

/*
 * cl_suspend --
 *	Suspend a curses screen.
 *
 * See common/signal.c for a long discussion of what's going on here.  Let
 * me put it this way, it's NOT my fault.
 */
int
cl_suspend(sp)
	SCR *sp;
{
	struct termios sv_term;
	FILE *savefp;
	sigset_t set;
	int oldx, oldy, rval;
	char *sbp, *t, sbuf[128];

	/*
	 * Block SIGALRM, because vi uses timers to decide when to paint
	 * busy messages on the screen.
	 */
	(void)sigemptyset(&set);
	(void)sigaddset(&set, SIGALRM);
	if (sigprocmask(SIG_BLOCK, &set, NULL)) {
		msgq(sp, M_SYSERR, "suspend: sigblock");
		return (1);
	}

	/* Save the current cursor position. */
	getyx(stdscr, oldy, oldx);

	/*
	 * Move the cursor to the bottom of the screen.
	 *
	 * XXX
	 * Some curses implementations don't turn off inverse video when
	 * standend() is called, waiting to see what the next character is
	 * going to be, instead.  Write a character to force inverse video
	 * off, and then clear the line.
	 */
	(void)move(O_VAL(sp, O_LINES) - 1, 0);
	(void)addch('.');
	refresh();
	(void)move(O_VAL(sp, O_LINES) - 1, 0);
	clrtoeol();
	refresh();

	/* Restore the cursor keys to normal mode. */
	(void)cl_keypad(sp, 0);

	/* Send VE/TE. */
#ifdef SYSV_CURSES
	if ((t = tigetstr("cnorm")) != NULL && t != (char *)-1)
		(void)tputs(t, 0, vi_putchar);
	if ((t = tigetstr("rmcup")) != NULL && t != (char *)-1)
		(void)tputs(t, 0, vi_putchar);
#else
	sbp = sbuf;
	if ((t = tgetstr("ve", &sbp)) != NULL)
		(void)tputs(t, 0, vi_putchar);
	sbp = sbuf;
	if ((t = tgetstr("te", &sbp)) != NULL)
		(void)tputs(t, 0, vi_putchar);
#endif
	(void)fflush(stdout);

	/* Save current terminal settings, and restore the original ones. */
	if (tcgetattr(STDIN_FILENO, &sv_term)) {
		msgq(sp, M_SYSERR, "suspend: tcgetattr");
		return (1);
	}
	if (tcsetattr(STDIN_FILENO,
	    TCSASOFT | TCSADRAIN, &sp->gp->original_termios)) {
		msgq(sp, M_SYSERR, "suspend: tcsetattr original");
		return (1);
	}

	/* Push any waiting messages out on stdout. */
	savefp = sp->stdfp;
	sp->stdfp = stdout;
	(void)sex_refresh(sp);
	sp->stdfp = savefp;

	/* Stop the process group. */
	rval = 0;
	if (kill(0, SIGTSTP)) {
		msgq(sp, M_SYSERR, "suspend: kill");
		rval = 1;
	}

	/* Time passes ... */

	/* Restore current terminal settings. */
	if (tcsetattr(STDIN_FILENO, TCSASOFT | TCSADRAIN, &sv_term)) {
		msgq(sp, M_SYSERR, "suspend: tcsetattr current");
		rval = 1;
	}

	/* Send TI/VS. */
#ifdef SYSV_CURSES
	if ((t = tigetstr("smcup")) != NULL && t != (char *)-1)
		(void)tputs(t, 0, vi_putchar);
	if ((t = tigetstr("cvvis")) != NULL && t != (char *)-1)
		(void)tputs(t, 0, vi_putchar);
#else
	sbp = sbuf;
	if ((t = tgetstr("ti", &sbp)) != NULL)
		(void)tputs(t, 0, vi_putchar);
	sbp = sbuf;
	if ((t = tgetstr("vs", &sbp)) != NULL)
		(void)tputs(t, 0, vi_putchar);
#endif
	(void)fflush(stdout);

	/* Put the cursor keys into application mode. */
	(void)cl_keypad(sp, 1);

	/*
	 * If the screen changed size, do a full refresh.  Otherwise,
	 * System V has curses repaint it.  4BSD curses will repaint
	 * it in the wrefresh() call below.
	 */
	if (!cl_ssize(sp, 1))
		(void)cl_refresh(sp);
#ifdef SYSV_CURSES
	else
		redrawwin(stdscr);
#endif

	/* Restore the cursor, refresh the screen. */
	(void)move(oldy, oldx);
	(void)wrefresh(curscr);

	/* Reset the signals. */
	if (sigprocmask(SIG_UNBLOCK, &set, NULL)) {
		msgq(sp, M_SYSERR, "suspend: sigblock");
		rval = 1;
	}
	return (rval);
}
