////////////////////////////////////////////////////////////////////////////////
//  Implementations of camera classes. some platform dependend stuff is       //  
//  placed in the system-sub directory. The core ray-tracing stuff is taken   //  
//  from Paul S. Heckberts Article in "An Introduction to Raytracing". But,   //  
//  computation of refraction was replaced by an own algorithm                //
//  LAST EDIT: Fri Aug  5 08:55:03 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include "lookat.h"

const char *RTN_LOOKAT_CAMERA = "LookatCamera";
const char *RTSN_LC = "LC";

RT_LookatCamera::RT_LookatCamera( char *_n, const RT_Vector &_vp, const RT_Vector &_rp): RT_RayCamera( _n ) {
    vp = _vp; rp = _rp; xmode = RTE_LC_SHADING; xtwist = 0;
    xangle = 90; xnear = 0.01; xfar = 100; 
    computeLookatMatrix();
}

RT_Primitive *boundingBox = 0;

int RT_LookatCamera::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Vector Vector} {Creates a new camera called {ARG 1 Name}, {ARG 2 Viewpoint} is the eye point and {ARG 3 Referencepoint} is the point the camera is looking at. Short name is ", RTSN_LC, ".}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 4 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <viewpoint[3]> <referencepoint[3]>! ", NULL );
	    return TCL_ERROR;
	}
	RT_Vector vp, rp; int dum;
	if (RT_getVector( &argv[2], vp, dum ) && RT_getVector( &argv[3], rp, dum )) { 
	    new RT_LookatCamera( argv[1], vp, rp ); 
	    RTM_classReturn;
	}
	Tcl_AppendResult( ip, "Bad parameters. <viewpoint[]> <referencepoint[]> must be float vectors! ", NULL );
	return TCL_ERROR;
    }
    return res; 
}

RT_Vector RT_LookatCamera::vpV; int RT_LookatCamera::vpF, RT_LookatCamera::vpG;
RT_Vector RT_LookatCamera::rpV; int RT_LookatCamera::rpF, RT_LookatCamera::rpG;
int RT_LookatCamera::moV, RT_LookatCamera::moF, RT_LookatCamera::moG;
int RT_LookatCamera::twV, RT_LookatCamera::twF, RT_LookatCamera::twG;
double RT_LookatCamera::anV; int RT_LookatCamera::anF, RT_LookatCamera::anG;
double RT_LookatCamera::nrV; int RT_LookatCamera::nrF, RT_LookatCamera::nrG;
double RT_LookatCamera::frV; int RT_LookatCamera::frF, RT_LookatCamera::frG;

RT_ParseEntry RT_LookatCamera::table[] = {
    { "-viewpoint", RTP_VECTOR, (char*)&vpV, &vpF, "Set {ARG 1 Viewpoint}.", RTPS_VECTOR },
    { "-get_viewpoint", RTP_NONE, 0, &vpG, "Get viewpoint.", RTPS_NONE },
    { "-refpoint", RTP_VECTOR, (char*)&rpV, &rpF, "Set {ARG 1 Referencepoint}.", RTPS_VECTOR },
    { "-get_refpoint", RTP_NONE, 0, &rpG, "Get refpoint.", RTPS_NONE },
    { "-twist", RTP_INTEGER, (char*)&twV, &twF, "Set {ARG 1 Twist} value - a right hand rotation about the z axis in the eye coordinate system.", RTPS_INTEGER },
    { "-get_twist", RTP_NONE, 0, &twG, "Return the twist value.", RTPS_NONE },
    { "-angle", RTP_DOUBLE, (char*)&anV, &anF, "Set {ARG 1 Angle} - the field-of-view angle in y direction in Degrees.", RTPS_DOUBLE },
    { "-get_angle", RTP_NONE, 0, &anG, "Return angle.", RTPS_NONE },
    { "-znear", RTP_DOUBLE, (char*)&nrV, &nrF, "Set {ARG 1 Distance } from the viewer to the closest clipping plane.", RTPS_DOUBLE },
    { "-get_znear", RTP_NONE, 0, &nrG, "Return near distance.", RTPS_NONE },
    { "-zfar", RTP_DOUBLE, (char*)&frV, &frF, "Set {ARG 1 Distance } from the viewer to the farthest clipping plane.", RTPS_DOUBLE },
    { "-get_zfar", RTP_NONE, 0, &frG, "Return far distance.", RTPS_NONE },
    { "-mode", RTP_INTEGER, (char*)&moV, &moF, 
#ifdef RTD_RSY
      "Set the {ARG 1 RenderingMode}.  Possible values of the integer are (from zero) {ENUM 1 \"Shading Raytracing Radiosity DiffuseOnlyRadiosity WireFrameRadiosity\"}.",
#else
      "Set the {ARG 1 RenderingMode}.  Possible values of the integer are (from zero) {ENUM 1 \"Shading Raytracing\"}.",
#endif	  
      RTPS_INTEGER },
    { "-get_mode", RTP_NONE, 0, &moG, "Return rendering mode.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

void RT_LookatCamera::event(RT_Event &ev) { 
    if (ev.isA( RTN_UPDATE_EVENT )) {
	if (get_mode() != RTE_LC_RAY_TRACING ) rendering();
    }
    else RT_Camera::event( ev );
}

int RT_LookatCamera::objectCMD(char *argv[]) {
    int r = RT_Camera::objectCMD( argv );
    RT_parseTable( argv, table );
    if (vpF) { viewpoint( vpV ); r++; };
    if (vpG) {
	r++;
	char tmp[60];
	RT_vector2string( get_viewpoint(), tmp );
	RT_Object::result( tmp );
    }
    if (rpF) { refpoint( rpV ); r++; };
    if (rpG) {
	r++;
	char tmp[60];
	RT_vector2string( get_refpoint(), tmp );
	RT_Object::result( tmp );
    }
    if (twF) { twist( twV ); r++; }
    if (twG) {
	r++;
	char tmp[20];
	RT_int2string( get_twist(), tmp );
	RT_Object::result( tmp );
    }
    if (moF) { 
	if (moV >= RTE_LC_SHADING && moV <= 

#ifndef RTD_RSY
	RTE_LC_RAY_TRACING 
#else 
	RTE_LC_WIREFRAME_RADIOSITY 
#endif
	    ) { mode( (RT_LookatCameraMode) moV ); r++; }
	else rt_Output->errorVar( get_name(), ": Bad camera mode.", 0 );
    }
    if (moG) {
	r++;
	char tmp[20];
	RT_int2string( get_mode(), tmp );
	RT_Object::result( tmp );
    }
    if (anF) { angle( anV ); r++; }
    if (anG) {
	r++;
	char tmp[20];
	RT_double2string( get_angle(), tmp );
	RT_Object::result( tmp );
    }
    if (nrF) { znear( nrV ); r++; }
    if (nrG) {
	r++;
	char tmp[20];
	RT_double2string( get_znear(), tmp );
	RT_Object::result( tmp );
    }
    if (frF) { zfar( frV ); r++; }
    if (frG) {
	r++;
	char tmp[20];
	RT_double2string( get_zfar(), tmp );
	RT_Object::result( tmp );
    }
    return r;
}

void RT_LookatCamera::rendering() {
    if (!get_pixmap() || !get_scene()) return;
    switch (xmode) {
      case RTE_LC_SHADING:
	RT_Camera::rendering();
	shading();
	break;
      case RTE_LC_RAY_TRACING: {
	  RT_RayCamera::rendering();
	  // update the WC box inside the primitives:
	  RT_PrimitiveBoundingFunc func;
	  get_scene()->doWithElements( &func );
	  int pw = get_pixmap()->getW();
	  int ph = get_pixmap()->getH();
	  double wc = 1.0 / (pw-1);
	  double hc = 1.0 / (ph-1);
	  RT_Ray ray;
	  for (int j = ph-1; j >= 0; j--) {
	      double yy = hc * (double)j;
	      for (int i = 0; i < pw; i++) {
		  double xx = wc * (double)i;
		  RT_Color colr;
		  buildRay( xx, yy, ray );
		  traceCnt = 0;
		  get_pixmap()->putPixel( i, j, RT_Color( 1,0,0) );
		  trace( 0, 1, ray, colr );
		  get_pixmap()->putPixel( i, j, colr );
	      }
	  }
	  break;
      }
#ifdef RTD_RSY
      case RTE_LC_RADIOSITY:
      case RTE_LC_DIFFUSE_RADIOSITY:
      case RTE_LC_WIREFRAME_RADIOSITY:  
	RT_Camera::rendering();
	RSYshading();
#endif
    }
}

void RT_LookatCamera::computeLookatMatrix() {
    RT_Vector tr; // translation eye and world coordinate system
    RT_Vector dr; // lookat direction
    double twist; // the z axis rotation angle 

    // set start values:
    dr = rp - vp; // direction
    twist = -xtwist * M_PI / 180.0;

    // unitize direction vector:
    dr = dr.UNITIZE();

    // compute y vector:
    RT_Vector ye;
    ye.x = 0; ye.y = 1; ye.z = 0;  // defaults

    // is dr kollinear with ye ?:
    if((dr.x==0) && (dr.y==1) && (dr.z==0)) { ye.x = 0; ye.y = 0; ye.z = 1; }
    
    if((dr.x==0) && (dr.y==-1) && (dr.z==0)) { ye.x=0; ye.y =0; ye.z = -1; }
    
    // compute eye point xe vector:
    RT_Vector xe;
    xe = ye.CROSS( dr); 

    // unitize xe vector:
    xe = xe.UNITIZE();

    // compute ye vector:
    ye = dr.CROSS( xe); 

    RT_Matrix trRt; // translation and rotation matrix
    trRt.set(0,0, xe.x); trRt.set(1,0, xe.y); trRt.set(2,0, xe.z); 
    trRt.set(0,1, ye.x); trRt.set(1,1, ye.y); trRt.set(2,1, ye.z); 
    trRt.set(0,2, dr.x); trRt.set(1,2, dr.y); trRt.set(2,2, dr.z);

    trRt.set(0,3, vp.x ); trRt.set(1,3, vp.y ); trRt.set(2,3, vp.z );
    trRt.set(3,0, 0); trRt.set(3,1, 0); trRt.set(3,2, 0); trRt.set(3,3, 1);

    RT_Matrix tw;
    tw.set(0,0, cos(twist));
    tw.set(0,1, -sin(twist));
    tw.set(1,0, sin(twist));
    tw.set(1,1, cos(twist));
        
    lam = (trRt*tw);
}

void RT_LookatCamera::buildRay( double x, double y, RT_Ray &ray) {
    double aspect = xpixmap->getW()/(double)xpixmap->getH();
    
    // transformation of the ray coordinates 0,1->-1,1    
    RT_Vector vn;  // normalized vector
    vn.x = (-2*x+1)*aspect; vn.y = 2*y-1; vn.z = 1/tan( xangle/2.0 * M_PI / 180.0 );

    RT_Vector vw;  // world coordinates
    vw= lam * vn;
    
    RT_Vector dir;
    dir.x = vw.x-vp.x; dir.y = vw.y-vp.y; dir.z = vw.z-vp.z;
    
    //result: 
    ray.dir = dir.UNITIZE(); ray.pt = vp ;
    
    // set the tmin value to the near plane and tmax to far:
    
    // first define the points for near and far plane
    RT_Vector pnear( 0, 0, get_znear());
    pnear = lam * pnear;
    RT_Vector pfar( 0, 0, get_zfar());
    pfar = lam * pfar;

    // define the normal:
    RT_Vector vnorm = ((lam * RT_Vector( 0,0,1 )) - (lam * RT_Vector( 0,0,0 ))).UNITIZE();

    // the near and far distance in wc
    double dnear = vnorm.DOT(pnear);
    double dfar  = vnorm.DOT(pfar);
    // calculate the t values for tmin and tmax
    double t1 = vnorm.DOT(ray.pt);
    double t2 = vnorm.DOT(ray.dir);
    ray.tmin = (dnear - t1)/t2; 
    ray.tmax = (dfar  - t1)/t2;
}

const char *RTN_RAY_IMAGE_LOOKAT_CAMERA ="RayImageLookatCamera";
const char *RTSN_RILC = "RILC";

void RT_RayImageLookatCamera::buildRay( double x, double y, RT_Ray &ray) {
    double aspect = ximage->get_width()/(double)ximage->get_height();
    
    // transformation of the ray coordinates 0,1->-1,1    
    RT_Vector vn;  // normalized vector
    vn.x = (-2*x+1)*aspect; vn.y = 2*y-1; vn.z = 1/tan( xangle/2.0 * M_PI / 180.0 );

    RT_Vector vw;  // world coordinates
    vw=(lam*vn);
    
    RT_Vector dir;
    dir.x = vw.x-vp.x; dir.y = vw.y-vp.y; dir.z = vw.z-vp.z;
    
    //result: 
    ray.dir = dir.UNITIZE(); ray.pt = vp ;
    
    // set the tmin value to the near plane and tmax to far:
    
    // first define the points for near and far plane
    RT_Vector pnear(0,0,get_znear());
    pnear = lam*pnear;
    RT_Vector pfar(0,0,get_zfar());
    pfar = lam*pfar;

    // define the normal
    RT_Vector vnorm = ((lam * RT_Vector( 0,0,1 )) - (lam * RT_Vector( 0,0,0 ))).UNITIZE();
    
    // the near and far distance in wc
    double dnear = vnorm.DOT(pnear);
    double dfar  = vnorm.DOT(pfar);

    // calculate the t values for tmin and tmax
    double t1 = vnorm.DOT(ray.pt);
    double t2 = vnorm.DOT(ray.dir);
    ray.tmin = (dnear - t1)/t2; 
    ray.tmax = (dfar  - t1)/t2; 
}

int RT_RayImageLookatCamera::classCMD(ClientData cd, Tcl_Interp *ip, int argc, char *argv[]) { 
    int res;
    res = _classCMD(cd, ip, argc, argv);
    if (res == TCL_HELP) {
	Tcl_AppendResult( ip, "{", argv[0], " {String Vector Vector} {Creates a new camera called {ARG 1 Name}, {ARG 2 Viewpoint} is the eye point and {ARG 3 Referencepoint} ist the point the camera is looking at. Short name is ", RTSN_RILC, ".}}", 0 );
	return TCL_OK;
    }
    if ( res  == TCL_OK ) {  
	if ( argc != 4 ) {
	    Tcl_AppendResult( ip, "Bad syntax. Must be: ", argv[0], " <name> <viewpoint[3]> <referencepoint[3]>! ", NULL );
	    return TCL_ERROR;
	}
	RT_Vector vp, rp; int dum;
	if (RT_getVector( &argv[2], vp, dum ) && RT_getVector( &argv[3], rp, dum )) { 
	    new RT_RayImageLookatCamera( argv[1], vp, rp ); 
	    RTM_classReturn;
	}
	Tcl_AppendResult( ip, "Bad parameters. <viewpoint[]> <referencepoint[]> must be float vectors! ", NULL );
	return TCL_ERROR;
    }
    return res; 
}

int RT_RayImageLookatCamera::nsV, RT_RayImageLookatCamera::nsF, RT_RayImageLookatCamera::nsG;
double RT_RayImageLookatCamera::jtV; int RT_RayImageLookatCamera::jtF, RT_RayImageLookatCamera::jtG;
char *RT_RayImageLookatCamera::imgV; int RT_RayImageLookatCamera::imgF, RT_RayImageLookatCamera::imgG;

RT_ParseEntry RT_RayImageLookatCamera::table[] = {
    { "-image", RTP_STRING, (char*)&imgV, &imgF, "Specify an {ARG 1 Image} device.", "Object" },
    { "-get_image", RTP_NONE, 0, &imgG, "Return the current image.", RTPS_NONE },
    { "-samples", RTP_INTEGER, (char*)&nsV, &nsF, "Set {ARG 1 Samples} value - number of samples per pixel in each direction.", RTPS_INTEGER },
    { "-get_samples", RTP_NONE, 0, &nsG, "Return the number of samples.", RTPS_NONE },
    { "-jitter", RTP_DOUBLE, (char*)&jtV, &jtF, "Set {ARG 1 Jitter } value - amount of jitter on a sample.", RTPS_DOUBLE },
    { "-get_jitter", RTP_NONE, 0, &jtG, "Return jitter.", RTPS_NONE },
    { 0, RTP_END, 0, 0, 0, 0 }
};

int RT_RayImageLookatCamera::objectCMD(char *argv[]) {
    int r = RT_LookatCamera::objectCMD( argv );
    RT_parseTable( argv, table );
    if (nsF) { samples( nsV ); r++; }
    if (nsG) {
	r++;
	char tmp[20];
	RT_int2string( get_samples(), tmp );
	RT_Object::result( tmp );
    }
    if (jtF) { jitter( jtV ); r++; }
    if (jtG) {
	r++;
	char tmp[20];
	RT_double2string( get_jitter(), tmp );
	RT_Object::result( tmp );
    }
    // look for scene or pixmap values:
    if (imgF) {
	RT_Object *obj = RT_Object::getObject( imgV );
	if (!obj || !(obj->isA( RTN_IMAGE ))) 
	    rt_Output->errorVar( get_name(), ": No such image object: ", imgV, "!", 0 );
	else { image( (RT_Image*)obj ); r++;}
    }
    if (imgG) { RT_Object::result( get_image() ? get_image()->get_name() : "" ); r++; } 
    return r;
} 

#ifndef RAND_MAX
#define RAND_MAX 32767
#endif

void RT_RayImageLookatCamera::rendering() {
    if (!get_image() || !get_scene()) return;
    RT_RayCamera::rendering();
    // update the WC box inside the primitives:

    RT_PrimitiveBoundingFunc func;
    get_scene()->doWithElements( &func );
    int iw = get_image()->get_width();
    int ih = get_image()->get_height();
    double wc = 1.0 / (iw-1);
    double hc = 1.0 / (ih-1);
    double rgb[3];
    RT_Ray ray;
    RT_Color pixel_colr;
    for (int j = ih-1; j >= 0; j--) {
	static int pc = -1;
	int tpc = ((ih-1-j)*100)/ih;
	if (tpc > pc) {
	    char tmp[5];
	    sprintf( tmp, "%i", tpc );
	    rt_Output->messageVar( get_name(), ": ", tmp, "%.", 0 );
	}
	pc = tpc;
        for (int i = 0; i < iw; i++) {
	    rgb[0] = rgb[1] = rgb[2] = 0;
	    for (int l = 0; l < xsamples; l++) {
		double yy = hc * ((double)j + (l + rand() *
			xjitter / RAND_MAX) / xsamples);
		for (int k = 0; k < xsamples; k++) {
		    double xx = wc * ((double)i + (k + rand() *
			    xjitter / RAND_MAX) / xsamples);
		    RT_Color colr;
		    buildRay( xx, yy, ray );
		    traceCnt = 0;
		    trace( 0, 1, ray, colr );
		    rgb[0] += colr.r;
		    rgb[1] += colr.g;
		    rgb[2] += colr.b;
		}
	    }
	rgb[0] /= xsamples * xsamples;
	rgb[1] /= xsamples * xsamples;
	rgb[2] /= xsamples * xsamples;
	pixel_colr.setRGB( rgb );
        ximage->color( i, j, pixel_colr );
	}
    }
    ximage->write();
}

void RT_RayImageLookatCamera::image(RT_Image *im) { 
    if (ximage) {
	if (long(im) == -1) im = 0;
	else ximage->removeRelatedObject( this );
    }
    ximage = im; 
    if (ximage) {
	ximage->addRelatedObject( this );
	addRelatedObject( ximage );
    }
}


