////////////////////////////////////////////////////////////////////////////////
// Declaration of the base class of all depictable primitives.                //  
//  LAST EDIT: Wed Mar  8 13:56:03 1995 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#ifndef __PRIMITIVE_H__
#define __PRIMITIVE_H__

#ifndef RTD_CPP_INCLUDES
extern "C" {
#endif

#include <string.h>
#include <math.h>    

#ifndef RTD_CPP_INCLUDES
}
#endif

#include "rlist.h"
#include "vector.h"
#include "surface.h"
#include "object.h"
#include "matrix.h"
#include "mapping.h"
#include "predattr.h"
#include "lowlevel.h"

#ifdef RTD_RSY
#include "rsy/rsyscene.h"
#endif

class RT_InterSection;
class RT_InterSectionList;
class RT_Camera;
class RT_LookatCamera;

// enumerations:

typedef enum RT_Pickability {
    RTE_NOT_PICKABLE = 0,
    RTE_PICKABLE,             // default pick mode for non-parts
    RTE_PICK_FATHER
};

typedef enum RT_BoxMode {
    RTE_NO_BOX = 0, // default
    RTE_OBJECT_BOX,
    RTE_HIERARCHICAL_BOX,
    RTE_FILLED_BOX,
    RTE_FILLED_HIERARCHICAL_BOX
};

// class RT_Primitive:

extern const char *RTN_PRIMITIVE;

class RT_Primitive: public RT_Object, public RT_SurfaceAttrImpl, public RT_MappingAttrImpl {
    //##### some friends:
    friend class RT_AttributeList;
    friend class RT_TransferModelFunc;
    //##### privates:
    RT_Bounds wcbounds; // the hierarchical bounds in WC
    int rayInBounds(const RT_Ray&);
    // returns 1 if ray fits the bounding box

    void createSurface();

    void createModel() { 
	if (!hasModel) {
	    setModel( new RT_Modelling ); 
	    hasModel = 1; 
	    if (isPart && get_father()) get_father()->partModelling = 1;
	    // force the father to transfer down changes of his matrix
	    // on default partModelling is turned off - that saves
	    // a lot of time for e.g. OFF objects 
	    if (get_father()) model->concat( get_father()->model );
	}
    }

    void replacedFatherModelling( RT_Modelling *);
    // update the pointer to the modelling matrices
    // of an object that does not have private
    // modelling matrices:

    int parseAttributes(char *[]);
    // parse attributes 

    int xhide;
    // true if the object is hidden
    // flase if the object is accessible

    RT_SurfaceAttribute *xsurface;
    // private reference to the surface attribute

    RT_MappingAttribute *xmapping;
    // private reference to the mapping attribute

    RT_BoxMode xbox;
    // if 1 draw a bounding box

    RT_Modelling *model;
    // the modelling matrices

    static RT_ParseEntry table[], attrTable;
    //#### the statics for parameter parsing:
    static int rotF, scaF, traF, matF, matG, wmatG, fatF, fatG, chiG;
    static int visF, invF, visG, bndG, hbdG;
    static char *fatV;
    static int hdF, unF, hdG;
    static RT_Vector rotV, scaV, traV;
    static RT_Matrix matV;
    static int clAtF; static char *clAtV;
    static int bxF, bxV, bxG, updF, atLiG;
    static int pmF, pmV, pmG;
    static int cpF; static char *cpV;
    //##### private flags:
    int hasModel;
    // if this is 1 the object has own modelling matrices
    int geomChangedFlag;
    //  if this is 1 the private geometry was changed
    int attrsChangedFlag;
    // this is TRUE when the attribute list of this object or of a child was changed 
    // e.g. attribute removed or inserted 
    int matrixChangedFlag;
    // this is TRUE when the !global! or local transformation was changed 
    // will be reset when rendered

  protected:
    void removeChild( RT_Primitive *c ) { 
	if (c->isPart) parts.remove( c );
	else children.remove( c ); 
	primChanged();
    }
    void addChild( RT_Primitive *c) {
	if (c->isPart) parts.append( c ); 
	else children.append( c ); 
    }

    RT_Primitive *xfather;
    // pointer to father object

    RT_GeneralList children;
    // children are separate objects, that are inserted and removed after
    // object creation time

    RT_GeneralList parts;
    // parts are built-in primitives from that the object is built of
    // all parts are inserted in one display list
    // parts will not saved and dont have a name!!
    // parts cannot have childs but parts!
    // the primitive itself has to raytrace/bound all
    // parts correctly

    RT_AttributeList *attributes;
    // private attribute list

    //##### essential flags:
    int dlChangedFlag;
    // this is TRUE when the object was changed 
    // or when a part of it or an attribute was changed
    // so that the display list must be regenerated
#ifdef RTD_RSY
    int rsyChangedFlag;
    // this is TRUE if the radiosity geometry has to be regenerated
#endif
    int partModelling;
    // if this is true - compute the modelling matrices of the part
    int xvis;
    // if the prim (and its parts) are visible - this is 1 (default)
    RT_Pickability xpick;
    // pickmode:
  public:
    void changedFatherModelling( RT_Modelling *m){ 
	// update the modelling matrices and or flags
	// if the contents of fathers modelling matrices
	// have been changed
	if (hasModel) model->concat( m ); 
	matrixChanged();
    }

    RT_Vector mc2wc(const RT_Vector &a) const  { return model->mc2wc( a ); }
    RT_Vector wc2mc(const RT_Vector &a) const { return model->wc2mc( a ); }

    void setModel( RT_Modelling *);
    // set new modelling matrices (temporary or for ever)

    RT_Modelling *getModel() {return model; }
    // set new modelling matrices (temporary or for ever)

    int isPart;
    // if this is 1 the prim is part of an other object
    // 0: it is seperate object
    RT_Attribute *get_privAttribute(const char *cls) {
	// get the attr with the specified class name from the private list
	return attributes->get( cls );
    }
    int isA(const char *_c) const { return RTM_isA(_c, RTN_PRIMITIVE ) || RT_Object::isA( _c ); }
    //##### essential 'private' routines:
    // private because not accessible from the official API (Tcl)

    void attrsChanged(); 
    // set recursively the attrs changed flag
    void geomChanged(); 
    // should be called from prims when their private geometry was changed
    void primChanged();
    // should called when anything of the primitive has changed
    // ( parts, surface, matrix, geometry)
    void matrixChanged(); 
    // should called when the matrix was changed

    //#### overloadable functions called before rendering when prim has changed 
    virtual void createReferences(const RT_AttributeList &);
    // link own pointers to attributes in the list
    // or to default attributes

    virtual void checkAttributes(); 
    // check if attributes have changed so that object must be reimplemented

    virtual void newMatrix() {}
    // this overloadable function will be called when the modeling matrix was changed 

    virtual void create() {}
    // this function will called when the private geometry of the
    // primitive has been changed

    // #### Tcl/C++ methods:
    RT_Primitive(char * = 0 );
    virtual ~RT_Primitive(); 

    virtual void update( int cr = 0 );
    // update the geometry, properties and matrices of the primitive and its children
    // if cr (create references) == 1 -> create references before 

    void referencing( const RT_AttributeList * = 0);
    // let the primitive bind attribute references 
    // if the list is not 0 then a complete
    // attribute context ( down transferred from root)
    // is in the list

    void father(RT_Primitive *f = 0);
    // if f == -1 the father primitive deletes its 
    // own parts (in destructor) 
    // thus don't remove this from the fathers list
    RT_Primitive *get_father() const { return xfather; }
    const RT_GeneralList &get_children() const { return children; }

    void hide() { xhide = 1; }
    void unhide() { xhide = 0; }
    // (un)hide the object
    int get_hide() { return xhide; }

    void pickmode( RT_Pickability p ) { xpick = p; }
    RT_Pickability get_pickmode() const { return xpick; }
    
    void box( RT_BoxMode _b) { xbox = _b; } 
    RT_BoxMode get_box() const { return xbox; }
    
    void visible() { xvis = 1; }
    void invisible() { xvis = 0; }
    int get_visible() { return xvis; }
    
    virtual int copy(RT_Primitive *p) const {
	rt_Output->errorVar( get_name(), ": Cannot copy a ", get_class(), " into a ", p->get_class(), ".", 0 );
	return 0;
    }
    // copy into another primitive
    // return 1 if successful

    //#### shader interface:
    void renderHierarchical();
    // render the primitive and its children 
    
    virtual void render();
    // default: render the parts
    
    //#### raytracer interface:
    int intersectHierarchical(const RT_Ray &, RT_InterSectionList&);
    // intersect with the primitive and all its children

    virtual int intersect(const RT_Ray&, RT_InterSectionList&);
    // default: intersect with all parts

    virtual void normal(const RT_Vector &p, RT_Vector &n) { 
	rt_Output->warningVar( "You called the DUMMY routine for the computing of the normal of object ",
			      get_name(), ". Did you forget to implement it?", 0 );
	n = p; 
    }
    // compute Normal n at given Point p
    // dummy implementation
    // this should never be called!

    void mappedSurface(const RT_Vector &WC, RT_Surface &SURFACE); 
    // change the overgiven SURFACE in result of a texture mapping

    void mappedNormal(const RT_Vector &WC, RT_Vector &NORMAL);
    // get the normal vector of the primitive at the WC point
    // and (optionally) change it by bump mapping
 
    RT_Bounds mobox();
    // compute bounding box inclusive modelling transformation

    virtual RT_Bounds get_bounds(); 
    // get object bounds im MC
    // default: return boundaries of the parts 

    RT_Bounds get_hierarchBounds();
    // get the bounds inclusive childs in MC

    virtual RT_Point inverse(const RT_Vector &) const {
	// inverse mapping of the overgiven MC point into a (1,1) area in dependence
	// on the primitive; dummy implementation - should be overloaded: 
	RT_Point p; p.x = p.y = -1; return p;
    }

#ifdef RTD_RSY
    //#### radiosity interface:
    void toRSYHierarchical(RT_RSYScene *);
    // insert the primitive and its childs into a radiosity scene
    virtual void toRSY(RT_RSYScene *); 
#endif
    
    //#### interface of SURFACE attribute:
    void surface(const RT_Surface &s) { createSurface(); xsurface->surface( s ); }
    const RT_Surface &get_surface() const { return xsurface->get_surface(); }
    void ambient( const RT_Color &ambi) { createSurface(); xsurface->ambient( ambi ); }
    const RT_Color &get_ambient() const { return xsurface->get_ambient(); }
    void diffuse(const RT_Color &diff) { createSurface(); xsurface->diffuse( diff ); }
    const RT_Color &get_diffuse() const { return xsurface->get_diffuse(); }
    void emission(const RT_Color &colr)  { createSurface(); xsurface->emission( colr ); }
    const RT_Color &get_emission() const { return xsurface->get_emission(); }
    void specular(const RT_Color &spec)  { createSurface(); xsurface->specular( spec ); }
    const RT_Color &get_specular() const { return xsurface->get_specular(); }
    void transmission(const double trans) { createSurface(); xsurface->transmission( trans ); }
    double get_transmission() const { return xsurface->get_transmission(); }
    void refraction(const double refr)  { createSurface(); xsurface->refraction( refr ); }
    double get_refraction() const { return xsurface->get_refraction(); }
    void shininess(const double shin)  { createSurface(); xsurface->shininess( shin ); }
    double get_shininess() const { return xsurface->get_shininess(); }
    void specfunc(const char *s) { createSurface(); xsurface->specfunc( s ); }
    const char *get_specfunc() const { return xsurface->get_specfunc(); }
    //#### the tcl commands:
    int objectCMD(char *[]);
    //#### access to attributes:
    void attribute( const RT_Attribute&);
    const RT_Attribute &get_attribute(const char *) const;
    // get the specified attr by retraversing
    void attrNames(const char*, const char*);
    // set a list of class and object names for a attribute
    void attrClear( const char * );
    // clear an attribute from primitives list
    const RT_AttributeList *get_attrList() { return attributes; }
    // access to the attribute list of the object

    //#### modelling operations:
    void matrix(const RT_Matrix &m) {
	createModel();
	model->matrix( m );
	matrixChanged();
    }
    const RT_Matrix &get_matrix() const { return hasModel ? model->get_matrix() : RT_IDENTITY; }
    const RT_Matrix &get_worldMatrix() const { return model->get_concatMatrix(); }
    void rotate( const RT_Vector &x) {
	createModel();
	model->rotate( x );
	matrixChanged();
    }
    void scale( const RT_Vector &x) {
	createModel();
	model->scale( x );
	matrixChanged();
    }
    void translate( const RT_Vector &x) {
	createModel();
	model->translate( x );
	matrixChanged();
    }
    void mapping( RT_Mapping *ma) {
	RT_Attribute *at = attributes->get( RTN_MAPPING_ATTR );
	if (at) attributes->remove( at );
	attributes->insert( xmapping = new RT_MappingAttribute( ma )); 
    }
    RT_Mapping *get_mapping() const { return xmapping->get_mapping(); }

    //#### etc. 

    void printHierarchical( FILE *);
    // print out the primitive and its childs

    void print(FILE *) const;
};

class RT_PrintObjectNameFunc: public RT_GeneralListFunctoid {
    RT_String *str;
  public:
    RT_PrintObjectNameFunc(RT_String *_str) { str = _str; }
    void exec(RT_GeneralListEntry *e, void * = 0) {
	*str += ((RT_Object*)e)->get_name();
	*str += ' ';
    }
}; 

#endif

