////////////////////////////////////////////////////////////////////////////////
//                    Functions using matrices and vectors.                   //
//  LAST EDIT: Fri Aug  5 08:55:05 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <values.h>

#include "rs_defs.h"
#include "rs_vec.h"
#include "rs_matrx.h"
#include "rs_io.h"
#include "../global.h"
#include "../error.h"

/////////////// FRIENDS OF RT_RS_Matrix AND RT_RS_Vector ////////////////////

RT_RS_Vector operator * (RT_RS_Vector& a, RT_RS_Matrix& b)
{
  int i,j;
  float sum;

#ifdef RS_DEBUG
  if(a.v == NULL || b.m == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Vector or RT_RS_Matrix in operator * (RT_RS_Vector&, RT_RS_Matrix&)", NULL);
  if(a.nc != b.nr)
    rt_Output->fatalVar("Dimensions do not match in operator * (RT_RS_Vector&, RT_RS_Matrix&)", NULL);
#endif
  RT_RS_Vector Temp(a.nr, b.nc, 'y');
  Temp.set(0.);
  for(i = 0; i < a.nelem; i++) {
    sum = 0.;
    for(j = 0; j < b.nr ; j++)
      sum += a.v[j] * b.m[j][i];
    Temp.v[i] = sum;
  }
  a.freet();
  b.freet();
  return Temp;
}

RT_RS_Vector operator * (RT_RS_Matrix& a, RT_RS_Vector& b)
{
  int i,j;
  double sum;

#ifdef RS_DEBUG
  if(a.m == NULL || b.v == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Vector or RT_RS_Matrix in operator * (RT_RS_Matrix&, RT_RS_Vector&)", NULL);
  if(a.nc != b.nr)
    rt_Output->fatalVar("Dimensions do not match in operator * (RT_RS_Matrix&, RT_RS_Vector&)", NULL);
#endif
  RT_RS_Vector Temp(a.nr, b.nc, 'y');
  Temp.set(0.);
  for(i = 0; i < b.nelem; i++) {
    sum = 0.;
    for(j = 0; j < a.nc; j++)
      sum += a.m[i][j]*b.v[j];
    Temp.v[i] = sum;
  }
  a.freet();
  b.freet();
  return(Temp);
}

RT_RS_Matrix operator * (RT_RS_Vector& a, RT_RS_Vector& b)
{
  int i,j,k;
  float sum;

#ifdef RS_DEBUG
  if(a.v == NULL || b.v == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Vector in operator * (RT_RS_Vector&, RT_RS_Vector&)", NULL);
  if(a.nc != b.nr)
    rt_Output->fatalVar("Dimensions do not match in operator * (RT_RS_Vector&, RT_RS_Vector&)", NULL);
#endif
  RT_RS_Matrix Temp(a.nr, b.nc, 'y');
  Temp.set(0.);
  for(i = 0; i < a.nr; i++)
    for(j = 0; j < b.nc; j++) {
      sum = 0.;
      for(k = 0; k < a.nc; k++)
        sum += a.v[k] * b.v[k];
      Temp.m[i][j] = sum;
    }
  a.freet();
  b.freet();
  return(Temp);
}

RT_RS_Matrix operator * (RT_RS_Matrix& a, RT_RS_Matrix& b)
{
  int i, j, k;
  float sum;

#ifdef RS_DEBUG
  if(a.m == NULL || b.m == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Matrix in operator * (RT_RS_Matrix&, RT_RS_Matrix&)", NULL);
  if(a.nc != b.nr)
    rt_Output->fatalVar("Dimensions do not match in operator * (RT_RS_Matrix&, RT_RS_Matrix&)", NULL);
#endif
  RT_RS_Matrix Temp(a.nr, b.nc, 'y');
  Temp.set(0.);
  for(i = 0; i < a.nr; i++)
    for(j = 0; j < b.nc; j++) {
      sum = 0.;
      for(k = 0; k < a.nc; k++)
        sum += a.m[i][k] * b.m[k][j];
      Temp.m[i][j] = sum;
    }
  a.freet();
  b.freet();
  return Temp;
}

// Solution of simultaneous linear equations
// (T_RS_Matrix)A * (T_RS_Vector)X = (T_RS_Vector)B,
// by Gaussian elimination with partial pivoting
// The contents of matrix A will be destroyed.
// Set flag = 0 to solve.
// Set flag = -1 to do a new back substitution for different B vector
// using the same A matrix previously reduced when flag = 0.
// Adaptation of simq (CEPHES Library by Stephen L. Moshier)
int MatEqSys(RT_RS_Matrix& A, RT_RS_Vector& X, RT_RS_Vector& B, int flag)
{
  static int IPS[RS_MAX_EQSYS];
  static int nm1;
  static int n;

  int i, j, ij, ip;
  int idxpiv, iback;
  int k, kp, kp1;
  double em, q, rownrm, big, size, pivot, sum;

#ifdef RS_DEBUG
  if(A.m == NULL || X.v == NULL || B.v == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Vector or RT_RS_Matrix in MatEqSys()", NULL);
#endif

  if( flag != 0 )
    goto solve;

  n = A.nr;
#ifdef RS_DEBUG
  if(n != A.nc || n > X.nelem || n > RS_MAX_EQSYS)
    rt_Output->fatalVar("Dimensions do not match in MatEqSys()", NULL);
#endif

  //  Initialize IPS and X
  ij=0;
  for( i=0; i<n; i++ )
  {
    IPS[i] = i;
    rownrm = 0.0;
    for( j=0; j<n; j++ )
    {
      q = fabs( A.m[i][j] );
      if( rownrm < q )
      rownrm = q;
      ++ij;
    }
    if( fabs(rownrm) < epsilon )
      return -1;
     X.v[i] = 1.0/rownrm;
  }

  //  Gaussian elimination with partial pivoting
  nm1 = n-1;
  for( k=0; k<nm1; k++ )
  {
    big= 0.0;
    for( i=k; i<n; i++ )
    {
      ip = IPS[i];
      size = fabs( A.m[ip][k] ) * X.v[ip];
      if( size > big )
      {
        big = size;
        idxpiv = i;
      }
    }

    if( fabs(big) < epsilon )
      return -2;
    if( idxpiv != k )
    {
      j = IPS[k];
      IPS[k] = IPS[idxpiv];
      IPS[idxpiv] = j;
    }
    kp = IPS[k];
    pivot = A.m[kp][k];
    kp1 = k+1;
    for( i=kp1; i<n; i++ )
    {
      ip = IPS[i];
      em = -A.m[ip][k]/pivot;
      A.m[ip][k] = -em;
      for( j=kp1; j<n; j++ )
        A.m[ip][j] += em * A.m[kp][j];
    }
  }
  if( fabs(A.m[IPS[n-1]][ n - 1]) < epsilon )
    return -3;
  //  back substitution
  solve:
  ip = IPS[0];
  X.v[0] = B.v[ip];
  for( i=1; i<n; i++ )
  {
    ip = IPS[i];
    sum = 0.0;
    for( j=0; j<i; j++ )
      sum += A.m[ip][j] * X.v[j];
    X.v[i] = B.v[ip] - sum;
  }

  X.v[n-1] = X.v[n-1]/A.m[IPS[n-1]][n - 1];

  for( iback=1; iback<n; iback++ )
  {
    // i goes (n-1),...,1
     i = nm1 - iback;
    ip = IPS[i];
    sum = 0.0;
    for( j=i+1; j<n; j++ )
      sum += A.m[ip][j] * X.v[j];
    X.v[i] = (X.v[i] - sum)/A.m[ip][i];
  }
  return 0;  // Ok
}

// Solution of simultaneous linear equations with a tridiagonal matrix
// (G. Engeln-Muellges, F. Reutter: "Formelsammlung zur numerischen Mathematik",
// BI Wissenschaftsverlag Mannheim/Wien/Zrich 1990
void TriMatEqSys(int n, RT_RS_Vector& lower, RT_RS_Vector& diag, RT_RS_Vector& upper,
                        RT_RS_Vector& XB, int flag)
{
  int i;
#ifdef RS_DEBUG
  if((n < 2) || (lower.nelem < n) || (upper.nelem < n) || (XB.nelem < n))
    rt_Output->fatalVar("Dimensions do not match in TriMatEqSys()", NULL);
  if(lower.v == NULL || diag.v == NULL || upper.v == NULL || XB.v == NULL)
    rt_Output->fatalVar("Reference to a deleted RT_RS_Vector or RT_RS_Matrix in TriMatEqSys()", NULL);
#endif 

  if(flag == 0) {
    for(i=1;i<n;i++) {
#ifdef RS_DEBUG
      if(fabs(diag.v[i-1]) < epsilon)
        rt_Output->fatalVar("A diagonal element is zero in TriMatEqSys()", NULL);
#endif
      lower.v[i] /= diag.v[i-1];
      diag.v[i] -= lower.v[i] * upper.v[i-1];
    }
  }
#ifdef RS_DEBUG
  if(fabs(diag.v[n-1]) < epsilon)
    rt_Output->fatalVar("Last diagonal element is zero in TriMatEqSys()", NULL);
#endif
  for(i=1;i<n;i++)                     //forward elimination
    XB.v[i] -= lower.v[i] * XB.v[i-1];

  XB.v[n-1] /= diag.v[n-1];               //backward
  for(i=n-2;i>=0;i--)
    XB.v[i] = (XB.v[i] - upper.v[i] * XB.v[i+1]) / diag.v[i];
}

//compute the coefficients of a non-parametric cubic spline
// (G. Engeln-Muellges, F. Reutter: "Formelsammlung zur numerischen Mathematik",
// BI Wissenschaftsverlag Mannheim/Wien/Zrich 1990
void nspline(RT_RS_Vector& x, RT_RS_Vector& y,
             RT_RS_Vector& b, RT_RS_Vector& c, RT_RS_Vector& d)
{
  int i;
  int n = x.nelem - 1;
  int nmin1 = n - 1;

#ifdef RS_DEBUG
  if((n < 3) || (y.nelem < n+1) || (b.nelem < n) ||
                (c.nelem < n) || (d.nelem < n))
    rt_Output->fatalVar("Dimensions do not match in nspline()", NULL);
#endif
  RT_RS_Vector a(n+1);
  RT_RS_Vector h(n);

  for(i=0;i<n;i++) {
    h.v[i] = x.v[i+1] - x.v[i];
#ifdef RS_DEBUG
    if(h.v[i] < epsilon)
      rt_Output->fatalVar("x-vector is not monoton in nspline()", NULL);
#endif
  }

  for(i=0;i<nmin1;i++) {
    a.v[i] = 3. * ((y.v[i+2] - y.v[i+1]) / h.v[i+1] -
                   (y.v[i+1] - y.v[i]) / h.v[i]);
    b.v[i] = h.v[i];
    c.v[i] = h.v[i+1];
    d.v[i] = 2. * (h.v[i] + h.v[i+1]);
  }

  switch(nmin1) {
    case 1:  { c.v[1] = a.v[0] / d.v[0]; break; }
    default: { TriMatEqSys(nmin1, b, d, c, a, 0);
               for(i=0; i<nmin1;i++)
                 c.v[i+1] = a.v[i];
             }
  }
  c.v[0] = 0.0;

  for(i=0;i<nmin1;i++) {
    b.v[i] = (y.v[i+1] - y.v[i]) / h.v[i] -
             h.v[i] * (c.v[i+1] + 2. * c.v[i]) / 3.;
    d.v[i] = (c.v[i+1] - c.v[i]) / (3. * h.v[i]);
  }
  b.v[nmin1] = (y.v[n] - y.v[nmin1]) / h.v[nmin1] -
             h.v[nmin1] * (2. * c.v[nmin1]) / 3.;
  d.v[i] = -c.v[nmin1] / (3. * h.v[nmin1]);
}

// evaluate a cubic spline at x0;
// (G. Engeln-Muellges, F. Reutter: "Formelsammlung zur numerischen Mathematik",
// BI Wissenschaftsverlag Mannheim/Wien/Zrich 1990
float spval(float x0,
            RT_RS_Vector& a, RT_RS_Vector& b, RT_RS_Vector& c,
            RT_RS_Vector& d, RT_RS_Vector& x)
{
  int m;
  int n = x.nelem;
  int i = 0;
  int k = n;

#ifdef RS_DEBUG
  m = n-1;
  if((a.nelem < m) || (b.nelem < m) || (c.nelem < m) || (d.nelem < m))
    rt_Output->fatalVar("Dimensions do not match in spval()", NULL);
  if((x0 < x.v[0]) || (x0 > x.v[m]))
    rt_Output->fatalVar("x0 is out of range in spval()", NULL);
#endif
  while(m = (i+k) >> 1, m != i)
    if(x0 < x.v[m]) k = m; else i = m;
  x0 -= x.v[i];
  return(((d.v[i] * x0 + c.v[i]) * x0 + b.v[i]) * x0 + a.v[i]);
}
