////////////////////////////////////////////////////////////////////////////////
// YART interface to OpenGL 1.0                                               //  
// LAST EDIT: Thu Mar  9 06:12:07 1995 by wicht(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRGHT which should be distributed with this  //
//  file. If COPYRGHT is not available or for more info please contact:       //
//                                                                            //  
//		yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //  
// (C) Copyright 1993 - 1995 YART team                                        //
////////////////////////////////////////////////////////////////////////////////

#include "../high/lookat.h"
#include "../high/usefligh.h"
#include "../quadmesh.h"
#include "../pixmap.h"
#include "../poly.h"
#include "../polyhdrn.h"

#ifndef RTD_CPP_INCLUDES
extern "C" {
#endif

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <GL/glx.h>
#include <GL/gl.h>
#include <GL/glu.h>

#ifndef RTD_CPP_INCLUDES
}
#endif

#include <unistd.h>


// Window Types
#define RGB		0
#define RGBA		RGB
#define INDEX		1
#define SINGLE		0
#define DOUBLE		2
#define DIRECT		0
#define INDIRECT	4
#define ACCUM		8
#define ALPHA		16
#define DEPTH		32
#define STENCIL		64


// Window Masks
#define IS_RGB(x)		(((x) & INDEX) == 0)
#define IS_INDEX(x)		(((x) & INDEX) != 0)
#define IS_SINGLE(x)		(((x) & DOUBLE) == 0)
#define IS_DOUBLE(x)		(((x) & DOUBLE) != 0)
#define HAS_ACCUM(x)		(((x) & ACCUM) != 0)
#define HAS_ALPHA(x)		(((x) & ALPHA) != 0)
#define HAS_DEPTH(x)		(((x) & DEPTH) != 0)
#define HAS_STENCIL(x)		(((x) & STENCIL) != 0)
#define IS_DIRECT(x)		(((x) & INDIRECT) != 0)
#define IS_INDIRECT(x)		(((x) & INDIRECT) == 0)

void RT_dlOpen(long id) {
    if (glIsList((GLuint) id)) glDeleteLists((GLuint) id, (GLsizei) 1);
    glNewList((GLuint) id, GL_COMPILE);
}

void RT_dlClose(long) { glEndList(); }

void RT_dlDelete(long id) { if (glIsList((GLuint) id)) glDeleteLists((GLuint) id, (GLsizei) 1);}
	
void RT_dlMatrix(const RT_Matrix &mat) {
    GLfloat m[4][4];
    for (int i = 0; i < 4; i++)
	for (int j = 0; j < 4; j++)
	    m[i][j] = mat.get(j, i);
    glPushMatrix();
    glMultMatrixf((GLfloat*)m);
}

void RT_dlUnMatrix() { glPopMatrix(); }

void RT_vtSurface(const RT_Surface &surface) {
    GLfloat ambi[4], diff[4], spec[4], emmi[4];

    surface.ambi.get_float(ambi);
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambi);
    surface.diff.get_float(diff);
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diff);
    surface.spec.get_float(spec);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, spec);
    surface.emmi.get_float(emmi);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emmi);
}

void RT_dlSurface(const RT_Surface &surface) {
    glColorMaterial(GL_FRONT_AND_BACK, GL_DIFFUSE) ;
    glColor3f(surface.diff.r, surface.diff.g, surface.diff.b);
    glColorMaterial(GL_FRONT_AND_BACK, GL_EMISSION);
    glColor3f(surface.emmi.r, surface.emmi.g, surface.emmi.b);
    glColorMaterial(GL_FRONT_AND_BACK, GL_SPECULAR);
    glColor3f(surface.spec.r, surface.spec.g, surface.spec.b);
    glColorMaterial(GL_FRONT_AND_BACK, GL_AMBIENT);
    glColor3f(surface.ambi.r, surface.ambi.g, surface.ambi.b);
}

void RT_dlCall(long id) { glCallList((GLuint) id); }

static int waitForNotify(Display*, XEvent *e, char *arg) { 
    return (e->type == MapNotify ) && (e->xmap.window == (Window)arg );
}

#if !defined TK

// the first pixmap will create a new graphics context
// all other link this
static GLXContext globalContext = 0;

// the class PixmapDisplay:
class RT_PixmapDisplayData {
  public:
    Display *display;
    Window window;
    int screen;
    XVisualInfo *visualInfo;
    Colormap colorMap;
    GLXContext context;
    int left, middle, right, width, height;
};

RT_PixmapDisplay::RT_PixmapDisplay(char *_name, int _w, int _h): RT_Pixmap(_name, _w, _h) {
    // config for doublebuffer mode
    int configuration_db[] = {
	GLX_RGBA,
	GLX_RED_SIZE, 1,
        GLX_GREEN_SIZE, 1,
        GLX_BLUE_SIZE, 1,
        GLX_DOUBLEBUFFER,
        GLX_DEPTH_SIZE, 1,
        None,
        };
    // config for singlebuffer mode
    int configuration_sb[] = {
        GLX_DEPTH_SIZE, 1,
        None,
        };
    XSetWindowAttributes wa;
    XEvent e;
    int erb, evb; // error base and event base
    unsigned long mask;

    data = new RT_PixmapDisplayData;
    data->width = _w;
    data->height = _h;
    data->left = data->middle = data->right = 0; 
    
    // open a connection to the X server 
    data->display=XOpenDisplay(0);
    if (!data->display) rt_Output->fatal ("Can't connect to display !");
    // is the GLX extension present ?
    if (!glXQueryExtension(data->display, &erb, &evb)) 
	rt_Output->fatal ("X server has no OpenGL GLX extension !");
    data->screen = DefaultScreen(data->display);
    sglBuf=0;
    data->visualInfo = glXChooseVisual( data->display, data->screen, configuration_db );
    if ( !data->visualInfo ) {
        sglBuf=1;
        data->visualInfo=glXChooseVisual(data->display, data->screen, configuration_sb);
        if ( !data->visualInfo ) 
	    rt_Output->fatal ("no appropriate RGB visual with depth buffer");
    }
    // create OpenGL context
    data->context=glXCreateContext( data->display, data->visualInfo, globalContext, GL_TRUE );
    if(!data->context) rt_Output->fatal ("Can't create a context !"); 
    if (!globalContext)	globalContext = data->context;
    // set standard properties
    data->colorMap=XCreateColormap(data->display, RootWindow(data->display, data->screen), data->visualInfo->visual, AllocNone);
    wa.colormap=data->colorMap;
    wa.border_pixel=0;
    wa.background_pixmap=None;
    wa.event_mask=StructureNotifyMask | ExposureMask | KeyPressMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
    mask=CWColormap | CWBorderPixel | CWEventMask | CWBackPixmap;
    data->window=XCreateWindow(data->display, RootWindow(data->display, data->visualInfo->screen), 0, 0, _w, _h, 0, data->visualInfo->depth, InputOutput, data->visualInfo->visual, mask, &wa);
    XSetStandardProperties(data->display, data->window, _name, _name, None, 0, 0, 0);
    // request the X window to be displayed on the screen
    XMapWindow(data->display, data->window);
    XIfEvent( data->display, &e, waitForNotify, (char*)data->window );
    // connect the context to the window
    if (!glXMakeCurrent(data->display, data->window, data->context)) 
	rt_Output->fatal ("Can't connect context and window !");
    XFlush( data->display );
}	
	  
RT_PixmapDisplay::~RT_PixmapDisplay() {
    if (data->display) {
	glFlush();
	glFinish();
	XDestroyWindow(data->display, data->window);
	glXDestroyContext(data->display, data->context);
	XFreeColormap(data->display, data->colorMap);
	XFree((char*)data->visualInfo);
    }
    delete(data);
}  


// activate the pixmap, called before rendering
// SGI GL drawing commands are related to the
// topical window
// thus set the topical window usin' this method
void RT_PixmapDisplay::activate() {
    if (!glXMakeCurrent(data->display, data->window, data->context)) 
	rt_Output->fatal("Can't connect context and window.");
}


void RT_PixmapDisplay::singlebuffer() { rt_Output->warning( "Can't change in singlebuffer mode !"); }
void RT_PixmapDisplay::doublebuffer() { rt_Output->warning( "Can't change in doublebuffer mode !"); }

// evaulation of events (if there are any)
void RT_PixmapDisplay::event() {
    XEvent ev;
    while ( XPending( data->display ) ) {
	XNextEvent( data->display, &ev );
	if( ev.xany.window != data->window) {
	    XPutBackEvent( data->display, &ev);
	    return;
	} 
	RT_Event *event = NULL;
	switch( ev.type ) {
	  case ButtonPress: { 
	      XButtonEvent *evb = (XButtonEvent *) &ev;
	      RT_ButtonEvent *pev = new RT_ButtonEvent;
	      if (evb->button == Button1) {pev->left = 1;}
	      if (evb->button == Button2) {pev->middle = 1;}
	      if (evb->button == Button3) {pev->right = 1;}
	      pev->w = w;
	      pev->h = h;
	      pev->x = evb->x;  
	      pev->y = h-evb->y;
	      event = (RT_Event*) pev; 
	      break;
	  }
	  case MotionNotify : {
	      XPointerMovedEvent *evm = (XPointerMovedEvent *) &ev;
	      RT_MotionEvent *pev = new RT_MotionEvent;
	      if (evm->state & Button1Mask) {pev->left = 1;}
	      if (evm->state & Button2Mask) {pev->middle = 1;}
	      if (evm->state & Button3Mask) {pev->right = 1;}
	      pev->w = w;
	      pev->h = h;
	      pev->x = evm->x;  
	      pev->y = h-evm->y;
	      event = (RT_Event*) pev;
	      break; 
	  }
	  break;
	  case Expose: event = new RT_UpdateEvent;
	} // switch 

	if (event && xcamera) {
	    xcamera->event( *event );
	    delete event;
	}
    } 
} 

RT_Color RT_PixmapDisplay::getPixel(int x, int y) {
    if (!checkIndices( x, y)) return RT_Color( 0,0,0);
    GLfloat c[3];
    glReadPixels(x, y, 1, 1, GL_RGB, GL_FLOAT, c);
    return(RT_Color(c[0], c[1], c[2]));
}

void RT_PixmapDisplay::putPixel(int x, int y, const RT_Color &colr) {
    GLfloat f[3];
    f[0] = colr.r; f[1] = colr.g; f[2] = colr.b;
    if (x==0 && y==data->height-1) {
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_LIGHTING);
	clear(RT_Color(0,0,0));
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluOrtho2D(0.0, (GLfloat) data->width , 0.0, (GLfloat) data->height);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
    }
    if (sglBuf==0) glDrawBuffer(GL_FRONT_AND_BACK);
    glColor3fv (f);
    glRecti(x, y, x+1 ,y+1);
    glFlush();
}

void RT_PixmapDisplay::clear(const RT_Color &colr) {
    float f[3];
    f[0]=colr.r; f[1]=colr.g; f[2]=colr.b; 
    glClearColor(f[0], f[1], f[2], 0.0);
    glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
}

#endif // TK

class RT_SetVtDataFunctoid: public RT_GeneralListFunctoid {
    RT_Polyvertex *pv;
    const RT_Surface *surface;
    float normal[4], ambi[4], spec[4], diff[4], emmi[4];
    void globalNorm() const;
    void globalMat() const;
    virtual void exec(RT_GeneralListEntry *, void*); 
  public:
    RT_SetVtDataFunctoid(RT_Polyvertex *_pv) {
	pv = _pv;
	surface = &pv->get_surface();
	GLfloat trans = 1-surface->tran;
	ambi[3] = spec[3] = diff[3] = emmi[3] = trans;
	pv->get_gnormal().get(normal); 
	surface->ambi.get_float(ambi);
	surface->spec.get_float(spec);
	surface->diff.get_float(diff);
	surface->emmi.get_float(emmi);
	globalMat();
	globalNorm();
    }
};

void RT_SetVtDataFunctoid::globalNorm() const { glNormal3fv(normal); }

void RT_SetVtDataFunctoid::globalMat() const {
    glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, ambi);
    glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, spec);
    glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, diff);
    glMaterialfv(GL_FRONT_AND_BACK, GL_EMISSION, emmi);
}

void RT_SetVtDataFunctoid::exec(RT_GeneralListEntry *_v, void*) {
    RT_Vertex *v = (RT_Vertex*)_v;
    const RT_Surface *vs;
    if ((vs=v->getSurface())!=NULL) {
	GLfloat tmp[4];
	tmp[3] = 1-vs->tran;
	vs->ambi.get_float( tmp );
	glMaterialfv(GL_FRONT_AND_BACK, GL_AMBIENT, tmp);
	vs->spec.get_float(tmp);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, tmp);
	vs->diff.get_float(tmp); 
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE, tmp);
	vs->emmi.get_float(tmp);
	glMaterialfv (GL_FRONT_AND_BACK, GL_EMISSION, tmp);
    } 
    const RT_Vector *vn=v->getNormal();
    if (vn) { float n[4]; vn->get(n); glNormal3fv(n); }
    float p[4];
    v->getPoint().get(p);
    glVertex3fv(p);
    if (vs) globalMat();
    if (vs || vn) globalNorm();
}

// rendering of the low level primitives:
void RT_Polymarker::render() {
    glBegin(GL_POINTS);
    RT_SetVtDataFunctoid func(this);
    doWithElements(&func);
    glEnd();
}

void RT_Polyline::render() {
    glBegin(GL_LINE_STRIP);
    RT_SetVtDataFunctoid func(this);
    doWithElements(&func);
    glEnd();
}

void RT_Polygon::render() {
    if (get_fillstyle()) glBegin(GL_POLYGON);
    else glBegin(GL_LINE_LOOP);
    RT_SetVtDataFunctoid func(this);
    doWithElements(&func);
    glEnd();
}

void RT_Quadmesh::render() {
    const RT_Surface *surface;
    surface = &get_surface();
    
    RT_Vertex *v;
    const RT_Vector *vn;
    const RT_Surface *vs;
    GLfloat ng[4],n[4],f[4];
    int rows=get_x(), cols=get_y(), nr=0;
    if (get_fillstyle()) {
	RT_dlSurface(*surface);
	get_gnormal().get(ng);
	glNormal3fv( ng );
	for(int i=0; i<rows-1; i++) {
	    glBegin(GL_QUAD_STRIP);
	    for(int j=0; j<cols; j++) {
		for( int p=0; p<2; p++) {
		    v = get(i+p,j);
		    if ((vs=v->getSurface())!=NULL) RT_vtSurface(*vs); 
		    if ((vn=v->getNormal())!=NULL) { vn->get(n); glNormal3fv(n); }
		    v->getPoint().get(f);
		    glVertex3fv(f);
		    if (vs) RT_vtSurface(*surface);
		    if (vs||vn) glNormal3fv(ng);
		}
	    }   
	    glEnd();
	}
    } else {
	RT_dlSurface(*surface);
	for(int i=0; i<rows; i++) {
	    glBegin (GL_LINE_STRIP);
	    for(int j=0; j<cols; j++) {
		v=get(i,j);  
		v->getPoint().get(f);
		glVertex3fv(f);
	    }
	    glEnd();
	}
	for(int j=0; j<cols; j++) {
	    glBegin (GL_LINE_STRIP);
	    for(int i=0; i<rows; i++) {
		const RT_Surface *vs = v->getSurface();
		if (vs) RT_vtSurface(*vs);
		if ((vn=v->getNormal())!=NULL) { vn->get(n); glNormal3fv(n); }
		v = get(i,j);  
		v->getPoint().get(f);
		glVertex3fv(f);
		if (vs) RT_vtSurface(*surface);
		if (vs||vn) glNormal3fv(ng);
	    }
	    glEnd();
	}
    }
}

void RT_Polyhedron::render() {
    RT_Vertex *v;
    const RT_Vector *vn;
    const RT_Surface *vs;
    float n[4],f[4];
    const RT_Surface *surface = &get_surface();
    int fstyle=get_fillstyle();

    RT_dlSurface(*surface);
    int fc=get_nfacets();
    for (int i=0; i<fc; i++) {
    	if (fstyle) glBegin(GL_POLYGON);
   	else glBegin (GL_LINE_LOOP);
	int nb=facets[i].get_number(); 
	for (int j=0; j<nb; j++) {
	    v=get(i, j);
	    if ((vs=v->getSurface())!=NULL) RT_dlSurface(*vs);
	    if ((vn=v->getNormal())!=NULL) vn->get(n); glNormal3fv(n); 
	    v->getPoint().get(f);
	    glVertex3fv(f);
	    if (vs) RT_dlSurface(*surface);
	}
	glEnd();
    }
}

// rendering of the light sources
// the systems provides for the correct sequence
// meaning that lights will be rendered before
// primitives will be rendered
static int GL_LightNumber = 0;

void RT_AmbientLight::render() {
    int maxLights;

    if (!get_on()) return;
    glGetIntegerv(GL_MAX_LIGHTS, &maxLights);
    if (GL_LightNumber<maxLights) {
	GLfloat ambient[4], specular[4], diffuse[4];
	RT_Color color = get_color();
	int ln = GL_LIGHT0+GL_LightNumber;
	
	ambient[0]=color.r; ambient[1]=color.g; ambient[2]=color.b; ambient[3]=1.0;
	specular[0]=0.0; specular[1]=0.0; specular[2]=0.0; specular[3]=0.0;
	diffuse[0]=0.0; diffuse[1]=0.0; diffuse[2]=0.0; diffuse[3]=0.0;
	glLightfv(ln, GL_AMBIENT, ambient);
	glLightfv(ln, GL_SPECULAR, specular);
	glLightfv(ln, GL_DIFFUSE, diffuse);
	glEnable(ln);
	GL_LightNumber++;
    } else {
	char error[100];
	sprintf(error, "GL can't define more then %d lights\n", maxLights);
	rt_Output->warning(error);
    }
}

void RT_PointLight::render() {
    int maxLights;

    if (!get_on()) return;
    glGetIntegerv(GL_MAX_LIGHTS, &maxLights);
    if (GL_LightNumber<maxLights) {
	GLfloat ambient[4], specular[4], diffuse[4];
	GLfloat position[4], direction[3], spot_exp[1], spot_cutoff[1];
	RT_Vector vector = get_origin();
	int n=0, ln = GL_LIGHT0+GL_LightNumber;
	RT_Color c = get_color();
	RT_Color cs(c.r*0.2, c.g*0.2, c.b*0.2);
	RT_Color cd(c.r*0.8, c.g*0.8, c.b*0.8);

	n = 0; 
	ambient[n++] = 0.0; 
	ambient[n++] = 0.0; 
	ambient[n++] = 0.0; 
	ambient[n++] = 1.0;
	glLightfv(ln, GL_AMBIENT, ambient);

	n = 0;
	specular[n++] = cs.r; 
	specular[n++] = cs.g; 
	specular[n++] = cs.b; 
	specular[n++] = 1.0;
	glLightfv(ln, GL_SPECULAR, specular);

	n = 0;
	diffuse[n++] = cd.r; 
	diffuse[n++] = cd.g; 
	diffuse[n++] = cd.b; 
	diffuse[n++] = 1.0;
	glLightfv(ln, GL_DIFFUSE, diffuse);

	n = 0;
	position[n++] = -vector.x; 
	position[n++] = -vector.y; 
	position[n++] = -vector.z; 
	position[n++] = 0.0;
	glLightfv(ln, GL_POSITION, position);

	n = 0;
	direction[n++] = 0.0; 
	direction[n++] = 0.0; 
	direction[n++] = -1.0;
	glLightfv(ln, GL_SPOT_DIRECTION, direction);         

	n = 0;
	spot_exp[n++] = 0.0;
	glLightfv(ln, GL_SPOT_EXPONENT, spot_exp); 

	n = 0;
	spot_cutoff[n++] = 180.0;
	glLightfv (ln, GL_SPOT_CUTOFF, spot_cutoff); 

	glEnable(ln);
	GL_LightNumber++;
    } else {
	char error[100];
	sprintf(error, "GL can't define more then %d lights", maxLights);
	rt_Output->warning(error);
    }
}

static RT_Vector calcVUP(const RT_Vector vp, const RT_Vector rp, double twist) {
    RT_Vector xa, ya, vup;
    RT_Vector dr = rp -vp;
    if (dr.x==0 && dr.z==0) { xa = RT_Vector(1,0,0); ya = RT_Vector(0,0,1); } 
    else { xa=dr.CROSS(RT_Vector(0,1,0)); ya=dr.CROSS(xa); }
    if (twist==0.0) vup=ya;
    else {
	RT_Vector xan=xa*sqrt(1/(xa.x*xa.x + xa.y*xa.y + xa.z*xa.z));
	RT_Vector yan=ya*sqrt(1/(ya.x*ya.x + ya.y*ya.y + ya.z*ya.z));
	RT_Vector d; 
	double length;
	if (twist>0 && twist<=M_PI_2) {
	    d=xan-yan;
	    length=sin(twist)/sin(M_PI-(M_PI_4+twist));
	    vup=yan+d*sqrt(length*length/(d.x*d.x + d.y*d.y + d.z*d.z));
	}
	if (twist>M_PI_2 && twist<=M_PI) {
	    twist-=M_PI_2;
	    d = yan*(-1)-xan; 
	    length=sin(twist)/sin(M_PI-(M_PI_4+twist));
	    vup=xan+d*sqrt(length*length/(d.x*d.x + d.y*d.y + d.z*d.z));
	}
	if (twist>M_PI && twist<=M_PI+M_PI_2) {
	    twist-=M_PI;
	    d = yan-xan;
	    length=sin(twist)/sin(M_PI-(M_PI_4+twist));
	    vup=yan*(-1)+d*sqrt(length*length/(d.x*d.x + d.y*d.y + d.z*d.z));
	}
	if (twist>M_PI+M_PI_2 && twist<=M_PI+M_PI) {
	    twist-=M_PI+M_PI_2;
	    d=yan-xan*(-1);
	    length=sin(twist)/sin(M_PI-(M_PI_4+twist));
	    vup=xan*(-1)+d*sqrt(length*length/(d.x*d.x + d.y*d.y + d.z*d.z));
	}
    }    
    return vup;
}


// shading method for the lookat camera:
void RT_LookatCamera::shading() {
    if (!get_pixmap()||!get_scene()) return; 
    xpixmap->activate();
    xpixmap->clear(get_scene()->get_background());

    glEnable(GL_DEPTH_TEST);
    glEnable(GL_LIGHTING);
    glShadeModel(GL_SMOOTH);
    glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);
    glEnable(GL_COLOR_MATERIAL);

    glPushMatrix();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluPerspective((GLdouble)xangle, (GLdouble) xpixmap->getW()/(double)xpixmap->getH(), (GLdouble)xnear, (GLdouble)xfar );
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    RT_Vector vup=calcVUP(vp, rp, xtwist*M_PI/180.0);
    gluLookAt(vp.x, vp.y, vp.z, rp.x, rp.y, rp.z, -vup.x, -vup.y, -vup.z);
    GL_LightNumber = 0;	
    RT_Camera::rendering();
    RT_RenderLightFunc lfunc;
    RT_RenderPrimitiveFunc pfunc;  
    get_scene()->doWithElements( &lfunc );
    get_scene()->doWithElements( &pfunc );
    feedback->doWithElements( &pfunc );
    if (xpixmap->isA(RTN_PIXMAP_DISPLAY)) {
	if ( ((RT_PixmapDisplay*)xpixmap)->sglBuf ) glFlush();
	else glXSwapBuffers(((RT_PixmapDisplay*)xpixmap)->data->display, ((RT_PixmapDisplay*)xpixmap)->data->window);
    }
    glPopMatrix();
}

#ifdef RTD_RSY
void RT_LookatCamera::RSYshading() {
    float max_rend = -BIGFLOAT;
    RT_RS_Scene* rscene = ((RT_RSYScene*)get_scene())->get_rscene(); //the radiosity scene to render
    long i, cnt,
    a_dcnt = rscene->Areas->get_diffcnt(),      //number of diffuse patches
    a_icnt = rscene->Areas->get_inscnt(),       //number of non-diffuse patches
    p_dcnt = rscene->Points->get_diffcnt(),     //number of diffuse patches
    p_icnt = rscene->Points->get_inscnt();      //number of non-diffuse patches
    int j, k, m, corners;
    int render_nondiff = xmode == RTE_LC_RADIOSITY; //render non-diffuse components ? (if any)
    int wireframe = xmode == RTE_LC_WIREFRAME_RADIOSITY;
    int auto_view_scale = rscene->Points->view_scale < epsilon && !wireframe;
    float max_scale = rscene->Points->max_scale;
    RT_RS_Area* ar;                            //the current patch
    RT_RS_3DVector prp(vp.x, vp.y, vp.z);      //projection reference point
    RT_Vector vv(rp - vp);                     //view vector                                 
    vv = vv.UNITIZE();                         //norm. view vector
    RT_RS_3DVector vdir(vv.x, vv.y, vv.z);
    RT_RS_Vtx *vtx[4],                         //the vertices of the current patch
    view_dir( prp, vdir ); //view direction
    
    RT_RS_DiffRGB xcolr[4];                    //the colors of the vertices
    
    if (xpixmap) { xpixmap->activate(); xpixmap->clear( get_scene()->get_background() ); }  
    glEnable( GL_DEPTH_TEST );
    glPushMatrix();
    glMatrixMode( GL_PROJECTION );
    glLoadIdentity();
    gluPerspective( (GLdouble)xangle, (GLdouble) xpixmap->getW()/(double)xpixmap->getH(), 
		   (GLdouble)xnear, (GLdouble)xfar );
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    RT_Vector vup = calcVUP(vp, rp, xtwist*M_PI/180.0);
    gluLookAt( vp.x, vp.y, vp.z, rp.x, rp.y, rp.z, -vup.x, -vup.y, -vup.z );

    rscene->Points->reset_vis();
    for(m = 0; m < 2; m++) {  //for all (diffuse + non-diffuse patches)
	cnt = m ? p_icnt : p_dcnt;
	for(i = 0; i < cnt; i++) {
	    if(m) *vtx = rscene->Points->get_ins(i);
	    else *vtx = rscene->Points->get_diff(i); //get the patch
	    if((*vtx)->get_view_val(*xcolr, view_dir, render_nondiff))
		if(auto_view_scale) {
		    if(xcolr->r > max_rend && xcolr->r < max_scale) max_rend = xcolr->r;
		    if(xcolr->g > max_rend && xcolr->g < max_scale) max_rend = xcolr->g;
		    if(xcolr->b > max_rend && xcolr->b < max_scale) max_rend = xcolr->b;
		}
	}
    }
    if(auto_view_scale) rscene->Points->view_scale = 1.0 / max_rend;
    rscene->Points->scale_view_vals();
    
    float crd[4][3], colr[3], Green[3], White[3];
    Green[0] = 0;  Green[1] = 1; Green[2] = 0;
    White[0] = 1;  White[1] = 1; White[2] = 1;
    
    // render:
    for(k = wireframe ? 0 : 1; k < 2; k++) {
	for(m = 0; m < 2; m++) {  //for all (diffuse + non-diffuse patches)
	    cnt = m ? a_icnt : a_dcnt;
	    for(i = 0; i < cnt; i++) {
		if(m) ar = rscene->Areas->get_ins(i); 
		else ar = rscene->Areas->get_diff(i); //get the patch
		corners = ar->vidx[2] == ar->vidx[3] ? 3 : 4;
		int xbackface = 0;
		for(j = 0; j < corners; j++) {
		    vtx[j] = ar->get_vtx(ar->vidx[j]);   //get the vertices
		    //get the vertex intensity in direction to point of view
		    if(!vtx[j]->get_view_val(xcolr[j], view_dir, render_nondiff))
			xbackface++;  //this vertex is back-facing 
		}
		xbackface = xbackface == corners;
		if(wireframe && (xbackface && !k || !xbackface && k) || !wireframe && !xbackface) {  //render this patch
		    for(j = 0; j < corners; j++) {
			crd[j][0] = vtx[j]->pnt.x; 
			crd[j][1] = vtx[j]->pnt.y; 
			crd[j][2] = vtx[j]->pnt.z;
		    }  
		    if (wireframe) {    //draw as a wireframe
			glBegin( GL_LINE_LOOP );
			if ( xbackface ) glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, Green );
			else glMaterialfv( GL_FRONT_AND_BACK, GL_EMISSION, White );
			for (j = 0; j < corners; j++) glVertex3fv( crd[j] ); 
			glEnd();
		    } else {		
			glBegin( GL_POLYGON );
			for (j = 0; j < corners; j++) {
			    colr[0] = xcolr[j].r; colr[1] = xcolr[j].g; colr[2] = xcolr[j].b; 
			    glColor3fv(colr); 
			    glVertex3fv(crd[j]);
			}         
			glEnd();
		    }
		}
	    }
	}
    }
    if(auto_view_scale) rscene->Points->view_scale = 0.;
    if ( ((RT_PixmapDisplay*)xpixmap)->sglBuf ) glFlush();
    else glXSwapBuffers(((RT_PixmapDisplay*)xpixmap)->data->display, ((RT_PixmapDisplay*)xpixmap)->data->window);
    glPopMatrix();
}

#endif
