#ifndef lint
static       char    rcsid[] = "$Header: polyline.c,v 1.2 90/12/12 15:30:21 zhang Exp $";
#endif

/*
 * $Log:	polyline.c,v $
 * 
 * Revision 1.1  90/06/17  03:34:51  zhang
 * Initial revision
 * 
 * Revision 1.2  90/12/12  15:30:21  zhang
 * Add backend for RADIANCE output
 */

#include "defs.h"

/*
 * a POLYLINE may have starting width and ending width. right now the two width
 * number should be the same.
 *
 * a POLYLINE may be a curve or spline. i cannot handle those POLYLINEs as i
 * do not know what exactly the fomula used in AutoCAD to tesselate (or
 * regenerate) them. So for a curve or spline, they have to be converted into
 * polyline first in AutoCAD before outputing them into DXF files
 *
 * a POLYLINE may be a mesh. only the polygon meshes can be handled. other mesh
 * types for smooth surfaces, i.e. quadratic B-spline surfaces, cubic B-spline
 * surfaces, Bezier surfaces and Coons surfaces, cannot be handled, because i
 * do not know how to regenerate them as the same way as AutoCAD did. so for
 * smooth surfaces, they have to be converted into polygon meshes first in
 * AutoCAD before outputing them into DXF files
 *
 * a segment of a POLYLINE may be a arc or a line. this information is stored
 * in the VERTEX data structure.
 *
 * a VERTEX is always and only associated with a POLYLINE entity.
 *
 * a VERTEX may have starting and ending widths. right now they are not used
 * as it is very difficult to implement. only the widths of the POLYLINE are
 * used, but they should be the same. as there is no smooth surface allowed,
 * there are only 3D polyline vertices or 3D polygon mesh vertices in VERTEX
 * sequences following a POLYLINE entity.
 *
 * a VERTEX may have curve fit tangent. but i cannot handle it as i cannot
 * handle curves and curve surfaces.
 *
 * the bulge value of a VERTEX is the tangent of 1/4 the included angle for
 * an ARC segment, made negative if the arc goes clockwise from the start
 * point to the emd point; a bulge of 0 indicates a straight segment, and
 * a bulge of 1 is a semi-circle.
 *
 *
 * surfaces, i.e. tetragons, defined by a POLYLINE entity are treated as
 * follows:
 *
 * 1. if it is just a simple polygon
 *    1. if it has nonzero width only, then the surfaces are a set of tetragons
 *       on the extrusion plane
 *    2. if it has thickness only, then the surfaces are a set of tetragons
 *       orthaxical to the extrusion, i.e. parallel to the extrusion direction.
 *       if the polyline is closed, and its layer name has an "_I" suffix,
 *       but the reference points is not defined, the orientation of the faces
 *       defined by the polyline have to be reversed.
 *    3. if it has both width and thickness, then the surfaces will defined
 *       a closed object with its bottom in the extrusion plane and its cup
 *	 parallel to the extrusion plane, and a set of side faces parallel to
 *       the extrusion direction. if the layer name of the polygon has an "_I"
 *       suffix, but the reference points is not defined, the orientation of
 *       the faces defined by the polyline have to be reversed.
 *
 *    the tesselation of the surfaces defined will be handled as follows:
 *    1. if there are arcs as segments of the polyline, the minimum tesselation
 *       length for surfaces, i.e. tetragons, on, or parallel to, the extrusion,
 *       is the minimum tesselation length of all those arcs.
 *    2. if there is no arc segment, the minimum tesselation is the width of
 *       the polyline when there is nonzero width.
 *    3. if the polyline is closed and has thickness, but no width, the minimum
 *       tesselation is the minimum length of all line segments. but if the
 *       thickness is smaller than the minimum length, no treatment for the
 *       side face parallel to the extrusion direction
 *
 * 2. if it is a 3D polyline, i.e. its vertices do not lie in the same plane,
 *    no surface will be defined
 *
 * 3. if it is a 3D mesh, it cannot be extruded, nor have width, the surfaces
 *    are just a set of tetragons
 *    no tesselation will be performed
 *
 * warning:
 *
 * if the object defined by a POLYLINE with width is intersected with itself,
 * unexpected results will occur.
 *
 * additional notes:
 *
 * in the POLYLINE data structure there are two things not following the
 * description of the DXF file format in "AutoCAD 10.0 Reference Manual".
 * one is that "10, 20, 30" groups occurs in very POLYLINE entity. this
 * is not mentioned in the manual. i do not know what the point means.
 * another is that the polyline flag should be always occurs in a DXF file.
 * but it is true. so i would assume it is an option with default value 0
 * the same thing happens to the starting and ending widths.
 */

/*
 * define polyline flags (bit-coded)
 */

#define	POLYLINE_POLYGON_CLOSED	1	/* this is a closed polyline, */
					/* or a polygon mesh colsed in */
					/* the M directio*/
#define	POLYLINE_CURVE_FIT	2	/* curve-fit vertices have been added */

#define	POLYLINE_SPLINE_FIT	4	/* spline-fit vertices have been added*/

#define	POLYLINE_POLYLINE	8	/* this is a 3D polyline */

#define	POLYLINE_MESH		16	/* this is a 3D polygon mesh */
					/* group 75 indicates the smooth */
					/* face type, see follows */
#define	POLYLINE_MESH_CLOSED	32	/* the polygon mesh is closed in */
					/* N direction */

/*
 * define group 75 values.
 * this group is optional and its default value is 0
 */

#define	POLYLINE_NOSMOOTH	0	/* no smooth surface fitted */
#define	POLYLINE_2BSPLINE	5	/* quadratic b-spline surface */
#define	POLYLINE_3BSPLINE	6	/* cubic b-spline surface */
#define	POLYLINE_BEZIER		8	/* cubic b-spline surface */

/*
 * define polyline data structure
 */

typedef	struct	polyline	{
	FLOAT	point[3];		/* this is not described in AutoCAD */
					/* reference manual. do not know what */
					/* it means */
	INT	vertexflag;		/* vertics follows flag, should be */
					/* always 1 */
	INT	polylineflag;		/* polyline flag. this is not */
					/* described as an option, but it */
					/* does not always in DXF files */
					/* so it is assumed as an option */
					/* with default value 0 */
	FLOAT	startwidth;		/* default starting width */
	FLOAT	endwidth;		/* default ending width */
	INT	Mmeshcount;		/* polygon mesh M vertex count, */
					/* default 0 */
	INT	Nmeshcount;		/* polygon mesh N vertex count, */
					/* default 0 */
	FLOAT	Mdensity;		/* smooth surface M density, default 0*/
	FLOAT	Ndensity;		/* smooth surface N density, default 0*/
	INT	smoothtype;		/* smooth surface type, default 0 */
	INT	nvertices;		/* number of vertices */
	VERTEX	**vertices;		/* list of vertices of the polyline */
} POLYLINE;	

/*
 * define data structure for tesselation of a polyline
 */

typedef	struct	segment	{
	FLOAT	point[3];		/* starting point of the segment */
					/* ending point in the next segment */
	FLOAT	plusoffset[3];		/* point offseted in "outer" */
					/* direction */
	FLOAT	minusoffset[3];		/* point offseted in "inner" */
} SEGMENT;

/*
 * parse a general POLYLINE
 */

ENTITY	*PolylineDxfParse(layerlist)
LAYER	**layerlist;
{
	ENTITY	*entity;
	POLYLINE	*polyline;
	INT	pointset = 0;
	INT	vertexflagset = 0;
	INT	polylineflagset = 0;
	INT	startwidthset = 0;
	INT	endwidthset = 0;
	INT	Mmeshcountset = 0;
	INT	Nmeshcountset = 0;
	INT	Mdensityset = 0;
	INT	Ndensityset = 0;
	INT	smoothtypeset = 0;
	INT	defaultflag = 0;
	OPTIONS	*seqendoptions;
	VERTEX	*vertex;
	VERTEX	**vertices;

	entity = Malloc(ENTITY, 1);
	polyline = Malloc(POLYLINE, 1);
	entity->type = ENTITY_POLYLINE;
	entity->data = (VOID *) polyline;

	entity->layer = LayerDxfParse(layerlist);

	polyline->point[0] = 0.0;
	polyline->point[1] = 0.0;
	polyline->point[2] = 0.0;
	polyline->startwidth = 0.0;
	polyline->endwidth = 0.0;
	polyline->polylineflag = 0;
	polyline->Mmeshcount = 0;
	polyline->Nmeshcount = 0;
	polyline->Mdensity = 0;
	polyline->Ndensity = 0;
	polyline->vertices = NULL;
	polyline->nvertices = 0;
	polyline->smoothtype = 0;

	do {
		GetNextGroup();
		switch(Group->code) {
		case 10:
			/*
			 * 10, 20, 30
			 *
			 * this is not described in the AutoCAD manual
			 * i do not know what it means
			 */

			if (pointset != 0)
				DXFERR("duplicated point for a POLYLINE %s", "\n");

			pointset = 1;
			CoordDxfParse(0, polyline->point);
			break;
		case 66:
			/*
			 * vertices follow flag
			 */

			if (vertexflagset != 0)
				DXFERR("duplicated vertices flow flag for a POLYLINE %s", "\n");

			vertexflagset = 1;
			polyline->vertexflag = Group->intnum;
			if (polyline->vertexflag != 1)
				DXFERR("wrong vertices flow flag for a POLYLINE %s", "\n");
			break;
		case 70:
			/*
			 * polyline flags
			 */

			if (polylineflagset != 0)
				DXFERR("duplicated polyline flags for a POLYLINE %s", "\n");

			polylineflagset = 1;
			polyline->polylineflag = Group->intnum;
			break;
		case 40:
			/*
			 * default starting width
			 */

			if (startwidthset != 0)
				DXFERR("duplicated starting width for a POLYLINE %s", "\n");

			startwidthset = 1;
			polyline->startwidth = Group->fltnum;
			break;
		case 41:
			/*
			 * default ending width
			 */

			if (endwidthset != 0)
				DXFERR("duplicated starting width for a POLYLINE %s", "\n");

			endwidthset = 1;
			polyline->endwidth = Group->fltnum;
			break;
		case 71:
			/*
			 * polygon mesh M count
			 */

			if (Mmeshcountset != 0)
				DXFERR("duplicated mesh M count for a POLYLINE %s", "\n");

			Mmeshcountset = 1;
			polyline->Mmeshcount = Group->intnum;
			break;
		case 72:
			/*
			 * polygon mesh N count
			 */

			if (Nmeshcountset != 0)
				DXFERR("duplicated mesh N count for a POLYLINE %s", "\n");

			Nmeshcountset = 1;
			polyline->Nmeshcount = Group->intnum;
			break;
		case 73:
			/*
			 * smooth surface M density
			 */

			if (Mdensityset != 0)
				DXFERR("duplicated smooth surface M density for a POLYLINE %s", "\n");

			Mdensityset = 1;
			polyline->Mdensity = Group->intnum;
			break;
		case 74:
			/*
			 * smooth surface N density
			 */

			if (Ndensityset != 0)
				DXFERR("duplicated smooth surface N density for a POLYLINE %s", "\n");

			Ndensityset = 1;
			polyline->Ndensity = Group->intnum;
			break;
		case 75:
			/*
			 * smooth surface type
			 */

			if (smoothtypeset != 0)
				DXFERR("duplicated smooth surface N density for a POLYLINE %s", "\n");

			smoothtypeset = 1;
			polyline->smoothtype = Group->intnum;
			break;

		default:
			if (OptionsDxfParse(&entity->options) == 0)
				defaultflag = 1;
			break;
		}
	} while (defaultflag == 0);

	/*
	 * check if vertices follow flag, polyline flags, default starting
	 * width, default ending width are all defined
	 */

	if (vertexflagset == 0)
		DXFERR("undefined vertices flow flag for a POLYLINE %s", "\n");

	/*
	 * the polyline flag is not an option according to the
	 * AutoCAD manual. but it does not occur in some DXF files.
	 * so i would assume it is an option with default value 0.
	 * the same thing happens to the starting and ending widths.
	 *
	 * if (polylineflagset == 0)
	 * 	DXFERR("undefined polyline flag for a POLYLINE %s", "\n");
	 *
	 * if (startwidthset == 0)
	 * 	DXFERR("undefined starting width for a POLYLINE %s", "\n");
	 *
	 * if (endwidthset == 0)
	 * 	DXFERR("undefined ending width for a POLYLINE %s", "\n");
	 */

	/*
	 * check if curve-fit or spline-fit vertices added
	 * as i cannot handle curves and curve surfaces
	 */

	if ((polyline->polylineflag & POLYLINE_CURVE_FIT) != 0 ||
	    (polyline->polylineflag & POLYLINE_SPLINE_FIT) != 0)
	    	DXFERR("POLYLINE has curve-fit and/or spline-fit vertices %s", "\n");

	/*
	 * check the smooth surface flag if it is not a smooth surface, i.e.
	 * it is a simple mesh
	 * i cannot handle smooth surfaces
	 */

	/*
	 * originally i use
	 * 	if (polyline->smoothtype != POLYLINE_NOSMOOTH)
	 * for error checking, but as it happens in DXF files,
	 * group 75 could be followed by type 7, which is not
	 * described in the manual
	 */

	if (polyline->smoothtype == POLYLINE_2BSPLINE ||
	    polyline->smoothtype == POLYLINE_3BSPLINE ||
	    polyline->smoothtype == POLYLINE_BEZIER)
		DXFERR("POLYLINE is smooth surface %s", "\n");

	/*
	 * check if the starting width is the same as the ending width
	 */

	if (fabs(polyline->startwidth - polyline->endwidth) > TOE)
		DXFERR("POLYLINE has different starting and ending widths %s", "\n");

	/*
	 * get all vertices associated with the polyline
	 */

	do {
		vertex = VertexDxfParse(layerlist);
#ifdef	DEBUG
		fprintf(stderr, "%g %g %g\n",
			vertex->point[0], vertex->point[1], vertex->point[2]);
#endif

		/*
		 * for a 3D polyline, it should have no arc segment
		 */

		if ((polyline->polylineflag & POLYLINE_POLYLINE) != 0 &&
		    fabs(vertex->bulge) > TOE)
			DXFERR("3D POLYLINE should have no arc segments %s", "\n");

		if (polyline->nvertices == 0) {
			polyline->vertices = Malloc(VERTEX *, 1);
			polyline->vertices[0] = vertex;
			polyline->nvertices++;
		} else
		if (CoordSame(vertex->point,
		    polyline->vertices[polyline->nvertices - 1]->point) != 0)

			/*
			 * the vertex is duplicated
			 */

			free(vertex);
		else {
			/*
			 * insert the vertex into the list
			 */

			vertices = Malloc(VERTEX *, polyline->nvertices + 1);
			(VOID) bcopy(polyline->vertices, vertices, polyline->nvertices * sizeof(VERTEX *));
			free(polyline->vertices);
			vertices[polyline->nvertices] = vertex;
			polyline->vertices = vertices;
			polyline->nvertices++;
		}

	} while (CmpGroupString(Group, 0, "SEQEND") == 0);

	/*
	 * check if there is any vertex defined
	 */

	if (polyline->nvertices == 0)
		DXFERR("undefined VERTEX for a POLYLINE %s", "\n");

	(VOID) LayerDxfParse(layerlist);
	while (OptionsDxfParse(&seqendoptions) != 0)
		free(seqendoptions);

	GetNextGroup();

	return(entity);
}

#ifdef	NEVER_DEFINED
/*
 * not used any more
 */

/*
 * calculate the intersection point of two 3D line
 *
 * method
 *
 * assume two lines are
 *	(S0, E0) and (S1, E1)
 * let
 *	U = E0 - S0, V = E1 - S1
 * then two lines are
 *	P = S0 + U . t0
 *	P = S1 + V . t1
 * for their intersetion point, we have
 *	S0 + U . t0 = S1 + V . t1
 * i.e.
 *	U . t0 - V . t1 = S1 - S0
 * so that
 *      (U . t0 - V . t1) x V = (S1 - S0) x V
 *      (U . t0 - V . t1) x U = (S1 - S0) x U
 * i.e.
 *	(U x V) * t0 = (S1 - S0) x V
 *	(U x V) * t1 = (S1 - S0) x U
 */

VOID	LineXLine(start0, end0, start1, end1, xpoint)
FLOAT	start0[3];
FLOAT	end0[3];
FLOAT	start1[3];
FLOAT	end1[3];
FLOAT	xpoint[3];
{
	FLOAT	t0, t1;
	FLOAT	UV[3], SU[3], SV[3];
	FLOAT	U[3], V[3], S[3], D[3];

	VecSub(start1, start0, S);
	VecSub(end0, start0, U);
	VecSub(end1, start1, V);
	VecCross(U, V, UV);
	VecCross(S, U, SU);
	VecCross(S, V, SV);

	if (fabs(UV[0]) > TOE) {
		t0 = SV[0] / UV[0];
		t1 = SU[0] / UV[0];
	} else
	if (fabs(UV[1]) > TOE) {
		t0 = SV[1] / UV[1];
		t1 = SU[1] / UV[1];
	} else
	if (fabs(UV[2]) > TOE) {
		t0 = SV[2] / UV[2];
		t1 = SU[2] / UV[2];
	} else
		GENERR("LineXLine - the length of (U x V) is %g\n", VecLength(UV));

	VecScale(t0, UV, D);
	VecSub(D, SV, D);

	if (VecDot(D, D) > TOE)
		GENERR("LineXLine - the length of ((U x V) . t0 - S x V) is %g",
			VecLength(D));

	VecScale(t1, UV, D);
	VecSub(D, SU, D);

	if (VecDot(D, D) > TOE)
		GENERR("LineXLine - the length of ((U x V) . t1 - S x U) is %g",
			VecLength(D));

	VecScale(t0, U, U);
	VecAdd(U, start0, U);

	VecScale(t1, V, V);
	VecAdd(V, start1, V);

	VecSub(U, V, D);
	if (VecDot(D, D) > TOE)
		GENERR("LineXLine - the distance of two points lying on the two line is %g\n", VecLength(D));

	VecCopy(U, xpoint);
}
#endif

#ifdef	NEVER_DEFINED

/*
 * Note: This function is not correct duo to some error
 *       use the one defined before
 */

/*
 * calculate the intersection point of two 3D line
 *
 * method
 *                          V
 *                         / e1
 *              p1      p/
 *        s0 .---------/---- U
 *            \ |    /     e0
 *              \  /
 *              |/p0
 *           s1 .
 *
 * assume: U = e0 - s0, V = e1 - s1, and U, V are normalized
 *         S0 = s1 - s0, S1 = s0 - s1 (i.e. S0 = - S1)
 * so if we know the distance between p and s0 or s1, we will know
 *         p = s0 + U . |s0-p| = s1 + V . |s1-p|
 *
 * as the two triangles, s0-p-p0, s1-p-p1 are similar, we have
 *
 *        |s0-p|     |s0-p0|
 *       -------- = ---------
 *        |s1-p|     |s1-p1|
 *
 * althouth |s0-p0| and |s1-p1| are unknown directly, we have
 *        |s0-p1| = (s1 - s0) . U = S0 . U
 *        |s1-p0| = (s0 - s1) . V = S1 . V
 * and
 *        |s0-s1|**2 = |s0-p0|**2 + |s1-p0| ** 2
 *        |s0-s1|**2 = |s1-p1|**2 + |s0-p1| ** 2
 */

#ifdef	NEVER_DEFINED
VOID	LineXLine(start0, end0, start1, end1, xpoint)
#else
VOID	LineXLine(start0, end0, end1, start1, xpoint)
#endif
FLOAT	start0[3];
FLOAT	end0[3];
FLOAT	start1[3];
FLOAT	end1[3];
FLOAT	xpoint[3];
{
	FLOAT	U[3], V[3], W[3];
	FLOAT	S0[3], S1[3];
	FLOAT	s0s1;
	FLOAT	s0p1, s1p0;
	FLOAT	s0p0, s1p1;
	FLOAT	s1p1s0p0;
	FLOAT	s0p, s1p;

	VecSub(start1, start0, S0);
	VecSub(start0, start1, S1);
	s0s1 = VecDot(S0, S0);

	/*
	 * if the two starting points are the same
	 */

	if (fabs(s0s1) < TOE) {
		VecCopy(start0, xpoint);
		return;
	}

	VecSub(end0, start0, U);
	VecSub(end1, start1, V);

	/*
	 * check if the lines are degenerated
	 */

	if ((s0p0 = VecNormalize(U)) < TOE)
		GENERR("LineXLine - length of the first line is %g\n", s0p0);

	if ((s1p1 = VecNormalize(V)) < TOE)
		GENERR("LineXLine - length of the second line is %g\n", s1p1);

	s0p1 = VecDot(S0, U);
	s1p0 = VecDot(S1, V);

	s0p0 = s0s1 - s1p0 * s1p0;
	s1p1 = s0s1 - s0p1 * s0p1;

	if (s0p0 < -TOE)
		GENERR("LineXLine - s0-p0 = %g\n", s0p0);

	if (s1p1 < -TOE)
		GENERR("LineXLine - s1-p1 = %g\n", s1p1);

	if (fabs(s0p0) < TOE) {
		VecCopy(start0, xpoint);
		return;
	}

	if (fabs(s1p1) < TOE) {
		VecCopy(start1, xpoint);
		return;
	}

	/*
	 * now s0p0 > TOE && s1p1 > TOE
	 */

	s0p0 = (FLOAT) sqrt(s0p0);
	s1p1 = (FLOAT) sqrt(s1p1);

	s1p1s0p0 = s1p1 / s0p0;
	VecScale(s1p1s0p0, V, W);
	VecSub(U, W, W);

	/*
	 * W . s0p = S0
	 */

	fprintf(stderr, "W = %g %g %g  S0 = %g %g %g\n", W[0], W[1], W[2],
			S0[0], S0[1], S0[2]);

	s0p = VecDot(W, W);

	if (fabs(s0p) < TOE)
		GENERR("LineXLine - the length of (U - V . s1p1 / s0p0) is %g\n",
			s0p);

	s0p = (FLOAT) sqrt(s0s1 / s0p);
	s1p = s0p * s1p1s0p0;

	VecScale(s0p, U, U);
	VecScale(s1p, V, V);
	VecAdd(start0, U, U);
	VecAdd(start1, V, V);
	VecSub(U, V, W);

	fprintf(stderr, "line %g %g %g - %g %g %g = %g %g %g - %g %g %g\n",
		start0[0], start0[1], start0[2], end0[0], end0[1], end0[2],
		start1[0], start1[1], start1[2], end1[0], end1[1], end1[2]);
	fprintf(stderr, "xpoint %g %g %g - %g %g %g = %g %g %g\n",
		U[0], U[1], U[2], V[0], V[1], V[2], W[0], W[1], W[2]);

	if (VecDot(W, W) > TOE)
		GENERR("LineXLine - distance of two points on two lines is %g\n",
			VecLength(W));

	VecCopy(U, xpoint);
}
#endif

/*
 * calculate a point within a tetragon
 */

VOID	TetragonPoint(x, y, p, q)
FLOAT	x, y;
FLOAT	p[4][3];
FLOAT	q[3];
{
	FLOAT	x1, y1;

	x1 = 1.0 - x;
	y1 = 1.0 - y;

	q[0] =	x1 * y1 * p[0][0] + x  * y1 * p[1][0] +
		x1 * y  * p[3][0] + x  * y  * p[2][0];
	q[1] =	x1 * y1 * p[0][1] + x  * y1 * p[1][1] +
		x1 * y  * p[3][1] + x  * y  * p[2][1];
	q[2] =	x1 * y1 * p[0][2] + x  * y1 * p[1][2] +
		x1 * y  * p[3][2] + x  * y  * p[2][2];
}

/*
 * calculate the normal of a tetragon, the vertices of which
 * is (i, j) (i + 1, j) (i + 1, j + 1) and (i, j + 1)
 * here, -1 < i < M, -1 < j < N.
 *
 * if all the 4 vertices exsit and the normal is not zero
 * it returns 1 else 0
 * the normal is normalized
 */

INT	MeshTetragonNormal(polyline, M, N, i, j, normal)
POLYLINE	*polyline;
INT	M, N, i, j;
FLOAT	normal[3];
{
	INT	m, n;
	INT	index0, index1, index2, index3;
	FLOAT	tetragon[4][3];
	FLOAT	d;

	m = i + 1;
	n = j + 1;

#ifdef	DEBUG
	fprintf(stderr, "MeshTetragonNormal [%d %d %d %d] flag %d\n",
			i, j, m, n, polyline->polylineflag);
#endif

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0 && i < 0)
	   	return(0);
	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0 && m >= M)
	   	return(0);
	if ((polyline->polylineflag & POLYLINE_MESH_CLOSED) == 0 && j < 0)
	   	return(0);
	if ((polyline->polylineflag & POLYLINE_MESH_CLOSED) == 0 && n >= N)
	   	return(0);

	if (i < 0)
		i = M - 1;
	if (m >= M)
		m = 0;
	if (j < 0)
		j = N - 1;
	if (n >= N)
		n = 0;

	index0 = N * i + j;
	index1 = N * i + n;
	index2 = N * m + n;
	index3 = N * m + j;
	VecCopy(polyline->vertices[index0]->point, tetragon[0]);
	VecCopy(polyline->vertices[index1]->point, tetragon[1]);
	VecCopy(polyline->vertices[index2]->point, tetragon[2]);
	VecCopy(polyline->vertices[index3]->point, tetragon[3]);

	PlaneEquation(4, tetragon, &normal[0], &normal[1], &normal[2], &d);

#ifdef	DEBUG
	fprintf(stderr, "MeshTetragonNormal [%d %d %d %d] %g %g %g\n", i, j, m, n,
			normal[0], normal[1], normal[2]);
#endif

	return(1);
}

/*
 * calculate the normal of a mesh vertex
 */

VOID	MeshVertexNormal(polyline, M, N, i, j, normal)
POLYLINE	*polyline;
INT	M, N, i, j;
FLOAT	normal[3];
{
	FLOAT	tnormal[3];
	INT	nnormal;

	nnormal = 0;
	normal[0] = normal[1] = normal[2] = 0.0;

	if (MeshTetragonNormal(polyline, M, N, i - 1, j - 1, tnormal) != 0) {
		nnormal++;
		VecAdd(normal, tnormal, normal);
	}

	if (MeshTetragonNormal(polyline, M, N, i - 1, j    , tnormal) != 0) {
		nnormal++;
		VecAdd(normal, tnormal, normal);
	}

	if (MeshTetragonNormal(polyline, M, N, i    , j - 1, tnormal) != 0) {
		nnormal++;
		VecAdd(normal, tnormal, normal);
	}

	if (MeshTetragonNormal(polyline, M, N, i    , j   , tnormal) != 0) {
		nnormal++;
		VecAdd(normal, tnormal, normal);
	}

	if (nnormal == 0)
		GENERR("MeshVertexNormal - length of normal is 0.0", "\n");

	if (VecNormalize(normal) < TOE)
		GENERR("MeshVertexNormal - length of normal is 0.0", "\n");
}
	
/*
 * tesselation a tetragon into DEF/NFF/DXF/RAD file
 */

VOID	TetragonTesselation(m, n, tetragon, entity)
INT	m;
INT	n;
FLOAT	tetragon[4][3];
ENTITY	*entity;
{
	INT	i, j;
	FLOAT	x, y;
	FLOAT	dx, dy;
	FLOAT	polygon[4][3];

	if (OutFileType == FILE_NFF || OutFileType == FILE_RAD) {
		OutputPolygon(4, tetragon, entity);
		return;
	}

	dx = 1.0 / (FLOAT) m;
	dy = 1.0 / (FLOAT) n;

	for (i = 0, x = 0.0; i < m; i++, x += dx)
		for (j = 0, y = 0.0; j < n; j++, y += dy) {
			TetragonPoint(x     , y     , tetragon, polygon[0]);
			TetragonPoint(x + dx, y     , tetragon, polygon[1]);
			TetragonPoint(x + dx, y + dy, tetragon, polygon[2]);
			TetragonPoint(x     , y + dy, tetragon, polygon[3]);
			OutputPolygon(4, polygon, entity);
		}
}

/*
 * output polygon meshes into DEF/NFF/DXF/RAD file
 */

VOID	MeshOutput(entity, polyline)
ENTITY	*entity;
POLYLINE	*polyline;
{
	INT	i, j;
	INT	m, n;
	INT	M;
	INT	N;
	INT	index0;
	INT	index1;
	INT	index2;
	INT	index3;
	FLOAT	tetragon[4][3];
	FLOAT	normal[4][3];

	M = polyline->Mmeshcount;
	N = polyline->Nmeshcount;

	if (polyline->nvertices != (M * N))
		GENERR("Polyline Mesh - number of vertices (%d) is not M x N\n",
			polyline->nvertices);

	for (i = 0; i < M; i++) {
		m = i + 1;
		if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0
		    && m == M)
			continue;
		if (m == M)
			m = 0;
		for (j = 0; j < N; j++) {
			n = j + 1;
			if ((polyline->polylineflag & POLYLINE_MESH_CLOSED) == 0
			    && n == N)
				continue;
			if (n == N)
				n = 0;

			index0 = N * i + j;
			index1 = N * i + n;
			index2 = N * m + n;
			index3 = N * m + j;
			VecCopy(polyline->vertices[index0]->point, tetragon[0]);
			VecCopy(polyline->vertices[index1]->point, tetragon[1]);
			VecCopy(polyline->vertices[index2]->point, tetragon[2]);
			VecCopy(polyline->vertices[index3]->point, tetragon[3]);
			if (OutFileType != FILE_NFF) {
				OutputPolygon(4, tetragon, entity);
				continue;
			}

			/*
			 * calculate the normal of the vertices of the tetragon
			 */
			
			MeshVertexNormal(polyline, M, N, i, j, normal[0]);
			MeshVertexNormal(polyline, M, N, i, n, normal[1]);
			MeshVertexNormal(polyline, M, N, m, n, normal[2]);
			MeshVertexNormal(polyline, M, N, m, j, normal[3]);

			/*
			 * output a phong patch
			 */

			OutputNffPhongPolygon(4, tetragon, normal, entity);
		}
	}
}

/*
 * output faces defined by a 3d polyline into DEF/NFF/DXF/RAD file
 */

VOID	Polyline3DOutput(entity, polyline)
ENTITY	*entity;
POLYLINE	*polyline;
{
	INT	i, j;
	INT	M;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	tetragon[4][3];

	if (OptionsThickness(entity->options, &thickness) == 0)
		return;

	extrusion = OptionsExtrusion(entity->options);

	VecScale(thickness, extrusion, delta);

	M = polyline->nvertices;
	for (i = 0; i < M; i++) {
		j = i + 1;
		if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0
		    && j == M)
			continue;
		if (j == M)
			j = 0;
		VecCopy(polyline->vertices[i]->point, tetragon[0]);
		VecCopy(polyline->vertices[j]->point, tetragon[1]);
		VecCopy(polyline->vertices[j]->point, tetragon[2]);
		VecCopy(polyline->vertices[i]->point, tetragon[3]);
		VecAdd(tetragon[2], delta, tetragon[2]);
		VecAdd(tetragon[3], delta, tetragon[3]);
		OutputPolygon(4, tetragon, entity);
	}
}

/*
 * calculate the center point and starting angle of an arc segment
 *
 * method: 
 *			 c
 *		         .
 *
 *
 *		s .      .      . e
 *			 p
 *
 * 1. calculate absolute delta angle: s-c-e = 4.0 * atan(bulge)
 * 2. calculate vector s-e and normalize it
 * 3. calculate point p which is the center between s and e
 * 4. calculate vector p-c = extrusion x s-e and normalize it
 * 5. c lies on c-p, and it has
 *    |p-s|                    2 * bulge
 *    ----- = tan(s-c-p) = -----------------
 *    |p-c|                1 - bulge * bulge
 *
 * 6. determine the center point
 *    1. bulge > 0 i.e count-clockwise arc
 *       1. 0 < bulge < 1, i.e. 0 < s-c-e < 180
 *		c = p + |p-c| . c-p
 *       2. 1 < bulge, i.e. 180 < s-c-e 360
 *		c = p - |p-c| . c-p
 *    2. bulge < 0 i.e clockwise arc
 *       1. -1 < bulge < 0, i.e. -180 < s-c-e < 0
 *		c = p - |p-c| . c-p
 *       2. bulge < -1, i.e. -360 < s-c-e < -180
 *		c = p + |p-c| . c-p
 * 7. calculate the starting and ending angles by transforming center ,
 *    starting and ending points onto the extrusion plane so that all their
 *    z-componets will be zero and calculate the angles by calling atan2(3)
 */

VOID	ArcCenterCalculation(start, end, bulge, extrusion,
	center, radius, startangle, endangle, deltaangle)
FLOAT	start[3];
FLOAT	end[3];
FLOAT	bulge;
FLOAT	*extrusion;
FLOAT	center[3];
FLOAT	*radius;
FLOAT	*startangle;
FLOAT	*endangle;
FLOAT	*deltaangle;
{
	MATRIX	matrix[2];
	FLOAT	vector[3];
	FLOAT	distance;
	FLOAT	centerline[3];
	FLOAT	centerpoint[3];
	FLOAT	centerdistance;

	if (fabs(bulge) < TOE)
		GENERR("POLYLINE arc bugle smaller than %g\n", TOE);

	VecSub(end, start, vector);
	VecAdd(end, start, centerpoint);
	VecScale(0.5, centerpoint, centerpoint);
	distance = 0.5 * VecLength(vector);
	VecCross(extrusion, vector, centerline);
	(VOID) VecNormalize(centerline);

	if (fabs(distance) < TOE) {
		VecCopy(start, center);
		*radius = *startangle = *endangle = *deltaangle = 0.0;
		return;
	}

	*deltaangle = 4.0 * atan(bulge);
	centerdistance = distance * fabs((1.0 - bulge * bulge) * 0.5 / bulge);
	VecScale(centerdistance, centerline, vector);

	if (bulge > 0.0 && bulge <= 1.0)
		VecAdd(centerpoint, vector, center);
	else
	if (bulge >= 1.0)
		VecSub(centerpoint, vector, center);
	else
	if (bulge < 0.0 && bulge >= -1.0)
		VecSub(centerpoint, vector, center);
	else
		VecAdd(centerpoint, vector, center);

	*radius = (FLOAT) sqrt(centerdistance * centerdistance
				+ distance * distance);

	if ((fabs(distance = *radius - CoordDistance(center, start))) > TOE)
		GENERR("ArcCenterCalculation: wrong radius %g\n", distance);

	MatrixExtrusion(matrix, extrusion);
	XformPoint(matrix[1], center, centerpoint);
	XformPoint(matrix[1], start, vector);
	VecSub(vector, centerpoint, vector);
	*startangle = atan2(vector[1], vector[0]);
	XformPoint(matrix[1], end, vector);
	VecSub(vector, centerpoint, vector);
	*endangle = atan2(vector[1], vector[0]);

#ifdef	DEBUG
	fprintf(stderr, "\nstart point: %g %g %g\n  end point: %g %g %g\n",
			start[0], start[1], start[2], end[0], end[1], end[2]);
	fprintf(stderr, "center: %g %g %g, start angle %g end angle %g\n",
			center[0], center[1], center[2],
			RAD_TO_DEG(*startangle), RAD_TO_DEG(*endangle));
#endif
}

/*
 * output faces defined by sweeping a planar POLYLINE into DEF/NFF/DXF/RAD file
 */

VOID	PolylineSweepOutput(entity, polyline, thickness)
ENTITY	*entity;
POLYLINE	*polyline;
FLOAT	thickness;
{
	INT	i;
	INT	m, M;
	INT	ntesselation;
	INT	nheighttesselation;
	FLOAT	delta[3];
	FLOAT	*extrusion;
	VERTEX	*startvertex;
	VERTEX	*endvertex;
	FLOAT	center[3];
	MATRIX	matrix[2];
	FLOAT	radius;
	FLOAT	startangle;
	FLOAT	endangle;
	FLOAT	deltaangle;
	FLOAT	startpoint[3];
	FLOAT	endpoint[3];
	FLOAT	distance;
	FLOAT	mindistance;
	FLOAT	p[3];
	FLOAT	rectangle[4][3];
	FLOAT	vector[3];

	mindistance = 2.0 * M_PI * MinCircleRadius / (FLOAT) MinCircleTesselation;
	nheighttesselation = max(1, (INT) (fabs(thickness) / mindistance + 0.5));

	extrusion = OptionsExtrusion(entity->options);
	MatrixExtrusion(matrix, extrusion);
	VecScale(thickness, extrusion, delta);

	M = polyline->nvertices;
	for (m = 0; m < M; m++) {
		if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0
		    && m == (M - 1))
		    	continue;
		startvertex = polyline->vertices[m];
		if (m == (M - 1))
			endvertex = polyline->vertices[0];
		else
			endvertex = polyline->vertices[m + 1];

#ifdef	DEBUG
		fprintf(stderr, "%d - %g %g %g %g %g %g\n",
			m,
			startvertex->point[0],
			startvertex->point[1],
			startvertex->point[2],
			endvertex->point[0],
			endvertex->point[1],
			endvertex->point[2]);
#endif

		/*
		 * check if it is a straight segment or an arc segment
		 */

		if (fabs(startvertex->bulge) < TOE) {
			distance = CoordDistance(startvertex->point,
					endvertex->point);
			ntesselation = (INT) (distance / mindistance + 0.5);
			ntesselation = max(1, ntesselation);
			VecSub(endvertex->point, startvertex->point, vector);
			VecScale(1.0 / (FLOAT) ntesselation, vector, vector);
			VecCopy(startvertex->point, startpoint);
			for ( i = 1; i <= ntesselation; i++) {
				VecAdd(startpoint, vector, endpoint);
				if (i == ntesselation)
					VecCopy(endvertex->point, endpoint);
				VecCopy(startpoint, rectangle[0]);
				VecCopy(endpoint, rectangle[1]);
				VecAdd(endpoint, delta, rectangle[2]);
				VecAdd(startpoint, delta, rectangle[3]);
				if (OutFileType == FILE_NFF || OutFileType == FILE_RAD)
					OutputPolygon(4, rectangle, entity);
				else
					CircleRectangleTesselation(1,
						nheighttesselation,
						rectangle[0], rectangle[1],
						rectangle[2], rectangle[3],
						entity);
				VecCopy(endpoint, startpoint);
			}
		} else {
			VecCopy(startvertex->point, startpoint);
			VecCopy(endvertex->point, endpoint);
			ArcCenterCalculation(startpoint, endpoint,
				startvertex->bulge, extrusion,
				center, &radius, &startangle,
				&endangle, &deltaangle);
			ntesselation = ArcTesselationNumber(radius,
					fabs(deltaangle));
			deltaangle /= (FLOAT) ntesselation;
			VecCopy(startvertex->point, startpoint);
			for (i = 1; i <= ntesselation; i++) {
				p[0] = radius * cos(startangle +
							deltaangle * (FLOAT) i);
				p[1] = radius * sin(startangle +
							deltaangle * (FLOAT) i);
				p[2] = 0;
				XformPoint(matrix[0], p, endpoint);
				VecAdd(center, endpoint, endpoint);
				if (i == ntesselation)
					VecCopy(endvertex->point, endpoint);
				VecCopy(startpoint, rectangle[0]);
				VecCopy(endpoint, rectangle[1]);
				VecAdd(endpoint, delta, rectangle[2]);
				VecAdd(startpoint, delta, rectangle[3]);
				if (OutFileType == FILE_NFF || OutFileType == FILE_RAD)
					OutputPolygon(4, rectangle, entity);
				else
					CircleRectangleTesselation(1,
						nheighttesselation,
						rectangle[0], rectangle[1],
						rectangle[2], rectangle[3],
						entity);
				VecCopy(endpoint, startpoint);
			}
		}
	}
}

/*
 * print contents of segments
 */

VOID	SegmentsPrint(n, s)
INT	n;
SEGMENT	s[];
{
	INT	i;

	for (i = 0; i < n; i++)
		fprintf(stderr, "segments %d - %g %g %g\n",
				i, s[i].point[0], s[i].point[1], s[i].point[2]);
}

/*
 * tesselate all arc segments, into smaller segments
 * according to the min distance
 */

SEGMENT	*PolylineArcTesselation(entity, polyline, nsegments)
ENTITY	*entity;
POLYLINE	*polyline;
INT	*nsegments;
{
	SEGMENT	*segments;
	SEGMENT	*tsegments;
	INT	i;
	INT	m, M;
	INT	ntesselation;
	FLOAT	*extrusion;
	VERTEX	*startvertex;
	VERTEX	*endvertex;
	FLOAT	center[3];
	MATRIX	matrix[2];
	FLOAT	radius;
	FLOAT	startangle;
	FLOAT	endangle;
	FLOAT	deltaangle;
	FLOAT	startpoint[3];
	FLOAT	endpoint[3];
	FLOAT	p[3];

	extrusion = OptionsExtrusion(entity->options);
	MatrixExtrusion(matrix, extrusion);

	M = polyline->nvertices;
	*nsegments = 0;
	for (m = 0; m < M; m++) {
		if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0
		    && m == (M - 1)) {
			if (*nsegments == 0)
				segments = Malloc(SEGMENT, 1);
			else {
				tsegments = Malloc(SEGMENT, *nsegments + 1);
				bcopy(segments, tsegments,
						*nsegments * sizeof(SEGMENT));
				free(segments);
				segments = tsegments;
			}

			VecCopy(polyline->vertices[m]->point,
						segments[*nsegments].point);
			(*nsegments)++;

#ifdef	DEBUG
			SegmentsPrint(*nsegments, segments);
#endif
		    	continue;
		}

		startvertex = polyline->vertices[m];
		if (m == (M - 1))
			endvertex = polyline->vertices[0];
		else
			endvertex = polyline->vertices[m + 1];

#ifdef	DEBUG
		fprintf(stderr, "%d - %g %g %g %g %g %g\n",
			m,
			startvertex->point[0],
			startvertex->point[1],
			startvertex->point[2],
			endvertex->point[0],
			endvertex->point[1],
			endvertex->point[2]);
#endif

		if (*nsegments == 0)
			segments = Malloc(SEGMENT, 1);
		else {
			tsegments = Malloc(SEGMENT, *nsegments + 1);
			bcopy(segments, tsegments, *nsegments * sizeof(SEGMENT));
			free(segments);
			segments = tsegments;
		}

		VecCopy(startvertex->point, segments[*nsegments].point);
		(*nsegments)++;

#ifdef	DEBUG
			SegmentsPrint(*nsegments, segments);
#endif

		/*
		 * check if it is a straight segment or an arc segment
		 */

		if (fabs(startvertex->bulge) > TOE) {
			VecCopy(startvertex->point, startpoint);
			VecCopy(endvertex->point, endpoint);
			ArcCenterCalculation(startpoint, endpoint,
				startvertex->bulge, extrusion,
				center, &radius, &startangle,
				&endangle, &deltaangle);
			ntesselation = ArcTesselationNumber(radius,
					fabs(deltaangle));
			deltaangle /= (FLOAT) ntesselation;
			VecCopy(startvertex->point, startpoint);
			for (i = 1; i < ntesselation; i++) {
				p[0] = radius * cos(startangle +
							deltaangle * (FLOAT) i);
				p[1] = radius * sin(startangle +
							deltaangle * (FLOAT) i);
				p[2] = 0;
				XformPoint(matrix[0], p, endpoint);
				VecAdd(center, endpoint, endpoint);
				if (*nsegments == 0)
					segments = Malloc(SEGMENT, 1);
				else {
					tsegments = Malloc(SEGMENT,
						*nsegments + 1);
					bcopy(segments, tsegments,
						*nsegments * sizeof(SEGMENT));
					free(segments);
					segments = tsegments;
				}

				VecCopy(endpoint, segments[*nsegments].point);
				(*nsegments)++;
#ifdef	DEBUG
				SegmentsPrint(*nsegments, segments);
#endif
			}
		}
	}

	return(segments);
}

/*
 * tesselate all line segments, into smaller segments
 * after all arcs are tesselated
 */

SEGMENT	*PolylineLineTesselation(nsegments, segments, nnewsegments)
INT	nsegments;
SEGMENT	*segments;
INT	*nnewsegments;
{
	SEGMENT	*newsegments;
	SEGMENT	*tsegments;
	INT	i;
	INT	m;
	INT	ntesselation;
	SEGMENT	*startvertex;
	SEGMENT	*endvertex;
	FLOAT	startpoint[3];
	FLOAT	endpoint[3];
	FLOAT	vector[3];
	FLOAT	plusstartpoint[3];
	FLOAT	plusendpoint[3];
	FLOAT	plusvector[3];
	FLOAT	minusstartpoint[3];
	FLOAT	minusendpoint[3];
	FLOAT	minusvector[3];
	FLOAT	distance;
	FLOAT	mindistance;

	mindistance = 2.0 * M_PI * MinCircleRadius / (FLOAT) MinCircleTesselation;
	*nnewsegments = 0;
	for (m = 0; m <= nsegments; m++) {
		if (m == nsegments) {
			if (*nnewsegments == 0)
				newsegments = Malloc(SEGMENT, 1);
			else {
				tsegments = Malloc(SEGMENT, *nnewsegments + 1);
				bcopy(newsegments, tsegments,
					*nnewsegments * sizeof(SEGMENT));
				free(newsegments);
				newsegments = tsegments;
			}

			bcopy(&segments[nsegments], &newsegments[*nnewsegments],
					sizeof(SEGMENT));
			(*nnewsegments)++;

		    	continue;
		}

		startvertex = &segments[m];
		if (m == nsegments)
			endvertex = &segments[0];
		else
			endvertex = &segments[m + 1];

		if (*nnewsegments == 0)
			newsegments = Malloc(SEGMENT, 1);
		else {
			tsegments = Malloc(SEGMENT, *nnewsegments + 1);
			bcopy(newsegments, tsegments,
					*nnewsegments * sizeof(SEGMENT));
			free(newsegments);
			newsegments = tsegments;
		}

		bcopy(startvertex, &newsegments[*nnewsegments], sizeof(SEGMENT));
		(*nnewsegments)++;

		VecCopy(startvertex->point      ,      startpoint);
		VecCopy(  endvertex->point      ,        endpoint);
		VecCopy(startvertex->plusoffset ,  plusstartpoint);
		VecCopy(  endvertex->plusoffset ,    plusendpoint);
		VecCopy(startvertex->minusoffset, minusstartpoint);
		VecCopy(  endvertex->minusoffset,   minusendpoint);

		distance = CoordDistance(startpoint, endpoint);
		ntesselation = max(1, (INT) (distance / mindistance + 0.5));

		VecSub(     endpoint,      startpoint,      vector);
		VecSub( plusendpoint,  plusstartpoint,  plusvector);
		VecSub(minusendpoint, minusstartpoint, minusvector);

		VecScale(1.0 / (FLOAT) ntesselation,      vector,      vector);
		VecScale(1.0 / (FLOAT) ntesselation,  plusvector,  plusvector);
		VecScale(1.0 / (FLOAT) ntesselation, minusvector, minusvector);

		for (i = 1; i < ntesselation; i++) {
			VecAdd(     startpoint,      vector,      endpoint);
			VecAdd( plusstartpoint,  plusvector,  plusendpoint);
			VecAdd(minusstartpoint, minusvector, minusendpoint);
		
			if (*nnewsegments == 0)
				newsegments = Malloc(SEGMENT, 1);
			else {
				tsegments = Malloc(SEGMENT, *nnewsegments + 1);
				bcopy(newsegments, tsegments,
					*nnewsegments * sizeof(SEGMENT));
				free(newsegments);
				newsegments = tsegments;
			}

			VecCopy(     endpoint,
					newsegments[*nnewsegments].point);
			VecCopy( plusendpoint,
					newsegments[*nnewsegments].plusoffset);
			VecCopy(minusendpoint,
					newsegments[*nnewsegments].minusoffset);

			(*nnewsegments)++;

			VecCopy(     endpoint,      startpoint);
			VecCopy( plusendpoint,  plusstartpoint);
			VecCopy(minusendpoint, minusstartpoint);
		}
	}

	return(newsegments);
}

/*
 * calculate the offset points
 */

VOID	PolylineOffseting(entity, polyline, nsegments, segments)
ENTITY	*entity;
POLYLINE	*polyline;
INT	nsegments;
SEGMENT	*segments;
{
	INT	i;
	FLOAT	width;
	FLOAT	*extrusion;
	FLOAT	startvector[3];
	FLOAT	endvector[3];
	FLOAT	offset[3];
	FLOAT	t;

	width = 0.5 * polyline->startwidth;
	extrusion = OptionsExtrusion(entity->options);

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0) {
		VecSub(segments[1].point, segments[0].point, endvector);
		VecCross(extrusion, endvector, offset);
		VecNormalize(offset);
		VecScale(width, offset, offset);
		VecAdd(segments[0].point, offset, segments[0].plusoffset);	
		VecSub(segments[0].point, offset, segments[0].minusoffset);	
	} else {
		VecSub(segments[0].point, segments[nsegments].point, startvector);
		VecSub(segments[1].point, segments[0        ].point, endvector);

		VecCross(extrusion, startvector, offset);
		VecNormalize(offset);
		VecCopy(offset, startvector);
		VecCross(extrusion, endvector, offset);
		VecNormalize(offset);
		VecCopy(offset, endvector);
		VecAdd(startvector, endvector, offset);
		VecNormalize(offset);
		t = VecDot(startvector, endvector);

		/*
		 * if two lines are in the reversed direction
		 */

		if (fabs(t + 1.0) < TOE)
			GENERR("angle of two lines is two small (cos(a) = %g\n", t);

		t = width / (FLOAT) sqrt(0.5 + 0.5 * t);

		VecScale(t, offset, offset);
		
		VecAdd(segments[0].point, offset, segments[0].plusoffset);	
		VecSub(segments[0].point, offset, segments[0].minusoffset);	
	}

	for (i = 1; i < nsegments; i++) {
		VecSub(segments[i].point, segments[i - 1].point, startvector);
		VecSub(segments[i + 1].point, segments[i].point, endvector);

		VecCross(extrusion, startvector, offset);
		VecNormalize(offset);
		VecCopy(offset, startvector);
		VecCross(extrusion, endvector, offset);
		VecNormalize(offset);
		VecCopy(offset, endvector);
		VecAdd(startvector, endvector, offset);
		VecNormalize(offset);
		t = VecDot(startvector, endvector);

		/*
		 * if two lines are in the reversed direction
		 */

		if (fabs(t + 1.0) < TOE)
			GENERR("angle of two lines is two small (cos(a) = %g\n", t);

		t = width / (FLOAT) sqrt(0.5 + 0.5 * t);

		VecScale(t, offset, offset);

		VecAdd(segments[i].point, offset, segments[i].plusoffset);	
		VecSub(segments[i].point, offset, segments[i].minusoffset);	
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0) {
		VecSub(segments[nsegments].point, segments[nsegments - 1].point,
			 endvector);
		VecCross(extrusion, endvector, offset);
		VecNormalize(offset);
		VecScale(width, offset, offset);
		VecAdd(segments[nsegments].point, offset,
					segments[nsegments].plusoffset);	
		VecSub(segments[nsegments].point, offset,
					segments[nsegments].minusoffset);	
	} else {
		VecSub(segments[nsegments].point,
				segments[nsegments - 1].point, startvector);
		VecSub(segments[0].point, segments[nsegments].point, endvector);

		VecCross(extrusion, startvector, offset);
		VecNormalize(offset);
		VecCopy(offset, startvector);
		VecCross(extrusion, endvector, offset);
		VecNormalize(offset);
		VecCopy(offset, endvector);
		VecAdd(startvector, endvector, offset);
		VecNormalize(offset);
		t = VecDot(startvector, endvector);

		/*
		 * two lines are in the reversed direction
		 */

		if (fabs(t + 1.0) < TOE)
			GENERR("angle of two lines is two small (cos(a) = %g\n", t);

		t = width / (FLOAT) sqrt(0.5 + 0.5 * t);

		VecScale(t, offset, offset);

		VecAdd(segments[nsegments].point, offset,
					segments[nsegments].plusoffset);	
		VecSub(segments[nsegments].point, offset,
					segments[nsegments].minusoffset);	
	}
}

/*
 * output a polyline with width only into DXF/DEF/NFF/RAD file
 */

VOID	PolylineWidthOutput(entity, polyline, nsegments, segments)
ENTITY	*entity;
POLYLINE	*polyline;
INT	nsegments;
SEGMENT	*segments;
{
	INT	i;
	FLOAT	tetragon[4][3];
	FLOAT	mindistance;
	INT	widthtesselation;

	mindistance = 2.0 * M_PI * MinCircleRadius / (FLOAT) MinCircleTesselation;
	widthtesselation = max(1, (INT) (polyline->startwidth / mindistance + 0.5));

	for (i = 0; i < nsegments ; i++) {
		VecCopy(segments[i    ].minusoffset, tetragon[0]);
		VecCopy(segments[i + 1].minusoffset, tetragon[1]);
		VecCopy(segments[i + 1].plusoffset , tetragon[2]);
		VecCopy(segments[i    ].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) != 0) {
		VecCopy(segments[nsegments].minusoffset, tetragon[0]);
		VecCopy(segments[0        ].minusoffset, tetragon[1]);
		VecCopy(segments[0        ].plusoffset , tetragon[2]);
		VecCopy(segments[nsegments].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}
}

/*
 * output a polyline with width and thickness into DXF/DEF/NFF/RAD file
 */

VOID	PolylineWidthSweepOutput(entity, polyline, nsegments, downsegments)
ENTITY	*entity;
POLYLINE	*polyline;
INT	nsegments;
SEGMENT	*downsegments;
{
	INT	i;
	FLOAT	thickness;
	FLOAT	*extrusion;
	FLOAT	delta[3];
	FLOAT	tetragon[4][3];
	FLOAT	mindistance;
	INT	widthtesselation;
	INT	heighttesselation;
	SEGMENT	*upsegments;

	mindistance = 2.0 * M_PI * MinCircleRadius / (FLOAT) MinCircleTesselation;
	widthtesselation = max(1, (INT) (polyline->startwidth / mindistance + 0.5));

	(VOID) OptionsThickness(entity->options, &thickness);

	heighttesselation = max(1, (INT) thickness / mindistance + 0.5);

	extrusion = OptionsExtrusion(entity->options);
	VecScale(thickness, extrusion, delta);

	upsegments = Malloc(SEGMENT, nsegments + 1);

	for (i = 0; i <= nsegments; i++) {
		VecAdd(downsegments[i].point, delta, upsegments[i].point);
		VecAdd(downsegments[i].plusoffset, delta,
						upsegments[i].plusoffset);
		VecAdd(downsegments[i].minusoffset, delta,
						upsegments[i].minusoffset);
	}

	/*
	 * here the orientations of all tetragon have to be
	 * checked, but i do not know how to do it
	 *
	 * may be later to do this
	 */

	/*
	VecCopy(segments[0].minusoffset, tetragon[0]);
	VecCopy(segments[1].minusoffset, tetragon[1]);
	VecCopy(segments[1].plusoffset , tetragon[2]);
	VecCopy(segments[0].plusoffset , tetragon[3]);

	PlaneEquation(4, tetragon, &a, &b, &c, &d);
	d = a * extrusion[0] + b * extrusion[1] + c * extrusion[2];
	if (d > TOE)
		SegmentsReverse(nsegments, downsegments);
	else
	if (d < -TOE)
		SegmentsReverse(nsegments, upsegments);
	else
		GENERR("cannot extrude a POLYLINE %s", "\n");
	*/

	/*
	 * for up and bottom faces
	 */

	for (i = 0; i < nsegments ; i++) {
		VecCopy(upsegments[i    ].minusoffset, tetragon[0]);
		VecCopy(upsegments[i + 1].minusoffset, tetragon[1]);
		VecCopy(upsegments[i + 1].plusoffset , tetragon[2]);
		VecCopy(upsegments[i    ].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) != 0) {
		VecCopy(upsegments[nsegments].minusoffset, tetragon[0]);
		VecCopy(upsegments[0        ].minusoffset, tetragon[1]);
		VecCopy(upsegments[0        ].plusoffset , tetragon[2]);
		VecCopy(upsegments[nsegments].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}

	for (i = 0; i < nsegments ; i++) {
		VecCopy(downsegments[i    ].minusoffset, tetragon[0]);
		VecCopy(downsegments[i + 1].minusoffset, tetragon[1]);
		VecCopy(downsegments[i + 1].plusoffset , tetragon[2]);
		VecCopy(downsegments[i    ].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) != 0) {
		VecCopy(downsegments[nsegments].minusoffset, tetragon[0]);
		VecCopy(downsegments[0        ].minusoffset, tetragon[1]);
		VecCopy(downsegments[0        ].plusoffset , tetragon[2]);
		VecCopy(downsegments[nsegments].plusoffset , tetragon[3]);
		TetragonTesselation(1, widthtesselation, tetragon, entity);
	}

	/*
	 * for side faces
	 */

	for (i = 0; i < nsegments; i++) {
		VecCopy(downsegments[i    ].minusoffset, tetragon[0]);
		VecCopy(downsegments[i + 1].minusoffset, tetragon[1]);
		VecCopy(  upsegments[i + 1].minusoffset, tetragon[2]);
		VecCopy(  upsegments[i    ].minusoffset, tetragon[3]);
		TetragonTesselation(1, heighttesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) != 0) {
		VecCopy(downsegments[nsegments].minusoffset, tetragon[0]);
		VecCopy(downsegments[0        ].minusoffset, tetragon[1]);
		VecCopy(  upsegments[0        ].minusoffset, tetragon[2]);
		VecCopy(  upsegments[nsegments].minusoffset, tetragon[3]);
		TetragonTesselation(1, heighttesselation, tetragon, entity);
	}

	for (i = 0; i < nsegments; i++) {
		VecCopy(downsegments[i    ].plusoffset, tetragon[0]);
		VecCopy(downsegments[i + 1].plusoffset, tetragon[1]);
		VecCopy(  upsegments[i + 1].plusoffset, tetragon[2]);
		VecCopy(  upsegments[i    ].plusoffset, tetragon[3]);
		TetragonTesselation(1, heighttesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) != 0) {
		VecCopy(downsegments[nsegments].plusoffset, tetragon[0]);
		VecCopy(downsegments[0        ].plusoffset, tetragon[1]);
		VecCopy(  upsegments[0        ].plusoffset, tetragon[2]);
		VecCopy(  upsegments[nsegments].plusoffset, tetragon[3]);
		TetragonTesselation(1, heighttesselation, tetragon, entity);
	}

	if ((polyline->polylineflag & POLYLINE_POLYGON_CLOSED) == 0) {
		VecCopy(downsegments[0        ].minusoffset, tetragon[0]);
		VecCopy(downsegments[0        ].plusoffset , tetragon[1]);
		VecCopy(  upsegments[0        ].plusoffset , tetragon[2]);
		VecCopy(  upsegments[0        ].minusoffset, tetragon[3]);
		TetragonTesselation(widthtesselation, heighttesselation,
				tetragon, entity);
		VecCopy(downsegments[nsegments].minusoffset, tetragon[0]);
		VecCopy(downsegments[nsegments].plusoffset , tetragon[1]);
		VecCopy(  upsegments[nsegments].plusoffset , tetragon[2]);
		VecCopy(  upsegments[nsegments].minusoffset, tetragon[3]);
		TetragonTesselation(widthtesselation, heighttesselation,
				tetragon, entity);
	}

	free(upsegments);
}

/*
 * output faces defined by a POLYLINE into DEF/NFF/DXF/RAD file
 */

VOID	PolylineDefOutput(entity)
ENTITY	*entity;
{
	POLYLINE	*polyline;
	FLOAT	width;
	FLOAT	thickness;
	INT	widthflag;
	INT	thicknessflag;
	INT	nsegments;
	INT	nlinesegments;
	SEGMENT	*segments;
	SEGMENT	*linesegments;

	polyline = (POLYLINE *) entity->data;

	/*
	 * if it is a polygon mesh
	 * other meshes cannot be handled in the parsing stage
	 */

	NameOutput(entity->layer);

	if ((polyline->polylineflag & POLYLINE_MESH) != 0) {
		MeshOutput(entity, polyline);
		return;
	}

	if ((polyline->polylineflag & POLYLINE_POLYLINE) != 0) {
		Polyline3DOutput(entity, polyline);
		return;
	}

	width = polyline->startwidth;
	widthflag = fabs(width) < TOE ? 0 : 1;
	thicknessflag = OptionsThickness(entity->options, &thickness);

	if (widthflag == 0 && thicknessflag == 0)
		return;

	if (widthflag == 0 && thicknessflag != 0) {
		PolylineSweepOutput(entity, polyline, thickness);
		return;
	}

	/*
	 * now the polyline has width
	 *
	 * for a general case, some segments will vanish and some loops
	 * may occur. but it is very difficult to handle those ill-fated
	 * cases, even AutoCAD is unable to do that. so if a ill-defined
	 * polyline occurs, unexpected results will occur. it is your own
	 * risk for defining such a kind polyline.
	 *
	 * now we face a simple case, i.e. no segement will dispears after
	 * offseting. first the polyline is tesselated into short line
	 * segments and then two offset polylines are generated.
	 */

	segments = PolylineArcTesselation(entity, polyline, &nsegments);
	nsegments--;
	if (OutFileType == FILE_NFF || OutFileType == FILE_RAD) {
		PolylineOffseting(entity, polyline, nsegments, segments);
		nlinesegments = nsegments;
		linesegments = segments;
	} else {
		PolylineOffseting(entity, polyline, nsegments, segments);
		linesegments = PolylineLineTesselation(nsegments, segments,
				&nlinesegments);
		nlinesegments--;
		free(segments);
	}

	/*
	 * if it has width but no thickness
	 */

	if (widthflag != 0 && thicknessflag == 0)
		PolylineWidthOutput(entity, polyline,
			nlinesegments, linesegments);
	else
		PolylineWidthSweepOutput(entity, polyline,
			nlinesegments, linesegments);

	free(linesegments);
}
