/********************************************************************************
 *  Projektname		: AERO
 *  Filename		: integration.h
 *  Filetyp		: Header
 ********************************************************************************
 *  Modulname		: integration.o
 *  Version           	: 0.0.3
 *  letzte Aenderung	: 20.01.93
 *  Autor		: Horst Stolz (HUS)
 *  Status:		: ungetestetes Dummy-File
 *  
 *  Beschreibung:
 *    Modul zur Loesung von DiffGleichungen mittels numerischer Integration
 *    
 *  Exportierte Funktionen:
 *    TReal *IErg()
 *    TReal *IInit(int numdgls)
 *    void ISchritt(TReal t)
 *    void INach(TReal *t, TReal tend)
 *    TReal IConfig(TConfig was, TReal r)
 *    void IStatistik(TIStatistik *istat)
 *
 *  Versionsgeschichte:
 *  -------------------
 *  
 *  20.12.92     Parameter t von ISchritt von *t nach t geaendert
 *  20.01.93     IStatistik() eingefuehrt und ConfigParameter zur Schrittweiten-
 *               begrenzung erweitert
 *
 ********************************************************************************/
#ifndef INTEGRATION_H
#define INTEGRATION_H

#ifndef GRUNDTYPEN_H
#include "grundtypen.h"
#endif


/* Konfigurations-Steuerwort
 */
typedef enum EConfig {
	EPS, ISCHRITT, ISCHRITT_MIN, ISCHRITT_MAX, CONSTSTEP, STEPCONTROL
} TConfig;


/* Struktur fuer Statistik-Daten 
 */
typedef struct {
    long AnzFehlversuche;
    long AnzSchritte;
    long AnzInterpolAufrufe;
    long AnzFunkAuswertungen;
} TIStatistik;


TReal IConfig(TConfig was, TReal r);
/****************************************************************************
 * Zweck:
 *   Konfiguration des Integrationsverfahrens.
 *   Zweckmaessig sind Werte: 
 *         EPS>0.0, 
 *         0.0<ISCHRITT_MIN<=ISCHRITT<=ISCHRITT_MAX.
 *   CONSTSTEP und STEPCONTROL sind Schalter um die automatische Schrittweiten-
 *   steuerung AN und AUS zu schalten. Default ist an.
 *   
 */



TReal *IInit(int numdgls, void (*funk)(TReal *, TReal *, TReal), 
	     void (*inach)(TReal *, TReal), void (*fschritt)(TReal ),
	     void (*fiswieder)(TReal));
/****************************************************************************
 * Zweck:
 *   Initialisierung des Integrationsverfahrens. Muss vor allen anderen 
 *   Routinen aufgerufen werden.
 *   
 * Parameter:
 *   numdgls  - Anzahl der zu verarbeitenden Differentialgleichungen 
 *   funk     - Differentialgleichungs-Funktion den Zielvektor(Ableitung), den 
 *              momentanen Zustandsvektor(t) und die Zeit(t) als Parameter
 *              erhaelt.
 *   inach    - Routine wird nach der Interpolation aufgerufen
 *   fschritt - Routine beim durchfueheren eines Integrationsschritte von 
 *              allen Routinen, also auch funk aufgerufen wird.
 *   fiswieder- Routine die bei einer Wiederholgung des Integrationsschrittes\
 *              wegen Ungenauigkeit aufgerufen wird.
 *
 * Rueckgabewert:
 *   Zeiger auf einem vom Aufrufer benutzbaren Vektor in den er seine 
 *   Startwerte des Zustandsvektors hereinschreiben muss.
 * 
 */


void INach(TReal *t, TReal tend);
/****************************************************************************
 * Zweck:
 *   Integration vom Zeitpunkt in *t bis zur Endzeit in tend.
 *   Der Startvektor ist der in y0=IInit(..). Als Funktion dient funk(..).
 *
 * Rueckgabewert:
 *   Bei Beendigung ist *t==tendt oder es wurde extern die Integration gestoppt
 *   mittels IStoppen().
 */


TReal *IErg();
/****************************************************************************
 * Zweck:
 *   Liefert den Ergebnisvektor nach einer Integration mit INach(..)
 *
 * Rueckgabewert:
 *   Zeiger auf Ergebnisvektor. (Groesse von IInit(numgl,..) festgelegt)
 */


void IStatistik(TIStatistik *istat);
/****************************************************************************
 * Zweck:
 *   Schreibt die bisher gesammelten Statistik in die Struktur *istat
 *   und setzt dann die bisherigen Werte wieder auf ihren Anfangszustand (0).
 */


TReal ISchritt(TReal t);
/****************************************************************************
 * Zweck:
 *   Fuehrt ein Integrationsschritt mit einem eingebettetem RK-Verfahren
 *   durch. ystart(t) -> yschritt(Rueckgabewert von ISchritt)
 *   
 * Parameter:
 *   TReal t   ... Startzeit des Integrationsschrittes
 *   
 * Rueckgabewert:
 *   TReal     ... Neu Zeit nach der Integration
 *
 * Benutzte Variablen:
 *   Lesend     : ystart - Startvektor
 *                
 *   Aufrufend  : f - Funktionsauswertungsfunktion
 *   Veraendernd: k - speicherung der RK-Zwischenwerte,
 *                    stehen danach fuer Interpolation zur Verfuegung!
 *                ytemp - Zwischenvektor 
 *   Schreibend : yschritt - Nach Aufruf steht hier der Ergebnisvektor!
 *
 */


void IStoppen();
/****************************************************************************
 * Zweck:
 *   Die Integration mittels INach wird zum fruehesten moeglichen Zeitpunkt
 *   abgebrochen(Korrekt!) ohne dass tend erreicht wird.
 *
 * Anwendung:
 *   Innerhalb von funk().
 *
 */


TReal IMinInterpolZeit();
/****************************************************************************
 * Zweck:
 *   Minimale Interpolationszeit bis zu der zurueckinterpoliert werden kann
 *   ohne einen neuen Funktionsaufruf durchfuehren zu muessen.
 *
 * Rueckgabewert:
 *   Liefer minimale Interpolationszeit.
 *   Diese ist <= tend!
 */



extern int i_debug_level;	/* Interne Benutzung Horst ! */

#endif

