/***********************************************************************/
/*                                                                     */
/* Project      : GRAV                                                 */
/* Current Rel. : 3.1                                                  */
/* Creator      : Michael Knigge                                       */
/* Creation Date: 01/21/95                                             */
/* Description  : Entry-Module                                         */
/*                                                                     */
/* Functions    : main()                                               */
/*                                                                     */
/***********************************************************************/

#include <stdio.h>
#include <ncurses.h>
#include "standard.h"
#include "gbm.h"
#include "defines.h"
#include "misc.h"
#include "directory.h"
#include "selector.h"
#include "screen.h"
#include "error.h"
#include "print.h"
#include "view.h"
#include "info.h"
#include "convert.h"

int main(int argc, char *argv[])
{
   int    rc;
   int    tmp;
   int    mode_index;
   int    page;
   int    index;
   int    error_id;
   int    use_color;
   int    max_files;
   int    count_tagged;
   int    need_redraw;
   int    loop_again;
   int    chipset;

   DIRTAB dirtab[MAX_DIRTAB];
   DIRTAB single[1];
   GMODE  gmode[MAX_MODES];



   printf("GRAV 3.1                                ");
   printf("               (C) Michael Knigge, 1995\n");
   printf("----------------------------------------");
   printf("---------------------------------------\n");

   grav_init_ncurses();
   grav_init(&use_color, gmode);

   rc = grav_interpret_params(argc, argv, &use_color, &error_id);

   if (error_id != GRAV_NO_ERROR)
   {
      endwin();
      fprintf(stderr,"\x1b[H\x1b[J");
      grav_show_error(MSG_CONSOLE, TRUE, error_id);      
      return(FALSE);
   }

   grav_read_config(&mode_index);

   /*
    * if rc is not 0 there was a filename specified at the command
    * line and we have to show a single file without showing the
    * menu. we just show the file and quit......
    */
    
   if (rc != 0)
   {
      printf("Please wait - reading file %s...\n", argv[rc]);
      error_id  = grav_view_single_file(argv[rc], mode_index, gmode);
      endwin();
      fprintf(stderr,"\x1b[H\x1b[J");
      if (error_id != GRAV_NO_ERROR)
         grav_show_error(MSG_CONSOLE, TRUE, error_id);
      return(TRUE);
   }

   /*
    * if we are here, we just have to do the normal work. First, read 
    * the current directory, show the grav mainscreen, show the directory
    * get a file, and so on....
    */

   page         = 0;
   index        = 0;
   count_tagged = 0;
   need_redraw  = TRUE;
   loop_again   = TRUE;
   error_id     = GRAV_NO_ERROR;
   chipset      = vga_getcurrentchipset();

   grav_read_directory(".", &max_files, dirtab);   

   /*
    * loop until we want to exit GRAV
    */

   do
   {
      if (need_redraw)
      {
         grav_draw_main_screen(use_color);
         grav_display_mode(use_color, mode_index, gmode);
         grav_display_card(use_color, chipset); 
         grav_show_directory(use_color, page, max_files, dirtab);
         need_redraw = FALSE;
      }
      
      if (error_id != GRAV_NO_ERROR)
         grav_show_error(MSG_WINDOW, use_color, error_id);
      
      switch(grav_file_selector(use_color, &page, max_files, &index, &count_tagged, dirtab))
      {
      case DO_CHDIR:    grav_change_directory(&error_id, index, dirtab);
                        grav_read_directory(".", &max_files, dirtab);
                        grav_show_directory(use_color, page, max_files, dirtab);
                        index        = 0;
                        page         = 0;
                        count_tagged = 0;
                        break;
                     
      case DO_QUIT:     loop_again = FALSE;
                        break;
                     
      case DO_VIEW:     error_id = grav_view_file(use_color, &count_tagged, dirtab, mode_index, gmode);
                        need_redraw = TRUE;
                        break;
                     
      case DO_INFO:     grav_view_fileinfo(&error_id, use_color, index, dirtab);
                        grav_show_directory(use_color, page, max_files, dirtab);
                        break;
                     
      case DO_HELP:     grav_show_help(use_color);
                        grav_display_mode(use_color, mode_index, gmode);
                        grav_display_card(use_color, chipset);
                        break;

      case DO_PRINT:    error_id    = grav_print_file();
                        need_redraw = TRUE;
                        break;

      case DO_INPUT:    error_id = grav_input_filename(use_color, single);
                        if ( error_id != GRAV_NO_ERROR ||
                             strlen(single[0].filename) == 0)
                          break;
                        tmp         = 1;
                        need_redraw = TRUE;
                        error_id = grav_view_file(use_color, &tmp, single, mode_index, gmode);
                        break;
                     
      case DO_CONVERT:  error_id    = grav_convert_file();
                        need_redraw = TRUE;
                        break;
                       
      case DO_INC_MODE: grav_change_mode(&mode_index, +1, gmode);
                        grav_display_mode(use_color, mode_index, gmode);
                        break;
                        
      case DO_DEC_MODE: grav_change_mode(&mode_index, -1, gmode);
                        grav_display_mode(use_color, mode_index, gmode);
                        break;
      }
   }
   while(loop_again == TRUE);
   
   /*
    * here we will do a global cleanup (exit ncurses etc.)
    */

   grav_write_config(mode_index);    
   grav_goodbye(use_color);
   grav_cleanup();   

   /*
    * go back to our loved Linux-Prompt ;-)
    */
    
   return(TRUE);
}
