/* 
 * rspfd: Radio Shortest Path Daemon. A router for packet radio networks.
 * Copyright (C) 1995 Craig Small VK2XLZ
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <arpa/inet.h>
#include <linux/ax25.h>
#include <linux/if_ether.h>
#include <netinet/in.h>
#include <netinet/ip_icmp.h>
#include <netinet/ip.h>
#include <net/if.h>
#include <net/if_route.h>
#include <sys/ioctl.h>
#include <sys/socket.h> 
#include <sys/types.h>
#include <netdb.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#include "rspfd.h"
#include "rspfif.h"
#include "rspf_out.h"
#include "queue.h"
#include "rspfroute.h"


extern struct rspf_mib rspf_stats;
extern struct queue *routerq;
extern struct queue *outfragq;


/*
 * send_rrh()
 *
 * Send a RRH (Router-Router Hello) onto specifed interface
 *
 * Returns:
 *	Nothing
 * Arguments:
 *	char*: Interface name to send RRH
 */
void send_rrh(char *iface)
{
	static u_char outpack[128];
	struct rspfrrh *rrh =(struct rspfrrh*) outpack;
	struct sockaddr_in addr;
	u_long saddr, daddr;

	rrh->version = RSPF_VERSION;
	rrh->type = TYPE_RRH;
	rrh->checksum = 0;
	
	addr = get_iface_addr(iface);
	if( (rrh->addr = addr.sin_addr.s_addr) == INADDR_NONE)
	{
		syslog(LOG_DAEMON | LOG_WARNING, "send_rrh(): Cannot get interface %s address.", iface);
		return;
	}
	
	rrh->tx_pkts = htons(get_tx_pkts(iface));
	rrh->flags = 1;		/* wizzer - this needs to be changed */
	
	saddr = addr.sin_addr.s_addr;
	
	/* Work out where to send it */
	addr = get_bcast_addr(iface);
	if( (daddr = addr.sin_addr.s_addr) == INADDR_NONE)
	{
		syslog(LOG_DAEMON | LOG_WARNING, "send_rrh(): Cannot get broadcast address for %s.", iface);
		return;
	}

	
	/* Calculate the checksum, we need a pseudo header like TCP/UDP :<
	 */
	rrh->checksum = rspf_check((char*)rrh, RSPF_RRH_LEN, saddr, daddr);	
	
	send_rspf(saddr, daddr, outpack, RSPF_RRH_LEN, iface);
	
	rspf_stats.rspfOutRrhs++;
}

/*
 * send_ping()
 * Sends pings (ICMP echo) to a specified address
 *
 * Returns:
 *	int		The sequence number of the ping
 *
 * Arguments:
 *	u_long		node to ping
 */
int send_ping(u_long daddr, char *iface)
{	
	static u_short pingseq = 1;
	u_char outpack[sizeof(struct icmphdr)];
	struct icmphdr *icp = (struct icmphdr*)&outpack;
	int s;
	struct sockaddr whereto;
	struct sockaddr_in *to = (struct sockaddr_in*) &whereto;
	struct protoent *proto;
	struct rtentry *rtp;
	struct rtentry rt;
	

	/* We cannot have a sequence number of 0 */
	if (pingseq == 0)
		pingseq++;
			
	/* Create the socket to send the echo on */
	bzero((char*)&whereto, sizeof(struct sockaddr));
	to->sin_family = AF_INET;
	to->sin_addr.s_addr = daddr;
	
	if ((proto = getprotobyname("icmp")) == NULL) {
		syslog(LOG_DAEMON | LOG_CRIT, "send_ping(): unknown protocol icmp (%m)");
		return 0;
	}
	if ((s = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		syslog(LOG_DAEMON | LOG_ERR, "send_ping(): socket failed (%m)");
		return 0;
	}
	
	/* Create ICMP header */
	icp->type = ICMP_ECHO;
	icp->code = 0;
	icp->checksum = 0;
	icp->un.echo.id = getpid();
	icp->un.echo.sequence = pingseq++;
	icp->checksum = in_cksum((u_short*)outpack, sizeof(struct icmphdr));

	/*
	 * We have to set a dummy route temporarily, if none is there
	 */
	if ( (rtp = get_route(daddr, 32)) != NULL)
	{
		rt = *rtp;
		add_route(daddr, 32, 0, 1);
		sendto(s, outpack, sizeof(struct icmphdr), 0, &whereto, sizeof(struct sockaddr));
		/*
		 * Restore the route entry
		 */
		if (ioctl(s, SIOCADDRT, &rt) < 0)
		{
			syslog(LOG_DAEMON | LOG_ERR, "send_ping(): ioctl to restor route failed. (%m)");
		}
	}			
	else
	{
		/* No route, so we can go for it */
		add_route(daddr, 32, 0 ,1);
		sendto(s, outpack, sizeof(struct icmphdr), 0, &whereto, sizeof(struct sockaddr));
		del_route(daddr, 32);
	}
	close(s);
	return icp->un.echo.sequence;
}	

/*
 * send_news()
 *
 * Sends a good or bad news packet about one link on all interfaces
 *
 * Returns:
 *	Nothing
 *
 * Arguments:
 *	u_long		Source address of link
 *	u_long		Destination address of link
 *	u_char		Significant bits of dest addr
 *	u_char		Cost of link
 *	char*		Name of port (ignored - wizzer remove it)
 */
void send_news(u_long saddr, u_long daddr, u_char sigbits, u_char cost, char *port)
{
	u_char buf[RSPFNODE_LEN + RSPFLINK_LEN + RSPFADJ_LEN + 3];
	struct rspfnode_hdr *nodehdr = (struct rspfnode_hdr*)buf;
	struct rspflink_hdr *linkhdr = (struct rspflink_hdr*)(buf + RSPFNODE_LEN);
	struct rspfadj_hdr *adjhdr = (struct rspfadj_hdr*)(buf + RSPFNODE_LEN + RSPFLINK_LEN);
	
	bcopy((char*)&saddr, nodehdr->addr, 4);
	nodehdr->seq_no = htons(rspf_stats.SequenceNumber);
	nodehdr->sub_seq_no = ++rspf_stats.SubSequenceNumber;
	nodehdr->links = 1;
	
	/* Wizzer get horizon */
	linkhdr->horizon = get_horizon(daddr, port);
	linkhdr->erp = 0;
	linkhdr->cost = cost;
	linkhdr->adjacencies = 1;
	
	adjhdr->sig_bits = sigbits | RSPFADJ_LASTFLAG;
	bcopy((char*)&daddr, adjhdr->addr, 4);

	/* Send to get stuck in an envelope and send */
	send_rspf_env(buf, sizeof(buf), 1, NULL);
		
}

int send_rspf(u_long saddr, u_long daddr, u_char *buf, int buflen,  char *iface)
{
	struct sockaddr_in addr;
	struct protoent *proto;
	int skt;
	int i;
/*	FILE *fp;*/

	if ( (proto = getprotobyname("rspf")) == NULL) {
		syslog(LOG_DAEMON | LOG_CRIT, "send_rrh(): unknown protocol rspf (%m)");
		return -1;
	}
			
	if ( (skt = socket(AF_INET, SOCK_RAW, proto->p_proto)) < 0) {
		perror("RSPFd: send_rrh(): socket");
		close(skt);
		return -1 ;
	}
	
	/* We have to tell the system we really want to send a b'cast packet */
	i = 1;
	if (setsockopt(skt, SOL_SOCKET, SO_BROADCAST, (char*)&i, sizeof(i)) < 0) {
		perror("RSPFd: send_rrh(): setsockopt");
		close(skt);
		return -1;
	}
	
	/* Copy over address */
	addr = get_bcast_addr(iface);
	if( addr.sin_addr.s_addr == INADDR_NONE)
	{
		syslog(LOG_DAEMON | LOG_WARNING, "send_rspf(): Cannot get broadcast address for interface %s.", iface);
		close(skt);
		return -1;
	}
	
	i = sendto(skt, buf, buflen, 0, (struct sockaddr*)&addr, sizeof(struct sockaddr) );
	if (i < 0) {
		syslog(LOG_DAEMON | LOG_ERR,"send_rspf(): sendto failed (%m)");
		close(skt);
		return -1;
	}
	rspf_stats.rspfOutMsgs++;
	close(skt);
#ifdef DEBUG_LOGTX	
	/*
	 * Update log file - xlz remove
	 */
	if ( (fp = fopen("rspftx.log", "a")) == NULL)
	{
		syslog(LOG_ERR | LOG_DAEMON, "send_rspf(): log file couldn't open. (%m).");
	} else		
	{
		for(i = 0; i < buflen; i++)
			fprintf(fp, "%u ", buf[i]);
		fprintf(fp, "\n\n");
		fclose(fp);
	}
#endif
	return 0; 
} /* send_rspf */


/*
 * send_full_bulletin
 *
 * Sends a full routing bulletin on the specified interface
 *
 * Returns:
 *	Nothing
 *
 * Arguments:
 *	char*	Name of interface, NULL means all interfaces
 */
void send_full_bulletin(char *iface)
{
	struct router *rtr;
	qmark rtr_qm;
	u_char buf[4096];
	int bufcnt, nodecnt;
	char hostname[64];
	struct hostent *phe;
	u_long addr;
	
	bufcnt = 0;
	nodecnt = 1;

	/* Get our hostname so we can put our routes in*/	
	gethostname(hostname, sizeof(hostname));
	if ((phe = gethostbyname(hostname)) == NULL)
	{
		syslog(LOG_ERR | LOG_DAEMON, "send_full_bulletin(): gethostbyname failed (%m)");
		return;
	}
	bcopy(phe->h_addr, (char*)&addr, 4);

	rspf_stats.SubSequenceNumber = 0;
	bufcnt += get_rtr_bull(addr, buf, sizeof(buf));

	/* Get routing bulletins from other systems */	
	rtr = (struct router*)qmove_first(routerq, &rtr_qm);
	while (rtr != NULL)
	{
		bufcnt += get_rtr_bull(rtr->addr, buf + bufcnt, sizeof(buf) - bufcnt);		
		rtr = (struct router*)qmove_next(routerq, &rtr_qm);		
		nodecnt++;
	}

	send_rspf_env(buf, bufcnt, nodecnt, iface);
}
	
void send_rspf_env(u_char *buf, int size, int nodecnt, char *iface)
{
	char outbuf[2048];
	char *startbuf, *endbuf;
	struct rspfroute_hdr *rthdr = (struct rspfroute_hdr*)outbuf;
	char ifbuf[256];
	char *ifbptr = ifbuf;
	int ifcount;
	struct sockaddr_in sin;
	u_long saddr, daddr;
	struct ifreq ifr;
	int sync, frag_tot, mtu, frag;
	u_int nodes, links, adjs;
	int skt;
	struct rspfnode_hdr *nodehdr;
	struct rspflink_hdr *linkhdr;
	struct outfrag *frg;
	qmark ofrg_qm;

	/* Create socket */
	if ( (skt = socket(AF_INET, SOCK_DGRAM,0)) < 0) 
	{
 		syslog(LOG_DAEMON | LOG_ERR, "send_rspf_env(): socket failed. (%m)");
 		(void) close(skt);
 		return;
 	}
	/* Fill in what we can in routing envelope header */
	rthdr->version = RSPF_VERSION;
	rthdr->type = TYPE_ROUTING;
	rthdr->checksum = 0;
	rthdr->nodes = nodecnt;
	rthdr->env_no = htons(rspf_stats.EnvelopeNumber++);
	
	ifcount = rspf_ifaces(ifbuf, 256);
	while (ifcount-- > 0) {
		strcpy(ifr.ifr_name, ifbptr);
		ifbptr += strlen(ifr.ifr_name) + 1;
		
		/* If we have a specified interface, check it */
		if (iface != NULL && (strcmp(iface, ifr.ifr_name) != 0))
			continue;
			
		frag_tot = 1;	
		nodes = nodecnt;
		sync = 0;
		endbuf = startbuf = buf;
		
		/* We now chop up the packet into MTU sized fragments */

		if (ioctl(skt, SIOCGIFMTU, &ifr) < 0)
		{
			syslog(LOG_DAEMON | LOG_WARNING, "send_rspf_env(): ioctl to get MTU failed for port %s (%m)", ifr.ifr_name);
			continue;
		}
		mtu = ifr.ifr_mtu - RSPFROUTE_LEN ;
		
		if (mtu < RSPFNODE_LEN)
		{
			syslog(LOG_DAEMON | LOG_WARNING, "send_rspf_env(): Interface %s has too small MTU of %d.", ifr.ifr_name, ifr.ifr_mtu);
			continue;
		}
		
		/* Now move through the buffer */
		while(nodes-- > 0)
		{
			if ( (endbuf - startbuf) + RSPFNODE_LEN > mtu)
			{
				/* Too big, have to send it */
				add_outfrag(frag, sync, startbuf, endbuf);
				sync = 0;
				frag++;
				startbuf = endbuf;
			}
			nodehdr = (struct rspfnode_hdr*)endbuf;
			/* Get relative value of sync byte, if not already set */
			if (sync == 0)
				sync = 4 + ((int)endbuf - (int)buf);
			endbuf += RSPFNODE_LEN;
			links = nodehdr->links;
			while(links-- > 0)
			{
				if ( (endbuf - startbuf) + RSPFLINK_LEN > mtu)
				{
					/* Too big, have to send it */
					add_outfrag(frag, sync, startbuf, endbuf);
					sync = 0;
					frag++;
					startbuf = endbuf;
				}
				linkhdr = (struct rspflink_hdr*)endbuf;
				endbuf += RSPFLINK_LEN;
				adjs = linkhdr->adjacencies;
				while(adjs-- > 0)
				{
					if ( (endbuf - startbuf) + RSPFLINK_LEN > mtu)
					{
						/* Too big, have to send it */
						add_outfrag(frag, sync, startbuf, endbuf);
						sync = 0;
						frag++;
						startbuf = endbuf;
					}
				 	endbuf += RSPFADJ_LEN;
				 }/*adjs*/
			} /*links*/
		} /* nodes */
		/* Add last fragment, if there is one */
		if (endbuf > startbuf)
			add_outfrag(frag, sync, startbuf, endbuf);
		/* Now have all the fragments in the queue, we know how many fragments
		 * we need, so we can send them
		 */
		/* Get source and destination addresses */
		sin = get_bcast_addr(ifr.ifr_name);
		if( (daddr = sin.sin_addr.s_addr) == INADDR_NONE)
		{
			syslog(LOG_DAEMON | LOG_WARNING, "send_rspf_env(): Cannot get broadcast address for interface %s.", ifr.ifr_name);
			continue;
		}

		sin = get_iface_addr(ifr.ifr_name);
		if( (saddr = sin.sin_addr.s_addr) == INADDR_NONE)
		{
			syslog(LOG_DAEMON | LOG_WARNING, "send_rspf_env(): Cannot get address for interface %s.", ifr.ifr_name);
			continue;
		}

		frag_tot = frag - 1;
		frag = 1;
		frg = (struct outfrag*)qmove_first(outfragq, &ofrg_qm);
		while(frg != NULL)
		{
			rthdr->frag = frag++;
			rthdr->frag_tot = frag_tot;
			rthdr->sync = sync < 256 ? sync : 0;
			if (frg->datalen > sizeof(outbuf) - RSPFROUTE_LEN)
			{
				syslog(LOG_DAEMON | LOG_ERR, "send_rspf_env(): Fragment of %d bytes too big, dropping.");
			} else {			
				bcopy(frg->data, outbuf + RSPFROUTE_LEN, frg->datalen);
				rthdr->checksum = 0;
				rthdr->checksum = rspf_check(outbuf, RSPFROUTE_LEN + frg->datalen, saddr, daddr);				
				send_rspf(saddr, daddr, outbuf, RSPFROUTE_LEN + frg->datalen, ifr.ifr_name);
			}
			del_qnode(outfragq, ofrg_qm, 1);		
			frg = NULL;			
			/* Use move first because we're destructing as we go */
			frg = (struct outfrag*)qmove_first(outfragq, &ofrg_qm);
		}
	} /* next interface */
	rspf_stats.rspfOutRouteEnvs++;
	(void) close(skt);
}	
			
void add_outfrag(int frag, int sync, u_char *startbuf, u_char *endbuf)
{
	struct outfrag *frg;
	int size = endbuf - startbuf;
	
	if (size <= 0)
	{
		syslog(LOG_DAEMON | LOG_ERR, "add_outfrag() fragment size of %d (%m)", size);
		return;
	}
	
	frg = (struct outfrag*)malloc(sizeof(struct outfrag) + size);
	if (frg == NULL)
	{
		syslog(LOG_DAEMON | LOG_ERR, "add_outfrag() Memory sequeze (%m)");
		return;
	}
	frg->frag = frag;
	frg->sync = sync;
	frg->data = (u_char*)frg + sizeof(struct outfrag);
	frg->datalen = size;
	bcopy(startbuf, frg->data, size);

	add_qnode(outfragq, (void*)frg, NULL);
}
