/*
 * feature.c
 *
 * (c)16/2/1994 Stuart N. John
 *
 *
 * History:
 *
 *   16-2-1994 - Created module
 *
 */

#include <sys/types.h>
#include <sys/times.h>
#include <stdio.h>
#include <math.h>

#include "glove.h"
#include "posture.h"
#include "feature.h"


/* derived continuous features from Sturman's thesis

path segment vector:            vi = (xi - xi-1, yi - yi-1, zi - zi-1)
length of path segment:         ||vi|| = sq (x^2 + y^2 + z^2)
normalized path segment vector: ~vi = vi/||vi||  e.g. (-0.5, .2, .8) -1.0 < 0 < 1.0
path segment speed:             si = vi/ti - ti-1 ie xyz each axis / tick time diff
linear speed:                   `si = ||si|| = ||vi|| / ti - ti-1
normalised cross-product:       ci = ~vi-1 x ~vi = (xi * xi-1, yi * yi-1, zi * zi-1)
normalised dot-product:         di = ~vi-1 . ~vi = xi * xi-1 + yi * yi-1 + zi * zi - 1

N.B. The tick time difference in the glove data structure is converted
     to seconds in the speed calculation functions.

N.B. The glove measurements are not consistent, each unit in Z is a real
     distance of about 14mm and each unit in X and Y is about 3mm.
     We convert the units to mm in the feature functions.

future work: direction switches, zones, hand postures, direction distances etc.

*/


/*
 * path segment vector: vi = (xi - xi-1, yi - yi-1, zi - zi-1)
 *
 */
void path_segment(Glove_t *g, Glove_t *lastg, Vector_t *v)
{
  v->x = g->x - lastg->x;
  v->y = g->y - lastg->y;
  v->z = g->z - lastg->z;

  /* convert the glove units to mm */

  v->x *= 3;
  v->y *= 3;
  v->z *= 14;

#ifdef DEBUGFEATURE
  fprintf(stderr, "ps [%f,%f,%f]\n", v->x, v->y, v->z);
#endif
}


/*
 * length of path segment: ||vi|| = sqrt (x^2 + y^2 + z^2)
 *
 */
double length_path_segment(Vector_t *v)
{
#ifdef DEBUGFEATURE
  fprintf(stderr, "lps %f\n", sqrt((v->x * v->x) + (v->y * v->y) + (v->z * v->z)));
#endif

  return sqrt((v->x * v->x) + (v->y * v->y) + (v->z * v->z));
}


/*
 * normalized path segment vector: ~vi = vi/||vi||
 * e.g. (-0.5, .2, .8)
 * i.e. normal is -1.0 < 0 < 1.0
 *
 */
void norm_path_segment(Vector_t *v, double length, Vector_t *nv)
{
  nv->x = v->x / length;
  nv->y = v->y / length;
  nv->z = v->z / length;

#ifdef DEBUGFEATURE
  fprintf(stderr, "nps [%f,%f,%f]\n", nv->x, nv->y, nv->z);
#endif
}


/*
 * path segment speed: si = vi/ti - ti-1 - each axis / tick time diff
 *
 */
void segment_speed(Vector_t *v, Glove_t *g, Vector_t *sv)
{
  /* convert the glove tick difference value to seconds */
  double sec = (double) g->tick / CLK_TCK;

  sv->x = v->x / sec;
  sv->y = v->y / sec;
  sv->z = v->z / sec;

#ifdef DEBUGFEATURE
  fprintf(stderr, "ss [%f,%f,%f]\n", sv->x, sv->y, sv->z);
#endif
}


/*
 * linear (abs) speed: `si = ||si|| = ||vi|| / ti - ti-1
 *
 */
double linear_speed(double length, Glove_t *g)
{
  /* convert the glove tick difference value to seconds */
  double sec = (double) g->tick / CLK_TCK;

#ifdef DEBUGFEATURE
  fprintf(stderr, "ls %f\n", length / sec);
#endif

  return length / sec;
}


/*
 * normalised cross-product: ci = ~vi-1 x ~vi = (xi * xi-1, yi * yi-1, zi * zi-1)
 *
 */
void norm_cross_product(Vector_t *nv, Vector_t *lastnv, Vector_t *ncp)
{
  ncp->x = lastnv->y * nv->z - lastnv->z * nv->y;
  ncp->y = lastnv->z * nv->x - lastnv->x * nv->z;
  ncp->z = lastnv->x * nv->y - lastnv->y * nv->x;

#ifdef DEBUGFEATURE
  fprintf(stderr, "ncp [%f,%f,%f]\n", ncp->x, ncp->y, ncp->z);
#endif
}


/*
 * normalised dot-product: di = ~vi-1 . ~vi = xi * xi-1 + yi * yi-1 + zi * zi-1
 *
 */
double norm_dot_product(Vector_t *nv, Vector_t *lastnv)
{
#ifdef DEBUGFEATURE
  fprintf(stderr, "ndp %f\n", (nv->x * lastnv->x + nv->y * lastnv->y + nv->z * lastnv->z));
#endif

  return (nv->x * lastnv->x + nv->y * lastnv->y + nv->z * lastnv->z);
}


/*
 * classify the current hand posture
 *
 */
postureData_t *hand_posture(Glove_t *g)
{
  postureData_t *pos;

  pos = classify_posture(getThumb(g), getIndex(g), getMiddle(g), getRing(g), getRotation(g));

#ifdef DEBUGFEATURE
  if (pos)
    fprintf(stderr, "pos '%s' [%d]\n", pos->name, pos->id);
  else
    fprintf(stderr, "pos (none)\n");
#endif

  return pos;
}


/* derived total features from Rubine's thesis

   bounding box volume: (xmin, xmax, ymin, ymax, zmin, zmax)
   -> cumulative length: Ei=0..n ||vi||

*/


/*
 * bounding box volume: (xmin, xmax, ymin, ymax, zmin, zmax)
 *
 */
double bounding_box(Vector_t *minv, Vector_t *maxv, Box_t *box)
{
  double length, width, height;

  box->min.x = minv->x;
  box->min.y = minv->y;
  box->min.z = minv->z;

  box->max.x = maxv->x;
  box->max.y = maxv->y;
  box->max.z = maxv->z;

  if (box->min.x < 0.0 && box->max.x > 0.0)
    length = box->max.x + fabs(box->min.x);
  else
    length = box->max.x - box->min.x;

  if (box->min.y < 0.0 && box->max.y > 0.0)
    width = box->max.y + fabs(box->min.y);
  else
    width = box->max.y - box->min.y;

  if (box->min.z < 0.0 && box->max.z > 0.0)
    height = box->max.z + fabs(box->min.z);
  else
    height = box->max.z - box->min.z;

  return length * width * height;
}
