/* powerglove.c
 *
 * (c)9/3/1994 Stuart N. John
 *
 *
 * History:
 *
 *   9-3-1994 - Created module
 *
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/times.h>
#include <errno.h>
#include <fcntl.h>
#include <stdio.h>

#include "powerglove.h"
#include "mode.h"        /* for filter mode flag */
#include "filter.h"


/* global variables */

static int fd = -1;             /* file descriptor for serial device */
static struct termio tio, tin;  /* tty control interface             */

#ifdef DEBUGGLOVE
static char buffer[] = {'0', '0', '0', '0', '0', '0', '0', '0', '\0'};
#endif


/* prototypes */

static int glove_sync(void);

#ifdef DEBUGGLOVE
static char *binary(u_char num, char *buf);
#endif


/* functions to handle glove input */

/*
 * Try and open the serial driver and get glove data
 *
 * return:  -1 on error
 *           0 if OK
 *
 */
int glove_init(void)
{
  int    sfd;        /* serial device file descriptor                */
  char   byte;       /* for M box mode command                       */
  int    retry = 2;  /* two retry attempts to find glove serial data */
  int    count;      /* check number of characters read              */

  /* open serial device for reading glove data */

  fprintf(stderr, "powerglove: opening serial device...\n");

  sfd = open(PGDEVICE, O_RDWR | O_NONBLOCK);

  if (sfd < 0)
  {
    fprintf(stderr, "powerglove: can't open serial device\n");
    perror("powerglove");
    return -1;
  }

  /* get serial tty settings */

  ioctl(sfd, TCGETA, &tio);

  tin = tio;

  tin.c_iflag &= ~ISTRIP;  /* turn off eighth bit strip */

  tin.c_lflag &= ~ECHO;    /* turn off echo mode                      */
  tin.c_lflag &= ~ICANON;  /* turn off canonical (buffered line) mode */
  tin.c_lflag &= ~ISIG;    /* turn off interrupt chars (CTRL-C #3)    */

  /* emulate Unix V7/BSD RAW or CBREAK modes            */
  /* requests return immediately with single characters */

  tin.c_cc[VMIN]  = 1;
  tin.c_cc[VTIME] = 0;

  /* set new serial tty flags */

  ioctl(sfd, TCSETA, &tin);

  /* put the Menelli box in continuous mode */

  /* (I tried using request mode, but after a few requests
      the box just locks up and stops sending/receiving) */

  byte = 'C';
  write(sfd, &byte, 1);

  /* attempt to read data from the serial device */

  fprintf(stderr, "powerglove: checking for glove data...\n");

  sleep(1);
  count = read(sfd, &byte, 1);

  while (count < 0 && errno == EAGAIN && retry--)
  {
    perror("powerglove");
    sleep(1);
    count = read(sfd, &byte, 1);
  }

  if (count < 0)
  {
    fprintf(stderr, "powerglove: glove data not found\n");

    if (errno != EAGAIN)
      perror("powerglove");

    close(sfd);
    return -1;
  }

  close(sfd);

  /* re-open the serial device in bi-directional blocking mode */

  fd = open(PGDEVICE, O_RDONLY);  /* use global file descriptor */

  ioctl(fd, TCSETA, &tin);

  /* sync with glove data packets */

  if (glove_sync() < 0)
  {
    fprintf(stderr, "powerglove: failed to sync with glove data packets\n");
    close(fd);
    return -1;
  }

  return 0;
}


/*
 * Sync with glove data packets
 *
 * return:  -1 on error
 *           0 if OK
 *
 */
static int glove_sync(void)
{
  char   byte;           /* temp glove data byte */
  u_char bits;           /* temp glove data byte */
  int    attempt = 50;   /* number of attempts to sync with data packets */

  fprintf(stderr, "powerglove: syncing with glove data...\n");

  fprintf(stderr, "powerglove: flex fingers 3 times and press CENTER\n");
  sleep(3);

  /* get in sync with glove data packets */

  while (attempt--)
  {
    /* find the next possible header byte */

    read(fd, &bits, 1);
    while (bits != 0xA0) read(fd, &bits, 1);

    /* possibly in sync here, but check for glitches */

    read(fd, &byte, 1);   /* x pos */
    if (byte < -127 || byte > 127)
    {
      continue;  /* invalid x position */
    }

    read(fd, &byte, 1);   /* y pos */
    if (byte < -127 || byte > 127)
    {
      continue;  /* invalid y position */
    }

    read(fd, &byte, 1);   /* z pos */
    if (byte < -127 || byte > 127)
    {
      continue;  /* invalid z position */
    }

    read(fd, &byte, 1);   /* rotation */
    if (byte < 0 || byte > 11)
    {
      continue;  /* invalid wrist rotation value */
    }

    read(fd, &bits, 1);  /* fingers */
    if (bits == 0xA0)
    {
      /* the thumb and index finger can cause the 0xA0 value when
         they are both at value 2, and the middle and ring fingers
         are both 0 (straight), so it might be worth flexing your
         fingers during glove initialisation */

      fprintf(stderr, "powerglove: thumb & index finger produce header byte value\n");
      continue;
    }

    read(fd, &bits, 1);  /* keys */
    break;
  }

  if (attempt < 0)
  {
    fprintf(stderr, "powerglove: couldn't sync with glove data\n");
    return -1;
  }

  return 0;
}


/*
 * Sample latest glove data (and display)
 *
 */
int glove_update(char *data, char filtermode)
{
  char           byte;     /* temp input byte                             */
  struct tms     tmptime;  /* times() fills this out, but we don't use it */
  clock_t        tick;     /* to get elapsed system ticks                 */
  static clock_t last = 0; /* previous sample elapsed system ticks        */

  read(fd, &byte, 1);  /* packet header (discard)

  /* fix for stupid linux pl15 serial driver, please remove if you don't
     lose characters i.e. linux pl14 serial driver */
  while((u_char) byte != 0xA0) read(fd, &byte, 1);

  read(fd, &data[G_X], 1);  /* x pos */
  read(fd, &data[G_Y], 1);  /* y pos */
  read(fd, &data[G_Z], 1);  /* z pos */

  /* remove glitches with predictive digital filter */

  if (filtermode == PGFILTER_ON)
    filter(&data[G_X], &data[G_Y], &data[G_Z]);

  read(fd, &data[G_W], 1);  /* rotation */

  /* adjust rotation boundaries for glitches */
  /* (anybody got any ideas on predicting the wrist roll value ?) */

  if (data[G_W] < 0)
    data[G_W] = 0;
  if (data[G_W] > 11)
    data[G_W] = 11;

  read(fd, &data[G_F], 1);  /* fingers */

#ifdef DEBUGGLOVE
  fprintf(stderr, "x:%4d y:%4d z:%4d r:%2d ", data[G_X], data[G_Y], data[G_Z], data[G_W]);
  fprintf(stderr, "f:%s(%2x) ", binary((u_char) data[G_F], buffer), (u_char) data[G_F]);
#endif

  read(fd, &data[G_K], 1);  /* keys */

#ifdef DEBUGGLOVE
  fprintf(stderr, "k:%2x ", (u_char) data[G_K]);
#endif

  /* store the tick time difference to the last sample time */

  tick = times(&tmptime);
  data[G_T] = (char) tick - last;

  /* store current tick count */
  last = tick;

#ifdef DEBUGGLOVE
  fprintf(stderr, "t: %3d", (int) data[G_T]);
#endif

#ifdef DEBUGGLOVE
  fprintf(stderr, "     \r");
#endif

  return 0;
}


int glove_quit(void)
{
  if (fd >= 0)
  {
    fprintf(stderr, "powerglove: closing serial device\n");

    /* reset original serial tty flags */
    ioctl(fd, TCSETA, &tio);

    close(fd);  /* close serial device */
  }

  return 0;
}


#ifdef DEBUGGLOVE
static char *binary(u_char bits, char *buffer)
{
  int k;
  char *buf = buffer;

  for(k=0; k <= 7; k++) *buf++ = ((bits << k) & 0x80) ? '1' : '0';

  return buffer;
}
#endif
