# #
# $Date: 1995/07/12 14:51:14 $  $Author: frankp $   $Revision: 1.96.2.6 $ #

# #
# frankp, 11/10/94 #

#++
  Matrix.mu

	Matrix -- the domains of matrices 

	Matrix([R])

	R - (optional) coefficient domain (a Rng)

	A domain Matrix(R) represents all matrices over the Rng R.
	If the argument 'R' is missing then the coefficient domain 
	will be the domain 'ExpressionField()'. 

	An element of a domain created by Matrix (a matrix), has three
	operands 'r, c, a'. The first two operands give the dimension 
	[r,c] of the matrix and the third operand is a two dimensional 
	array a which holds the entries of the matrix.

	Methods:
	--------
	new, newThis, mkDense, _index, _index, set_index,
	convert, convert_to, dimen, nonZeros, transpose,
	stackMatrix, concatMatrix, setRow, setCol, row, col,
	delRow, delCol, swapRow, swapCol, _mult, _plus, _power,
	negate, iszero, equal, invert, divex, gaussElim, isDense,
	isSparse, TeX, map, subs, subsex, length, expr,
	randomDimen, random, trace, norm, exp
++#

Matrix := DomainConstructor(
# name #
    Matrix,
# arguments #
    [ R ],
# local variables #
    [ RcanonicalRep, Riszero, Rmult, Rplus, Rzero ],
# initialisation #
    (if args(0) > 1 then error("wrong no of args") end_if;
     if args(0) = 0 then R := ExpressionField()
     elif R::hasProp( Rng ) <> TRUE then
	error("coefficients must be from a Rng")
     end_if;
     RcanonicalRep := R::hasProp( canonicalRep );
     Riszero := R::iszero; Rzero := R::zero;
     Rplus := R::_plus; Rmult := R::_mult
    ),
# super-domains #
    BaseDomain,
# categories #
    [ MatrixCat(R) ],
# axioms #
    [ (if R::hasProp( canonicalRep ) then canonicalRep end_if) ],
# #
    "isSparse" = FALSE, "isDense" = TRUE,
#--
  new  --  return a new matrix
--#
    "new" = proc(x)
        local i, j, t, a, r, c, f;
    begin
        if testargs() then
            case args(0)
            of 1 do
		if x::hasProp( MatrixCat ) = TRUE then
                    break
                end_if;
                if domtype(x) = DOM_ARRAY then
                    if op(x,[0,1]) > 2 then
               	        error("array has invalid dimension")
                    end_if
		elif domtype(x) <> DOM_LIST then
		    error("invalid argument")
		end_if;
                break
            of 4 do
		if args(4) <> hold(Diagonal) then
                    error("expecting 'Diagonal' as option")
                end_if
            of 3 do
            of 2 do
		if not testtype( x,Type::PosInt ) or
                not testtype( args(2), Type::PosInt ) then
                    error("expecting dimension as positive integers")
                end_if;
                break
            otherwise
                error("wrong no of args")
            end_case
        end_if;

        case args(0)
	of 1 do
	    t := this::convert( x );
            if t <> FAIL then return( t ) end_if;
            t := x::convert_to( x,this );
            if t = FAIL then
		error("unable to convert the argument") 
	    end_if;
	    return( t )
        of 2 do
	    c := args(2);
            return( new(this,x,c,array(1..x,1..c,[[Rzero $ c] $ x])) )
        of 4 do
        of 3 do
	    t := args(3); c := args(2);
            f := fun(
                    (if domtype(args(1)) = R then args(1)
                     else
                         R::convert( args(1) );
                         if % <> FAIL then %
                         else (args(1))::convert_to( args(1),R )
                         end_if
                     end_if)
            );
            if domtype(t) = DOM_PROC or domtype(t) = DOM_EXEC then
                if args(0) = 4 then
                    a := array(1..x,1..c,[ [Rzero $ c] $ x]);
                    for i from 1 to min(x,c) do
                        a[i,i] := f(t(i,i));
                        if a[i,i] = FAIL then
                            error("invalid function value, unable to convert")
                        end_if
                    end_for;
                    return( new(this,x,c,a) )
                end_if;
                a := array(1..x,1..c);
                for i from 1 to x do
                    for j from 1 to c do
                        a[i,j] := f(t(i,j));
                        if a[i,j] = FAIL then
                            error("invalid function value, unable to convert")
                        end_if
                    end_for
                end_for;
                return( new(this,x,c,a) )
            end_if;

            if domtype(t) <> DOM_LIST then
                t := f(t);
                if t = FAIL then
                    error("invalid 3nd argument, unable to convert")
                end_if;
                if args(0) = 3 then
                    return( new(this,x,c,array(1..x,1..c,[[t$c]$x])) )
                else
		    return( new(this,x,c,array(1..x,1..c,
			    [[Rzero $ i-1, t, Rzero $ c-i] $ i=1..min(x,c),
			     [Rzero $ c] $ x-min(x,c)]))
		    )
		end_if
            end_if;

            if args(0) = 3 then
                t := this::mkDense( x,c,t );
                if t = FAIL then 
		    error("invalid type of list or unable to convert")
		end_if;
		return( new(this,op(t)) )
            end_if;
            # define diagonal matrix from list #
	    a := array( 1..x,1..c,[ [Rzero $ c] $ x ] );
	    for i from 1 to min(x,c,nops(t)) do
	        a[i,i] := f(t[i]);
		if a[i,i] = FAIL then
		    error("invalid entries, unable to convert")
		end_if
	    end_for;
	    return( new( this,x,c,a ) )
        end_case
    end_proc,
#--
  newThis  --  create matrices without testing the arguments
--# 
    "newThis" = proc(r)
	local i, t, a, c, diag;
    begin
	case args(0)
	of 2 do
	    c := args(2);
	    return( new(this,r,c,array(1..r,1..c,[[Rzero$c]$r])) )
	of 1 do
	    if domtype(r) = DOM_ARRAY then
	        return( new(this,op(r,[0,2,2]) - op(r,[0,2,1]) + 1,
                            op(r,[0,3,2]) - op(r,[0,3,1]) + 1,r) )
            elif domtype(r) = DOM_LIST then
                t := r; r := nops(t); c := 1;
                if domtype(t[1]) = DOM_LIST then
                    c := max( op(map(t,nops)) )
                end_if
            elif r::hasProp(MatrixCat) then
                return( new(this,extop(r,1),extop(r,2),extop(r,3)) )
	    else
	        error("invalid argument")
            end_if
	of 4 do
	    if args(0) = 4 then diag := TRUE end_if
	of 3 do
	    if args(0) > 2 then c := args(2); t := args(3) end_if;
	    if nops(t) = 0 then
	        return( new(this,r,c,array(1..r,1..c,[[Rzero$c]$r])) )
	    end_if;
	    if domtype(t) <> DOM_LIST then
		if diag <> TRUE then
		    return( new(this,r,c,array(1..r,1..c,[[t$c]$r])) )
		else
		    return( new(this,r,c,array(1..r,1..c,
			    [[Rzero $ i-1, t, Rzero $ c-i] $ i=1..min(r,c),
			     [Rzero $ c] $ r-min(r,c)]))
		    )
		end_if
	    end_if;
	    a := array( 1..r,1..c,[ [Rzero $ c] $ r ] );
	    if diag <> TRUE and r > 1 and c > 1 then
	    # define matrix by list of list #
		for i from 1 to nops(t) do
		    ( a[i,j] := op(t,[i,j]) ) $ hold(j)=1..nops(t[i])
		end_for
	    elif diag = TRUE then
                ( a[i,i] := t[i] ) $ i=1..min(r,c,nops(t))
	    elif r = 1 then
		if domtype(t[1]) = DOM_LIST then t := t[1] end_if;
		( a[1,i] := t[i] ) $ i=1..nops(t) 
	    else 
		(a[i,1] := t[i] ) $ i=1..nops(t)
	    end_if;
	    new( this,r,c,a )
	end_case
    end_proc,
#--
   mkDense  --  convert an expression to an array 
--#
    "mkDense" = proc(x)
	local i, j, r, c, a, f, t;
    begin
	f := fun(
	    (if domtype(args(1)) = R then
		args(1)
	     else
		R::convert( args(1) );
		if % <> FAIL then
		    %
		else
		    (args(1))::convert_to( args(1),R )
		end_if
	     end_if)
	      );
	case args(0)
	of 1 do
	    if domtype(x) = DOM_ARRAY then
	        a := map( [op(x)],f );
	        if contains( {op(a)},FAIL ) then return( FAIL ) end_if;  
	        r := op(x,[0,2,2]) - op(x,[0,2,1]) + 1; c := 1;
	    	if op(x,[0,1]) = 2 then 
		    c := op(x,[0,3,2]) - op(x,[0,3,1]) + 1 
		end_if;
	    	return( [r,c,
		  array(1..r,1..c,[[a[j] $ j=i*c-c+1..i*c] $ i=1..r])] 
		)
	    elif domtype(x) = DOM_LIST then
	        r := nops(x);
	        if r = 0 then return( FAIL ) end_if;
		if domtype(x[1]) <> DOM_LIST then
		    c := 1;
		    x:= map(x, fun([args(1)]));
		    break
		end_if;
		if nops({op(map(x,domtype))} minus {DOM_LIST}) > 0 then
		    return( FAIL )
		end_if;
		c := max(op(map(x,nops)));
		break
	    else return( FAIL )
	    end_if
	of 3 do
	    r := x; c := args(2); 
	    if not testtype( r,Type::PosInt ) 
	    or not testtype( c,Type::PosInt )
	    then return( FAIL ) 
	    end_if;
	    x := args(3);
	    if nops(x) > r then
		error("number of rows does not match")
	    end_if;
	    t := map({op(x)},domtype);
            if t = {DOM_LIST} then
                if max(op(map(x, nops))) > c then
                    error("number of columns does not match")
                end_if;
                break
            elif contains(t,DOM_LIST) then
                return( FAIL )
            else
                break
            end_if
	otherwise
	    return( FAIL )
	end_case;

	a := array(1..r,1..c,[[ Rzero $ c] $ r]);
	# define matrix by list of list #
	for i from 1 to nops(x) do
	    t := x[i];
	    if domtype(t) <> DOM_LIST then
               t:=[t] # allows to write matrix([a,b,c]) instead 
			of matrix([[a],[b],[c]]) #
            end_if;
	    for j from 1 to nops(t) do
		a[i,j] := f(t[j]);
		if a[i,j] = FAIL then return( FAIL ) end_if
	    end_for
	end_for;
	[r,c,a]
    end_proc,
#--
  _index  --  indexing of matrices 
--#
   "_index" = fun( 
	(if args(0) = 3 then
            _index(extop(args(1),3),args(2),args(3))
         elif extop(args(1),1) = 1 then
            _index(extop(args(1),3),1,args(2))
         else
            _index(extop(args(1),3),args(2),1)
         end_if)
	),
#--
  set_index  --  indexing of matrices 
--#
    "set_index" = fun(
	(if args(0) = 4 then
	    extsubsop( args(1),3=subsop( 
	    extop(args(1),3),(args(2)-1)*extop(args(1),2)+args(3)=args(4)) )
	 elif extop(args(1),1) = 1 then
	    extsubsop( args(1),3=subsop(extop(args(1),3),args(2)=args(3)) ) 
	 else
	    extsubsop( args(1),3=subsop(
		extop(args(1),3),(args(2)-1)*extop(args(1),2)+1=args(3)) ) 
	 end_if)
	),
#--
  print  --  print matrices in form of an array 
--#
    "print" = fun( map(extop(args(1),3),context) ),
#--
   expr -- returns the matrix as an element of DOM_ARRAY, whereby
           each entry was converted to an expression (using R::expr).
--#
    "expr" = fun(map(extop(args(1),3),R::expr)),
#--
   convert -- convert an expression to a matrix 
--#
    "convert" = fun(
	(if args(0) <> 1 then
	    FAIL
         elif domtype(args(1)) = this then 
	    args(1)
         elif domtype(args(1)) = DOM_ARRAY or domtype(args(1)) = DOM_LIST then
            this::mkDense( args(1) );
            if % = FAIL then FAIL else new( this,op(%) ) end_if
	 elif (args(1))::hasProp( MatrixCat ) = TRUE then
	    if (args(1))::coeffRing = R then
		extsubsop( args(1),0=this )
	    elif (args(1))::isSparse then
		FAIL # not implemented yet #
	    else
	        this::mkDense( extop( args(1),3 ) );
	        if % = FAIL then FAIL else new( this,op(%) ) end_if
	    end_if
	 else 
	    FAIL
	 end_if)),
#--
  convert_to  --  convert matrices 
--#
    "convert_to" = proc(e,F)
	local t;
    begin
	if args(0) <> 2 then return( FAIL ) end_if;
	if domtype(e) = F then return(e) end_if;
	if domtype(F) <> DOM_DOMAIN then F := domtype(F) end_if;
        if F = DOM_ARRAY then return( extop( e,3 ) ) end_if;
	if F::hasProp( MatrixCat ) = TRUE then
	    if F::isSparse then
	    # not implemented yet #
		return( FAIL )
	    end_if;
	    if F::hasProp(SquareMatrixCat) then
		if F::dimen <> e::dimen(e) then return( FAIL ) end_if
	    end_if;
	    if F::coeffRing = e::coeffRing then
		return( extsubsop( e,0=F ) )
	    end_if;
	    t := F::mkDense( extop( e,3 ) );
	    if t <> FAIL then 
	        return( new( F,op(t) ) )
	    end_if
	end_if;
	FAIL
    end_proc,
#--
  dimen  --  return the dimension of a matrix 
--#
    "dimen" = fun( [extop(args(1),1),extop(args(1),2)] ),
#--
  _mult  --  multiply matrices

  Syntax:

  _mult(x1 [,x2,x3])

  Synopsis:

  Multiplies two or more matrices.
  If an argument is not a matrix then this argument will be
  converted to an element of R if possible, and scalar multiplication
  is performed. Otherwise FAIL is returned for the multiplication of
  these two arguments.
--#
    "_mult" = (if R::hasProp( systemRep ) then
    proc(x)
	local y, r, c, i, j, cy;
    begin
	if args(0) = 1 then return( args(1) )
	elif args(0) > 2 then
	    r := _mult( args(i) $ i=1..(args(0) div 2) );
            c := _mult( args(i) $ i=((args(0) div 2) + 1)..args(0) );
            return( _mult(r,c) )
	end_if;
	y := args(2);
	if domtype(x) = this then
	    if domtype(y) = DOM_INT then
		if y = 1 then return( x )
		elif y = -1 then return( this::negate(x) )
		elif y = 0 then 
		    r := extop(x,1); c := extop(x,2);
		    return( new(this,r,c,array(1..r,1..c,[[Rzero $ c] $ r])) )
		elif y > 1 then
		    return( this::_plus(x $ y) )
		else
		    return( this::negate(this::_plus(x $ -y)) )
		end_if
	    elif domtype(y) = this then
		r := extop(x,1); c := extop(x,2);
		x := extop(x,3);
		if c <> extop(y,1) then
		    error("dimensions don't match") 
		end_if;
		cy := extop(y,2);
		y := extop(y,3);
		return( new(this,r,cy,array(1..r,1..cy,[
		  [Rplus(Rmult(x[k,i],y[i,j]) $ i=1..c ) $ j=1..cy] 
			$ hold(k)=1..r]
		)) )
	    elif testtype( y,R ) then
		return( 
		  new(this,extop(x,1),extop(x,2),
		      map(extop(x,3),Rmult,R::convert(y))
		  )
		)
	    else
	        return((domtype(y))::_mult(x, y))
	    end_if
	elif domtype(x) = DOM_INT then
	    return( this::_mult(y,x) )
	elif testtype( x,R ) then
	    x := R::convert(x);
	    r := extop(y,1); c := extop(y,2);
	    y := extop(y,3);
	    for i from 1 to r do
		(y[i,j] := Rmult(x,y[i,j])) $ j=1..c
	    end_for;
	    return( new(this,r,c,y) )
	end_if;
	FAIL
    end_proc
else 
    proc(x)
        local r, c, i, j, cy, y;
    begin
        if args(0) = 1 then return( args(1) )
        elif args(0) > 2 then
            r := _mult( args(i) $ i=1..(args(0) div 2) );
            c := _mult( args(i) $ i=((args(0) div 2) + 1)..args(0) );
            return( _mult(r,c) )
        end_if;
	y := args(2);
	case domtype(x)
	of this do
	    case domtype(y)
            of DOM_INT do
                if y = 1 then return( x )
                elif y = -1 then return( this::negate(x) )
                elif y = 0 then
                    r := extop(x,1); c := extop(x,2);
                    return( new(this,r,c,array(1..r,1..c,[[Rzero $ c] $ r])) )
                elif y > 1 then
                    return( this::_plus(x $ y) )
                else
                    return( this::negate(this::_plus(x $ -y)) )
                end_if;
	    of R do
		 return( new(this,extop(x,1),extop(x,2),
		     map( extop(x,3),Rmult,y )
		 ) )
            of this do
                r := extop(x,1); c := extop(x,2);
                x := extop(x,3);
                if c <> extop(y,1) then
                    error("dimensions don't match")
                end_if;
                cy := extop(y,2);
                y := extop(y,3);
                return( new(this,r,cy,array(1..r,1..cy,[
                  [Rplus(Rmult(x[k,i],y[i,j]) $ i=1..c ) $ j=1..cy] $ hold(k)=1..r]
                )) );
	    otherwise
		if testtype(y,R) then
		    return( new(this,extop(x,1),extop(x,2),
                        map( extop(x,3),Rmult,R::convert(y) )
                    ) )
		else
	            return((domtype(y))::_mult(x, y))
		end_if
	    end_case;
	    break
        of DOM_INT do
	    return( this::_mult(y,x) )
        of R do 
            r := extop(y,1); c := extop(y,2);
            y := extop(y,3);
            for i from 1 to r do
                (y[i,j] := Rmult(x,y[i,j])) $ j=1..c
            end_for;
            return( new(this,r,c,y) )
	otherwise
	    if testtype(x,R) then
		x := R::convert(x);
	        r := extop(y,1); c := extop(y,2);
                y := extop(y,3);
                for i from 1 to r do
                    (y[i,j] := Rmult(x,y[i,j])) $ j=1..c
                end_for;
                return( new(this,r,c,y) )
	    end_if
        end_case;
        FAIL
    end_proc
end_if),
#--
   _plus  --  add matrices 
--#
    "_plus" = proc(x)
	local y, r, c, i, j, k;
    begin
	if args(0) = 1 then return( x ) end_if;
	if map({args()},domtype) <> {this} then
	    return( FAIL )
	end_if;
        r := extop(x,1); c := extop(x,2);
        x := extop(x,3);
        for i from 2 to args(0) do
            y := args(i);
            if [extop(y,1),extop(y,2)] <> [r,c] then
                error ("dimensions don't match")
            end_if;
            x := array(1..r,1..c,
                [[zip([op(x)],[op(extop(y,3))],Rplus)[j] $ j=k*c-c+1..k*c]
                    $ k=1..r]
            )
        end_for;
        new(this,r,c,x)
    end_proc,
#--
  _power  --  multiply a matrix with itselfes 

  Syntax:

  _power(A,i)

  A  -- matrix
  i  -- integer

  Synopsis:

  Multiplies A i times. There are three cases:
  * i > 0: The product A*A*...*A (i times) is computed.
  * i = 0: The identity matrix is returned if R has the unit 1.
  * i < 0: The product 1/A*...*1/A (-i times) is computed if
           A is regular.

  Returns FAIL if i is not an integer or in the case of
  negative i, if A is singular.
  If i = 0 and R has not the unit 1 then FAIL is returned too.
--#
    "_power" = proc(x,i)
	local n;
    begin
	if extop(x,1) <> extop(x,2) or domtype(i) <> DOM_INT then
	    return( FAIL )
	end_if;

	if i = 0 then
	    if R::one = FAIL then return( FAIL ) end_if;
	    n := extop( x,1 );
	    new( this,n,n,
	      array(1..n,1..n,[[Rzero $ j-1,R::one,Rzero $ n-j]
		$ hold(j)=1..n])
	    )
	elif i = 1 then x
	elif i > 1 then this::_mult( x$i )
	elif i = -1 then this::invert( x )
	else # i < -1 #
	    x := this::invert( x );
	    if x = FAIL then FAIL
	    else this::_mult( x$(-i) )
	    end_if
	end_if
    end_proc,
#--
  negate  --  negate a matrix 
--#
    "negate" =
	fun( extsubsop( args(1),3=map(extop( args(1),3 ),R::negate) )
    ),
#--
  invert  --  compute the invert of a matrix 

  Syntax:

  invert(A)

  A  --  matrix

  Synopsis:

  Returns A^(-1) if A is regular, FAIL otherwise.
--#
    "invert" = (if R::hasProp( IntegralDomain ) then 
    # in MuPAD an IntegralDomain has the unit element #
    proc(x)
	local n, i, j, t, k, Rdivex, Rnegate;
    begin
	if extop( x,1 ) <> extop( x,2 ) then return( FAIL ) end_if;

    # append the n x n identity matrix to x #
	n := extop( x,1 ); x := extop( x,3 );
	t := array( 1..n,1..2*n,
             [[ x[i,j]$j=1..n,Rzero$i-1,R::one,Rzero$n-i ] $ i=1..n]
	);

    # rank(x) < n ? #
	x := extop( op(this::gaussElim( new(this,n,2*n,t) ),1),3 );
	t := { iszero( x[n,i] ) $ i=1..n };
	if t = {TRUE} then return( FAIL ) end_if;

	Rdivex := R::divex; Rnegate := R::negate;

    # transform the submatrix x(1..n,1..n) to identity matrix 
      by backsolving
    #
	for k from n downto 1 do
	    t := x[k,k];
	    for j from n+1 to 2*n do
		x[k,j] := Rdivex( x[k,j],t );
		if x[k,j] = FAIL then return( FAIL ) end_if
	    end_for;
	    for i from k-1 downto 1 do
		t := x[i,k];
		for j from n+1 to 2*n do
		    if not Riszero( x[k,j] ) then
			x[i,j] := Rplus(x[i,j],Rnegate(Rmult(x[k,j],t)))
		    end_if
		end_for
	    end_for
	end_for;

	new( this,n,n,
	    array(1..n,1..n,[[x[i,j] $ hold(j)=n+1..2*n] $ hold(i)=1..n])
	)
    end_proc
    end_if ),
#--
  divex  --  compute the product A*1/A of a matrix A 
--#
    "divex" = ( if R::hasProp( IntegralDomain ) then
        fun( (this::invert( args(2) );
	  if % <> FAIL then
	      this::_mult( args(1),% )
	  else
	      FAIL
	  end_if)
	)
    end_if ),
#--
  iszero  --  test if a matrix is the zero matrix 
--#
    "iszero" = fun(
	(map({op(extop(args(1),3))},Riszero);	
	 if nops(%) > 1 then	
	     FALSE 
	 else 
	     op(%,1) 
	 end_if) 
	       ),
#--
  equal  --  test if two matrices are equal 
--#
    "equal" = proc(x,y)
	local r, c, i, j, t;
    begin
	r := extop(x,1); c := extop(x,2);
	if [r,c] <> [extop(y,1),extop(y,2)] then
	    return( FALSE )
	end_if;
	x := extop(x,3); 
	y := extop(y,3);
	t := R::equal;
	for i from 1 to r do
	    for j from 1 to c do
		if not t(x[i,j],y[i,j]) then return( FALSE ) end_if
	    end_for
	end_for;
	return( TRUE )
    end_proc,
#--
  gaussElim  --  Gaussian elimination

  Syntax:

  gaussElim(A)

  A  --  matrix

  Synopsis:

  Performs a two-step fraction free gaussian elimination on A and 
  returns a list with a row echolon form of A, the rank of A
  and the determinant of A if this is defined. Otherwise FAIL
  will be the third operand of the list.

  If R has a method "pivotSize" then the pivot element of
  smallest size will be choosen, whereby pivotSize must return
  a natural number representing 'the size' of an element.
--#
    "gaussElim" = (if R::hasProp( IntegralDomain ) then 
    proc(x)
	option remember;
	local n, m, i, j, l, f, f0, f1, f2, r, k, sig, i0, l0,
	      Rnegate, Rdivex, ps;
    begin
        Rnegate := R::negate; Rdivex := R::divex;
	ps := R::pivotSize; 
	# if ps = FAIL then no pivot strategy will be used, otherwise
	  choose the pivot of smallest size
	#

	m := extop(x,1); n := extop(x,2); 
	x := extop(x,3);
	sig := 1;
	f0 := R::one;
	k := 2; r := 2;
	for k from 2 to 2*(n div 2) step 2 do
	    if r > m then break end_if;
	    for i from r-1 to m do
		for j from i+1 to m do 
		    f := Rplus( Rmult( x[i,k-1],x[j,k] ),
                                Rnegate(Rmult( x[i,k],x[j,k-1] )) );
		    if not Riszero(f) then break end_if
		end_for;
		if j <= m then break end_if
	    end_for;
	    if i > m then
		userinfo(2,"  no non-zero determinant found in step ",k);
		break
	    end_if;
	    if i <> r-1 then 
		userinfo(1,"swap rows ",r-1," and ",i);
		for l from 1 to n do
		    f2 := x[r-1,l]; x[r-1,l] := x[i,l]; x[i,l] := f2
		end_for;
		sig := -sig
	    end_if;
	    if j <> r then
		userinfo(1,"swap rows ",j," and ",r);
		for l from 1 to n do
		    f2 := x[r,l]; x[r,l] := x[j,l]; x[j,l] := f2
		end_for;
		sig := -sig
	    end_if;
	    userinfo(1,"searching for pivot element");
            i0 := m+1; l0 := -1;
	    for i from r-1 to m do
		if not Riszero(x[i,k-1]) then
		    if ps = FAIL then i0 := i; break
		    elif l0 = -1 then l0 := ps(x[i,k-1]); i0 := i
                    elif (l:=ps(x[i,k-1])) < l0 then l0 := l; i0 := i
		    end_if
                end_if
	    end_for;
            i := i0;
	    if i <= m then 
		userinfo(2,"  found ",x[i,k-1]);
		if r-1 <> i then
		    userinfo(1,"swap rows ",r-1," and ",i);
		    for l from 1 to n do
			f2 := x[r-1,l]; x[r-1,l] := x[i,l]; x[i,l] := f2
		    end_for;
		    sig := -sig;
		    f := Rplus( Rmult( x[r-1,k-1],x[r,k] ),
                                Rnegate(Rmult( x[r-1,k],x[r,k-1] )) )
		end_if;
		f := Rdivex(f,f0);
		for i from r+1 to m do
		    f1 := Rdivex( Rplus( Rmult( x[r-1,k],x[i,k-1] ),
			  Rnegate(Rmult( x[r-1,k-1],x[i,k] )) ),f0 );
		    f2 := Rdivex( Rplus( Rmult( x[r,k-1],x[i,k] ),
			  Rnegate(Rmult( x[r,k],x[i,k-1] )) ),f0 );
		    for j from k+1 to n do
			x[i,j] := Rdivex(
				Rplus(Rmult(f,x[i,j]),Rmult(f1,x[r,j]),
                                Rmult(f2,x[r-1,j]) ),f0
	                );
		    end_for;
		    x[i,k] := Rzero; x[i,k-1] := Rzero
		end_for;
		for j from k+1 to n do
		    x[r,j] := Rdivex( Rplus( Rmult( x[r-1,k-1], x[r,j] ),
			       Rnegate(Rmult( x[r-1,j],x[r,k-1] )) ),f0 )
		end_for;
		x[r,k-1] := Rzero; x[r,k] := f; f0 := f; 
		r := r + 2
	    end_if
	end_for;
	r := r - 2;
	if n mod 2 = 1 or r <= n then
	    userinfo(1,"perform single-step algorithm");
	    f1 := k - 1; 
	    r := r + 1;
	    for k from f1 to n do
		if r > m then break end_if;
		userinfo(1,"searching for pivot element");
                i0 := m + 1; l0 := -1;
                for i from r to m do
                    if not Riszero(x[i,k]) then
			if ps = FAIL then i0 := i; break
			elif l0 = -1 then l0 := ps(x[i,k]); i0 := i
			elif (l:=ps(x[i,k])) < l0 then l0:=l; i0:=i 
			end_if
                    end_if
                end_for;
                i := i0;
		if i <= m then
		    userinfo(2,"  found ",x[i,k]);
		    if r <> i then 
			userinfo(1,"swap rows ",r," and ",i);
			for l from 1 to n do
			    f2 := x[r,l]; x[r,l] := x[i,l]; x[i,l] := f2
			end_for;
			sig := -sig
		    end_if;
		    for i from r+1 to m do
			for j from k+1 to n do
			    x[i,j] := Rdivex( Rplus( Rmult( x[r,k],x[i,j] ),
				      Rnegate(Rmult( x[r,j],x[i,k] )) ),f0 )
			end_for;
			x[i,k] := Rzero
		    end_for;
		    f := x[r,k]; 
		    r := r + 1
		end_if
	    end_for;
	    r := r - 1
	end_if;
	if n = m then
	    if r = m then if sig = -1 then f := Rnegate(f) end_if
	    else f := Rzero
	    end_if
	else f := FAIL
	end_if;

	[new( this,m,n,x ),r,f]
    end_proc
    end_if ),
#--
  transpose  --  compute the transpose of a matrix 
--#
    "transpose" = proc(x)
	local r, c, i, t;
    begin
	r := extop( x,1 ); c := extop( x,2 ); x := extop( x,3 );
	t := array( 1..c,1..r );
	for i from 1 to r do
	    ( t[j,i] := x[i,j] ) $ hold(j)=1..c
	end_for;

	new( this,c,r,t )
    end_proc,
#--
  nonZeros  --  return the number of non-zero elements of a matrix 
--#
    "nonZeros" = fun(
	nops( select([op(extop(args(1),3))],_not@iszero) )
    ),
#--
  stackMatrix  --  concat two matrices vertically 
--#
    "stackMatrix" = proc(a,b)
	local rb, r, c, t, j;
    begin
	if testargs() then
	    if args(0) <> 2 then
	 	error("wrong no of args")
	    end_if;
	    if a::constructor <> Matrix and a::constructor <> SquareMatrix
	    or b::constructor <> Matrix and b::constructor <> SquareMatrix
	    then
	        error("expecting matrices either of type Matrix or SquareMatrix")
	    end_if;
            if a::coeffRing <> R or b::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if extop( a,2 ) <> extop( b,2 ) then
		error("dimensions don't match")
	    end_if
	end_if;

	r := extop( a,1 ); c := extop( a,2 ); a := extop( a,3 );
	rb := extop( b,1 ); b := extop( b,3 );
	t := array(1..r+rb,1..c);

	for j from 1 to c do
            ( t[i,j] := a[i,j] ) $ hold(i)=1..r;
            ( t[r+i,j] := b[i,j] ) $ hold(i)=1..rb
	end_for;

	new( this,r+rb,c,t )
    end_proc,
#--
  concatMatrix  --  concat two matrices horizontally 
--#
    "concatMatrix" = proc(a,b)
	local cb, r, c, t, i;
    begin
	if testargs() then
	    if args(0) <> 2 then
	 	error("wrong no of args")
	    end_if;
            if a::constructor <> Matrix and a::constructor <> SquareMatrix
            or b::constructor <> Matrix and b::constructor <> SquareMatrix
            then
                error("expecting matrices either of type Matrix or SquareMatrix")
            end_if;
            if a::coeffRing <> R or b::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if extop( b,1 ) <> extop( a,1 ) then
		error("dimensions don't match")
	    end_if
	end_if;

	r := extop( a,1 ); c := extop( a,2 ); a := extop( a,3 );
	cb := extop( b,2 ); b := extop( b,3 );
	t := array(1..r,1..c+cb);

	for i from 1 to r do
            ( t[i,j] := a[i,j] ) $ hold(j)=1..c;
            ( t[i,c+j] := b[i,j] ) $ hold(j)=1..cb
	end_for;

	new( this,r,c+cb,t )
    end_proc,
#--
  setRow  --  replaces a row of a matrix by a new one 
--#
    "setRow" = proc(x,p,row)
	local a, c;
    begin
	if testargs() then
	    if args(0) <> 3 then
		error("wrong no of args")
	    end_if;
            if x::constructor <> Matrix and x::constructor <> SquareMatrix
            or row::constructor <> Matrix
            then
		error("expecting matrices either of type Matrix or SquareMatrix")
            end_if;
            if x::coeffRing <> R or row::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if extop( row,1 ) <> 1 or extop( row,2 ) <> extop( x,2 ) then
		error("dimensions don't match")
	    end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,1 ) then error("index out of range") end_if
	end_if;

	c := extop( x,2 ); a := extop( x,3 );
	row := extop( row,3 );
	( a[p,j] := row[1,j] ) $ hold(j)=1..c;

	extsubsop( x,3=a )
end_proc,
#--
  setCol  --  replaces a column of a matrix by a new one 
--#
    "setCol" = proc(x,p,col)
	local a, r;
    begin
	if testargs() then
	    if args(0) <> 3 then
		error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    or col::constructor <> Matrix
	    then
	       error("expecting matrices either of type Matrix or SquareMatrix")
	    end_if;
            if x::coeffRing <> R or col::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if extop( col,2 ) <> 1 or extop( col,1 ) <> extop( x,1 ) then
		error("dimensions don't match")
	    end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,2 ) then error("index out of range") end_if
	end_if;

	r := extop( x,1 ); a := extop( x,3 );
	col := extop( col,3 );
	( a[i,p] := col[i,1] ) $ hold(i)=1..r;

	extsubsop( x,3=a )
    end_proc,
#--
  row  --  extract a row of a matrix 
--#
    "row" = proc(x,p)
	local c;
    begin
	if testargs() then
	    if args(0) <> 2 then
	        error("wrong no of args")
	    end_if;
 	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    then
	        error("first argument must be either of type Matrix or SquareMatrix")
	    end_if;
            if x::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,1 ) then
		error("index out of range")
	    end_if
	end_if;

	c := extop( x,2 ); x := extop( x,3 );
	new( this,1,c,array( 1..1,1..c,[[x[p,j] $ hold(j)=1..c]] ) )
    end_proc,
#--
  col  --  extract a column of a matrix 
--#
    "col" = proc(x,p)
	local r;
    begin
	if testargs() then
	    if args(0) <> 2 then
	        error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    then
                error("first argument must be either of type Matrix or SquareMatrix")
            end_if;
	    if x::coeffRing <> R then
		error("expecting matrices over ".expr2text(R))
	    end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,2 ) then
		error("index out of range")
	    end_if
	end_if;

	r := extop( x,1 ); x := extop( x,3 );
	new( this,r,1,array( 1..r,1..1,[ [x[i,p]] $ hold(i)=1..r ] ) )
    end_proc,
#--
  delRow  --  delete a row of a matrix 
--#
    "delRow" = proc(x,p)
	local r, c, t, j;
    begin
	if testargs() then
	    if args(0) <> 2 then
	        error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    then
                error("first argument must be either of type Matrix or SquareMatrix")
            end_if;
            if x::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,1 ) then
		error("index out of range")
	    end_if
	end_if;

	r := extop( x,1 ); c := extop( x,2 );
	if r = 1 then return( NIL ) end_if;
	x := extop( x,3 );
	t := array( 1..r-1,1..c );
	( ( t[i,j] := x[i,j] ) $ j=1..c ) $ hold(i)=1..p-1;
	( ( t[i-1,j] := x[i,j] ) $ j=1..c ) $ hold(i)=p+1..r;

	new( this,r-1,c,t )
    end_proc,
#--
  delCol  --  delete a column of a matrix 
--#
    "delCol" = proc(x,p)
	local r, c, t, i;
    begin
	if testargs() then
	    if args(0) <> 2 then
	        error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    then
                error("first argument must be either of type Matrix or SquareMatrix")
            end_if;
            if x::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if not testtype( p,Type::PosInt ) then
		error("index must be a positive integer")
	    end_if;
	    if p > extop( x,2 ) then
		error("index out of range")
	    end_if
	end_if;

	r := extop( x,1 ); c := extop( x,2 );
	if c = 1 then return( NIL ) end_if;
	x := extop( x,3 );
	t := array( 1..r,1..c-1 );
	( ( t[i,j] := x[i,j] ) $ i=1..r ) $ hold(j)=1..p-1;
	( ( t[i,j-1] := x[i,j] ) $ i=1..r ) $ hold(j)=p+1..c;

	new( this,r,c-1,t )
    end_proc,
#--
  swapRow  --  swap two rows of a matrix 
--#
    "swapRow" = proc(x,k,l)
	local a, d;
    begin
	if testargs() then 
	    if args(0) <> 3 then
   	        error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix
	    and x::constructor <> SquareMatrix then
                error("first argument must be either of type Matrix or SquareMatrix")
            end_if;
            if x::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if not testtype( k,Type::PosInt )
	    or not testtype( l,Type::PosInt )
	    then
		error("indices must be positive integers")
	    end_if;
	    d := extop( x,1 );
	    if k > d or l > d then
		error("indices out of range")
	    end_if
	end_if;

	if k = l then return( x ) end_if;
	a := extop( x,3 );
	( d := a[l,j]; a[l,j] := a[k,j]; a[k,j] := d ) $ hold(j)=1..extop(x,2);

	extsubsop( x,3=a )
    end_proc,
#--
  swapCol  --  swap two columns of a matrix 
--#
    "swapCol" = proc(x,k,l)
	local a, d;
    begin
	if testargs() then
	    if args(0) <> 3 then
	        error("wrong no of args")
	    end_if;
	    if x::constructor <> Matrix and x::constructor <> SquareMatrix
	    then
                error("first argument must be either of type Matrix or SquareMatrix")
            end_if;
            if x::coeffRing <> R then
                error("expecting matrices over ".expr2text(R))
            end_if;
	    if not testtype( k,Type::PosInt )
	    or not testtype( l,Type::PosInt )
	    then
		error("indices must be positive integers")
	    end_if;
	    d := extop( x,2 );
	    if k > d or l > d then
		error("indices out of range")
	    end_if
	end_if;

	if k = l then return( x ) end_if;
	a := extop( x,3 );
	( d := a[i,l]; a[i,l] := a[i,k]; a[i,k] := d ) $ hold(i)=1..extop(x,1);

	extsubsop( x,3=a )
    end_proc,
#--
  TeX  --  'TeX' for matrices 
--#
    "TeX" = fun( TeX(extop(args(1),3)) ),
#--
  map  --  'map' for matrices 
--#
    "map" = fun( 
	extsubsop(args(1),3=map(extop(args(1),3),args(i) $ hold(i)=2..args(0)))
	    ),
#--
  subs  --  'subs' for matrices 
--#
    "subs" = fun( extsubsop(args(1),3=
	subs(extop(args(1),3),args(i) $ hold(i)=2..args(0))
	     ) ),
#--
  subsex  --  'subsex' for matrices 
--#
    "subsex" = fun( extsubsop(args(1),3=
	subsex(extop(args(1),3),args(i) $ hold(i)=2..args(0))
	     ) ),
#--
  length  --  returns the lenght of a matrix 

  The length of a matrix is defined by the length of its
  3rd operand, i.e. the length of an array.
--#
    "length" = fun( length(extop(args(1),3)) ),
#--
  conjugate  --  compute the complex conjugate of a matrix 
--#
    "conjugate" = (if R::conjugate <> FAIL then 
	fun( extsubsop(args(1),3=map(extop(args(1),3),R::conjugate)) )
    end_if),
#--
  randomDimen 
--#
    "randomDimen" = [10,10],
#--
  random  --  create random matrices

  Syntax:

  random()

  Synopsis:

  Returns a random matrix whereby the dimension is randomly choosen
  in the interval [1..this::randomDimen[1],1..this::randomDimen[2]]
  The elements of the random matrix are created by the method
  "random" of R. If such a method does not exist, then FAIL is 
  returned.
--#
    "random" = (if R::random <> FAIL then 
	proc()
	    local m,n,j;
	begin
	    m := (random() mod this::randomDimen[1]) + 1;
	    n := (random() mod this::randomDimen[2]) + 1;
 	    new( this,m,n,array( 1..m,1..n,
		[[R::random() $ j=1..n] $ hold(i)=1..m]
	    ))
	end_proc
    end_if),
#--
  trace  --  compute the trace of a square matrix

  Syntax:

  trace(A)

  A  --  matrix

  Synopsis:

  A trace of a square matrix A is defined to be the sum of the
  diagonal elements of A.
--#
    "trace" = proc(x)
	local n;
    begin
	n := extop(x,1);
        if n <> extop(x,2) then
            error("expecting a square matrix")
        end_if;
	x := extop(x,3);
	Rplus( Rzero,x[i,i] $ hold(i)=1..n )
    end_proc,
#--
  norm()  --  compute the norm of a matrix

  Syntax: 

  norm(A)
  norm(A,Frobenius)
  norm(v,k)

  A          --  matrix
  v          --  vector
  k          --  positive integer
  Frobenius  --  identifier

  Synopsis:

  norm(A) computes the infinity norm of a matrix.
  That is the maximum row sum, where the row sum is the sum of norms 
  of each element.
  The infinity norm of a vector is the maximum norm of all
  elements.

  norm(A,1) computes the 1-norm of a matrix which is the maximum sum 
  of the norm of the elements of each column.

  norm(A,Frobenius) computes the Frobenius norm of a matrix which is 
  the square root of the sum of the squares of the norm of each element.

  norm(v,k) computes the k-norm of a vector for k >= 2.
  This norm is defined to be the kth root of the sum of the norm of each 
  element raised to the kth power.
  For the 2-norm the function 'linalg::scalarProduct' will be used
  if she is defined. Otherwise the Euclidian scalar product of v will
  be returned, which is defined by

	v[1]*conjugate(v[1]) + v[2]*conjugate(v[2]) + ...

  If the coeffRing of v does not provide a method "conjugate"
  (which must return the complex conjugate of an element) then 
  the sum

            v[1]*v[1] + v[2]*v[2] + ...

  will be returned.


  This method returns FAIL if the result is not longer an element
  of the coefficient ring of A and v, respectively.

  For all cases except of the 2-norm, the coefficient ring of the
  matrix has to provide the method 'norm', which result for any element 
  has to be a number. FAIL will be returned if such a method not exists.
--#
    "norm" = proc(x)
	local i, j, c, r, n, Rnorm, nt;
    begin
        if args(0) < 1 or args(0) > 2 then
            error("wrong no of args")
        end_if;

	if nt <> 2 then
	    Rnorm := R::norm;
	    if Rnorm = FAIL then return( FAIL ) end_if
	end_if;

	r := extop(x,1); c := extop(x,2);
	if args(0) = 2 then nt := args(2) end_if;
	case nt
	of 1 do # compute the 1-norm #
	    x := this::transpose(x);
            n := c; c := r; r := n

	of hold(nt) do # compute the infinity norm #
	    x := extop(x,3);
	    n := max(_plus(Rzero,Rnorm(x[i,j]) $ j=1..c) $ i=1..r);
	    break

	of hold(Frobenius) do # compute the Frobenius norm #
	    x := extop(x,3);
            n := Rzero;
            (n := _plus(n,_mult(Rnorm(x[i,j]) $ 2) $ j=1..c)) $ i=1..r;
            n := n^(1/2);
            break

	of 2 do # compute the 2-norm of vector x #
	    if min(r,c) <> 1 then
                error("invalid matrix norm (expecting a vector)")
            end_if;

	    nt := linalg::scalarProduct;
	    if nt <> FAIL then
	        n := linalg::scalarProduct(x,x)^(1/2)
	    else
		x := extop(x,3);
		nt := R::conjugate;
		if nt <> FAIL then
		    if r = 1 then
        	        n := Rplus( Rmult(x[1,i],nt(x[1,i])) $ i=1..c )
		    else
			n := Rplus( Rmult(x[i,1],nt(x[i,1])) $ i=1..r )
		    end_if
    		elif r = 1 then
		    n := Rplus( Rmult(x[1,i],x[1,i]) $ i=1..c )
		else # c = 1 #
		    n := Rplus( Rmult(x[i,1],x[i,1]) $ i=1..r )
    		end_if;
		n := n^(1/2)
	    end_if;
	    break

	otherwise # compute the k-norm of vector x #
	    if not testtype( nt,Type::PosInt ) then
		error("invalid matrix norm")
	    end_if;
	    if r = 1 then
		x := extop(x,3);
	        n := _plus( Rnorm(x[1,i])^nt $ i=1..c )^(1/nt)
	    elif c = 1 then
		x := extop(x,3);
		n := _plus( Rnorm(x[i,1])^nt $ i=1..r )^(1/nt)
	    else
		error("invalid matrix norm (expecting a vector)")
            end_if
	end_case;

	if not testtype( n,R ) then return( FAIL ) end_if;
        r := R::convert(n);
        if r = FAIL then n::convert_to(n,R) else r end_if
    end_proc,
#--
	exp  --  compute the exponential of a matrix

	Syntax:

        exp(A [,x])

        A:  square matrix
        x:  undeterminate

	Synopsis:

        exp(A,x) computes the matrix exponential defined by

            exp(A,x) = I + A*x + 1/2!*A^2*x^2 + ...

        If the second argument is not given then 1 is set for x.
        When the eigenvalues of A can not be computed then an 
	error message will be returned.

        When the sum of the algebraic multiplicities of the
        eigenvalues of A is not equal to the dimension of A
        then FAIL will be returned.

	NOTE: This method needs the linalg functions
	      linalg::eigenValues and linalg::linearSolve.
--#
    "exp" = (if R::hasProp( Field ) then proc(A)
	local x, i, j, l, P, iP, st, r, J, t, linsolve, ev, n;
    begin
	loadlib("linalg");

	if args(0) < 1 or args(0) > 2 then
	    error("wrong no of args")
	end_if;
	if extop(A,1) <> extop(A,2) then
	    error("expecting a square matrix")
	end_if;
	if args(0) = 2 then
	    if poly(args(2)) = FAIL then
		error("invalid indeterminate")
	    end_if
	end_if;

	userinfo(2,"compute the eigenvalues");
	ev := linalg::eigenVectors(A);
	if ev = FAIL then return( FAIL ) end_if;

	linsolve := linalg::linearSolve;
	J := this::negate(A);

	userinfo(1,"compute Jordan transformation matrix");
	n := extop(A,1);
	P := 0;
	for i from 1 to nops(ev) do
	    ns := op(ev,[i,3]); # list of EV's #
	    r := nops(ns);
            l := 1;
	    if r < op(ev,[i,2]) then
	        iP := this::_plus(this::newThis(n,n,op(ev,[i,1]),Diagonal),J);
                while r < op(ev,[i,2]) do
		    userinfo(2,"solve the jordan chain for ",op(ev,[i,1]));
                    ns := append(ns,linsolve(iP,ns[l],hold(Special)));
                    r := r + 1;
                    l := l + 1
                end_while
	    end_if;

            if P = 0 then
                P := ns[1];
                (P := this::concatMatrix( P,ns[j] )) $ j=2..r
            else
                (P := this::concatMatrix( P,ns[j] )) $ j=1..r
            end_if
        end_for;
	if A::constructor = SquareMatrix then
	    P := this::convert(P)
	end_if;

        iP := this::invert(P);
        J := this::_mult( iP,A,P );

        if args(0) = 2 then x := args(2)
        else x := R::one
        end_if;

        ns := 0;
        for l from 1 to nops(ev) do
            r := op(ev,[l,2]);
            for i from ns+1 to ns+r do
                J[i,i] := R::convert(exp(x*op(ev,[l,1])));
		if J[i,i] = FAIL then return( FAIL ) end_if;
                if i < n then
                    if J[i,i+1] <> Rzero then
			t := R::convert(x^j/fact(j));
			if t = FAIL then return( FAIL ) end_if;
                        (J[i,i+j] := Rmult(J[i,i],J[i,i+1],t)) $ j=1..r-1
                    end_if
                end_if
            end_for;
            ns := ns + r
        end_for;

        this::_mult( P,J,iP )
    end_proc end_if)
):

# end of file #
