#
--   Title:    funcattr(solve,"float")
--   Created:  Tue Feb 02 1993
--    Author:   Xavier Gourdon
--      <gourdon@margaux.inria.fr>
--     translated to MuPAD by Paul Zimmermann, Wed Apr 12 1995
--
--  Description: approximate root finding of complex polynomials
--  Based on an algorithm of Schonhage and "Algorithmique du theoreme
--  fondamental de l'algebre" (Rapport de Recherche INRIA n 1852)
#


_fsolve_floatln:=funcattr(ln,"float"):
log10 := proc(x) local DIGITS; 
begin 
   DIGITS:=10; _fsolve_floatln(x)/2.302585093
end_proc:
# alias(ln(x)=_fsolve_floatln(x)): #
# LN:=proc(x) begin print(DIGITS,x); ln(x) end_proc: alias(ln(x)=LN(x)): #
alias(Re(x)=op(x,1)):
alias(Im(x)=(if type(x)=DOM_COMPLEX then op(x,2) else 0 end_if)):
ABS:=stdlib::abs:
floatln:=proc(x,n) local DIGITS; begin DIGITS:=n; _fsolve_floatln(x) end_proc:
alias(ln2=0.69314): # ln(2) #

# coeff(p,i) is faster than coeff(p,x,i) #
# poly(e,[x]) is faster than poly(e) #
# evalp(p,x=v) is faster than p(v) #

# main procedure which tests if the polynom has rational coefficients.
  If yes, tries squarefree decomposition #
_fsolve_Sqrfpoly := proc(p,precision,X) # type(p)=DOM_POLY #
local q,i,j,sqrf,fac,result,racine,Rep,Imp;
begin
    Rep:=mapcoeffs(p,Re);
    if not testtype(Rep,Type::PolyExpr(X,Type::Rational)) then
	return(op(_fsolve_Teste(p,precision,X)))
    end_if;
    userinfo(2,"real part has rational coefficients");
    q:=p;
    Imp:=mapcoeffs(p,Im);
    if not iszero(Imp) then
        userinfo(2,"non zero imaginary part");
	if not testtype(Imp,Type::PolyExpr(X,Type::Rational)) 
	    then return(op(_fsolve_Teste(p,precision,X)))
	end_if;
	# q:=subs(p,I=RootOf(_Z^2+1)); #
   else userinfo(2,"real coefficients")
   end_if;
    loadlib("faclib");
    sqrf:=faclib::psqrfree(q,1);
    # sqrf:=subs(sqrf,RootOf(_Z^2+1)=I); #
    result:=null();
    for i from 1 to nops(sqrf)/2 do
       racine:=_fsolve_Teste(sqrf[2*i-1],precision,X);
       result:=result,(op(racine)$hold(j)=1..sqrf[2*i]);
    end_for;
    result
end_proc:

_fsolve_Degre2 := proc(p,precision,X)
local a,b,c,delta,DIGITS;
begin
	DIGITS:=precision;
	a:=coeff(p,2);
	b:=coeff(p,1);
         c:=coeff(p,0);
	delta:=expand(b^2-4*a*c);
	delta:=float(sqrt(delta));
	(-b-delta)/(2*a),(-b+delta)/(2*a)
end_proc:

_fsolve_Cauchy := proc(p,n,X)
local cd,R,i,Rm,RM,q,DIGITS;
begin
    DIGITS:=4;
    q:=p;
    cd:=coeff(p,n);
    RM:=2*max((-coeff(p,i)/cd)^(1/(n-i))$hold(i)=0..n-1);
    Rm:=RM/(2*n);
    for i from 1 to 7 do
	R:=(RM+Rm)/2;
	if q(R)<0 then Rm:=R else RM:=R end_if;
    end_for;
    (RM+Rm)/2
end_proc:

_fsolve_Newtonbis :=proc(p,n,precision,X)
local R,RU,cd,prec,x,y,z0,y0,x0,yy0,discr,u,v,q,qd,qdd,norme,
	  coef,inf,i,j,N,k,n0,n1,uu,vv,z,z1,DIGITS;
begin
    userinfo(2,"deg=",degree(p),"prec=",precision);
    DIGITS:=3; norme:=norm(p,1);
    cd:=ABS(coeff(p,n));
    prec:=precision+trunc(float(log10(norme/cd)+n*log10(2)))+3;
    DIGITS:=prec;
    qd:=diff(p,X);
    DIGITS:=4;
    q:=_plus(ABS(coeff(p,i))*X^i$hold(i)=0..n-1);
    R:=_fsolve_Cauchy(poly(cd*X^n-q,[X]),n,X);
    q:=p;
    R:=R/2; 
    N:=n-(n mod 2);
    RU:=_fsolve_RacineUnite(N,trunc(n/4)+2);
    DIGITS:=trunc(n/4)+2;
    for i from 0 to n do coef[i]:=coeff(p,i)*R^i end_for;
    k:=0; inf:=max(1,R^n)*cd;
    n0:=trunc(n/2); n1:=trunc((n-1)/2);
    for i from 0 to N/2-1 do
	x:=_plus(coef[2*j]*RU[(2*i*j) mod N] $ hold(j)=0..n0);
	y:=_plus(coef[2*j+1]*RU[(i*(2*j+1)) mod N] $ hold(j)=0..n1);
	u:=expand(x+y); v:=expand(x-y);
	DIGITS:=4;
	if ABS(Re(u))+ABS(Im(u))<inf then k:=i;
	    inf:=ABS(Re(u))+ABS(Im(u)) end_if;
	if ABS(Re(v))+ABS(Im(v))<inf then k:=i+N/2; 
	    inf:=ABS(Re(v))+ABS(Im(v)) end_if;
	DIGITS:=trunc(n/4)+2;
    end_for;
    z0:=R*RU[k];
    DIGITS:=trunc(n/4)+5;
    x0:=q(z0);
    DIGITS:=4;
    inf:=float(ABS(Re(x0))+ABS(Im(x0)));
    R:=R*cos(1.0)+I*R*sin(1.0);
    for i from 2 to 3 do 
	z:=ABS(q(R*i/4));
	if z<inf then 
	    inf:=z; z0:=R*i/4 end_if;
    end_for;
    DIGITS:=3;
    for i from 1 to 40 do
        if not ((inf>norme*10.0^(-precision-3)) and DIGITS<2*prec) then break end_if;
	DIGITS:=trunc(float(-2*log10(min(1,inf/cd))+n/2))+3;
    	y0:=qd(z0);
      	if expr2text(y0)="0.0" then return(FAIL) end_if;
	u:=Re(y0); v:=Im(y0);
	z1:=z0-expand(x0*(u-I*v)/(u^2+v^2));
	x0:=q(z1);
	for j from 1 to 6 do
             if float(ABS(Re(x0))+ABS(Im(x0)))<=inf/1.1 then break end_if;
	    DIGITS:=DIGITS+2;
	    z1:=(z1+z0)/2;
	    x0:=q(z1);
	end_for;
	if j>=6 then # This could happen ! #
	    qdd:=diff(p,X,X);
	    yy0:=qdd(z0);
	    discr:=float(sqrt(y0^2-2*yy0*q(z0)));
	    u:=Re(yy0); v:=Im(yy0);
	    z1:=z0+(-y0+discr)*(u-I*v)/(u^2+v^2);
	    x0:=q(z1);
	    if ABS(x0)>inf then 
		z1:=z0+(-y0-discr)*(u-I*v)/(u^2+v^2);
		x0:=q(z1);
	    end_if;
	    if ABS(x0)>inf then return(FAIL) end_if; # this could happen ! #
	end_if;
	z0:=z1; 
	DIGITS:=3; inf:=ABS(Re(x0))+ABS(Im(x0));	
    end_for;
    if i>=40 then return(FAIL) end_if;
    DIGITS:=prec;
    if Im(z0)<>0 and ABS(Im(z0))<10.0^(-4) then
        userinfo(2,"small imaginary part, trying real root");
	if ABS(p(Re(z0)))<10.0^(-precision)*norme
	then return([Re(z0),divide(p,poly(X-Re(z0),[X]),Quo)]);
	end_if;
    end_if;
    q:=divide(p,poly(X-z0,[X]));
    x:=ABS(expr(q[2]));
    if x>10.0^(-precision)*norme then return(FAIL) end_if;
    if ABS(Im(z0))>10.0^(-4) and _fsolve_CoefReal(p,precision,X) then
	y:=Re(z0)-I*Im(z0);
	if ABS(q[1](y))<10.0^(-precision)*norme then
             userinfo(2,"found two conjugate roots");
	    return([z0,y,divide(q[1],poly(X-y,[X]),Quo)])
	end_if;
    end_if;
    [z0,q[1]]
end_proc:

_fsolve_Newton :=proc(p,n,precision,X)
local prec,x,y,u,v,RU,R,q,qd,pp,i,j,k,inf,val,valx,bary,coef,chif,norme,
	Norme,N,x0,y0,z0,x2,DIGITS;
begin
    DIGITS:=3; x:=ABS(coeff(p,n));
    userinfo(1,"deg=",degree(p),"prec=",precision,"x=",x);
    DIGITS:=precision+3; pp:=multcoeffs(p,1/x);
    DIGITS:=3; Norme:=norm(p,1); norme:=max(1,Norme);
    bary:=coeff(pp,n-1)/coeff(pp,n)/n;
    R:=ABS(bary);
    userinfo(2,"R=",R);
    prec:=precision+trunc(n*log10(1+n*R)+2*log10(n))+4
	    +trunc(log10(norme));
    DIGITS:=prec+2;
    bary:=coeff(pp,n-1)/coeff(pp,n)/n;
    q:=poly(pp(X-bary),[X]);
    R:=_fsolve_RootRadius(q,n,0.05,X);
    R:=float(R);
    q:=multcoeffs(poly(q(X*R),[X]),1/R^n);
    qd:=diff(q,X);
    N:=n*4;
    RU:=_fsolve_RacineUnite(N,trunc(log10(norme))+trunc(n/4)+2);
    # coef:=array(0..n); #
    for i from 0 to n do coef[i]:=coeff(q,i) end_for;
    k:=0; inf:=q(1.0); inf:=float(ABS(inf));
    for i from 0 to 2*n-1 do
	x:=_plus(coef[2*j]*RU[(2*i*j) mod N] $ hold(j)=0..trunc(n/2));
	y:=_plus(coef[2*j+1]*RU[(i*(2*j+1)) mod N] $ hold(j)=0..trunc((n-1)/2));
	u:=x+y; v:=x-y;
	DIGITS:=4;
	if ABS(u)<inf then k:=i; inf:=ABS(u) end_if;
	if ABS(v)<inf then k:=i+2*n; inf:=ABS(v) end_if;
	DIGITS:=trunc(log10(norme)+n/4)+2;
    end_for;
    z0:=RU[k];
    DIGITS:=3;
    inf:=min(inf,1.0); if expr2text(inf)="0.0" then inf:=10.0^(-precision) end_if;
    DIGITS:=2*trunc(-log10(inf)+log10(norme)+1);
    x:=z0*R-bary;
    valx:=p(x);
    i:=0;
    val:=float(ABS(valx));
    while (val>10.0^(-precision)*Norme/500) and (i<15 and 
	    inf>10.0^(-2*prec)) do
	i:=i+1;
	inf:=min(inf,1.0);
	DIGITS:=trunc(-4*log10(inf)+2*log10(n)+1+log10(norme));
	x0:=q(z0);
	chif:=DIGITS;
	DIGITS:=3; inf:=ABS(x0); DIGITS:=chif;
	y0:=qd(z0);
	if expr2text(y0)="0.0" then return(FAIL) end_if;
	u:=Re(y0); v:=Im(y0);
	z0:=z0-x0*(u-I*v)/(u^2+v^2);
	x:=z0*R-bary;
      	valx:=pp(x);
	DIGITS:=3;
	val:=ABS(valx);
    end_while;
    if i=15 then return(FAIL) end_if;
# Effectue a nouveau une iteration de Newton sur le polynome pp #
    if expr2text(val)="0.0" then val:=10.0^(-precision) end_if;
    DIGITS:=trunc(-1.15*log10(val)+2*log10(n)+1+log10(norme));
    z0:=p(x);
    y0:=diff(p,X)(x); 
    u:=Re(y0); v:=Im(y0);
    x:=x-z0*(u-I*v)/(u^2+v^2);
    DIGITS:=prec+2;
    if Im(x)<>0 and ABS(Im(x))<10.0^(-4) then
	if ABS(p(Re(x)))<10.0^(-precision)*Norme
	then x:=Re(x)
	end_if;
    end_if;

    if ABS(p(x))>10.0^(-precision)*Norme then return(FAIL) end_if;
    q:=[divide(p,poly(X-x,[X]))];
    if ABS(Im(x))>10.0^(-4) and _fsolve_CoefReal(p,precision,X) then
	y:=Re(x)-I*Im(x);
	if ABS(q[1](y))<10.0^(-precision)*Norme then
	    return([x,y,divide(q[1],poly(X-y,[X]),Quo)])
	end_if;
    end_if;
    [x,q[1]]
end_proc:

_fsolve_Racines:=proc(p,precision,X)
local f,g,q,m,n,liste,root,test,DIGITS;
begin
    DIGITS:=precision+trunc(log10(norm(p,1)))+1;
    n:=degree(p);
    userinfo(2,"degre : ",n);
    if n=1 then return(-coeff(p,0)/coeff(p,1)) end_if; # evalc #
    if n=2 then return(_fsolve_Degre2(p,precision+1,X)) end_if;
    q:=p;
    liste:=null();
    test:=TRUE;
    while test=TRUE do
	    m:=degree(q);
	    if m=2 then return(liste,_fsolve_Degre2(q,precision+1,X)) end_if;
		if m<20 then 
			f:=_fsolve_Newtonbis(q,m,precision,X);
			if f=FAIL then f:=_fsolve_Newton(q,m,precision,X) end_if;
		else 
		    f:=_fsolve_Newton(q,m,precision,X); 
		end_if;
		if f=FAIL then test:=false
		else
			if nops(f)=2 then
                                userinfo(1,"found one root:",op(f,1));
				liste:=liste,op(f,1);
				q:=op(f,2);
				test:=TRUE
			else
                           userinfo(1,"found two roots:",op(f,1),op(f,2));
		    	   liste:=liste,op(f,1),op(f,2);
		    	   q:=op(f,3);
		    	   test:=TRUE
			end_if;
		end_if;

	if degree(q)=1 then 
		return(liste,-coeff(q,0)/coeff(q,1)) # evalc #
	end_if;
	if degree(q)=0 then return(liste) end_if;
	
    end_while;

    userinfo(1,"DEGRE : ",degree(q));
    q:=_fsolve_Translate(q,degree(q),precision,X);
    DIGITS:=precision;
    f:=op(q,1);
    g:=op(q,2);
    if degree(f,X)=1 then 
	root:=-coeff(f,0)/coeff(f,1); # evalc #
	liste:=liste,float(root)
    else
	liste:=liste,_fsolve_Racines(f,precision,X);
   end_if;
    if degree(g,X)=1 then 
	root:=-coeff(g,0)/coeff(g,1); # evalc #
        liste:=liste,float(root);   
    else
	liste:=liste,_fsolve_Racines(g,precision,X);
    end_if;
    liste
end_proc:

# procedure principale qui ne renvoie le resultat que si la precision
- absolue demandee est obtenue. #
_fsolve_Teste :=proc(p,precision,X)
local epsilon,i,liste,result,x,x1,x2,taille1,taille2,taille,DIGITS;
begin
    epsilon:=1; i:=0; 
    while float(epsilon*10.0^precision)>1 do
	liste:=_fsolve_Solve(p,2^i*precision+iquo(degree(p,X),2),X);
 	epsilon:=op(liste,2);
	if epsilon*10.0^precision>1 then 
	    liste:=_fsolve_Raffine(p,op(liste,1),precision,X);
	    epsilon:=op(liste,2);
            userinfo(1,"after Raffine, erreur=",epsilon);
	end_if;
	i:=i+1;
    end_while;
    liste:=op(liste,1);
    result:=null();
    for i from 1 to degree(p,X) do
    x:=op(liste,i);
    x1:=Re(x); x2:=Im(x);
    if x1=0 then taille1:=-1000 
    else
	taille1:=trunc(log10(ABS(x1)));
    end_if;
    if x2=0 then taille2:=-1000 
    else
	taille2:=trunc(log10(ABS(x2)));
    end_if;
    taille:=max(1,taille1,taille2);
    DIGITS:=precision+1+taille;
    x1:=round(x1*10.0^(precision+1))/10.0^(precision+1);
    x2:=round(x2*10^(precision+1))/10.0^(precision+1);
    DIGITS:=max(1,taille1+1+precision);
    x:=float(x1)+I*float(x2);
    result:=result,x;
    end_for;
    [result]
end_proc:

_fsolve_Solve :=proc(p,precision,X)
local liste,r,n,prec,i,epsilon,DIGITS;
begin
    DIGITS:=3;
    n:=degree(p);
    prec:=precision+trunc(n*log10(2)+log10(n))+1;
    prec:=prec+trunc(log10(norm(mapcoeffs(multcoeffs(p,1/lcoeff(p)),float),1)));
    DIGITS:=prec+trunc(2*log10(n));
    liste:=_fsolve_Racines(mapcoeffs(p,float),prec,X);
    r:=poly(_mult(X-op([liste],i) $ hold(i)=1..n),[X]);
    DIGITS:=2*prec;
    epsilon:=norm(mapcoeffs(p-multcoeffs(r,lcoeff(p)),float),1);
    epsilon:=epsilon/ABS(lcoeff(p));
    if epsilon>4^(-n) then return([[liste],1]) end_if;
    DIGITS:=prec;
    epsilon:=_fsolve_Erreur(n,[liste],epsilon);
    userinfo(1,"erreur=",epsilon);
    [[liste],epsilon]
end_proc:

# Essaye de raffiner le calcul des racines a partir de Newton #
_fsolve_Raffine:=proc(p,liste,precision,X)
local pd,pp,x,y,z,result,r,epsilon,n,prec,i,DIGITS;
begin
    n:=degree(p);
    DIGITS:=5;
    result:=null();
    prec:=precision+trunc(n*log10(2)+log10(n))+1;
    prec:=prec+trunc(log10(norm(mapcoeffs(multcoeffs(p,1/lcoeff(p)),float),1)));
    DIGITS:=2*prec; 
    pp:=mapcoeffs(p,float);
    pd:=diff(pp,X);
    for i from 1 to n do
	x:=op(liste,i);
	y:=pd(x); # float(subs(pd,X=x)) #
	z:=pp(x); # float(subs(pp,X=x)) #
	userinfo(2,"root=",x); if expr2text(y)<>"0.0" then x:=x-z/y end_if;
	result:=result,x
    end_for;
    r:=poly(_mult(X-op([result],i) $ hold(i)=1..n),[X]);
    epsilon:=norm(mapcoeffs(pp-multcoeffs(r,lcoeff(p)),float),1);
    epsilon:=epsilon/ABS(lcoeff(p));
    if epsilon>4^(-n) then return([[result],1]) end_if;
    [[result],_fsolve_Erreur(n,[result],epsilon)]
end_proc:

_fsolve_Erreur := proc(n,liste,epsilon)
local d,listerreur,e1,e2,prod,i,j,v,precision,k,m,abser,DIGITS;
begin
    if expr2text(epsilon)="0.0" then return(0.0)
    else
    precision:=trunc(-log10(epsilon))+1 end_if;
    v:=array(1..n);
    d:=array(1..n);
    listerreur:=null();
    for i from 1 to n do
	v[i]:=op(liste,i)
    end_for;
    abser:=0;
    for i from 1 to n do
	DIGITS:=precision;
	e1:=4*max(1,ABS(v[i]))*epsilon^(1/n)/(1-4*epsilon^(1/n));
	for j from 1 to n do 
	    d[j]:=ABS(float(v[j]-v[i]))
	end_for;
	    e2:=e1; e1:=3*e1;
	    for j from 1 to 5 do
               if not ((e1/e2)>1.2 or j<3) then break end_if;
	    e1:=e2; m:=n; prod:=1;
	        for k from 1 to n do
		if i<>k and d[k]>1.25*e1 then 
		    prod:=prod*(d[k]-e1);
		    m:=m-1;
		end_if;
		end_for;
	    e2:=2^(2-1/m)*max(1,ABS(v[i])+e1)^(n/m)*epsilon^(1/m)
		/prod^(1/m);
	    e2:=float(e2);
	    end_for; 
	abser:=max(abser,e2);
    end_for;
    abser
end_proc:

# Calcule par Karatchouba le carre d'un polynome a coefficients entiers #
_fsolve_Square := proc(p,X)
local q,r,c1,c2,c3,dp,dq,i;
begin
    dp:=degree(p,X);
    dq:=trunc(dp/2);
    q:=poly(_plus(coeff(p,i)*X^i $ i=0..dq),[X]);
    r:=poly(_plus(coeff(p,i)*X^(i-dq-1) $ i=dq+1..dp),[X]);
    c1:=q^2; c2:=r^2; c3:=r+q; c3:=c3^2; 
    c1+poly(X^(dq+1),[X])*(c3-c1-c2)+poly(X^(2*dq+2),[X])*c2
end_proc:

# Calcule Graeffe(p)=f  ( f(x^2)=p(x)p(-x) ) en prenant en compte precision
- chiffres pour p #
_fsolve_Graeffe := proc(p,n,precision,X)
local coef,f,g,i,x,prec,DIGITS;
begin
  DIGITS:=3;
  prec:=trunc(log10(norm(p)))+1; # norm(p,infinity) #
  DIGITS:=prec+precision;
  for i from 0 to n do
   x:=coeff(p,i)*10.0^precision;
   coef[i]:=trunc(Re(x))+I*trunc(Im(x));
  end_for;
  f:=poly(_plus(coef[2*i]*X^i $ hold(i)=0..trunc(n/2)),[X]);
  g:=poly(_plus(coef[2*i+1]*X^i $ hold(i)=0..trunc((n-1)/2)),[X]);
  if (n<32) or (DIGITS<80) then
    f:=f^2-poly(X,[X])*g^2;
  else
    f:=_fsolve_Square(f,X)-poly(X,[X])*_fsolve_Square(g,X)
  end_if;
  DIGITS:=2*(prec+precision);
  f:=mapcoeffs(f,float);
  multcoeffs(f,10.0^(-2*precision))
end_proc:

# Renvoie le rayon racine R ( i.e le plus grand des modules des racines )
- du polynome p a un facteur exp(tau) pres #
_fsolve_RootRadius := proc(p,n,tau,X)
local param,q,precision,R,i,M,coef,beta,taux,alpha,j,qq,ro,auxi,DIGITS;
begin
  R:=0;
  param:=1;
  taux:=tau;
# nombre d'iterations de Graeffe #
  coef:=array(0..n);
  DIGITS:=4;
  M:=trunc((ln(ln(4.0*n))+ln(3/taux))/ln2)+1;
  precision:=trunc(n*log10(6/taux)/param+log10(n))+1;
  DIGITS:=precision+trunc(2*log10(n));
  q:=multcoeffs(p,1/ABS(coeff(p,n)));
  for j from 0 to n do coef[j]:=binomial(n,j) end_for;
  userinfo(2,"nb iterations Graeffe=",M);
  for i from 0 to M do
    DIGITS:=4;
    beta:=-100000;
    for j from 1 to n do
	auxi:=float(ABS(coeff(q,n-j)/coef[j]));
        if expr2text(auxi)<>"0.0" then
	   beta:=max(beta,float(ln(auxi)/ln2/j)) end_if;
    end_for;
    alpha:=trunc(beta);
    ro:=2^alpha;  
    DIGITS:=trunc(precision+2*log10(n));
    qq:=multcoeffs(poly(q(X*ro),[X]),1/ro^n);
    DIGITS:=5;
    R:=R+float(alpha/2^i);
    userinfo(3,"R=",R);
    taux:=taux*1.5;
    if taux>2 then taux:=2 end_if;
    if i<M then precision:=trunc(n/param*log10(1/(exp(taux/6)-1))+
				log10(n))+1; 
	    DIGITS:=precision;
       	    q:=_fsolve_Graeffe(qq,n,precision,X);
	    end_if;
  end_for;
  float(2^R)
end_proc:

# Determine k tel que r{k+1}<R<rk a exp(tau) pres #
_fsolve_Separe := proc(p,n,R,tau,X)
local q,i,t,k,x,m,prec,precision,M,DIGITS;
begin
  DIGITS:=4;
  prec:=float(n*5*log10(2)+log10(n+2));
  DIGITS:=trunc(prec+n*(log10(max(1/R,R))+log10(1/tau)))+1;
  q:=poly(p(X*R),[X]);
  DIGITS:=6;
  M:=trunc((ln(ln(4*n))+ln(2/tau))/ln2)+1;
  t:=tau;
  for i from 0 to M-1 do
   m:=norm(q); # norm(q,infinity) #
   DIGITS:=2;
   m:=1/m;
   DIGITS:=4;
   precision:=trunc(prec+n*(log10(exp(t)/t)))+1;
   DIGITS:=precision+2;
   q:=multcoeffs(q,m); # m*q #
   q:=_fsolve_Graeffe(q,n,precision,X);
   DIGITS:=4;
   t:=t*1.5;
  end_for; 
   m:=ABS(coeff(q,0));
   k:=0;
   for i from 1 to n do
    x:=ABS(coeff(q,i));
    if m<x then k:=i; m:=x end_if;
   end_for; 
 k
end_proc:

_fsolve_ten_to_minus_thousand:=10.0^(-1000):

# returns the points on the above part of the convex-hull 
  of the Newton diagram of p around k #
_fsolve_EnvConv := proc(p,n,k,precision,X)
local q,sommet,coeflog,h,i,j,l,pente,norme,DIGITS,st;
begin
	DIGITS:=5;
	for i from 0 to n do
            q[i]:=ABS(coeff(p,i));
	end_for;
	for i from 0 to n do
	    h[i]:=bool(q[i]<_fsolve_ten_to_minus_thousand);
	end_for;
	for i from 0 to n do
	    if h[i] then q[i]:=-1000
	    else q[i]:=text2expr(expr2text(q[i])); q[i]:=ln(q[i])/ln2
            end_if;
	end_for;
	for i from 0 to n do
	    coeflog[i]:=trunc(q[i])
	end_for;
	for i from 0 to n do sommet[i]:=FALSE end_for;
	sommet[0]:=TRUE;
       	i:=0;
	while i<n do
	    pente:=coeflog[i+1]-coeflog[i];
	    h:=i+1;
	    for j from i+1 to n do 
		if pente<(coeflog[j]-coeflog[i])/(j-i) then
		    h:=j; pente:=(coeflog[j]-coeflog[i])/(j-i); 
		end_if;
	    end_for;
	    i:=h;
	    sommet[h]:=TRUE;
	end_while;
	h:=k;
	while not sommet[h] do h:=h+1 end_while;
	l:=k-1;
	while not sommet[l] do l:=l-1 end_while;
       	pente:=round((coeflog[h]-coeflog[l])/(h-l));
	DIGITS:=precision+trunc(2*log10(n))+1;
	i:=2^pente;
	q:=poly(p(X/i),[X]);
	DIGITS:=3;
       	norme:=trunc(log10(norm(q,1)));
	DIGITS:=precision+1;
	q:=multcoeffs(q,10.0^(-norme));
        [q,pente,l,h]
end_proc:

# renvoie le k  ieme plus grand module des racines de p a exp(tau) pres #
_fsolve_RootModule := proc(p,n,k,tau,X)
local q,M,precision,r,i,taux,DIGITS;
begin
	if k=n then return(_fsolve_RootRadius(p,n,tau,X)) end_if;
	if k=1 then
	    DIGITS:=trunc(n*log10(6/tau)+log10(n))+1;
	    q:=poly(_plus(coeff(p,i)*X^(n-i) $ i=0..n),[X]);
	    DIGITS:=5;
	    return(1/_fsolve_RootRadius(q,n,tau,X))
	end_if;
	DIGITS:=3;
	taux:=tau;
	M:=trunc((ln(3/tau)+ln(ln(n*4)))/ln2)+1;
	precision:=trunc((
	    2*log10(2)+log10(6/tau)+log10(2.83*n))*n+log10(n+1))+1;
	DIGITS:=precision;
	q:=p;
	r:=0;
	for i from 0 to M do
	    q:=_fsolve_EnvConv(q,n,k,precision,X);
	    DIGITS:=3;
	    r:=r+op(q,2)/2^i;
	    if i<M then 
		taux:=taux*1.5;
		if taux>2 then taux:=2 end_if;
	        precision:=trunc((
		    2*log10(2)+log10(6/taux)+log10(2.83*n))*n+log10(n+1))+1;
		DIGITS:=precision;
		q:=op(q,1); 
		q:=_fsolve_Graeffe(q,n,precision,X); 
	    end_if;
	end_for;
	DIGITS:=5;
	float(1/2^r)
end_proc:

# Calcule les racines N iemes de l'unite et les place dans RU #
_fsolve_RacineUnite := proc(N,precision)
local RU,x,y,i,DIGITS;
begin
  DIGITS:=precision+trunc(log10(N))+1;
  RU[0]:=1;
  x:=float(cos(2*PI/N)+I*sin(2*PI/N));
  y:=1;
  for i from 1 to N-1 do 
	 y:=expand(y*x);
	 RU[i]:=y
  end_for;
  RU
end_proc:

# Calcule mu=inf {|P(z)|,|z|=1} et l'integrale gamma = int {1/|P(z)|} #
_fsolve_BornInf := proc(p,n,delta,X)
local precision,i,j,N,NN,n0,n1,gamma,mu,mesure,x,y,u,v,RU,coefpair,coefimpair,DIGITS;
begin
# Calcul des parametres #
  DIGITS:=3;
  mu:=norm(p,1);
  N:=trunc(n/(delta*ln(n))/2)+1;
  precision:=trunc(n*(log10(2*exp(delta))-log10(delta)))+1;
  NN:=2*N;
  # RU:=array(0..NN-1); #
  DIGITS:=precision;
  RU:=_fsolve_RacineUnite(NN,precision);  
  gamma:=0; mesure:=0;
# Calcul effectif de mu et gamma #
  n0:=trunc(n/2);
  n1:=trunc((n-1)/2);
  # coefpair:=array(0..n0); #
  # coefimpair:=array(0..n1); #
  for i from 0 to n0 do coefpair[i]:=coeff(p,2*i) end_for;
  for i from 0 to n1 do coefimpair[i]:=coeff(p,2*i+1) end_for;
  for i from 0 to N-1 do 
   DIGITS:=precision;
   x:=_plus(expand(coefpair[j]*RU[(2*j*i)mod NN]) $ j=0..n0);
   y:=_plus(expand(coefimpair[j]*RU[((2*j+1)*i)mod NN]) $ j=0..n1);
      u:=x+y; v:=x-y;
   DIGITS:=3;
   mu:=min(mu,ABS(u),ABS(v));
   gamma:=gamma+1/ABS(u)+1/ABS(v);
   mesure:=mesure+ln(ABS(u))+ln(ABS(v));
   end_for;
 gamma:=gamma/NN;
 mesure:=mesure/NN;
 [mu,gamma,exp(mesure)]
end_proc:

# calcule la norme2 du polynome p #
_fsolve_Norm2 := proc(p,n,X)
local x,i,DIGITS;
begin
    x:=0;
    DIGITS:=5;
    for i from 0 to n do 
	x:=x+ABS(coeff(p,i))^2/binomial(n,i);
    end_for;
    sqrt(x)
end_proc:

# p etant donne tel que ses racines s'eloignent du cercle unite a exp(delta)
- pres avec k de ses racines dans le cercle, renvoie [f,g] , ou f est
- le polynome dont les racines sont celles de p dans le cercle et g tels que
- |p-fg| < epsilon |p| #
_fsolve_Circle := proc(p,n,k,delta,epsilon,X)
local norme,expo,RU,N,NN,NNN,puissance,x,u,v,
            facteur,coefreal,aux,co,dd,der,exposant,indice,nfft,prec,
            fft,coefp,coefd,ip,c,
            pp,f,g,h,d,i,j,jj,kk,dl,init,mu,gamma,mesure,coef,alpha,
            produit,couple,precision,precaux,eps0,eps1,l,ll,DIGITS;
begin
# Calcul des parametres #
    userinfo(1,"n=",n,"k=",k,"delta=",delta,"epsilon=",epsilon);
    DIGITS:=4;
    coefreal:=_fsolve_CoefReal(p,trunc(-log10(epsilon)),X);
    userinfo(2,"CoefReal=",coefreal);
    mu:=_fsolve_BornInf(p,n,delta,X);
    gamma:=op(mu,2);
    mesure:=op(mu,3);
    mu:=op(mu,1);
    DIGITS:=3;
    norme:=float(log10(norm(p,1)));
    expo:=trunc(norme);
    gamma:=gamma*10.0^expo;
    mu:=mu*10.0^(-expo);
    DIGITS:=trunc(-log10(epsilon))+1;
    pp:=multcoeffs(p,10.0^(-expo));
    DIGITS:=3;
    mesure:=mesure*10.0^(-expo);
    produit:=float(2^(n/2)*_fsolve_Norm2(pp,n,X)*sqrt(binomial(n,k)));
    alpha:= min(produit,2^n*mesure);
    precision:=trunc(2*log10(k)+1.5*log10(n-k)
                +3*log10(alpha)+2.5*log10(gamma)-2*log10(mu)+1
                )+1;
    precaux:=trunc(log10(k)+2+log10(alpha)-log10(mu)+log10(n))+1;
    N:=k+trunc((3*ln(alpha)+ln(n)+3*ln(k)+1.5*ln(n-k)
                        +2.5*ln(gamma)-ln(mu))/delta)+1;
    userinfo(2,"N=",N);
    ll:=trunc(ln(n)/ln2); # parametre de FFT partielle #
    l:=2^ll; 
    userinfo(2,"divisions : ",l);
    N:=trunc(N/l/2)+1;
    NN:=l*(2*N+1); # ( l | NN  ) est necessaire pour mettre en oeuvre une 
             fft partielle #
    NNN:=2*N+1;
    if coefreal and l>2 then NNN:=N+1 end_if;
    DIGITS:=precision;
    RU:=_fsolve_RacineUnite(NN,precision);
    for i from 0 to n do co[i]:=coeff(pp,i) end_for;
    for i from 0 to n-1 do der[i]:=coeff(pp,i+1)*(i+1) end_for;
    for i from 1 to k do puissance[i]:=0; aux[i]:=0 end_for;
    for j from 0 to l-1 do
        init:=_fsolve_converttobase2(j+l);
        indice[j]:=_plus(2^(ll-jj)*op(init,jj) $ hold(jj)=1..ll);
    end_for;
    for j from 0 to l-1 do
        # facteur[j]:=float(exp(j*2*I*PI/l)) #
        facteur[j]:=float(cos(2*j*PI/l)+I*sin(2*j*PI/l));
    end_for;
# Calcule une approximation de f et de h en integrant sur le cercle #
    for i from 0 to NNN-1 do
        # Mise en oeuvre de la fft partielle #
# pour pp : #
        for j from 0 to l-1 do
            init:=indice[j];
            fft[j]:=_plus( co[init+kk*l]*RU[i*(init+kk*l) mod NN] $
                        hold(kk)=0..iquo(n-init,l));
        end_for;
        for exposant from 1 to ll do
            dl:=2^(exposant-1);
            for j from 0 to 2^(ll-exposant)-1 do
                for kk from 0 to dl-1 do
                    nfft[2*dl*j+kk]:=
                    fft[2*dl*j+kk]+fft[dl*(2*j+1)+kk];
                    nfft[dl*(2*j+1)+kk]:=
                    facteur[dl*indice[2*dl*j] mod l]
                *(fft[2*dl*j+kk]-fft[dl*(2*j+1)+kk]);
                end_for;
            end_for;
            fft:=map(nfft,expand); # copy(nfft) #
        end_for;
        coefp:=fft; # copy(fft); #
# pour sa derivee : #
        for j from 0 to l-1 do
            init:=indice[j];
            fft[j]:=_plus(der[init+kk*l]*RU[i*(init+kk*l) mod NN] $
                        hold(kk)=0..trunc((n-1-init)/l))
        end_for;
        for exposant from 1 to ll do
            dl:=2^(exposant-1);
            for j from 0 to 2^(ll-exposant)-1 do
                for kk from 0 to dl-1 do
                    nfft[2*dl*j+kk]:=
                fft[2*dl*j+kk]+fft[dl*(2*j+1)+kk];
                    nfft[dl*(2*j+1)+kk]:=
                    facteur[dl*indice[2*dl*j] mod l]
                *(fft[2*dl*j+kk]-fft[dl*(2*j+1)+kk]);
                end_for;
            end_for;
            fft:=map(nfft,expand) # copy(nfft) #
        end_for;
        coefd:=fft; # copy(fft) #
     userinfo(2,"fin fft partielle");
# fin de la fft partielle #
        for j from 0 to l-1 do
            u:=Re(coefp[j]); v:=Im(coefp[j]);
            ip[j]:=(u-I*v)/(u^2+v^2);
            c[j]:=coefd[j]*ip[j]; # expand #
        end_for;
# Calcul des valeurs permettant d'avoir les puissances j eme des racines
   dans le cercle unite #
        # fft:=array(0..l-1,[c[indice[j]] $ j=0..l-1]); #
        for j from 0 to l-1 do fft[j]:=c[indice[j]] end_for;
        for exposant from 1 to ll do
            dl:=2^(exposant-1);
            for j from 0 to 2^(ll-exposant)-1 do
                for kk from 0 to dl-1 do
                    nfft[2*dl*j+kk]:=
                fft[2*dl*j+kk]+fft[dl*(2*j+1)+kk];
                    nfft[dl*(2*j+1)+kk]:=
            facteur[dl*indice[2*dl*j] mod l]
                *(fft[2*dl*j+kk]-fft[dl*(2*j+1)+kk]);
                end_for;
            end_for;
            fft:=map(nfft,expand); # copy(nfft) #
        end_for;
        for j from 2 to k+1 do 
            if (coefreal and l>2) and i>0 then
                puissance[j-1]:=puissance[j-1]+2*Re(
                    fft[j mod l]*RU[(j*i) mod NN])
            else 
                puissance[j-1]:=puissance[j-1]+
                    fft[j mod l]*RU[(j*i) mod NN]
            end_if;
        end_for;
   userinfo(2,"calcul des valeurs");
# Calcul des valeurs permettant d'avoir le polynome auxiliaire h #
        DIGITS:=precaux;
        for j from 0 to l-1 do
            fft[j]:=ip[indice[j]]
        end_for;
        for exposant from 1 to ll do
            dl:=2^(exposant-1);
            for j from 0 to 2^(ll-exposant)-1 do
                for kk from 0 to dl-1 do
                    nfft[2*dl*j+kk]:=
                fft[2*dl*j+kk]+fft[dl*(2*j+1)+kk];
                    nfft[dl*(2*j+1)+kk]:=
            facteur[dl*indice[2*dl*j] mod l]
                *(fft[2*dl*j+kk]-fft[dl*(2*j+1)+kk])
                end_for;
            end_for;
            fft:=map(nfft,expand); # copy(nfft) #
        end_for;
        for j from 1 to k do
            if (coefreal and l>2) and i>0 then
                aux[j]:=aux[j]+2*Re(
                    fft[j mod l]*RU[(j*i) mod NN])
            else
                aux[j]:=aux[j]+fft[j mod l]*RU[(j*i) mod NN]
            end_if;
        end_for;
        DIGITS:=precision;
    end_for;
# Mise en oeuvre des formules de Newton pour avoir f a partir des sommes
   des puissances des racines dans le cercle unite #
    DIGITS:=precision+trunc(2*log10(k))+1;
    for i from 1 to k do
        if coefreal then puissance[i]:=Re(puissance[i]) end_if;
        puissance[i]:=puissance[i]/NN;
        x:=-puissance[i];
        for j from 1 to i-1 do
            x:=x-puissance[j]*coef[i-j]; # expand #
        end_for;
        coef[i]:=x/i;
    end_for;
    coef[0]:=1;
    f:=poly(_plus(coef[j]*X^(k-j) $ hold(j)=0..k),[X]);
    DIGITS:=2*DIGITS;
    couple:=[divide(pp,f)];
    dd:=couple[2];
    g:=couple[1];
# Calcul du polynome auxiliaire h a partir des valeurs int( t^m/pp(t)) #
    DIGITS:=precaux;
    for i from 1 to k do
        if coefreal then aux[i]:=Re(aux[i]) end_if;
        aux[i]:=aux[i]/NN
    end_for;
    h:=0;
    for i from 0 to k-1 do
        x:=0;
        for j from 1+i to k do
            x:=x+coef[k-j]*aux[j-i]
        end_for;
        h:=h+x*X^i;
    end_for;
    h:=poly(h,[X]);
# Mise en oeuvre de la methode de Newton pour affiner le calcul de f #
    DIGITS:=3;
    eps0:=norm(dd,1);
    alpha:=norm(f,1)*norm(g,1)*k;
    j:=0;
    while (eps0>epsilon) and (j<5)  do
    userinfo(2,"eps0=",eps0,"j=",j);
  # Raffinement du calcul de h #
            DIGITS:=trunc(-log10(eps0)+log10(gamma)+log10(alpha))+3;
            d:=poly(1,[X])-divide(h*g,f,Rem); 
            i:=0;
            DIGITS:=5;
            userinfo(2,"d=",d);
            eps1:=norm(d,1);
            userinfo(2,"eps1=",eps1);
            while (eps1>eps0^(9/10)) and (i<6) do
                userinfo(2," > ",eps1);
                precision:=min(trunc(-2*log10(eps1)
		+log10(gamma)+log10(alpha)),
                        trunc(-log10(eps0)+
			log10(gamma)+log10(alpha)))+3;
                DIGITS:=precision-trunc(-log10(eps1));
                d:=h*d;
                d:=divide(d,f,Rem); # op(reste(d,f,DIGITS,X),2); #
                DIGITS:=DIGITS+trunc(-log10(eps1));
                h:=h+d;
                d:=poly(1,[X])-divide(h*g,f,Rem); # -op(reste(h*g,f,DIGITS,X),2); #
                i:=i+1;   
                DIGITS:=5;
                eps1:=norm(d,1);
            end_while;
        DIGITS:=5;
        prec:=trunc(-log10(epsilon)+log10(norm(f,1)+norm(g,1))
	  +log10(gamma)+log10(alpha))+3;
        precision:=min(prec,trunc(-2*log10(eps0))+3);
        DIGITS:=precision;
        d:=divide(h*dd,f,Rem); # op(reste(h*dd,f,DIGITS,X),2); #
        DIGITS:=precision;
        f:=f+d;
        DIGITS:=min(2*DIGITS,prec);
        couple:=divide(pp,f); # reste(pp,f,min(2*DIGITS,prec),X); #
        g:=op(couple,1);
        dd:=op(couple,2);
        DIGITS:=3;
        eps0:=norm(dd,1);
        userinfo(2,"eps0=",eps0);
        j:=j+1;
    end_while;
    DIGITS:=3;
    DIGITS:=trunc(-log10(epsilon)+log10(norm(f,1)*norm(g,1))+
                        2*log10(n))+1;
    g:=multcoeffs(g,10^expo); 
    [f,g]
end_proc:

_fsolve_converttobase2:=proc(n) local l;
begin
   l:=[];
   while n<>0 do
      l:=append(l,n mod 2);
      n:=n div 2;
   end_while;
   l
end_proc:

#Effectue une transformation sur le polynome de sorte que les racines
- s'eloignent encore plus du cercle de separation 
- k designe le nombre de racines dans le cercle de separation, kk designe
- l'entier caracterisant la position du centre du cercle de separation. #
_fsolve_Eloigne := proc(p,n,k,kk,delta,epsilon,X)
local a,b,q,f,g,R,rmin,rmax,Delta,precision,rap,DIGITS;
begin
    a:=-expand(0.577*I^kk); 
    b:=-expand(0.577*(-I)^kk);
    precision:=trunc(-log10(epsilon)+2*n*log10(1.577)+3*log10(n))+5;
    q:=_fsolve_Homographie(p,n,a,b,precision,X);
    rmin:=_fsolve_RootModule(q,n,k,delta/10,X);
    rmax:=_fsolve_RootModule(q,n,k+1,delta/10,X);
    R:=sqrt(rmin*rmax);
    Delta:=floatln(rmax/R,5);
    userinfo(1,"delta = ",Delta);
    q:=poly(q(X*R),[X]);
    DIGITS:=3;
    rap:=norm(p,1)/norm(q,1);
    DIGITS:=trunc(-log10(epsilon)+n*log10(1.577)-n*log10(1-0.577^2)+
			3*log10(n)+n*log10(2))+5;
    q:=_fsolve_Circle(q,n,k,Delta,epsilon*(1-0.577^2)^n/(1.577)^n*rap,X);
    f:=op(q,1);
    g:=op(q,2);
    f:=poly(f(X/R),[X]);
    g:=poly(g(X/R),[X]);
    f:=_fsolve_Homographie(f,k,a,b,DIGITS,X);
    g:=_fsolve_Homographie(g,n-k,a,b,DIGITS,X);
    f:=multcoeffs(f,1/(1-0.577^2)^k);
    g:=multcoeffs(g,1/(1-0.577^2)^(n-k));
    userinfo(1,"ratio of norms=",norm(p-f*g,1)/norm(p,1));
    [f,g]
end_proc:

_fsolve_Homographie := proc(p,n,a,b,precision,X)
local pui,r,i,bx,xa,DIGITS;
begin
    DIGITS:=precision;
    xa:=poly(X-a,[X]);
    bx:=poly(b*X-1,[X]);
    pui:=bx;
    r:=poly(coeff(p,n),[X]);
    for i from n-1 downto 0 do
	r:=r*xa+multcoeffs(pui,coeff(p,i));
	pui:=pui*bx
    end_for;
    r
end_proc:

# renvoie TRUE si p est a coefficient reels, FALSE sinon. #
_fsolve_CoefReal := proc(p,precision,X)
local norme,DIGITS;
begin
   DIGITS:=3; norme:=norm(p,1);
   bool(norm(mapcoeffs(p,Im),1)<10^(-precision)*norme);
end_proc:

# renvoie le polynome apres le changement de repere
- precision designe le DIGITS avec lequel on veut la precision relative /p
- sur la factorisation de p en deux polynomes #
_fsolve_Translate := proc(p,n,precision,X)
local x,u,v,vv,reciproque,recibis,q,qq,f,g,t,rapport,rmax,rmin,rM,rm,
	i,j,k,kk,r,rr,R,prec,norme,delta,expo,coefreal,fin,DIGITS;
begin
 DIGITS:=3;
 expo:=trunc(log10(norm(p,1)));
 coefreal:=_fsolve_CoefReal(p,precision,X);
 DIGITS:=precision+1;
 q:=multcoeffs(p,10^(-expo));
 if ABS(coeff(q,0))<10^(-precision) then 
    return([poly(X,[X]),poly(_plus(coeff(p,i+1)*X^i $ hold(i)=0..n-1),[X])])
 end_if;
 reciproque:=FALSE;
 if _fsolve_Separe(p,n,2,0.1,X)<n then
    if _fsolve_Separe(q,n,0.5,0.1,X)=0 then
	reciproque:=TRUE;
	q:=poly(_plus(coeff(q,j)*X^(n-j) $ hold(j)=0..n),[X]);
    else 
	rmax:=_fsolve_RootRadius(q,n,0.1/n,X);
	rmin:=_fsolve_RootModule(q,n,1,0.1/n,X);
	delta:=_fsolve_FindCircle(q,n,rmin,rmax,X);
	j:=op(delta,1);
	R:=op(delta,2);
	delta:=op(delta,3);
	rmin:=_fsolve_RootModule(q,n,j,delta/10,X);
	rmax:=_fsolve_RootModule(q,n,j+1,delta/10,X);
	R:=sqrt(rmin*rmax);
	delta:=floatln(rmax/R,5);
       	DIGITS:=precision+trunc(log10(norm(q,1))+
				n*log10(max(1,R))+2*log10(n))+1;
	q:=poly(q(R*X),[X]);
	prec:=precision+trunc(
	    log10(norm(q,1))+n*log10(max(1,1/R)))+1;
	if j>n/2 then
	    recibis:=TRUE;
	    q:=poly(_plus(coeff(q,i)*X^(n-i) $ hold(i)=0..n),[X]);
	    j:=n-j
	else recibis:=FALSE
	end_if;
	qq:=_fsolve_Circle(q,n,j,delta,10.0^(-prec),X);
	f:=op(qq,1);
	g:=op(qq,2);  
	DIGITS:=3; norme:=norm(f,1)*norm(g,1);
	DIGITS:=precision+trunc(log10(norme)+n*log10(max(1,1/R))+2*log10(n))+1;
	if recibis then
	    f:=poly(_plus(coeff(f,i)*X^(j-i) $ hold(i)=0..j),[X]);
	    g:=poly(_plus(coeff(g,i)*X^(n-j-i) $ hold(i)=0..n-j),[X]);
	end_if;
	f:=poly(f(X/R),[X]);
	g:=poly(g(X/R),[X]);
	return([f,g])
    end_if
 end_if;
  DIGITS:=precision+trunc(n*log10(4)+1);
 # first operation #
  q:=poly(q(4*X),[X]);
  DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(1.5)+2*log10(n)+1);  
  x:=coeff(q,n);
  u:=Re(x);
  v:=Im(x);
  t:=coeff(q,n-1)*(u-I*v)/(u^2+v^2)/n;
 # second operation #
  q:=poly(q(X-t),[X]);
  DIGITS:=3;
  if ABS(coeff(q,0))<10^(-precision)
  then
	f:=poly(X,[X]);
	DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(1.5)+2*log10(n))+1;
	g:=poly(_plus(coeff(q,j+1)*X^j $ hold(j)=0..n-1),[X]);
	DIGITS:=3; norme:=norm(f,1)*norm(g,1);
	DIGITS:=precision+trunc(log10(norme)+n*log10(1.5)+2*log10(n))+1;
	f:=poly(f(X+t),[X]);
	g:=poly(g(X+t),[X]);
	DIGITS:=3; norme:=norm(f,1)*norm(g,1); 
	DIGITS:=precision+trunc(log10(norme)+2*log10(n))+1;
	f:=poly(f(X/4),[X]);
	g:=poly(g(X/4),[X]);
	if reciproque then
	    f:=poly(_plus(coeff(f,i)*X^(1-i) $ hold(i)=0..1),[X]);
	    g:=poly(_plus(coeff(g,i)*X^(n-1-i) $ hold(i)=0..n-1),[X]);
	end_if;
	return([f,g])
  end_if;
  r:=_fsolve_RootRadius(q,n,0.1/n,X);
DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(2)+log10(n))+1;
 # troisieme operation #
  q:=multcoeffs(poly(q(X*r),[X]),(1/r)^n);
  rr:=float(ABS(t));
  v:=array(0..3);
  rmax:=array(0..3);
  rmin:=array(0..3);
  v[0]:=2;
  v[1]:=2*I;
  v[2]:=-2;
  v[3]:=-2*I;
  rapport:=1;
  vv:=array(0..3);
  vv[0]:=0; vv[1]:=2; vv[2]:=1; vv[3]:=3;
  i:=0; fin:=4; if coefreal then fin:=3 end_if;
  while i<fin do
    DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(3*(1+rr)))+1;
   qq:=poly(q(X+v[vv[i]]),[X]);
    rmin[i]:=_fsolve_RootModule(qq,n,1,0.05,X);
    if 3.0/rmin[i]>rapport then 
	rmax[i]:=_fsolve_RootRadius(qq,n,0.05,X)
    else
	rmax[i]:=3.0
    end_if;
    DIGITS:=5;
    if rapport<rmax[i]/rmin[i] then k:=vv[i]; rapport:=rmax[i]/rmin[i] end_if;
    if rapport>2 then i:=fin end_if;
    if  (coefreal and rapport>1.5 and i=1) then k:=vv[i]; 
	i:=fin end_if;
    if (coefreal and i=1 and rmax[0]/rmin[0]>1.5) then 
	k:=0; i:=fin; rapport:=rmax[0]/rmin[0] end_if;
    i:=i+1;  
  end_while;  
  DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(3*(1+rr)))+1;
  q:=poly(q(X+v[k]),[X]);
  DIGITS:=5;
  userinfo(1,"rapport : ",rapport);
    rM:=_fsolve_RootRadius(q,n,ln(rapport)*0.1/n,X);
    DIGITS:=precision+trunc(log10(norm(q,1))+n*log10(3*(1+rr)))+1;
    qq:=poly(_plus(coeff(q,j)*X^(n-j) $ hold(j)=0..n),[X]);
    rm:=1/_fsolve_RootRadius(qq,n,ln(rapport)*0.1/n,X);
    delta:=_fsolve_FindCircle(q,n,rm,rM,X);
    j:=op(delta,1);
    R:=op(delta,2);
    delta:=op(delta,3);
  DIGITS:=precision+trunc(log10(norm(q,1))+
				n*log10(9*(1+rr))+2*log10(n))+1;
  q:=poly(q(R*X),[X]);
  prec:=precision+trunc(log10(norm(q,1))+n*log10(3*(1+rr)))+1;
  kk:=k;
  if j>n/2 then 
    recibis:=TRUE; 
    q:=poly(_plus(coeff(q,i)*X^(n-i) $ hold(i)=0..n),[X]);
    j:=n-j;
    kk:=-k
  else recibis:=FALSE
  end_if;  
  qq:=_fsolve_Eloigne(q,n,j,kk,delta,10.0^(-prec),X);
  f:=op(qq,1);
  g:=op(qq,2);
  DIGITS:=3; norme:=norm(f,1)*norm(g,1);
  DIGITS:=precision+trunc(log10(norme)+n*log10(3*(1+rr))+2*log10(n))+1;
  if recibis then
    f:=poly(_plus(coeff(f,i)*X^(j-i) $ hold(i)=0..j),[X]);
    g:=poly(_plus(coeff(g,i)*X^(n-j-i) $ hold(i)=0..n-j),[X]);
  end_if;
  f:=poly(f(X/R),[X]);
  g:=poly(g(X/R),[X]);
  DIGITS:=3; norme:=norm(f,1)*norm(g,1);
  DIGITS:=precision+trunc(log10(norme)+n*log10(3*(1+rr))+2*log10(n))+1;
  f:=poly(f(X-v[k]),[X]);
  g:=poly(g(X-v[k]),[X]);
  DIGITS:=3; norme:=norm(f,1)*norm(g,1);
  DIGITS:=precision+trunc(log10(norme)+n*log10(1+rr)+2*log10(n))+1;
  f:=multcoeffs(poly(f(X/r),[X]),r^j);
  g:=multcoeffs(poly(g(X/r),[X]),r^(n-j));
  DIGITS:=3; norme:=norm(f,1)*norm(g,1);
  DIGITS:=precision+trunc(log10(norme)+n*log10(1+rr)+2*log10(n))+1;
  f:=poly(f(X+t),[X]);
  g:=poly(g(X+t),[X]);
  DIGITS:=3; norme:=norm(f,1)*norm(g,1); 
  DIGITS:=precision+trunc(log10(norme)+2*log10(n))+1;
  f:=poly(f(X/4),[X]);
  g:=poly(g(X/4),[X]);
  if reciproque then
	f:=poly(_plus(coeff(f,i)*X^(j-i) $ hold(i)=0..j),[X]);
	g:=poly(_plus(coeff(g,i)*X^(n-j-i) $ hold(i)=0..n-j),[X]);
  end_if;
  x:=float(lcoeff(f));
  g:=multcoeffs(g,x);
  x:=1/x;
  f:=multcoeffs(f,x);
  [f,g]
end_proc:

# finds the radius of the separation circle #
_fsolve_FindCircle := proc(p,n,rmin,rmax,X)
local ii,jj,Inf,Max,poids,Rmin,Rmax,R,delta,l,ll,l0,nn,DIGITS;
begin
    DIGITS:=5;
    Rmin:=rmin;
    Rmax:=rmax;
    R:=sqrt(rmin*rmax);
    nn:=0;
    ll:=n;
    l:=_fsolve_Separe(p,n,R,ln(Rmax/Rmin)/ll,X)-nn;
    if l<n/2 then
	if (n/6<l) and (l<2*n/5+1) then 
	    Rmin:=R; ll:=n-l; nn:=l;
	else
	    Rmax:=R; ll:=l; 
	end_if
    else
	if (n/6 < (n-l)) and ((n-l)<2*n/5) then 
	    Rmax:=R; ll:=l;
	else
	    Rmin:=R; ll:=n-l; nn:=l;
	end_if;
    end_if;
    while ll>0 do 
	DIGITS:=5;
	ii:=nn; jj:=nn+ll; 
        userinfo(2,"ii,jj ",ii,jj);
       	Inf:=min(ii,n-jj); Max:=min(jj,n-ii);
	poids:=1+1.5*(1-float(ln(Inf+2)/ln(Max+2)));
	if ii<n-jj then
	    R:=float(exp( (ln(Rmin)+poids*ln(Rmax))/(1+poids) ));
	else
	    R:=float(exp( (ln(Rmax)+poids*ln(Rmin))/(1+poids) ));
	end_if;
        if ll=n then R:=sqrt(Rmin*Rmax) end_if;
	l:=_fsolve_Separe(p,n,R,ln(Rmax/Rmin)/ll,X)-nn;
	if 2*l < ll then 
	    Rmax:=R;l0:=l;
	end_if;
	if 2*l > ll then 
	    Rmin:=R; l0:=ll-l; nn:=nn+l;
	end_if;
	if 2*l=ll then 
	    if nn+l<n/2 then 
		Rmin:=R; nn:=nn+l;l0:=l;
	    else
		Rmax:=R; l0:=l;
	    end_if;
	end_if;
	ll:=l0;
    end_while;
    delta:=0.5*float(ln(Rmax/Rmin));
    R:=sqrt(Rmax*Rmin);
    [nn,R,delta]
end_proc:
