#  ode::autonomous(eq,y,x,n) solves the autonomous equation 
   eq=0 of order n in y(x),
   using the algorithm described in Zwillinger, section 36 page 153.

   Output: FAIL or a list of solutions for y(x).

   Example: ode::autonomous(diff(y(x),x,x)-diff(y(x),x)=2*y(x)*diff(y(x),x),y,x,2);
#
ode::autonomous := proc(eq,y,x,n) local u,i,solu,soly,s;
begin
   eq:=subs(eq,diff(y(x),x$i)=ode::auto_transform(y,u,i)$i=1..n,y(x)=y);
   if has(eq,x) then FAIL 
   else 
      userinfo(1,"autonomous equation");
      solu:=ode::solve_eq(eq,u,y);
      userinfo(2,"solutions for u(y) are",solu);
      # now each solution corresponds to u that is y'(x) #
      s:=solve(subs(eq,u(y)=0),y);
      if type(s)=DOM_LIST then soly:=s else soly:=[s] end_if;
      for u in solu do
         if u=0 then s:=genident("C") # y'=0 #
         elif u=FAIL then s:=null()
         else
            userinfo(2,"integrating",1/u);
            s:=solve(int(1/u,y)=x+genident("C"),y,1);
            if type(s)<>"solve" then s:=op(s) end_if;
            userinfo(2,"solution is",s);
         end_if;
         soly:=append(soly,s);
      end_for;
      soly
   end_if
end_proc:

# gives the value of diff(y(x),x$n) in terms of u(y)=diff(y(x),x) #
ode::auto_transform:=proc(y,u,n)
begin
   if n=1 then u(y)
   elif n=2 then u(y)*diff(u(y),y)
   elif n=3 then u(y)*diff(u(y),y)^2+u(y)^2*diff(u(y),y$2)
   elif n=4 then u(y)*diff(u(y),y)^3+4*diff(u(y),y$2)*diff(u(y),y)*u(y)^2
      +u(y)^3*diff(u(y),y$3)
   else FAIL
   end_if
end_proc:
