# #
# $Date: 1995/06/23 14:19:48 $ $Author: frankp $ $Revision: 1.24.2.1 $ #
# #
# frankp, 18.08.1994 #

#++
ReIm.mu

	Re -- A function for computing the real part of
              an expression 
	Im -- A function for computing the imaginary part of 
              an expression


	For computing the real part and the imaginary part
	of an expression x, the method "Re" and "Im" of the
	domain of x will be used respectivly.
	If such methods does not exist, then 'Re(x)' and
	'Im(x)' will be returned respectivly.

	The basis types DOM_INT, DOM_RAT, DOM_FLOAT, DOM_COMPLEX
	and DOM_EXPR be an exception. These types are treated by
	the procedures 'Re' and 'Im'.
	For expressions of type DOM_EXPR the procedures 'Re' and
	'Im' looks for a function attribute 'Re' and 'Im' res-
	pectivly, which will be used for x if such exist.
	Otherwise either, if the function environment of x is one
	of the built-in environments "_plus", "_mult" or "_power", 
	x will be treated by the procedure 'Re' and 'Im' or the
	expression 'Re(x)' and 'Im(x)' will be returned respectivly.

	For the following functions, the function attributes 'Re'
	and 'Im' are defined:

	    abs, exp, sin, sinh, cos, cosh, tan, tanh, ln, cot, 
	    coth, csc, csch, sec, sech
++#

#--
	the real part
--#

Re := func_env(
proc(x)
    name Re;
    option remember;
    local t, multSplit, a, ai, z;
begin
    if args(0) <> 1 then error("wrong no of args") end_if;

    if x::Re <> FAIL then return( x::Re(x) ) end_if;

    t := domtype(x);
    case t
    of DOM_INT     do 
    of DOM_RAT     do
    of DOM_FLOAT   do return( x );
    of DOM_COMPLEX do return( op(x,1) );
    of DOM_EXPR    do
	t := op(x,0);
	if domtype(level(t,2)) = DOM_FUNC_ENV then
	    if funcattr(level(t,2),"Re") <> FAIL then
		return( funcattr(level(t,2),"Re")(op(x)) )
	    end_if
	end_if;
    
	multSplit := proc(z) local t,a,b,ci,cr,d;
	    begin
 	    # split product into real and imaginary parts if possible #
		cr := 1; ci := 0;
		for t in op(z) do
       		    a := Re(t);
        	    if has(a,hold(Re)) then return( FAIL ) end_if;
         	    b := Im(t);
                    if has(b,hold(Im)) then return( FAIL ) end_if;
                    d := cr;
                    cr := cr*a - ci*b;
                    ci := d*b + ci*a
		end_for;
		[cr,ci]
	    end_proc:
 
	case t
	of hold(_plus) do
	    return( _plus( Re( op(x,i) ) $ hold(i)=1..nops(x) ) )
	of hold(_mult) do
	    # expressions of sign -1 must be treated separatedly #
	    if contains( {op(x)},-1 ) or contains( {op(x)},-I ) then
		return( -Re(-x) )
	    end_if;
	    if contains( {op(x)},-I ) then
		return( -subs(x,-I=I) )
	    end_if; 
		
	    # find constant terms #
	    t := _mult(op(map([op(x)],fun(
		(if testtype( args(1),Type::RealNum ) then
		     args(1)
		 else
		     1
		 end_if)
	    ))));
            # PZ: added t<>1.0 to fix PR-194, otherwise Re(1.0*PI) loops #
            if t<>1 and t<>1.0 then return( t*Re(normal(x/t)) ) end_if;

	    # split product into real and imaginary part if possible #
	    t := multSplit(x);
	    if t <> FAIL then return( t[1] ) end_if;

	    # test if the expression is a symbolic complex number #
	    if contains( {op(x)},I ) then
		t := subs( x,I=1 );
		if domtype(t) = DOM_EXPR then
		    if op(t,0) = hold(Re) or op(t,0) = hold(Im) then
			return( 0 )
		    end_if
		end_if;
		return( op( subs( x,t=Re(t)+Im(t)*I ),1 ) )
	    end_if;
	    return( procname(x) ) 
	of hold(_power) do
	    if op(x,2) = 1/2 then
		case domtype(op(x,1))
		of DOM_INT      do
		of DOM_RAT      do
		of DOM_FLOAT    do return( x )
		of DOM_COMPLEX  do
		    t := op( op(x,1) );
		    return( sqrt((op(t,1)+sqrt(op(t,1)^2+op(t,2)^2))/2) )
		end_case
	    elif domtype(op(x,2)) = DOM_INT then
		a := Re(op(x,1));
		ai := Im(op(x,1));
		if has(a,{Re,Im}) or has(ai,{Re,Im}) then
		    return( procname(x) )
		end_if;
		t := [a,ai];
		if op(x,2) > 0 then
		    t := multSplit( [(a+I*ai) $ op(x,2)] );
		    if t = FAIL then
			return( procname(x) )
		    else
			return( t[1] )
		    end_if
		end_if;
		if op(x,2) < -1 then
		    t := multSplit( [(a+I*ai) $ -op(x,2)] );
		    if t = FAIL then return( procname(x) ) end_if
		end_if;
		# case op(x,2) = -1 #
		return( t[1]/(t[1]^2+t[2]^2) )
	    elif testtype( op(x,2),NUMERIC ) then
		t := Re( hold(exp)(op(x,2)*ln(op(x,1))) );
		if type(t) <> "Re" then return( t ) end_if
	    end_if
	end_case;
	break
    otherwise
	if testtype( x,Type::Constant ) then
	    t := float(x);
	    if domtype(t) = DOM_FLOAT then return( x ) end_if
	end_if
    end_case;

    procname(x)
end_proc,
NIL,
table( "type"="Re", "print"="Re", "Re"=hold(Re)@id, "Im"=0,
       "info"="Re(x) -- the real part of x" )
):

#--
	the imaginary part
--#

Im := func_env(
proc(x)
    name Im;
    option remember;
    local t, multSplit, a, ai, z;
begin
    if args(0) <> 1 then error("wrong no of args") end_if;

    if x::Im <> FAIL then return( x::Im(x) ) end_if;

    t := domtype(x);
    case t
    of DOM_INT     do 
    of DOM_RAT     do
    of DOM_FLOAT   do return( 0 );
    of DOM_COMPLEX do return( op(x,2) );
    of DOM_EXPR    do
	t := op(x,0);
	if domtype(level(t,2)) = DOM_FUNC_ENV then
	    if funcattr(level(t,2),"Im") <> FAIL then
	        return( funcattr(level(t,2),"Im")(op(x)) )
	    end_if
	end_if;
    
        multSplit := proc(z) local t,a,b,ci,cr,d;
            begin
            # split product into real and imaginary parts if possible #
                cr := 1; ci := 0;
                for t in op(z) do
                    a := Re(t);
                    if has(a,Re) then return( FAIL ) end_if;
                    b := Im(t);
                    if has(b,Im) then return( FAIL ) end_if;
                    d := cr;
                    cr := cr*a - ci*b;
                    ci := d*b + ci*a
                end_for;
                [cr,ci]
            end_proc:

	case t
	of hold(_plus) do
	    return( _plus( Im( op(x,i) ) $ hold(i)=1..nops(x) ) );
        of hold(_mult) do
            # expressions of sign -1 must be treated separatedly #
            if contains( {op(x)},-1 ) then return( -Im(-x) ) end_if;

            # find constant terms #
            t := _mult(op(map([op(x)],fun(
                (if testtype( args(1),Type::RealNum ) then
		    args(1)
		 else
		    1
		 end_if)
            ))));
            # PZ: added t<>1.0 to fix PR-194 #
            if t<>1 and t<>1.0 then return( t*Im(normal(x/t)) ) end_if;

	    # split product into real and imaginary parts if possible #
	    t := multSplit(x);
            if t <> FAIL then return( t[2] ) end_if;

            # test if the expression is a symbolic complex number #
            if contains( {op(x)},I ) then
                t := subs( x,I=1 );
                if domtype(t) = DOM_EXPR then
                    if op(t,0) = hold(Re) or op(t,0) = hold(Im) then
                        return( t )
                    end_if
                end_if;
                return( subs(op(subs( x,t=Re(t)+Im(t)*I ),2),I=1) )
            end_if;
	    return( procname(x) )
        of hold(_power) do
            if op(x,2) = 1/2 then
                case domtype( op(x,1) )
                of DOM_INT     do
                of DOM_RAT     do
                of DOM_FLOAT   do return( 0 )
                of DOM_COMPLEX do
                    t := op( op(x,1) );
                    return( sqrt((-op(t,1)+sqrt(op(t,1)^2+op(t,2)^2))/2) )
                end_case
            elif domtype(op(x,2)) = DOM_INT then
                case domtype(op(x,1))
                of DOM_INT      do
                of DOM_RAT      do
                of DOM_FLOAT    do return( 0 )
                otherwise
                    a := Re(op(x,1));
                    ai := Im(op(x,1));
                    if has(a,{Re,Im}) or has(ai,{Re,Im}) then
                        return( procname(x) )
                    end_if;
                    t := [a,ai];
                    if op(x,2) > 0 then
                        t := multSplit( [(a+I*ai) $ op(x,2)] );
                        if t = FAIL then
			    return( procname(x) )
			else 
			    return( t[2] ) 
			end_if
                    end_if;
                    if op(x,2) < -1 then
                        t := multSplit( [(a+I*ai) $ -op(x,2)] );
                        if t = FAIL then return( procname(x) ) end_if
                    end_if;
                    # case op(x,2) = -1 #
                    return( -t[2]/(t[1]^2+t[2]^2) )
		end_case
            elif testtype( op(x,2),NUMERIC ) then
                # hold is *very* important, otherwise
		  exp simplifies back to op(x,1)^op(x,2)
                  and we get an infinite loop #
		t := Im(hold(exp)(op(x,2)*ln(op(x,1))));
		if type(t) <> "Im" then return( t ) end_if
            end_if
        end_case;
	break
    otherwise
        if testtype( x,Type::Constant ) then
            t := float(x);
            if domtype(t) = DOM_FLOAT then return( 0 ) end_if
        end_if
    end_case;

    procname(x)
end_proc,
NIL,
table( "type"="Im", "print"="Im", "Im"=0, "Re"=id,
       "info"="Im(x) -- the imaginary part of x" )
):

#--
	define the function attributes 'Im' and 'Re'
	       for some built-in functions
--#

abs := funcattr( abs, "Re", hold(abs) ):
abs := funcattr( abs, "Im", 0 ):

#-------#

exp := funcattr( exp, "Re", proc(x) 
        local r, s;
    begin
        r := Re(x);
        if r = x then return( exp(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then 
                return( exp(r)*cos(s) )
            end_if
        end_if;
        hold(Re)(exp(x))
    end_proc
):

exp := funcattr( exp, "Im", proc(x) 
	local r, s;
    begin
        r := Re(x);
	if r = x then return( 0 )
	elif not has(r,{hold(Re),hold(Im)}) then
	    s := Im(x);
	    if not has(s,{hold(Re),hold(Im)}) then
		return( exp(r)*sin(s) )
	    end_if
	end_if;
	hold(Im)(exp(x))
    end_proc
):
		
#-------#

ln := funcattr( ln, "Re", fun(
    (sign(args(1));
    if % = 1 then ln(args(1))
    elif % = -1 then ln(-args(1))
    elif iszero(%) then error("singularity")
    elif domtype(args(1)) = DOM_COMPLEX then
	abs(args(1));
	if domtype(%) = DOM_EXPR then 1/2*ln(%^2) else ln(%) end_if
    else
	ln(abs(args(1)))
    end_if)
) ):

ln := funcattr( ln, "Im", fun(
    (sign(args(1));
    if % = 1 then 0
    elif % = -1 then PI
    elif iszero(%) then error("singularity")
    elif domtype(args(1)) = DOM_COMPLEX then
	args(1);
	if op(%,1) = 0 then
	    return( 1/2*PI*sign(op(%,2)) )
	else
	    return( atan(op(%,1),op(%,2)) )
	end_if
    elif not has(%,hold(sign)) then
	atan(Re(%),Im(%))
    else
	hold(Im)(ln(args(1)))
    end_if)
) ):

#-------#

sin := funcattr( sin, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( sin(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sin(r)*cosh(s) )
            end_if
        end_if;
        hold(Re)(sin(x))
    end_proc
):

sin := funcattr( sin, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( cos(r)*sinh(s) )
            end_if
        end_if;
        hold(Im)(sin(x))
    end_proc
):

#-------#

cos := funcattr( cos, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( cos(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( cos(r)*cosh(s) )
            end_if
        end_if;
        hold(Re)(cos(x))
    end_proc
):

cos := funcattr( cos, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( -sin(r)*sinh(s) )
            end_if
        end_if;
        hold(Im)(cos(x))
    end_proc
):

#-------#

tan := funcattr( tan, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( tan(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sin(2*r) / (cos(2*r) + cosh(2*s)) )
            end_if
        end_if;
        hold(Re)(tan(x))
    end_proc
):

tan := funcattr( tan, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sinh(2*s) / (cos(2*r) + cosh(2*s)) )
            end_if
        end_if;
        hold(Im)(tan(x))
    end_proc
):

#-------#

sinh := funcattr( sinh, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( sinh(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sinh(r)*cos(s) )
            end_if
        end_if;
        hold(Re)(sinh(x))
    end_proc
):

sinh := funcattr( sinh, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( cosh(r)*sin(s) )
            end_if
        end_if;
        hold(Im)(sinh(x))
    end_proc
):

#-------#

cosh := funcattr( cosh, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( cosh(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( cosh(r)*cos(s) )
            end_if
        end_if;
        hold(Re)(cosh(x))
    end_proc
):

cosh := funcattr( cosh, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sinh(r)*sin(s) )
            end_if
        end_if;
        hold(Im)(cosh(x))
    end_proc
):

#-------#

tanh := funcattr( tanh, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( tanh(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sinh(r)*cosh(r)/(sinh(r)^2+cos(s)^2) )
            end_if
        end_if;
        hold(Re)(tanh(x))
    end_proc
):

tanh := funcattr( tanh, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sin(s)*cos(s)/(sinh(r)^2+cos(s)^2) )
            end_if
        end_if;
        hold(Im)(tanh(x))
    end_proc
):

#-------#

cot := funcattr( cot, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( cot(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
		return( sin(r)*cos(r) / (sin(r)^2+sinh(s)^2) )
            end_if
        end_if;
        hold(Re)(cot(x))
    end_proc
):

cot := funcattr( cot, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( -sinh(s)*cosh(s) / (sin(r)^2+sinh(s)^2) )
            end_if
        end_if;
        hold(Im)(cot(x))
    end_proc
):

#-------#

coth := funcattr( coth, "Re", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( coth(x) )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( sinh(r)*cosh(r) / (sinh(r)^2+sin(s)^2) )
            end_if
        end_if;
        hold(Re)(coth(x))
    end_proc
):

coth := funcattr( coth, "Im", proc(x)
        local r, s;
    begin
        r := Re(x);
        if r = x then return( 0 )
        elif not has(r,{hold(Re),hold(Im)}) then
            s := Im(x);
            if not has(s,{hold(Re),hold(Im)}) then
                return( -sin(s)*cos(s) / (sinh(r)^2+sin(s)^2) )
            end_if
        end_if;
        hold(Im)(coth(x))
    end_proc
):

#-------#

# sec is evaluated to 1/cos #
# sech is evaluated to 1/cosh #
# csc is evaluated to 1/sin #
# csch is evaluated to 1/sinh #

# end of file #
