#===========================================================================

contfrac -- continued fractions arithmetic

contfrac(e <,p>)	computes a continued fraction for e (with precision p)

CF::approx(cf)		gives an interval that contains cf

CF::unapprox(a,b)	gives a continued fraction for the interval [a,b]

CF::linear(x,a,b,c,d)	computes (a*x+b)/(c*x+d) where x is a continued fraction

a + b			adds two continued fractions

a * b			multiplies two continued fractions

1/a			inverts a continued fraction

a^n			computed the (integer) power of a continued fraction

Convention: "..." represents a real between 1 and infinity, thus
            for example the continued fraction of 0 is 1/..., that
            of an integer n is n+1/...

Examples:

>> a := contfrac(PI,5);

                                  1               
                        ---------------------- + 3
                                1                 
                        ------------------ + 7    
                              1                   
                        ------------- + 15        
                            1                     
                        --------- + 1             
                         1                        
                        --- + 292                 
                        ...                       

>> b := contfrac(sqrt(5),4);

                                   1             
                          ------------------- + 2
                                 1               
                          --------------- + 4    
                               1                 
                          ----------- + 4        
                             1                   
                          ------- + 4            
                           1                     
                          --- + 4                
                          ...                    

>> CF::approx(a);

                        103993/33102, 104348/33215


>> CF::unapprox(%); 

                                  1            
                           ---------------- + 3
                                1              
                           ------------ + 7    
                              1                
                           ------- + 15        
                            1                  
                           --- + 1             
                           ...                 

(we lost some precision, which is normal, because ... does not
 have a precise value)

Now we get a CF for the gold-ratio (1+sqrt(5))2 from that of sqrt(5),
via the fractional linear form L(x) = (x+1)/2:

>> CF::linear(b,1,1,0,2);

                                   1                           
            ----------------------------------------------- + 1
                                 1                             
            ------------------------------------------- + 1    
                               1                               
            --------------------------------------- + 1        
                             1                                 
            ----------------------------------- + 1            
                           1                                   
            ------------------------------- + 1                
                         1                                     
            --------------------------- + 1                    
                       1                                       
            ----------------------- + 1                        
                     1                                         
            ------------------- + 1                            
                   1                                           
            --------------- + 1                                
                 1                                             
            ----------- + 1                                    
               1                                               
            ------- + 1                                        
             1                                                 
            --- + 1                                            
            ...                                                

>> a+b; CF for PI+sqrt(5)

                                   1               
                        ----------------------- + 5
                                 1                 
                        ------------------- + 2    
                               1                   
                        --------------- + 1        
                             1                     
                        ----------- + 1            
                           1                       
                        ------- + 1                
                         1                         
                        --- + 5                    
                        ...                        

>> a*b; CF for PI*sqrt(5)

                                  1          
                             ------------ + 7
                                1            
                             ------- + 40    
                              1              
                             --- + 3         
                             ...             

>> 1/a; CF for 1/PI

                                    1             
                        --------------------------
                                  1               
                        ---------------------- + 3
                                1                 
                        ------------------ + 7    
                              1                   
                        ------------- + 15        
                            1                     
                        --------- + 1             
                         1                        
                        --- + 292                 
                        ...                       

>> a^3; CF for PI^3

                                  1           
                            ------------- + 31
                               1              
                            ------- + 159     
                             1                
                            --- + 3           
                            ...               

==========================================================================#

contfrac := proc(e) local l,n,pn,qn,pnold,qnold,hi,tmp,DIGITS;
begin
   if args(0)>=2 then DIGITS:=args(2) end_if;
   if not contains({DOM_FLOAT,DOM_RAT,DOM_INT},type(e)) then 
      e:=float(e);
      if type(e)<>DOM_FLOAT then return(CF::function(args())) end_if
   end_if;
   pnold:=1; qnold:=0;
   n:=trunc(e); 
   pn:=n; qn:=1; 
   l:=[n];
   hi := float(10^(DIGITS/2));
   while qn<hi do
      if e=n then break end_if; # for rationals #
      e:=1/(e-n);
      n:=trunc(e);
      l:=append(l,n);
      tmp:=n*pn+pnold; pnold:=pn; pn:=tmp;
      tmp:=n*qn+qnold; qnold:=qn; qn:=tmp;
   end_while;
   new(CF,l)
end_proc:

CF:=domain():

CF::name:="CF":

# continued fraction of a function, for example e = sin(z) #
CF::function := proc(e) local z,ord;
begin
   z:=indets(e);
   if nops(z)=1 then z:=op(z) else error("too many variables") end_if;
   if args(0)>=2 then ord:=args(2) else ord:=null() end_if;
   e:=series(e,z,ord);
   Puiseux::contfrac(e)
end_proc:

CF::approx:=proc(cf) # rational approximation #
local l,r,s,i;
begin
   l:=extop(cf,1);
   if l=[] then 0
   else
     r:=op(l,nops(l));
     s:=op(l,nops(l))+1;
     for i from nops(l)-1 downto 1 do 
        r:=op(l,i)+1/r;
        s:=op(l,i)+1/s;
     end_for;
     min(r,s),max(r,s)
   end_if
end_proc:

CF::print:=proc(cf) 
local l,res,i;
begin
   l:=extop(cf,1);
   res:="...";
   for i from nops(l) downto 1 do res:=op(l,i)+1/res end_for;
   res
end_proc:

CF::linear:=proc(x,a,b,c,d) # compute (ax+b)/(cx+d) [Zippel, page 34] #
local l,q,t,m,extract;
begin
   extract:=proc() local p,t; begin 
      while c<>0 do
         p:=trunc((a+b)/(c+d));
         if p<>trunc(a/c) then break end_if;
         m:=append(m,p);
         t:=a; a:=c; c:=t-p*c;
         t:=b; b:=d; d:=t-p*d;
      end_while
   end_proc;
   l:=extop(x,1); m:=[]; # for result #
   for q in l do # first try to extract some partial quotients#
      extract();
      t:=a; a:=a*q+b; b:=t;
      t:=c; c:=c*q+d; d:=t;
   end_for;
   extract();
   new(CF,m)
 end_proc:

# from Zippel, page 38 #
# computes the linear form (a*x*y+b*x+c*y+d)/(e*x*y+f*x+g*y+h) #
CF::linear2 := proc(x,y,a,b,c,d,e,f,g,h)
local lx,ly,m,extract,tmp,maxi;
begin
   maxi:=proc(a,b) begin 
      if b=0 then if a=0 then -1 else FAIL end_if else trunc(a/b) end_if
   end_proc;
   extract:=proc() local t,hi; begin
      while TRUE do
         # the minimum is #
         t:=trunc((a+b+c+d)/(e+f+g+h));
         hi:=maxi(a,e);
         if hi=FAIL then break end_if;
         # if a=e=0 then the form is (b*x+c*y+d)/(f*x+g*y+h) #
         # and the maximum is either b/f or c/g #
         hi:=max(maxi(b,f),maxi(c,g));
         if has(hi,FAIL) then break end_if;  
         if t<>hi then break end_if;
         m:=append(m,t);
         tmp:=a; a:=e; e:=tmp-t*e;
         tmp:=b; b:=f; f:=tmp-t*f;
         tmp:=c; c:=g; g:=tmp-t*g;
         tmp:=d; d:=h; h:=tmp-d*h;
      end_while
   end_proc;
   lx:=extop(x,1);
   ly:=extop(y,1);
   m:=[];
   for r in lx do
      # first try to extract: the linear form being monotonic, #
      # the extremes are obtained for x=y=1 and for x=y=infinity #
      # that is (a+b+c+d)/(e+f+g+h) and a/e #
      extract();
      # then absorb a partial quotient from x #
      tmp:=a; a:=r*a+c; c:=tmp;
      tmp:=b; b:=r*b+d; d:=tmp;
      tmp:=e; e:=r*e+g; g:=tmp;
      tmp:=f; f:=r*f+h; h:=tmp;
   end_for;
   for s in ly do
      extract();
      tmp:=a; a:=s*a+b; b:=tmp;
      tmp:=c; c:=s*c+d; d:=tmp;
      tmp:=e; e:=s*e+f; f:=tmp;
      tmp:=g; g:=s*g+h; h:=tmp;
   end_for;
   extract();
   new(CF,m)
end_proc:

CF::unapprox:=proc(a,b) # find the CF x \in [a,b] (we can have b<a) #
local m,t;
begin
   m:=[];
   repeat
      if (t:=trunc(a))<>trunc(b) then break end_if;
      m:=append(m,t);
      a:=1/(a-t); b:=1/(b-t);
   until FALSE end_repeat;
   new(CF,m,1)
end_proc:

CF::_plus:=proc(x,y) 
local l;
begin 
   # CF::linear2(x,y,0,1,1,0,0,0,0,1) #
   l:=zip([CF::approx(x)],[CF::approx(y)],_plus);
   CF::unapprox(op(l))
end_proc:

CF::_mult:=proc(x,y) 
local l,m;
begin 
   l:=zip([CF::approx(x)],[CF::approx(y)],_mult);
   CF::unapprox(op(l))
end_proc:

CF::invert:=proc(x) local l;
begin
   l:=extop(x,1);
   if op(l,1)=0 then l:=subsop(l,1=null()) else l:=[0,op(l)] end_if;
   # extsubsop(cf,1=l) # # does not work #
   new(CF,l)
end_proc:

CF::_power:=proc(x,n) local y;
begin
   if type(n)<>DOM_INT then FAIL
   elif n<0 then 1/x^(-n)
   elif n=0 then contfrac(1)
   else
      y:=x;
      while n>1 do y:=y*x; n:=n-1 end_while;
      y
   end_if
end_proc:
