# $Date: 1995/07/20 08:17:09 $ $Author: kg $ $Revision: 1.8.2.7 $ #

#++
infinity -- representing the real positive infinity

An infinite number is represented as an element of the domain
stdlib::Infinity. An element of this domain has two operands:

- The first operand is the sign, which is an non-zero expression.

- The second operand is a boolean value, TRUE indicates that
  sign/infinity is represented, otherwise sign*infinity is
  represented (where 'infinity' is the positive real infinity).

- Given the sign s the elements represent limes(s*a, a = infinity)
  or limes(s*a, a = 1/infinity) resp.

The domain element 'infinity' (which represents the positive real
infinity) is defined here. Other domain elements are only created
via the operations of this domain and NOT via 'new'.
++#

proc()
    local Inf;
begin

Inf:= domain():
Inf::name:= "stdlib::Infinity":

infinity:= new(Inf, 1, FALSE):

Inf::info:= "Domain 'stdlib::Infinity': representing infinity":
Inf::interface:= {}:

Inf::expr:= fun((
    if extop(args(1),2) then
        extop(args(1),1) / hold(infinity)
    else
        hold(infinity) * extop(args(1),1)
    end_if)):

Inf::print:= Inf::expr:

Inf::evaluate:= fun((
    extop(args(1),1);
    eval(%);
    if iszero(%) then % else extsubsop(args(1), 1 = %) end_if)):

Inf::TeX:= fun((
    if extop(args(1),2) then
	case extop(args(1),1)
	of 1 do "1 / \\infty"; break;
	of -1 do "-1 / \\infty"; break;
	otherwise TeX(extop(args(1),1))." / \\infty";
	end_case
    else
	case extop(args(1),1)
	of 1 do "\\infty"; break;
	of -1 do "- \\infty"; break;
	otherwise TeX(extop(args(1),1))." \\infty";
	end_case
    end_if)):

Inf::negate:= fun(extsubsop(args(1), 1 = -extop(args(1),1))):

Inf::invert:= fun(extsubsop(args(1), 
    1 = 1/extop(args(1),1),
    2 = not extop(args(1),2))):

Inf::not:= fun(error("invalid sign")):

Inf::_power:= fun((
    if domtype(args(1)) <> stdlib::Infinity then
	return(hold(_power)(args()))
    end_if;
    case domtype(args(2))
    of DOM_FLOAT do
        if iszero(args(2)) then return(1.0) end_if;
    of DOM_INT do
        if args(2) = 0 then return(1) end_if;
    of DOM_RAT do
	if args(2) < 0 then
	    (stdlib::Infinity)::_power(
	    		(stdlib::Infinity)::invert(args(1)), -args(2));
	    break;
	end_if;
	extsubsop(args(1), 1 = extop(args(1),1)^args(2));
	break;
    otherwise
	hold(_power)(args())
    end_case)):

Inf::_plus:= proc() local a, b, c; begin
    # select infinities and others #
    c:= select({args()}, (stdlib::Infinity)::thisDomain);
    b:= select(c, not (stdlib::Infinity)::isInverted);
    a:= select([args()], not (stdlib::Infinity)::thisDomain);
    if b = {} then
        b:= select(c, (stdlib::Infinity)::isInverted);
	# add all non-infinities #
	a:= _plus(op(a));
    else
	# add non-infinities which have infinity subexpressions #
	a:= _plus(op(select(a, has, infinity)));
    end_if;
    if nops(select(map(b, extop, 1), testtype, NUMERIC)) > 1 then
	error("undefined") 
    end_if;
    
    if a = 0 then
	if nops(b) = 1 then
	    op(b)
	else
	    hold(_plus)(op(b))
	end_if
    elif type(a) = "_plus" then  # flat operands of a #
        hold(_plus)(op(a), op(b))
    else
        hold(_plus)(a, op(b))
    end_if
end_proc:

Inf::_mult:= proc() local a, b, c, f; begin
    b:= [args()];
    # multiply in b the infinities and in c and a the other arguments #
    a:= select(b, not (stdlib::Infinity)::thisDomain);
    c:= select(a, not has, infinity);
    if nops(c) = nops(a) then
	a:= 1
    else
	a:= _mult(op(select(a, has, infinity)))
    end_if;
    # select infinities #
    b:= select(b, (stdlib::Infinity)::thisDomain);
    # infinity/infinity is undefined #
    if { op(map(b, extop, 2)) } = { TRUE, FALSE } then
        error("undefined")
    end_if;
    f:= b[1];
    b:= expand(_mult(op(c), op(map(b, extop, 1))));
    if iszero(b) then return(b) end_if;
    case type(b)
    of "_plus" do
        return(_plus(map(op(b), _mult, extsubsop(f, 1 = 1))) * a);
    of "_mult" do
        c:= op(b, nops(b));
        if iszero(c) then return(c) end_if;
        if testtype(c, Type::RealNum) then
            b:= subsop(b, nops(b) = (if c > 0 then 1 else -1 end_if))
        end_if;
        break;
    of DOM_INT do
    of DOM_RAT do
    of DOM_FLOAT do
        b:= (if b > 0 then 1 else -1 end_if);
    end_case;
    
    b:= extsubsop(f, 1 = b);

    if a = 1 then
	b
    elif type(a) = "_mult" then  # flat operands of a #
        hold(_mult)(op(a), b)
    else
        hold(_mult)(a, b)
    end_if
end_proc:

Inf::_less:= proc(x, y) begin
    if domtype(x) = stdlib::Infinity then
        if domtype(extop(x,1)) <> DOM_INT then
            return(hold(_less)(x, y))
        end_if;
        if domtype(y) = stdlib::Infinity then
	    if x = y then return(FALSE) end_if;
	    if domtype(extop(y,1)) <> DOM_INT then
		return(hold(_less)(x, y))
	    end_if;
	    case x
	    of -infinity do return(TRUE);
	    of -1/infinity do return(not bool(y = -infinity));
	    of 1/infinity do return(bool(y = infinity));
	    of infinity do return(FALSE);
	    end_case
	else
	    if not testtype(y, Type::RealNum) then
		return(hold(_less)(x, y))
	    end_if;
	    case x
	    of -infinity do return(TRUE);
	    of -1/infinity do return(bool(y >= 0));
	    of 1/infinity do return(bool(y > 0));
	    of infinity do return(FALSE);
	    end_case
	end_if
    elif domtype(y) = stdlib::Infinity then
        if domtype(extop(y,1)) <> DOM_INT then
            return(hold(_less)(x, y))
        end_if;
        if not testtype(x, Type::RealNum) then
            return(hold(_less)(x, y))
        end_if;
        not (stdlib::Infinity)::_less(y, x)
    else
        hold(_less)(x, y)
    end_if
end_proc:

Inf::_leequal:= proc(x,y)
    local c;
begin
    if x = y then
        TRUE 
    else
        c:= (stdlib::Infinity)::_less(x, y);
        if domtype(c) = DOM_BOOL then
            c
        else
	    hold(_leequal)(x, y)
        end_if
    end_if
end_proc:

Inf::min:= proc() local a, aa, b, c, f; begin
    b:= { args() };
    # collect in b infinities, in c real numbers and in a the other arguments #
    a:= select(b, not (stdlib::Infinity)::thisDomain);
    f:= fun(testtype(args(1), Type::RealNum));
    aa := select(a, f);
    if nops(aa) > 0 then
  	c := min(op(aa));	
    else
	c := infinity;
    end_if;
    a:= select(a, not f);

    # collect in b infinities #
    b:= select(b,(stdlib::Infinity)::thisDomain);
    f:= fun(contains({-1,1}, extop(args(1),1)));
    a:= a union select(b, not f);
    b:= select(b, f);

    if b = {} then
    	if c = infinity then
    	    b:= null()
    	else
    	    b:= c
    	end_if
    else
	# get minimal real infinity #
	if contains(b, -infinity) then b:= -infinity
	elif contains(b, -1/infinity) then b:= -1/infinity
	elif contains(b, 1/infinity) then b:= 1/infinity
	else b:= infinity
	end_if;
	if (stdlib::Infinity)::_less(c, b) then
	    b:= c
	end_if
    end_if;

    if a = {} then
        b
    else
        hold(min)(op(a), b)
    end_if
end_proc:

Inf::max:= proc() local a, aa, b, c, f; begin
    b:= { args() };
    # collect in b infinities, in c real numbers and in a the other arguments #
    a:= select(b, not (stdlib::Infinity)::thisDomain);
    f:= fun(testtype(args(1), Type::RealNum));
    aa := select(a, f);
    if nops(aa) > 0 then
  	c := max(op(aa));	
    else
	c := -infinity;
    end_if;
    a:= select(a, not f);

    # collect in b infinities #
    b:= select(b,(stdlib::Infinity)::thisDomain);
    f:= fun(contains({-1,1}, extop(args(1),1)));
    a:= a union select(b, not f);
    b:= select(b, f);
    
    if b = {} then
    	if c = -infinity then
    	    b:= null()
    	else
    	    b:= c
    	end_if
    else
	# get maximal real infinity #
	if contains(b, infinity) then b:= infinity
	elif contains(b, 1/infinity) then b:= 1/infinity
	elif contains(b, -1/infinity) then b:= -1/infinity
	else b:= -infinity
	end_if;
	if (stdlib::Infinity)::_less(b, c) then
	    b:= c
	end_if
    end_if;

    if a = {} then
        b
    else
        hold(max)(op(a), b)
    end_if
end_proc:

Inf::intmult:= fun((
    if iszero(args(2)) then 0
    elif args(2) > 0 then args(1)
    else extsubsop(args(1), 1 = -extop(args(1),1))
    end_if
)):

Inf::thisDomain:= fun(bool(domtype(args(1)) = stdlib::Infinity)):

Inf::isInverted:= fun(extop(args(1),2)):

Inf::simplify:= fun(
    extsubsop(args(1), 1 = simplify(extop(args(1),1)))
):

Inf::sign:= fun(sign(extop(args(1),1))):

Inf::subs:= fun((
    if domtype(op(args(2),1)) = stdlib::Infinity then
	eval(subs(subs(expr(args(1)), expr(op(args(2),1)) = op(args(2),2)),
		  hold(infinity) = infinity))
    else
	eval(subs(subs(expr(args(1)), args(2)),
		  hold(infinity) = infinity))
    end_if
)):

Inf::subsop:= fun(
    eval(subs(subsop(expr(args(1)), args(2)=args(3)),
	      hold(infinity) = infinity))
):

Inf::op:= fun(
    eval(subs(op(expr(args(1)), (if args(0) = 2 then args(2) else null() end_if)),
	      hold(infinity) = infinity))
):

stdlib::Infinity:= Inf:

end_proc():

# end of file #
