/*
   MATRIX.I
   Yorick interface to LINPACK matrix solving routines.

   $Id: matrix.i,v 1.1 1993/08/07 19:57:08 munro Exp $
 */
/*    Copyright (c) 1994.  The Regents of the University of California.
                    All rights reserved.  */

/* ------------------------------------------------------------------------ */

func unit(n, m)
/* DOCUMENT unit(n)
         or unit(n, m)
     returns n-by-n (or n-by-m) unit matrix, i.e.- matrix with diagonal
     elements all 1.0, off diagonal elements 0.0
 */
{
  if (is_void(m)) m= n;
  u= array(0.0, n, m);
  u(1:numberof(u):n+1)= 1.0;
  return u;
}

/* ------------------------------------------------------------------------ */

func TDsolve(c, d, e, b, which=)
/* DOCUMENT TDsolve(c, d, e, b)
         or TDsolve(c, d, e, b, which=which)

     returns the solution to the tridiagonal system:
        D(1)*x(1)       + E(1)*x(2)                       = B(1)
	C(1:-1)*x(1:-2) + D(2:-1)*x(2:-1) + E(2:0)*x(3:0) = B(2:-1)
	                  C(0)*x(-1)      + D(0)*x(0)     = B(0)
     (i.e.- C is the subdiagonal, D the diagonal, and E the superdiagonal;
     C and E have one fewer element than D, which is the same length as
     both B and x)

     B may have additional dimensions, in which case the returned x
     will have the same additional dimensions.  The WHICH dimension of B,
     and of the returned x is the one of length n which participates
     in the matrix solve.  By default, WHICH=1, so that the equations
     being solved involve B(,..) and x(+,..).
     Non-positive WHICH counts from the final dimension (as for the
     sort and transpose functions), so that WHICH=0 involves B(..,)
     and x(..,+).

     The C, D, and E arguments may be either scalars or vectors; they
     will be broadcast as appropriate.

  SEE ALSO: LUsolve, QRsolve, SVsolve, SVdec
 */
{
  /* check validity of b argument */
  if (structof(b)==complex) error, "expecting a non-complex RHS vector";
  dims= dimsof(b);
  ndb= is_void(dims)? 0 : dims(1);
  if (is_void(which)) which= 1;
  else if (which<=0) which+= ndb;
  if (!ndb) error, "RHS must have at least one dimension";
  n= dims(1+which);
  b= double(b);   /* copy of RHS to be transformed into solution */
  nrhs= numberof(b)/n;

  /* put first matrix dimension of b first */
  if (which!=1) b= transpose(b, [1,which]);

  /* copy, force to double, and broadcast matrix diagonals
     -- also will blow up on conformability error */
  cc= ee= array(0.0, n-1);
  dd= array(0.0, n);
  cc()= c;
  dd()= d;
  ee()= e;

  info= 0;
  _dgtsv, n, nrhs, cc, dd, ee, b, n, info;
  if (info) error, "tridiagonal element "+pr1(info)+" of became 0.0";

  /* restore proper order of result if necessary */
  if (which!=1) b= transpose(b, [1,which]);

  return b;
}

extern _dgtsv;
/* PROTOTYPE FORTRAN
   void dgtsv(long array n, long array nrhs, double array c, double array d,
              double array e, double array b, long array ldb, long array info)
 */
/* DOCUMENT _dgtsv
     LAPACK dgtsv routine.
 */

/* ------------------------------------------------------------------------ */

func LUsolve(a, b, which=, job=, det=)
/* DOCUMENT LUsolve(a, b)
         or LUsolve(a, b, which=which)
	 or a_inverse= LUsolve(a)

     returns the solution x of the matrix equation:
        A(,+)*x(+) = B
     If A is an n-by-n matrix then B must have length n, and the returned
     x will also have length n.

     B may have additional dimensions, in which case the returned x
     will have the same additional dimensions.  The WHICH dimension of B,
     and of the returned x is the one of length n which participates
     in the matrix solve.  By default, WHICH=1, so that the equations
     being solved are:
        A(,+)*x(+,..) = B
     Non-positive WHICH counts from the final dimension (as for the
     sort and transpose functions), so that WHICH=0 solves:
        x(..,+)*A(,+) = B

     If the B argument is omitted, the inverse of A is returned:
     A(,+)*x(+,) and A(,+)*x(,+) will be unit matrices.

     LUsolve works by LU decomposition using Gaussian elimination with
     pivoting.  It is the fastest way to solve square matrices.  QRsolve
     handles non-square matrices, as does SVsolve.  SVsolve is slowest,
     but can deal with highly singular matrices sensibly.

   SEE ALSO: QRsolve, TDsolve, SVsolve, SVdec, LUrcond
 */
{
  /* get n, m, dims, nrhs, checking validity of a and b */
  { local dims, n, m, nrhs; }
  _get_matrix, 1;
  if (m!=n) error, "expecting a square matrix";

  if (is_void(b)) {
    b= unit(n);
    nrhs= n;
    which= 1;
  }

  /* perform LU solve */
  pivot= array(0, n);
  info= 0;
  _dgesv, n, nrhs, a, n, pivot, b, n, info;
  /* row i interchanged with row pivot(i) --> permutation matrix P
     a now contains the L and U factors from the decomposition;
     original a= P*L*U */
  if (info) error, "matrix is (numerically) singular";

  /* restore proper order of result if necessary */
  if (which!=1) b= transpose(b, [1,which]);

  return b;
}

func LUrcond(a, one_norm=)
/* DOCUMENT LUrcond(a)
         or LUrcond(a, one_norm=1)
     returns the reciprocal condition number of the N-by-N matrix A.
     If the ONE_NORM argument is non-nil and non-zero, the 1-norm
     condition number is returned, otherwise the infinity-norm condition
     number is returned.

     The condition number is the ratio of the largest to the smallest
     singular value, max(singular_values)*max(1/singular_values) (or
     sum(abs(singular_values)*sum(abs(1/singular_values)) if ONE_NORM
     is selected?).  If the reciprocal condition number is near zero
     then A is numerically singular; specifically, if
          1.0+LUrcond(a) == 1.0
     then A is numerically singular.

   SEE ALSO: LUsolve
 */
{
  dims= dimsof(a);
  if (is_void(dims) || dims(1)!=2 || dims(2)!=dims(3) ||
      structof(a)==complex)
    error, "expecting a square 2D real matrix";
  n= dims(2);
  a= double(a);
  pivot= array(0, n);
  info= 0;
  _dgetrf, n, n, a, n, pivot, info;
  work= array(double, 4*n);
  iwork= array(0, n);
  rcond= 0.0;
  if (!one_norm) {
    one_norm= 0;
    anorm= abs(a)(max,sum);
  } else {
    one_norm= 1;
    anorm= abs(a)(sum,max);
  }
  _dgecox, one_norm, n, a, n, anorm, rcond, work, iwork, info;
  return rcond;
}

extern _dgesv;
/* PROTOTYPE FORTRAN
   void dgesv(long array n, long array nrhs, double array a, long array lda,
              long array pivot, double array b, long array ldb,
	      long array info)
 */
/* DOCUMENT _dgesv
     LAPACK dgesv routine.
 */

extern _dgetrf;
/* PROTOTYPE FORTRAN
   void dgetrf(long array m, long array n, double array a, long array lda,
               long array pivot, long array info)
 */
/* DOCUMENT _dgetrf
     LAPACK dgetrf routine.  Performs LU factorization.
 */

extern _dgecox;
/* PROTOTYPE FORTRAN
   void dgecox(long array norm, long array n, double array a, long array lda,
               double array anorm, double array rcond, double array work,
               long array iwork, long array info)
 */
/* DOCUMENT _dgecox
     LAPACK dgecon routine, except norm argument not a string.
 */

/* ------------------------------------------------------------------------ */

func QRsolve(a, b, which=)
/* DOCUMENT QRsolve(a, b)
         or QRsolve(a, b, which=which)

     Removed.  Use the C version in the MathC directory of the distribution.

   SEE ALSO: LUsolve, TDsolve, SVsolve, SVdec
 */
{
  error, "QRsolve removed from this Fortran version";
}

/* ------------------------------------------------------------------------ */

func SVsolve(a, b, rcond, which=)
/* DOCUMENT SVsolve(a, b)
         or SVsolve(a, b, rcond)
         or SVsolve(a, b, rcond, which=which)

     Removed.  Use the C version in the MathC directory of the distribution.

   SEE ALSO: SVdec, LUsolve, QRsolve, TDsolve
 */
{
  error, "SVsolve removed from this Fortran version";
}

func SVdec(a, &u, &vt, full=)
/* DOCUMENT s= SVdec(a, u, vt)
         or s= SVdec(a, u, vt, full=1)

     Removed.  Use the C version in the MathC directory of the distribution.

   SEE ALSO: SVsolve, LUsolve, QRsolve, TDsolve
 */
{
  error, "SVdec removed from this Fortran version";
}

/* ------------------------------------------------------------------------ */

func _get_matrix(b_optional)
{
  { extern dims, n, m, nrhs; }

  /* check validity of a argument */
  dims= dimsof(a);
  if (dims(1)!=2 || structof(a)==complex)
    error, "expecting a non-complex 2D matrix";
  a= double(a);  /* copy a to avoid clobbering, as well as force type */
  m= dims(2);
  n= dims(3);

  /* check validity of b argument */
  if (!b_optional || !is_void(b)) {
    dims= dimsof(b);
    ndb= is_void(dims)? 0 : dims(1);
    if (is_void(which)) which= 1;
    else if (which<=0) which+= ndb;
    if (!ndb || dims(1+which)!=m)
      error, "RHS dimensions not conformable with matrix dimensions";
    if (structof(b)==complex) error, "expecting a non-complex RHS vector";
    b= double(b);  /* copy to avoid clobbering, and force type */
    nrhs= numberof(b)/m;

    /* put first matrix dimension of b first */
    if (which!=1) b= transpose(b, [1,which]);

    /* be sure that the first dimension of b is at least n */
    if (n>m) {
      dims= dimsof(b);
      dims(2)= n;
      bn= array(0.0, dims);
      bn(1:m,..)= b;
      b= bn;
    }

  } else {
    nrhs= 0;
  }
}

/* ------------------------------------------------------------------------ */
